////////////////////////////////////////////////////////////////////////////////
//
// Utility for inspecting land use registers.
// ==========================================
//
// Copyright (C) Janicke Consulting, 88662 Ueberlingen, Germany, 2011-2018
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation; either version 2 of
// the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
//
// History:
//
// 2011-06-01 uj  1.0.0  extended version based on the project A2Kro
// 2018-11-22 uj  1.1.0  adjusted to TA Luft (2018) draft
//
////////////////////////////////////////////////////////////////////////////////

package de.janicke.gisviewluc;

import de.janicke.ibjutil.IBJarr;
import de.janicke.ibjutil.IBJarr.IntegerArray;
import de.janicke.ibjutil.IBJdmn;
import de.janicke.ibjutil.IBJhdr;
import de.janicke.ibjutil.IBJarr.StringArray;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Cursor;
import java.awt.Dimension;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.Point;
import java.awt.Rectangle;
import java.awt.RenderingHints;
import java.awt.Robot;
import java.awt.Toolkit;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.KeyEvent;
import java.awt.event.KeyListener;
import java.awt.event.MouseEvent;
import java.awt.event.MouseListener;
import java.awt.event.MouseMotionListener;
import java.awt.image.BufferedImage;
import java.io.BufferedReader;
import java.io.File;
import java.io.FileReader;
import java.net.URI;
import java.net.URL;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.Locale;
import java.util.Map;
import java.util.zip.CRC32;

import javax.imageio.ImageIO;
import javax.swing.ImageIcon;
import javax.swing.JButton;
import javax.swing.JComboBox;
import javax.swing.JFileChooser;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTextField;
import javax.swing.SwingUtilities;
import javax.swing.filechooser.FileFilter;

public class GisViewLuc {

  /** 
   * Program version.
   */
  public String Version = "1.1.0";
  
  public final static String TOOLTIP_DE =
    "<html>Inhalt des Textfeldes bei Ansicht eines Landnutzungs-Katasters:<br>" +
    "x-Koordinate (m), y-Koordinate (m), CORINE-Klasse, Beschreibung.<br<<br>" + 
    "Inhalt des Textfeldes bei Ansicht eines Rauigkeitsklassen-Katasters:<br>" +
    "x-Koordinate (m), y-Koordinate (m), Z0-Klasse, Z0 (m), zugeordnete CORINE-Klassen.<br><br>" +
    "Editieren des Textfeldes:<br>" +
    "Angabe von x-Koordinate und y-Koordinate (m), durch ein Leerzeichen getrennt,<br>" +
    "und Drücken der RETURN-Taste bewegt den Mauszeiger an diese Position.<br><br></html>";
    
  public final static String TOOLTIP_EN =
    "<html>Contents of the text field viewing a land use register:<br>" +
    "x-coordinate (m), y-coordinate (m), CORINE class, description.<br<<br>" + 
    "Contents of the text field viewing a roughness class register:<br>" +
    "x-coordinate (m), y-coordinate (m), Z0 class, Z0 (m), attributed CORINE classes.<br><br>" +
    "Editing the text field:<br>" +
    "Specification of x-coordinate and y-coordinate (m), separated by a blank,<br>" +
    "and bressing the RETURN key moves the mouse cursor to that position.<br><br></html>";
  
  /**
   * Roughness length values according to TA Luft
   */
  public static final double[] Z0_CLASSES = 
  { -999, 0.01, 0.02, 0.05, 0.1, 0.2, 0.5, 1., 1.5, 2. };
  
  /**
   * Assignment of CORNIE classes to Z0 classes according to TA Luft 2002 Annex 3.
   */
  public static final String[] LUC_2_Z0C_DEPRECATED = { 
    /* 1 */ "331 512",
    /* 2 */ "132 231 321 333 421 423 511 522",
    /* 3 */ "131 142 211 335 521",
    /* 4 */ "124 411 412 523",
    /* 5 */ "122 141 221 242 243 322 332",
    /* 6 */ "123 222 324",
    /* 7 */ "112 121 133 312",
    /* 8 */ "311 313",
    /* 9 */ "111" 
  };
  
  /**
   * Assignment of CORNIE classes to Z0 classes according to TA Luft 2018.
   */
  public static final String[] LUC_2_Z0C_2018 = { 
    /* 1 */ "331 512",
    /* 2 */ "333 421 423 511 522",
    /* 3 */ "131 132 142 335 521",
    /* 4 */ "124 211 231 334 411 412 523",
    /* 5 */ "122 141 221 321 322 332",
    /* 6 */ "123 222 324",
    /* 7 */ "112 121 133",
    /* 8 */ "312 313",
    /* 9 */ "111 311" 
  };
  
  /**
   * Classes (English)
   */
  public static final String LUC_EN =
    "902 Z0 class reduced by 8\n" +                        /* for diff maps   */
    "903 Z0 class reduced by 7\n" +                        /* ...             */
    "904 Z0 class reduced by 6\n" +
    "905 Z0 class reduced by 5\n" +
    "906 Z0 class reduced by 4\n" +
    "907 Z0 class reduced by 3\n" +
    "908 Z0 class reduced by 2\n" +
    "909 Z0 class reduced by 1\n" +
    "910 Z0 class unchanged\n" +
    "911 Z0 class increased by 1\n" +                        
    "912 Z0 class increased by 2\n" +                       
    "913 Z0 class increased by 3\n" +
    "914 Z0 class increased by 4\n" +
    "915 Z0 class increased by 5\n" +
    "916 Z0 class increased by 6\n" +
    "917 Z0 class increased by 7\n" +
    "918 Z0 class increased by 8\n" +
    "000 Areas outside the definition domain (0.05 m)\n" + /* TA Luft 2018 classes */
    "001 0.01 m (331 512)\n" +                            /* ...             */
    "002 0.02 m (333 421 423 511 522)\n" +
    "003 0.05 m (131 132 142 335 521)\n" +
    "004  0.1 m (124 211 231 334 411 412 523)\n" +
    "005  0.2 m (122 141 221 321 322 332)\n" +
    "006  0.5 m (123 222 324)\n" +
    "007  1.0 m (112 121 133)\n" +
    "008  1.5 m (312 313)\n" +
    "009  2.0 m (111 311)\n" +
    "111 Continuous urban fabric\n" +                      /* CORINE classes  */
    "112 Discontinuous urban fabric\n" +                   /* ...             */
    "121 Industrial or commercial units\n" +
    "122 Road and rail networks and associated land\n" +
    "123 Port areas\n" +
    "124 Airports\n" +
    "131 Mineral extraction sites\n" +
    "132 Dump sites\n" +
    "133 Construction sites\n" +
    "141 Green urban areas\n" +
    "142 Sport and leisure facilities\n" +
    "211 Non-irrigated arable land\n" +
    "212 Permanently irrigated land\n" +
    "213 Rice fields\n" +
    "221 Vineyards\n" +
    "222 Fruit trees and berry plantations\n" +
    "223 Olive groves\n" +
    "231 Pastures\n" +
    "241 Annual crops associated with permanent crops\n" +
    "242 Complex cultivation patterns\n" +
    "243 Land occupied by agriculture, significant areas of natural vegetation\n" +
    "244 Agro-forestry areas\n" +
    "311 Broad-leaved forest\n" +
    "312 Coniferous forest\n" +
    "313 Mixed forest\n" +
    "321 Natural grasslands\n" +
    "322 Moors and heathland\n" +
    "323 Sclerophyllous vegetation\n" +
    "324 Transitional woodland-shrub\n" +
    "331 Beaches, dunes, sands\n" +
    "332 Bare rocks\n" +
    "333 Sparsely vegetated areas\n" +
    "334 Burnt areas\n" +
    "335 Glaciers and perpetual snow\n" +
    "411 Inland marshes\n" +
    "412 Peat bogs\n" +
    "421 Salt marshes\n" +
    "422 Salines\n" +
    "423 Intertidal flats\n" +
    "511 Water courses\n" +
    "512 Water bodies\n" +
    "521 Coastal lagoons\n" +
    "522 Estuaries\n" +
    "523 Sea and ocean\n" +
    "999 Areas outside the definition domain\n";

  /**
   * Classes (German)
   */
  public static final String LUC_DE =
    "902 Z0-Klasse erniedrigt um 8\n" +                        /* for diff maps   */
    "903 Z0-Klasse erniedrigt um 7\n" +                        /* ...             */
    "904 Z0-Klasse erniedrigt um 6\n" +
    "905 Z0-Klasse erniedrigt um 5\n" +
    "906 Z0-Klasse erniedrigt um 4\n" +
    "907 Z0-Klasse erniedrigt um 3\n" +
    "908 Z0-Klasse erniedrigt um 2\n" +
    "909 Z0-Klasse erniedrigt um 1\n" +
    "910 Z0-Klasse unverädert\n" +
    "911 Z0-Klasse erhöht um 1\n" +                        
    "912 Z0-Klasse erhöht um 2\n" +                       
    "913 Z0-Klasse erhöht um 3\n" +
    "914 Z0-Klasse erhöht um 4\n" +
    "915 Z0-Klasse erhöht um 5\n" +
    "916 Z0-Klasse erhöht um 6\n" +
    "917 Z0-Klasse erhöht um 7\n" +
    "918 Z0-Klasse erhöht um 8\n" +
    "000 Flächen außerhalb des Bearbeitungsgebietes (z0=0.05 m)\n" + /* TA Luft classes */
    "001 0.01 m (331 512)\n" +                            /* ...             */
    "002 0.02 m (333 421 423 511 522)\n" +
    "003 0.05 m (131 132 142 335 521)\n" +
    "004  0.1 m (124 211 231 334 411 412 523)\n" +
    "005  0.2 m (122 141 221 321 322 332)\n" +
    "006  0.5 m (123 222 324)\n" +
    "007  1.0 m (112 121 133)\n" +
    "008  1.5 m (312 313)\n" +
    "009  2.0 m (111 311)\n" +
    "111 Flächen durchgängig städtischer Prägung\n" +          /* CORINIE classes */
    "112 Flächen nicht-durchgängig städtischer Prägung\n" +    /* ...             */
    "121 Industrie- und Gewerbeflächen\n" +
    "122 Straßen und Eisenbahn\n" +
    "123 Hafengebiete\n" +
    "124 Flughäfen\n" +
    "131 Abbauflächen\n" +
    "132 Deponien und Abraumhalden\n" +
    "133 Baustellen\n" +
    "141 Städtische Grünflächen\n" +
    "142 Sport und Freizeitanlagen\n" +
    "211 Nicht bewässertes Ackerland\n" +
    "212 Permanent bewässertes Ackerland\n" +
    "213 Reisfelder\n" +
    "221 Weinbauflächen\n" +
    "222 Obst- und Beerenobstbestände\n" +
    "223 Olivenhaine\n" +
    "231 Wiesen und Weiden\n" +
    "241 Mischung einjähriger Früchte mit Dauerkulturen\n" +
    "242 Komplexe Parzellenstrukturen\n" +
    "243 Landwirtschaft mit natürlicher Bodenbedeckung\n" +
    "244 Agroforstlich genutzte Flächen\n" +
    "311 Laubwald\n" +
    "312 Nadelwald\n" +
    "313 Mischwald\n" +
    "321 Natürliches Grasland\n" +
    "322 Heiden und Moorheiden\n" +
    "323 Hartlaubgewächse\n" +
    "324 Wald-Strauch-Übergangsstadien\n" +
    "331 Strände, Dünen und Sandflächen\n" +
    "332 Felsflächen ohne Vegetation\n" +
    "333 Flächen mit spärlicher Vegetation\n" +
    "334 Brandflächen\n" +
    "335 Gletscher und Dauerschneegebiete\n" +
    "411 Sümpfe\n" +
    "412 Torfmoore\n" +
    "421 Salzwiesen\n" +
    "422 Salinen\n" +
    "423 In der Gezeitenzone liegende Flächen\n" +
    "511 Gewässerläufe\n" +
    "512 Wasserflächen\n" +
    "521 Lagunen\n" +
    "522 Mündungsgebiet\n" +
    "523 Meere und Ozeane\n" +
    "999 Flächen außerhalb des Bearbeitungsgebietes\n";
  
  private static String sFile = "z0-utm@278000-5226000-100.png";
  private static boolean bExpert = false;
  private static final String[] LOCALES = { "en", "de" };
  private double X0, Y0, Dd, Xorg, Yorg;
  private double X0org, Y0org, Ddorg;
  private boolean zoomed;
  private int Nx, Ny, no_data;
  private int Nxorg, Nyorg;
  private int Nxz=41, Nyz=41;
  private HashMap<Integer, Color> mapColor;
  private HashMap<Integer, String> mapLuc;
  private Color clrUnknown;
  private BufferedImage bim, bimz, currentbim;
  private ImageIcon ico;
  private JFrame frame;
  private JPanel panel, pnl, pnlButtons;
  private JLabel lClassesUnknown;
  private JScrollPane scp;
  private JTextField txt;
  private int nClassesUnknown;
  private JFileChooser chooser;
  private JButton btnSave;
  private MouseMotionListener mml;
  private MouseListener ml;
  private File dirRoot;
  private File dirInst;
  private JComboBox cbLocale;
  private String sFilter;
  private String sClassesUnknown;
  private long lCrc;
  
  private class LUC implements Comparable {
    public String id;
    public String dsc;
    public char b;
    public int clr;

    @Override
    public int compareTo(Object o) {
      return id.compareTo(((LUC)o).id);
    }
    
    
  }

 
  public GisViewLuc() {
    setParams();
    setDirs();   
  }

  private void setParams() {
    clrUnknown = new Color(255, 200, 255);
    mapColor = new HashMap<Integer, Color>();
    //
    // Colour coding of the various classes.
    // CORINE classes with official CORINE RGB colours.
    mapColor.put(111, new Color(211,   0,   0));
    mapColor.put(112, new Color(239,  42,  71));
    mapColor.put(121, new Color(211,   0, 155));
    mapColor.put(122, new Color(126, 126, 126));
    mapColor.put(123, new Color(155, 155, 211));
    mapColor.put(124, new Color( 99,  99,  99));
    mapColor.put(131, new Color(155,  71,  42));
    mapColor.put(132, new Color(155,  13,  42));
    mapColor.put(133, new Color(183,  99, 126));
    mapColor.put(141, new Color( 99, 239,   0));
    mapColor.put(142, new Color(239,  71,   0));
    mapColor.put(211, new Color(239, 239, 126));
    mapColor.put(212, new Color(  0,   0,   1));
    mapColor.put(213, new Color(  0,   0,   2));
    mapColor.put(221, new Color(239, 155,  13));
    mapColor.put(222, new Color(239, 211, 155));
    mapColor.put(223, new Color(  0,   0,   3));
    mapColor.put(231, new Color(155, 211,  13));
    mapColor.put(241, new Color(  0,   0,   4));
    mapColor.put(242, new Color(239, 211,  99));
    mapColor.put(243, new Color(183, 183,  71));
    mapColor.put(244, new Color(  0,   0,   5));
    mapColor.put(311, new Color(  0, 183,   0));
    mapColor.put(312, new Color(  0, 126,  99));
    mapColor.put(313, new Color(  0, 126,   0));
    mapColor.put(321, new Color(155, 183,  99));
    mapColor.put(322, new Color(211, 211,   0));
    mapColor.put(323, new Color(  0,   0,   6));
    mapColor.put(324, new Color(183, 239,   0));
    mapColor.put(331, new Color(239, 239, 183));
    mapColor.put(332, new Color(211, 211, 155));
    mapColor.put(333, new Color(126, 211, 155));
    mapColor.put(334, new Color(  0,   0,   7));
    mapColor.put(335, new Color(  0,   0,   8));
    mapColor.put(411, new Color(183,  71, 239));
    mapColor.put(412, new Color(126,  71, 211));
    mapColor.put(421, new Color(183, 126, 211));
    mapColor.put(422, new Color(  0,   0,   9));
    mapColor.put(423, new Color(239, 183, 239));
    mapColor.put(511, new Color(  0,  99, 239));
    mapColor.put(512, new Color(  0, 155, 239));
    mapColor.put(521, new Color(  0, 211, 239));
    mapColor.put(522, new Color( 99, 211, 239));
    mapColor.put(523, new Color(159, 215, 248));
    mapColor.put(999, new Color(255, 255, 255));
    mapColor.put(   0, new Color(250, 250, 250));
    mapColor.put(   1, mapColor.get(512).brighter());
    mapColor.put(   2, mapColor.get(124).brighter());
    mapColor.put(   3, mapColor.get(231).brighter());
    mapColor.put(   4, mapColor.get(211).brighter());
    mapColor.put(   5, mapColor.get(243).brighter());
    mapColor.put(   6, mapColor.get(324).brighter());
    mapColor.put(   7, mapColor.get(112).brighter());
    mapColor.put(   8, mapColor.get(313).brighter());
    mapColor.put(   9, mapColor.get(111).brighter());     
    mapColor.put( 902, new Color(242, 242, 200));
    mapColor.put( 903, mapColor.get(132).brighter());
    mapColor.put( 904, mapColor.get(131).brighter());
    mapColor.put( 905, mapColor.get(411).brighter());
    mapColor.put( 906, mapColor.get(122).brighter());
    mapColor.put( 907, mapColor.get(123).brighter());
    mapColor.put( 908, mapColor.get(121).brighter());
    mapColor.put( 909, mapColor.get(311).brighter());
    mapColor.put( 910, new Color(213, 213, 213));       
    mapColor.put( 911, mapColor.get(331).darker());
    mapColor.put( 912, mapColor.get(132).darker());
    mapColor.put( 913, mapColor.get(131).darker());
    mapColor.put( 914, mapColor.get(411).darker());
    mapColor.put( 915, mapColor.get(122).darker());
    mapColor.put( 916, mapColor.get(123).darker());
    mapColor.put( 917, mapColor.get(121).darker());
    mapColor.put( 918, mapColor.get(311).darker());
    String[] ss = LUC_DE.split("\n");
    mapLuc = new HashMap<Integer, String>();
    try {
      for (String s: ss) {
        int luc = Integer.parseInt(s.substring(0, 3));
        Color c = mapColor.get(luc);
        if (c == null)
          throw new Exception("no colour defined for class "+luc);
        if (mapLuc.get(c.getRGB()) != null)
          throw new Exception("Colour assignments not unique ("+luc+")!");
        mapLuc.put(c.getRGB(), s);        
      }
    }
    catch (Exception e) {
      e.printStackTrace(System.out);
      System.exit(1);
    }
    //
    mml = new MouseMotionListener() {
      @Override
      public void mouseDragged(MouseEvent e) {
      }
      @Override
      public void mouseMoved(MouseEvent e) {
        mouseAt(e.getX(), e.getY());
      }
    };
    //
    ml = new MouseListener() {
      @Override
      public void mouseClicked(MouseEvent e) {}
      @Override
      public void mouseEntered(MouseEvent e) {}
      @Override
      public void mouseExited(MouseEvent e) {}
 
      @Override
      public void mouseReleased(MouseEvent e) {}

      @Override
      public void mousePressed(MouseEvent e) {
        if (SwingUtilities.isLeftMouseButton(e)) {
          zoom(e.getX(), e.getY());
        }
      }
    };
    bExpert = false;
    lCrc = -1;
    nClassesUnknown = 0;
  }
  
  private void setDirs() {
    Class<?> cls = getClass();
    String name = cls.getName();
    URL u2 = cls.getResource(cls.getSimpleName() + ".class");
    String s2 = null;
    try {
      URI uri = u2.toURI();
      String sScheme = uri.getScheme();
      s2 = uri.getSchemeSpecificPart();
      if (sScheme.equals("jar")) {
        if (s2.startsWith("file:")) {
          s2 = s2.substring(5);
        }
      } 
      else if (!sScheme.equals("file")) {
        return;
      }
    } catch (Exception e1) {
      e1.printStackTrace(System.out);
      System.exit(1);
    }
    int k = s2.indexOf(name.replace('.', '/'));
    if (k > 0) {
      s2 = s2.substring(0, k - 1);
    }
    if (s2.endsWith(".jar!")) { //$NON-NLS-1$
      s2 = s2.substring(0, s2.length() - 1);
      dirRoot = (new File(s2)).getParentFile();
    } 
    else {
      dirRoot = new File(s2);
    }
    dirInst = dirRoot.getParentFile();
    chooser = new JFileChooser(dirInst);
  }
  
  /**
   * Reads a CORINE land use register from a GRIDASCII file.
   * @param fn The file name.
   * @throws Exception
   */
  public void readLUC(String fn) throws Exception {
    File f = new File(fn);
    System.out.printf("reading %s...\n", fn);
    chooser.setCurrentDirectory(f.getParentFile());
    FileReader fr = new FileReader(f);
    BufferedReader br;
    br = new BufferedReader(fr);
    String line = null;
    Map<String, Integer> hdr = new HashMap<String, Integer>();
    for (int i=0; i<6; i++) {
      line = br.readLine();
      String[] ss = line.split("[ ]+");
      hdr.put(ss[0], Integer.parseInt(ss[1]));
    }
    Nx = hdr.get("ncols");
    Ny = hdr.get("nrows");
    no_data = hdr.get("NODATA_value");
    X0 = hdr.get("xllcorner");
    Y0 = hdr.get("yllcorner");
    Dd = hdr.get("cellsize");
    //
    bim = new BufferedImage(Nx, Ny, BufferedImage.TYPE_INT_ARGB);
    for (int j=0; j<Ny; j++) {
      line = br.readLine();
      String[] ss = line.trim().split("[ ]+");
      for (int i=0; i<Nx; i++) {
        int luc = Integer.parseInt(ss[i]);
        if (luc == no_data || luc < 0)
          luc = 999;
        Color c = mapColor.get(luc);
        if (c == null) {
          c = clrUnknown;
          System.out.printf("*** unknown class "+luc+"\n");
          nClassesUnknown++;
        }
        bim.setRGB(i, j, c.getRGB());
      }
    }
    br.close();
    System.out.printf("...done.\n");
  }
  
  /**
   * Reads a CORINE land use register from a PNG file. Each pixel correspnds to
   * one grid cell. The RGB value must correspond to a pre-defined values that
   * either matches a CORINE land use class or a roughness length class of
   * AUSTAL. The lower left corner and the mesh width of the register are 
   * provided in the file name. 
   * @param fn The file name.
   * @throws Exception
   */
  public void readPNG(String fn) throws Exception {
    File f = new File(fn);
    chooser.setCurrentDirectory(f.getParentFile());
    String name = f.getName();
    String[] ss = name.split("[@]");
    if (ss.length != 2)
      throw new Exception("invalid file name format!");
    ss = ss[1].split("[\\-.]");
    if (ss.length != 4)
      throw new Exception("invalid file name format!");
    X0 = IBJhdr.parseDoubleOf(ss[0]);
    Y0 = IBJhdr.parseDoubleOf(ss[1]);
    Dd = IBJhdr.parseDoubleOf(ss[2]);
    ico = new ImageIcon(fn);
    Nx = ico.getIconWidth();
    Ny = ico.getIconHeight();
    bim = new BufferedImage(Nx, Ny, BufferedImage.TYPE_INT_ARGB);
    Graphics g = bim.getGraphics();
    g.drawImage(ico.getImage(), 0, 0, null);   
  }
  
  /**
   * Reads a roughness length register from a DMN file as applied by AUSTAL.
   * @param fn The file name.
   * @throws Exception
   */
  public void readZ0C(String fn) throws Exception {
    File f = new File(fn);
    chooser.setCurrentDirectory(f.getParentFile());
    IBJarr arr = IBJdmn.readDmn(fn);
    X0 = arr.getHeader().getDouble("xmin");
    Y0 = arr.getHeader().getDouble("ymin");
    Dd = arr.getHeader().getDouble("delta");
    StringArray sa = (StringArray)arr.getArray("Classes");
    int dim = sa.getStructure().getDims();
    if (dim != 2)
      throw new Exception("invalid dimension (must be 2)");
    int i1 = sa.getStructure().getFirstIndex()[0];
    int i2 = sa.getStructure().getLastIndex()[0];
    if (i1 != 0 || i2 != 0)
      throw new Exception("invalid bounds for i (must be 0)");
    int j1 = sa.getStructure().getFirstIndex()[1];
      int j2 = sa.getStructure().getLastIndex()[1];
    Ny = j2 - j1 + 1;
    Nx = sa.get(0, j1).length();
    bim = new BufferedImage(Nx, Ny, BufferedImage.TYPE_INT_ARGB);
    //
    StringBuilder sb = new StringBuilder(Nx*Ny);
    String line;
    for (int j=j1; j<=j2; j++) {
      line = sa.get(0, j);
      sb.append(line).append("\0");
      if (line.length() != Nx)
        throw new Exception("invalid line length: "+line.length()+", expected: "+Nx);        
      for (int i=0; i<Nx; i++) {
        int z0c = Integer.parseInt(""+line.charAt(i));      
        Color c = mapColor.get(z0c);
        if (c == null) {
          c = clrUnknown;
          nClassesUnknown++;
        }
        bim.setRGB(i, j2-j, c.getRGB());
      }
    } 
    lCrc = getCrc(sb.toString());
    sb = new StringBuilder(1);
    sb = null;
    sa = null;
    arr = null;  
  }
  
  /**
   * Difference of 2 Z0 class registers
   * @param fn The file names, separated by a ";".
   * @throws Exception
   */
  private void readZ0CDIFF(String fn) throws Exception {
    String[] fa = fn.split("[;]");      
    File f = new File(fa[1]);
    chooser.setCurrentDirectory(f.getParentFile());
    IBJarr arrold = IBJdmn.readDmn(fa[0]);
    IBJarr arr = IBJdmn.readDmn(fa[1]);
    X0 = arr.getHeader().getDouble("xmin");
    Y0 = arr.getHeader().getDouble("ymin");
    Dd = arr.getHeader().getDouble("delta");
    double x0old = arrold.getHeader().getDouble("xmin");
    double y0old = arrold.getHeader().getDouble("ymin");
    double ddold = arrold.getHeader().getDouble("delta");
    if (x0old != X0 || y0old != Y0 || ddold != Dd)
      throw new Exception("inconsistent grid!");
    StringArray saold = (StringArray)arrold.getArray("Classes");
    StringArray sa = (StringArray)arr.getArray("Classes");
    int dim = sa.getStructure().getDims();
    if (dim != 2)
      throw new Exception("invalid dimension (must be 2)");
    int i1 = sa.getStructure().getFirstIndex()[0];
    int i2 = sa.getStructure().getLastIndex()[0];
    if (i1 != 0 || i2 != 0)
      throw new Exception("invalid bounds for i (must be 0)");
    int j1 = sa.getStructure().getFirstIndex()[1];
    int j2 = sa.getStructure().getLastIndex()[1];
    Ny = j2 - j1 + 1;
    Nx = sa.get(0, j1).length();
    //
    int i1old = saold.getStructure().getFirstIndex()[0];
    int i2old = saold.getStructure().getLastIndex()[0];
    if (i1old != 0 || i2old != 0)
      throw new Exception("invalid bounds for iold (must be 0)");
    int j1old = saold.getStructure().getFirstIndex()[1];
    int j2old = saold.getStructure().getLastIndex()[1];
    int nyold = j2old - j1old + 1;
    int nxold = saold.get(0, j1old).length();
    if (nxold != Nx || nyold != Ny)
      throw new Exception("inconsistent dimensions!");
    //
    bim = new BufferedImage(Nx, Ny, BufferedImage.TYPE_INT_ARGB);
    //
    int ndiff = 0;
    StringBuilder sb = new StringBuilder();
    for (int j=j1; j<=j2; j++) {
      String lineold = saold.get(0, j);
      String line = sa.get(0, j);
      sb.append(line).append("\0");
      if (line.length() != Nx)
        throw new Exception("invalid line length: "+line.length()+", expected: "+Nx);
      if (lineold.length() != Nx)
        throw new Exception("invalid lineold length: "+lineold.length()+", expected: "+Nx);
      for (int i=0; i<Nx; i++) {
        int z0cold = Integer.parseInt(""+lineold.charAt(i));
        int z0c = Integer.parseInt(""+line.charAt(i));
        if (z0c != z0cold)
          ndiff++;
        Color c = mapColor.get(910 + z0c - z0cold);
        if (c == null) {
          c = clrUnknown;
          nClassesUnknown++;
        }
        bim.setRGB(i, j2-j, c.getRGB());
      }
    } 
    System.out.printf("file 1: %s\nfile 2: %s\n%d fields with deviations.\n",
        fa[0], fa[1], ndiff);
  }
   
  private long getCrc(String s) {
    long cc = 0;
    try {
      byte[] bb = s.getBytes("ISO-8859-1");
      CRC32 crc32 = new CRC32();
      crc32.reset();
      crc32.update(bb);
      cc = crc32.getValue();
    } 
    catch (Exception e) {
      e.printStackTrace(System.out);
    }
    return cc;
  }
  
  private void createFrame() {
    frame = new JFrame();
    pnl = new JPanel() {
      @Override
      public void paint(Graphics g) {
        super.paint(g);
        g.drawImage(currentbim, 0, 0, null);
      }
    };
    cbLocale = new JComboBox(LOCALES);
    cbLocale.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent e) {
        String buf = null;
        String loc = ((String)cbLocale.getSelectedItem()).toLowerCase();        
        setTitle(loc);
        if (loc.equalsIgnoreCase("de"))         
          buf = LUC_DE;
        /*
        else if (((String)cbLocale.getSelectedItem()).equalsIgnoreCase("RO"))
          buf = LUC_RO;
        */
        else
          buf = LUC_EN;
        String[] ss = buf.split("\n");
        try {
          for (String s: ss) {
            int luc = Integer.parseInt(s.substring(0, 3));
            Color c = mapColor.get(luc);
            mapLuc.put(c.getRGB(), s);
          }
        }
        catch (Exception ee) {
          ee.printStackTrace(System.out);
        }
      }
    });
    pnl.setPreferredSize(new Dimension(Nx, Ny));
    pnl.addMouseMotionListener(mml);
    pnl.addMouseListener(ml);
    pnl.setCursor(new Cursor(Cursor.CROSSHAIR_CURSOR));
    panel = new JPanel(new BorderLayout());
    scp = new JScrollPane(pnl);
    panel.add(scp, BorderLayout.CENTER);
    lClassesUnknown = new JLabel("");
    lClassesUnknown.setForeground(Color.gray);
    pnlButtons = new JPanel();
    txt = new JTextField(60);
    txt.setEditable(true);
    txt.addKeyListener(new KeyListener() {
        @Override
        public void keyPressed(KeyEvent e) {}
        
        @Override
        public void keyReleased(KeyEvent e) {}
        
        @Override
        public void keyTyped(KeyEvent e) {
          if (e.getKeyChar() == KeyEvent.VK_ENTER) {          
            mouseTo(txt.getText());
            e.consume();
          }
          
        }         
    });
    pnlButtons.add(lClassesUnknown);
    pnlButtons.add(txt);
    pnlButtons.add(cbLocale);
    pnlButtons.add(new JLabel("  "));
    btnSave = new JButton("Save Picture...");
    btnSave.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent e) {
        save_data(pnl);
      }
    });
    pnlButtons.add(btnSave);
    panel.add(pnlButtons, BorderLayout.SOUTH);
    frame.setContentPane(panel);
    frame.setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);
    URL u = GisViewLuc.class.getResource("logo.gif");
    if (u != null)
      frame.setIconImage(frame.getToolkit().getImage(u));
    setTitle(Locale.getDefault().getLanguage());
  }
  
  private void display(String fn) {
    try {
      if (fn.endsWith(".png"))
        readPNG(fn);
      else if (bExpert && fn.indexOf(";") > 0)
        readZ0CDIFF(fn);
      else if (fn.endsWith(".dmna"))
        readZ0C(fn);
      else
        readLUC(fn);
      currentbim = bim;
      createFrame();
      if (!bExpert) 
        pnlButtons.remove(btnSave);
      setTitle(Locale.getDefault().getLanguage());
      if (nClassesUnknown > 0)
        lClassesUnknown.setText("("+nClassesUnknown + sClassesUnknown+")");
      Dimension sz = Toolkit.getDefaultToolkit().getScreenSize();
      frame.setSize(new Dimension((int)(0.9*sz.width), (int)(0.9*sz.height)));
      frame.setLocationRelativeTo(null);
      frame.setVisible(true);
      cbLocale.setSelectedItem(Locale.getDefault().getLanguage());
      scp.getVerticalScrollBar().setUnitIncrement(Ny/100);
      scp.getHorizontalScrollBar().setUnitIncrement(Nx/100);
      zoomed = false;     
    }
    catch (Exception e) {
      e.printStackTrace(System.out);
      JOptionPane.showMessageDialog(null, e.toString());
    }
  }

  private void save_data(JPanel pnl) {
    if (sFile.endsWith(".asc"))
      chooser.setDialogTitle("Specify the name of the output file (PNG or DMN)");
    else
      chooser.setDialogTitle("Specify the name of the output file (PNG)");
    int r = chooser.showSaveDialog(frame);
    if (r != JFileChooser.APPROVE_OPTION)
      return;
    File f = chooser.getSelectedFile();
    String s = f.getPath();
    if (f.exists()) {
    r = JOptionPane.showOptionDialog(frame,
        "Overwrite existing file?",
          "Please select",
          JOptionPane.YES_NO_OPTION,
          JOptionPane.QUESTION_MESSAGE,
          null,
          new String[] { "Yes", "No" },
          "No");
      if (r != JOptionPane.YES_OPTION)
        return;
    }
    String xtn = s.substring(s.lastIndexOf('.')+1);
    int width = pnl.getWidth();
    int height = pnl.getHeight();
    if (xtn.equalsIgnoreCase("png")) {
      BufferedImage bi = new BufferedImage(width, height,
        BufferedImage.TYPE_4BYTE_ABGR);
      Graphics2D g = bi.createGraphics();
      RenderingHints hints = new RenderingHints(RenderingHints.KEY_ANTIALIASING,
        RenderingHints.VALUE_ANTIALIAS_ON);
      hints.put(RenderingHints.KEY_RENDERING, RenderingHints.VALUE_RENDER_QUALITY);
      g.setRenderingHints(hints);
      pnl.paint(g);
      g.dispose();
      try {
        ImageIO.write(bi, xtn, f);
      }
      catch (Exception e) {
        e.printStackTrace(System.out);
      }
    }
    else if (sFile.endsWith(".asc") && xtn.equalsIgnoreCase("dmna")) {
      try {
        if (f.getName().startsWith("z0"))
          save_dmn_z0(f.getAbsolutePath());  // convert LUC to Z0 and save it as dmna
        else
          save_dmn(f.getAbsolutePath());     // simply save as dmna with class indices in the header
       }
      catch (Exception e) {
        e.printStackTrace(System.out);
      } 
    }
    else {
      System.out.printf("*** invalid output, data not saved.\n");
    }
  }

  private void save_dmn_z0(String fn) {
    String[] sa = new String[9];
    System.out.printf("Converting to Z0 class register %s...\n", fn);
    try {
      //
      // some security checks
      if (LUC_2_Z0C_2018.length != 9)
        throw new Exception("invalid dimension of LUC_2_Z0C_2018");
      for (int i=0; i<9; i++) {
        sa[i] = " "+LUC_2_Z0C_2018[i].trim()+" ";
        //
        // check chars
        String valid = " 0123456789";
        for (int j=0; j<sa[i].length(); j++)
          if (valid.indexOf(sa[i].charAt(j)) < 0)
            throw new Exception("invalid char in LUC_2_Z0C_2018 entry: "+sa[i]);
        //
        // check entries
        String[] saa = sa[i].trim().split("[ ]+");
        if (saa.length == 0)
          throw new Exception("invalid entries in LUC_2_Z0C_2018 entry: "+sa[i]);
        for (String saa1 : saa) {
          if (saa1.length() != 3) {
            throw new Exception("invalid entries in LUC_2_Z0C_2018 entry: "+sa[i]);             
          }
        }
      }
      //
      // automatically set GCCS
      File fin = new File(sFile);
      String idnt = fin.getName();
      String ggcs = null;
      if (idnt.contains("gk"))
        ggcs = "GK";
      else if (idnt.contains("utm"))
        ggcs = "UTM";
      else
        throw new Exception("can't determine GCCS from file name");      
      //
      // convert and write
      IBJarr arr = new IBJarr("Classes", 1, Ny); 
      arr.setFirstIndex(0, 1);
      IBJhdr hdr = arr.getHeader();
      hdr.putString("creator ", this.getClass().getName()+" "+Version, true);
      hdr.putDate("created", IBJhdr.getDate(new Date()));
      hdr.putString("clsb", "0123456789", true);
      String[] za = new String[Z0_CLASSES.length];
      for (int i=0; i<Z0_CLASSES.length; i++)
        za[i] = String.format(Locale.ENGLISH, "%d", i);
      hdr.putStrings("clsi", za, true);
      for (int i=0; i<Z0_CLASSES.length; i++)
        za[i] = String.format(Locale.ENGLISH, "%1.2f m", Z0_CLASSES[i]);
      hdr.putStrings("clsd", za, true);
      hdr.putString("ggcs", ggcs, true); 
      hdr.putDouble("xmin", X0, "%1.1f");
      hdr.putDouble("ymin", Y0, "%1.1f");
      hdr.putDouble("delta", Dd, "%1.1f");
      hdr.putString("vldf", "V", true);
      hdr.putString("sequ", "j-,i+", true);
      hdr.putInteger("buff", 1000000, "%1d");
      hdr.putString("mode", "text", true);
      hdr.putInteger("cmpr", 9, "%1d");      
      arr.createArray("Classes%"+String.format("%d.%ds", Nx, Nx+1));
      StringArray out = (StringArray)arr.getArray("Classes");      
      StringBuilder sb = new StringBuilder();
      StringBuilder line = new StringBuilder(Nx+1);
      String notfound = " ";
      for (int j=Ny-1; j>=0; j--) {
        System.out.printf("%04d\r", j);
        line.setLength(0);
        for (int i=0; i<Nx; i++) {
          int icolor = currentbim.getRGB(i, j);
          String label = mapLuc.get(icolor).substring(0, 3);
          int iclass = -1; 
          for (int k=0; k<9; k++)
            if (sa[k].contains(" "+label+" ")) {
              iclass = k+1;
              break;
            }
          if (iclass < 0) {
            if (!notfound.contains(" "+label+" "))
              notfound += label+" ";
            iclass = 0;
          }         
          line.append(iclass);
        }
        out.set(line.toString(), 0, Ny-j);
        sb.append(line.toString()).append("\0");        
      }
      if (notfound.trim().length() > 0)
        System.out.printf("*** the following land use classes could not be assigned:\n%s\n", 
            notfound.trim());
      hdr.putString("idnt", idnt+String.format("(%x)", getCrc(sb.toString())), true); 
      File f = new File(fn);
      IBJdmn.writeDmn(arr, f.getAbsolutePath()); 
      System.out.printf("Z0 class register written to %s.\n", f.getAbsolutePath());
    }
    catch (Exception e) {
      e.printStackTrace(System.out);
    }
    System.out.printf("...done.\n");
  }
  
  private void save_dmn(String fn) {
    System.out.printf("Writing class register %s...\n", fn);
    try {
      //
      // collect class indices
      StringBuilder sb = new StringBuilder();
      ArrayList<LUC> va = new ArrayList<LUC>();
      for (int j=Ny-1; j>=0; j--) {
        for (int i=0; i<Nx; i++) {
          int icolor = currentbim.getRGB(i, j);
          String s = mapLuc.get(icolor);
          if (s == null)
            throw new Exception("class "+icolor+" not set internally");
          String id = s.substring(0, 3).trim();
          if (!sb.toString().contains(id)) {
            sb.append(" ").append(id);
            LUC luc = new LUC();
            luc.id = id;
            luc.clr = icolor;
            luc.dsc = s.substring(3).trim();
            va.add(luc);
          }
        }
      }
      Collections.sort(va);
      String[] ia = new String[va.size()];
      String[] da = new String[ia.length];
      char[] ca = new char[ia.length];
      int b0 = 'a';
      int off = 0;
      for (int i=0; i<ia.length; i++) {
        LUC luc = va.get(i);
        ia[i] = luc.id;
        da[i] = luc.dsc;
        char c = (char)(b0 + i + off);
        if (c > 'z') {
          c = 'A';
          b0 = c;
          off = -i;
        }
        if (b0 != 'a' && c > 'Z')
          throw new Exception("too many land use classes!");
        luc.b = c;
        ca[i] = c;
      }
      System.out.printf("Used land use classes: %d\n", ca.length);
      for (int i=0; i<ca.length; i++)
        System.out.printf("%2d: %c --> %s (%s)\n", i+1, ca[i], ia[i], da[i]);     
      //
      // automatically set GCCS
      File fin = new File(sFile);
      String idnt = fin.getName();
      String ggcs = null;
      if (idnt.contains("gk"))
        ggcs = "GK";
      else if (idnt.contains("utm"))
        ggcs = "UTM";
      else
        throw new Exception("can't determine GCCS from file name");      
      //
      // write
      IBJarr arr = new IBJarr("Classes", 1, Ny);
      arr.setFirstIndex(0, 1);
      IBJhdr hdr = arr.getHeader();
      hdr.putString("creator ", this.getClass().getName()+" "+Version, true);
      hdr.putDate("created", IBJhdr.getDate(new Date()));
      sb.setLength(0);
      for (char c : ca)
        sb.append(c);
      hdr.putString("clsb", sb.toString(), true);
      hdr.putStrings("clsi", ia, true);
      hdr.putStrings("clsd", da, true);
      hdr.putString("ggcs", ggcs, true); 
      hdr.putDouble("xmin", X0, "%1.1f");
      hdr.putDouble("ymin", Y0, "%1.1f");
      hdr.putDouble("delta", Dd, "%1.1f");
      hdr.putString("vldf", "V", true);
      hdr.putString("sequ", "j-,i+", true);
      hdr.putInteger("buff", 1000000, "%1d");
      hdr.putString("mode", "text", true);
      hdr.putInteger("cmpr", 9, "%1d");  
      arr.createArray("Classes%"+String.format("%d.%ds", Nx, Nx+1));
      StringArray out = (StringArray)arr.getArray("Classes");
      //
      IBJarr arr2 = new IBJarr("Classes", Nx, Ny);
      arr2.setFirstIndex(1, 1);
      arr2.setHeader(hdr);
      IBJhdr hdr2 = arr2.getHeader();
      arr2.createArray("Classes%3d");
      IntegerArray out2 = (IntegerArray)arr2.getArray("Classes");
      //
      StringBuilder line = new StringBuilder(Nx+1);
      String notfound = " ";
      sb.setLength(0);
      for (int j=Ny-1; j>=0; j--) {
        if (j%100 == 0) System.out.printf("\r%04d", j);
        line.setLength(0);
        for (int i=0; i<Nx; i++) {
          int icolor = currentbim.getRGB(i, j);
          for (int ii=0; ii<va.size(); ii++) {
            if (va.get(ii).clr == icolor) {
              line.append(va.get(ii).b);
              out2.set(Integer.parseInt(va.get(ii).id), i+1, Ny-j);
              break;
            }  
          }
        } 
        out.set(line.toString(), 0, Ny-j);
        sb.append(line.toString()).append("\0");        
      }
      hdr.putString("idnt", idnt+String.format("(%x)", getCrc(sb.toString())), true); 
      hdr2.putString("idnt", idnt, true); 
      File f = new File(fn);
      IBJdmn.writeDmn(arr, f.getAbsolutePath()); 
      System.out.printf("\rclass register written to %s.\n", f.getAbsolutePath());
      /*
      File f2 = new File(fn);
      int ip = f2.getName().indexOf(".");
      f2 = new File(f2.getParent(), f2.getName().substring(0,ip)+"-2d.dmna");
      IBJdmn.writeDmn(arr2, f2.getAbsolutePath()); 
      System.out.printf("\nclass register (2d) written to %s.\n", f2.getAbsolutePath());
      */
    }
    catch (Exception e) {
      e.printStackTrace(System.out);
    }
    System.out.printf("...done.\n");
  }
    
  private void zoom(int x, int y) {
    if (zoomed) {
      txt.setEditable(true);
      zoomed = false;
      X0 = X0org;
      Y0 = Y0org;
      Dd = Ddorg;
      Nx = Nxorg;
      Ny = Nyorg;
      currentbim = bim;
      pnl.setPreferredSize(new Dimension(Nx, Ny));
      pnl.updateUI();
    }
    else {
      txt.setEditable(false);
      zoomed = true;
      int fac = 50;
      X0org = X0;
      Y0org = Y0;
      Ddorg = Dd;
      Nxorg = Nx;
      Nyorg = Ny;
      if (bimz == null) {
        if (Nxz > Nx)
          Nxz = Nx;
        if (Nyz > Ny)
          Nyz = Ny;
        bimz = new BufferedImage(Nxz*fac, Nyz*fac, BufferedImage.TYPE_INT_ARGB);     
      }
      double[] da = mouseAt(x, y);
      Xorg = da[0];
      Yorg = da[1];     
      x -= Nxz/2;
      y -= Nyz/2;
      if (x < 0)
        x = 0;
      if (x + Nxz >= Nx)
        x = Nx - Nxz - 1;
      if (y < 0)
        y = 0;
      if (y + Nyz >= Ny)
        y = Ny - Nyz - 1;
      da = mouseAt(x, y);
      X0 = da[0] - Dd*0.5;
      Y0 = da[1] - Nyz*Dd + Dd*0.5;
      Nx = Nxz*fac;
      Ny = Nyz*fac;
      Dd = Ddorg/fac;     
      for (int i=x; i<x+Nxz; i++) {
        for (int j=y; j<y+Nyz; j++) {
          int clr = bim.getRGB(i, j);
          for (int ii=0; ii<fac; ii++)
            for (int jj=0; jj<fac; jj++)
              bimz.setRGB((i-x)*fac+ii, (j-y)*fac+jj, clr);
        }
      }
      currentbim = bimz;      
    }
    pnl.setPreferredSize(new Dimension(Nx, Ny));
    pnl.updateUI();
    pnl.repaint();      
    scp.setViewportView(pnl);
    scp.repaint();
    mouseTo(String.format("%1.1f %1.1f", Xorg, Yorg));
  }
    
  private void mouseTo(String text) {    
    double xm = -1;
    double ym = -1;
    try {
      String s = text.trim();
      String[] sa = s.split("[ ;]+");
      xm = IBJhdr.parseDoubleOf(sa[0]);
      ym = IBJhdr.parseDoubleOf(sa[1]);
      if (xm < X0 || xm > X0+Nx*Dd)
        xm = -1;
      if (ym < Y0 || ym > Y0+Ny*Dd)
        ym = -1;
    }
    catch (Exception e) {
      e.printStackTrace(System.out);
      xm = -1;
      ym = -1;
    }
    if (xm < 0 || ym < 0) {
      System.out.printf("*** xm=%1.1f, ym=%1.1f, X0=%1.1f, Y0=%1.1f, Dd=%1.1f, Nx=%d, Ny=%d\n",
          xm, ym, X0, Y0, Dd, Nx, Ny);
      txt.setText("Invalid coordinate specification!");
      return;
    }
    int x = (int)((xm - X0 - 0.5*Dd)/Dd + 0.5);
    int y = (int)(Ny-1 - (ym - Y0 - 0.5*Dd)/Dd + 0.5);
    //
    Rectangle viewrect = scp.getViewport().getViewRect();
    int xx = x;
    int yy = y;
    x -= viewrect.getWidth()/2;
    if (x < 0)
      x = 0;
    y -= viewrect.getHeight()/2;
    if (y < 0)
      y = 0;
    scp.getViewport().setViewPosition(new Point((int)x,(int)y));
    pnl.requestFocus();
    viewrect = scp.getViewport().getViewRect();
    try {
      Robot r = new Robot();
      x = frame.getContentPane().getLocationOnScreen().x + xx - viewrect.x + 1;
      y = frame.getContentPane().getLocationOnScreen().y + yy - viewrect.y + 1;     
      r.mouseMove(x, y);
    }
    catch (Exception e) {
      e.printStackTrace(System.out);
    }   
  }

  private double[] mouseAt(int x, int y) {
    if (x < 0 || x >=Nx || y < 0 || y >= Ny)
      return null;
    int ic = currentbim.getRGB(x, y);
    String label = mapLuc.get(ic);
    if (label == null)
      label = "unknown";
    double xm = X0 + x*Dd + 0.5*Dd;
    double ym = Y0 + (Ny-1 - y)*Dd + 0.5*Dd;
    txt.setText(String.format(" %6.0f %6.0f : %s", xm, ym, label));
    double[] da = { xm, ym };
    return da;
  }

  private String getFile() {
    FileFilter filter = new FileFilter() {
      @Override
      public boolean accept(File f) {
        String name = f.getName().toLowerCase();
        return (f.isDirectory() || name.endsWith(".asc") || name.endsWith(".png") || name.endsWith(".dmna"));
      }
      @Override
      public String getDescription() {
        return sFilter;
      }
    };
    chooser.setFileFilter(filter);
    int rc = chooser.showOpenDialog(frame);
    if (rc != JFileChooser.APPROVE_OPTION)
      System.exit(0);
    return chooser.getSelectedFile().getPath();
  }
  
  private void setTitle(String locale) {
    if (frame == null)
      return;
    String s = sFile;
    if (lCrc > 0)
      s += String.format(" [%x]", lCrc);
    if (locale.equalsIgnoreCase("de")) {
      btnSave.setText("Speichern");
      frame.setTitle("ViewLuc "+Version+" (C) 2019 Ing.-Büro Janicke - "+s);
      sFilter = "PNG-Bild (*.png), Landnutzungs-Kataster (*.asc) oder Z0-Klassen-Kataster (*.dmn)";
      sClassesUnknown = " Felder mit unbekannter Klasse";
      txt.setToolTipText(TOOLTIP_DE);
    }
    else {
      btnSave.setText("Save");
      frame.setTitle("ViewLuc "+Version+" (C) 2019 Janicke Consulting - "+s);
      sFilter = "PNG picture (*.png), land use register (*.asc), or Z0 class register (*.dmn)";
      sClassesUnknown = " fields with unknown classes";
      txt.setToolTipText(TOOLTIP_EN);
    }
  }

  //----------------------------------------------------------------------------
  public static void main(String[] args) {
    GisViewLuc g = new GisViewLuc();
    
    sFile = null;
    String sout = null;
    boolean cz0 = false;
    bExpert = false; 
    try {
      /*
      /////////////////////////////////////////////////////////////////
      // convert 2018-11-22
      String dir = "D:/austal/LBM/LBM-2017";
      String foutc, foutz;
      String[] ta = { "gk3", "utm32" };
      for (String t : ta) {
        sFile = "clcopti1ha_100m_"+t+"_ex.asc";
        foutc = "clcopti1ha_100m_"+t+"_ex.dmna";
        String s = (t.startsWith("gk") ? "gk" : "utm");
        foutz = "z0-"+s+".dmna";
        g.readLUC(((new File(dir, sFile)).getAbsolutePath()));
        g.currentbim = g.bim;
        g.save_dmn(((new File(dir, foutc)).getAbsolutePath()));
        g.save_dmn_z0(((new File(dir, foutz)).getAbsolutePath()));
      }
      if (true) return;
      /////////////////////////////////////////////////////////////////
      */
      
      
      for (String arg : args) {
        if (arg.startsWith("--expert")) {
          bExpert = true;
        }
        else if (arg.startsWith("--input=")) {
          sFile = arg.substring(8);   
        }
        else if (arg.startsWith("--output=")) {
          sout = arg.substring(9);   
        }
        else if (arg.startsWith("--convert")) {
          cz0 = true;   
        }
      }
      if (sFile != null && sout != null) {
        g.readLUC(((new File(args[0], sFile)).getAbsolutePath()));
        g.currentbim = g.bim;
        if (!cz0)
          g.save_dmn(((new File(args[0], sout)).getAbsolutePath()));
        else
          g.save_dmn_z0(((new File(args[0], sout)).getAbsolutePath()));
      }
      else {
        g.createFrame();
        if (sFile == null)
          sFile = g.getFile();
        g.display(sFile);
      }
    }
    catch (Exception e) {
      e.printStackTrace(System.out);
    }
    
  }

}
