// ================================================================= IBJdmn.java
//
// Utility for reading and writing DMN files
// =========================================
//
// Copyright 2005-2018 Janicke Consulting, 88662 Überlingen
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation; either version 2 of
// the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
//
// History:
//
// 2007-01-04 uj 1.1  change header entry "zone" to "tmzn"
// 2007-02-19 uj 1.2  skip line comments in formatted data part
// 2008-11-14 lj      readDmnHeader() added
// 2008-12-05 lj 1.3  merged with uj
// 2009-05-07 lj 1.4  user can provide character set name
// 2010-01-25 lj 1.5  readHeader() uses setLocale() and setTimeZone()
// 2010-01-29 lj 1.6  main(): adjust header for selection with artp="C"
// 2010-02-05 lj 1.7  putGA(), putCSV()
// 2010-02-10 lj 1.8  mode "artm" (use tab as separator in header)
// 2010-05-19 lj 1.9  chaining of error messages (errlog not used)
// 2011-11-01 lj 1.10 readASC()
// 2012-02-03 lj 1.11 handling of encoding (see note), putASC() corrected
// 2012-08-03 uj 1.12 handling of "fact" in main() revised, check in putDmn()
// 2012-09-03 uj 1.13 checking of "." adjusted
// 2012-10-15 uj 1.14 readHeader: skip comment in line
// 2012-10-22 uj 1.15 static routines addDry2Wet() and adjustHeader()
// 2013-07-27 uj 1.16 make header fact and arr factor consistent before writing
// 2013-08-19 uj 1.17 quoting and factor setting corrected
// 2018-02-08 uj 1.18 output type XYZ
//
// =============================================================================

package de.janicke.ibjutil;

import de.janicke.ibjutil.IBJarr.AbstractArray;
import de.janicke.ibjutil.IBJarr.FloatArray;
import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.FileReader;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.LineNumberReader;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.PrintWriter;
import java.nio.ByteOrder;
import java.nio.charset.Charset;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Date;
import java.util.Locale;
import java.util.Map;
import java.util.zip.GZIPInputStream;
import java.util.zip.GZIPOutputStream;

/**
 * 
 * <p>Copyright 2005-2013 Janicke Consulting, 88662 Überlingen</p>
 *
 * Reading and writing of <code>IBJarr</code> objects as DMN files (formatted or binary, 
 * optionally with compressed data). Static methods are provided for reading 
 * and writing. The method <code>main</code> can be used like the C-program 
 * <code>IBJdata</code> for copying and transforming DMN-files.
 * <p>
 * The following parameters in the header of the <code>IBJarr</code> object are
 * recognized:
 * <table>
 * <tr><th>Name</th> <th>Usage</th></tr>
 * <tr><td valign="top"><code>chsn</code></td> 
 * <td>The name of the characterset used for conversion between characters (in 
 * internal strings) and bytes (in files). Common values are {@code iso-8859-1} 
 * (Latin-1), {@code iso-8859-15} (Latin-1 with EURO symbol) and {@code utf-8}.
 * </td></tr>
 * <tr><td valign="top"><code>cmpr</code></td> 
 * <td>Compression level (0..9) using GZIP. A value of 0 means "no compression".
 * Compression can be applied to formatted and unformatted output.
 * </td></tr>
 * <tr><td valign="top"><code>data</code></td> 
 * <td>Name of the data file including file name 
 * extension. If this parameter
 * is ommitted or if it has the value "*" the following standard actions are 
 * taken:
 * <ul>
 * <li>formatted uncompressed: data are appended to the header file.</li>
 * <li>formatted compressed: data file is separate
 * with extension <code>.dmnt.gz</code></li>
 * <li>binary: data file is separate
 * with extension <code>.dmnb</code>.</li>
 * </ul>
 * If a file name is provided, the path may be relative to the directory containing
 * the header file or absolut.<br>
 * Compressed data files get the additional extension <code>.gz</code>.
 * </td></tr>
 * <tr><td valign="top"><code>drop</code></td> 
 * <td>
 * Specifies which lines should be dropped when reading formatted data. 
 * Basic patterns are:
 * <table>
 * <tr><td><i>count</i> </td>
 * <td>drop the first <i>count</i> lines</td></tr>
 * <tr><td>{@code =}<i>string</i></td>
 * <td>drop all lines starting with <i>string</i></td></tr>
 * <tr><td>{@code =[}<i>string</i>{@code ]}</td>
 * <td>drop all lines starting with a character
 * in <i>string</i> ({@code @} stands for all letters a..z and A..Z).</td></tr>
 * </table>
 * If the character {@code =} is replaced by {@code !}, the selection is inversed.
 * The first pattern might be combined with one of the others by separating it
 * by a semicolon. Examples:
 * <table>
 * <tr><td>{@code 5;[@]}</td>
 * <td>drop the first 5 lines and all of the following lines starting
 * with a letter</td></tr>
 * <tr><td>{@code !AK}</td>
 * <td>drop all lines not starting with the string "AK".</td></tr>
 * </table>
 * </td></tr>
 * <tr><td valign="top"><code>fact</code></td> 
 * <td>A factor applied to all elements of type <i>FLOAT</i> or <i>DOUBLE</i>
 * in formatted output.
 * </td></tr>
 * <tr><td valign="top"><code>locl</code></td> 
 * <td>
 * The localisation used in formatting floating point numbers. If the value is 
 * {@code C} then a decimal point is used. If the value is 
 * {@code german} then a decimal comma is used. 
 * </td></tr>
 * <tr><td valign="top"><code>mode</code></td> 
 * <td>If the value is {@code binary} the data are stored unformatted in a separate file.
 * Otherwise the data are stored formatted according to the formats given by the
 * parameter <code>form</code> 
 * (see {@link IBJarr.AbstractArray#createArray(String) IBJarr}).
 * </td></tr>
 * <tr><td valign="top"><code>sequ</code></td>
 * <td>
 * Specifies the sequence (reordering and/or selection)
 *  of data elements for writing (see {@link IBJarr#getMapped(String, String)
 *  getMapped}).
 * If the sequence represents a pure reordering then it is written unchanged into
 * the header of the output file.
 * If a selection is made, the parameter {@code sequ} is cleared.<br><br>
 * If the parameter is specified in the header of a DMN-file, then a reordering is 
 * reversed when this file is read, i.e. the arrangement of the data elements in 
 * the array read is the same as in the original array written. If a selection 
 * is made either in the header of the array written or in the header of the 
 * file read usually the arrangement and the index range is changed.
 * </td></tr>
 * <tr><td valign="top"><code>tmzn</code></td> 
 * <td>
 * The time zone used in formatting dates. The format is GMT[+|-]HH:mm.
 * </td></tr>
 * </table>
 * </p>
 * Required classes: {@link IBJdcl IBJdcl}, {@link IBJarr IBJarr}, 
 * {@link IBJhdr IBJhdr}.
 * 
 * @author Janicke Consulting, Überlingen
 * @version 1.17 of 2013-08-19
 */

 
public class IBJdmn implements IBJdcl {
  /** Enables check output to {@code System.err}. */
  public static boolean CHECK = false;
  /** The current version. */
  public static String version = "1.18";
  /** Date of last change. */
  public static String last_change = "2018-02-08";
  IBJhdr header;
  String name;
  String data;
  String mode;
//  String locl;
  String vldf;
  String form;
  String lsbf;
  String sequ;
//  String tmzn;
  String chsn;
  String drop;
  float fact;
  int cmpr;
  int dims;
  int[] lowb;
  int[] hghb;
  int[] length;
  ArrayList<String> vfrm = new ArrayList<String>();
  Drop drp;
  boolean artm = false;                                           //-2010-02-10
  String default_chsn;                                            //-2012-02-03
  
  public IBJdmn() {                                              //-2011-11-15
    default_chsn = Charset.defaultCharset().name();
  }
  
  public IBJdmn(String cset) {                                   //-2012-02-03
    default_chsn = Charset.defaultCharset().name();
    chsn = (cset != null && Charset.isSupported(cset)) ? cset : default_chsn;
  }

  public boolean setArtm(boolean artm) {                         //-2010-02-10
    boolean old_value = this.artm;
    this.artm = artm;
    return old_value;
  }

  private String readHeader(BufferedReader br) throws Exception {
    int i, l, ll;
    String line;
    String key;
    String value;
    String encoding = null;
    header = new IBJhdr();
    while (null != (line = br.readLine())) {
      l = line.length();
      if (l < 1)
        continue;
      char c = line.charAt(0);
      if (c == '*')
        break;
      if (!Character.isLetter(c))
        continue;
      for (i = 0; i < l; i++)
        if (Character.isWhitespace(line.charAt(i)))
          break;
      if (i >= l)
        continue;
      //                                                            -2010-01-25
      key = line.substring(0, i).toLowerCase();
      value = line.substring(i + 1).trim();
      //
      // skip comment in line                                     //-2012-10-15
      i = value.indexOf('\'');
      l = value.indexOf('\"');
      ll = value.lastIndexOf('\"');
      if (i >= 0) {
        if (l < 0 || (l >= 0 && i < l) || (ll >= 0 && i > ll))
          value = value.substring(0, i).trim();
      }     
      //
      if (key.equals("locl") || key.equals("locale"))
        header.setLocale(IBJhdr.unquote(value));
      else if (key.equals("tmzn") || key.equals("zone"))
        header.setTimeZone(IBJhdr.unquote(value));
      else if (key.equals("chsn") || key.equals("cset"))   
        encoding = IBJhdr.unquote(value.toUpperCase());
      else 
        header.putString(key, value, false);
      //
    }
    return encoding;
  }

  private void analyseHeader(String fn) throws Exception {
    name = header.getString("file", true);
    mode = header.getString("mode", true);
    data = header.getString("data", true);
    vldf = header.getString("vldf|valdef", true);
    form = header.getString("form|format", true);
    lsbf = header.getString("lsbf", true);
    sequ = header.getString("sequ", true);
    cmpr = header.getInteger("cmpr");
    dims = header.getInteger("dims");
    lowb = header.getIntegers("lowb"); 
    hghb = header.getIntegers("hghb");
    fact = header.getFloat("fact");
    drop = header.getString("drop", true);
    //
    // check data
    //
    if (form == null)
      throw new Exception("missing format description");
    if (mode == null) mode = "text";
    if (data == null) data = "*";
    if (lsbf == null) lsbf = "1";
    if (chsn == null) chsn = Charset.defaultCharset().name();
    if (dims < 1 || dims > 5)
      throw new Exception("invalid number of dimensions");
    if (lowb == null || lowb.length != dims)
      throw new Exception("no or invalid lower bounds");
    if (hghb == null || hghb.length != dims)
      throw new Exception("no or invalid upper bounds");
    length = new int[dims];
    for (int i = 0; i < dims; i++) {
      length[i] = hghb[i] - lowb[i] + 1;
      if (length[i] <= 0)
        throw new Exception("invalid bounds");
    }
    if (cmpr < 0 || cmpr > 9)
      throw new Exception("invalid compression mode");
    if (data.equals("*") && (mode.equals("binary") || cmpr > 0)) {
      String n = new File(fn).getName();
      if (n.endsWith(".dmna")) n = n.substring(0, n.length() - 5);
      data = n + ((mode.equals("binary")) ? ".dmnb" : ".dmnt");
      if (cmpr > 0) data += ".gz";
    }
    if (Float.isNaN(fact)) fact = 1;
    if (drop != null) drp = new Drop(drop);
    //
    // check sequence
    //
    if (sequ != null) {
      int l = sequ.length();
      StringBuilder sb = new StringBuilder();
      for (int i = 0; i < l; i++) {
        char c = sequ.charAt(i);
        if (Character.isLetter(c)) {
          sb.append(c);
          c = (i < l - 1) ? sequ.charAt(i + 1) : ';';
          if (",;".indexOf(c) >= 0) sb.append('+');
        }
        else {
          if (c == ',') c = ';';
          sb.append(c);
        }
      }
      sequ = sb.toString();
      if ("i+;j+;k+;l+;m+;".startsWith(sequ)) {
        if (CHECK) System.err.println("removing sequ");
        header.p.remove("sequ");
        sequ = null;
      }
      else {
        header.putString("sequ", sequ, true);
      }
    }
    //
    // save format
    //
    String[] forms = IBJarr.Descriptor.expandFormat(form);
    vfrm.addAll(Arrays.asList(forms));
  }

  private LineNumberReader getReader(String fn, String chsn, int cmpr)
      throws Exception {
    if (chsn == null) chsn = Charset.defaultCharset().name();
    if (CHECK) System.err.println("reader: using chsn=" + chsn);
    FileInputStream is = new FileInputStream(fn);
    InputStream in = is;
    if (cmpr > 0) {
      in = new GZIPInputStream(is);
      if (CHECK) System.err.println("reading compressed data");
    }
    InputStreamReader sr = new InputStreamReader(in, chsn);
    LineNumberReader lr = new LineNumberReader(sr);
    return lr;
  }

  private PrintWriter getWriter(String fn, String chsn, int cmpr)
      throws Exception {
    if (chsn == null) chsn = Charset.defaultCharset().name();
    if (CHECK) System.err.println("writer: using chsn=" + chsn);
    FileOutputStream os = new FileOutputStream(fn);
    OutputStream out = os;
    if (cmpr > 0) {
      out = new GZIPOutputStream(out);
      if (CHECK) System.err.println("writing compressed data");
    }
    OutputStreamWriter sw = new OutputStreamWriter(out, chsn);
    PrintWriter pw = new PrintWriter(sw, true);
    return pw;
  }

  private BufferedInputStream getInputStream(String fn, int cmpr)
      throws Exception {
    FileInputStream fi = new FileInputStream(fn);
    InputStream is = fi;
    if (cmpr > 0) {
      is = new GZIPInputStream(fi);
      if (CHECK) System.err.println("reading compressed data");
    }
    BufferedInputStream bi = new BufferedInputStream(is);
    return bi;
  }

  private BufferedOutputStream getOutputStream(String fn, int cmpr)
      throws Exception {
    FileOutputStream fo = new FileOutputStream(fn);
    OutputStream os = fo;
    if (cmpr > 0) {
      os = new GZIPOutputStream(fo);
      if (CHECK) System.err.println("writing compressed data");
    }
    BufferedOutputStream bo = new BufferedOutputStream(os);
    return bo;
  }

  private IBJarr getDmn(String fn, boolean force) throws Exception {
    IBJarr arr = null;
    IBJarr brr = null;
    LineNumberReader lr = null;
    BufferedInputStream bi = null;
    if (fn.endsWith(".dmna")) fn = fn.substring(0, fn.length() - 5);
    String gn = fn + ".dmna";
    File f = new File(gn);
    try {
      lr = getReader(f.getPath(), chsn, 0);
      String cn = readHeader(lr);
      if (!force && cn != null && !cn.equals(chsn)) {
        lr.close();
        header.p.clear();
        lr = getReader(gn, cn, 0);
        readHeader(lr);
        chsn = cn;
      }
      header.putString("cset", chsn, true);                       //-2012-02-03
      analyseHeader(fn);
      if (name == null) 
        name = fn;
      arr = createContainer();
      if (mode.equals("binary")) {
        lr.close();
        lr = null;
        f = new File(fn);
        File g = f.getParentFile();
        f = new File(g, data);
        bi = getInputStream(f.getPath(), cmpr);
        if (sequ == null) {
          brr = arr;
        }
        else {
          brr = arr.getSelected("mapped", sequ);
        }
        readBinary(bi, brr);
        bi.close();
        bi = null;
      }
      else {
        if (!data.equals("*")) {
          lr.close();
          lr = null;
          f = new File(fn);
          File g = f.getParentFile();
          f = new File(g, data);
          lr = getReader(f.getPath(), chsn, cmpr);
        }
        if (sequ == null) {
          brr = arr;
        }
        else {
          brr = arr.getSelected("mapped", sequ);
        }
        readText(lr, brr);
        lr.close();
        lr = null;
      }
    }
    catch (Exception e) {
      String emsg = e.toString();
      if (lr != null) {
        emsg = String.format("*** error in file %s, line %d%n", f.getPath(),
          lr.getLineNumber()) + emsg;
        try {
          lr.close();
        }
        catch (Exception x) {}
      }
      if (bi != null) {
        emsg = String.format("%s%n", "*** error in binary read") + emsg;
        try {
          bi.close();
        }
        catch (Exception x) {}
      }
      if (CHECK) e.printStackTrace(System.out);
      throw new Exception(String.format("Can't read file \"%s\"%n%s", fn, emsg));
    }
    finally {
      try { lr.close(); } catch (Exception e) {}
      try { bi.close(); } catch (Exception e) {}
    }
    return arr;
  }

  private IBJarr createContainer() throws Exception {
    IBJarr arr = null;
    try {
      arr = new IBJarr(name, length);
      arr.setFirstIndex(lowb);
      arr.setHeader(header);
      for (int i = 0; i < vfrm.size(); i++) {
        String frm = vfrm.get(i);
        IBJarr.AbstractArray aa = arr.createArray(frm, null);
        if (vldf != null && vldf.length() > i)
          aa.valdef = vldf.substring(i, i + 1);
      }
      arr.setFactor(fact);
      arr.setCharset(chsn);
      arr.setLSBF(!lsbf.equals("0"));
    }
    catch (Exception e) {
      String emsg = e.toString();
      if (CHECK) e.printStackTrace(System.out);
      throw new Exception(String.format("Can't create container%n%s", emsg));
    }
    return arr;
  }

  private int readBinary(BufferedInputStream bi, IBJarr arr) throws Exception {
    int n = 0;
    try {
      arr.setLSBF(!lsbf.equals("0"));
      int na = arr.arrays.size();
      if (na == 0) 
        return 0;
      AbstractArray aa = arr.arrays.get(0);
      IBJarr.Structure struc = arr.getStructure();
      int[] ff = struc.getFirstIndex();
      int[] ll = struc.getLastIndex();
      int[] ii = (int[]) ff.clone();
      int ns = struc.getDims();
      int i = 0;
      while (i >= 0) {
        for (int k = 0; k < na; k++) {
          if (na > 1) aa = arr.arrays.get(k);
          int nb = bi.read(aa.bytes, 0, aa.bytes.length);
          if (nb != aa.bytes.length)
            throw new Exception("Only " + nb + " bytes of " + aa.bytes.length
                + " available");
          aa.fromBuffer(ii);
          n++;
        }
        for (i = ns - 1; i >= 0; i--) {
          ii[i]++;
          if (ii[i] <= ll[i])
            break;
          else {
            ii[i] = ff[i];
          }
        }
      }
    }
    catch (Exception e) {
      String emsg = e.toString();
      if (CHECK) e.printStackTrace(System.out);
      throw new Exception(String.format("Can't read binary data (n=%d)%n%s",
        n, emsg));
    }
    return n;
  }

  private int readText(BufferedReader br, IBJarr arr) throws Exception {
    int n = 0;
    try {
      int na = arr.arrays.size();
      if (na == 0)
        return 0;
      AbstractArray aa = arr.arrays.get(0);
      IBJarr.Structure struc = arr.getStructure();
      int[] ff = struc.getFirstIndex();
      int[] ll = struc.getLastIndex();
      int[] ii = (int[]) ff.clone();
      int ns = struc.getDims();
      StringBuffer sb = new StringBuffer();
      if (drp != null)  drp.active = true;
      int i = 0;
      while (i >= 0) {
        for (int k = 0; k < na; k++) {
          if (na > 1) aa = arr.arrays.get(k);
          nextToken(br, sb);
          aa.parse(sb.toString(), ii);
          n++;
        }
        for (i = ns - 1; i >= 0; i--) {
          ii[i]++;
          if (ii[i] <= ll[i])
            break;
          else {
            ii[i] = ff[i];
          }
        }
      }
    }
    catch (Exception e) {
      String emsg = e.toString();
      if (CHECK) e.printStackTrace(System.out);
      throw new Exception(String.format("Can't read text data (n=%d)%n%s",
        n, emsg));
    }
    return n;
  }

  private boolean nextToken(BufferedReader br, StringBuffer sb) throws Exception {
    sb.setLength(0);
    char c;
    boolean string = false;
    boolean checking = false;
    boolean dropping = false;
    int pos = 0;
    int chk = 0;
    int n = 0;
    int i = 0;
    if (drp != null && drp.active) {
      checking = true;
      pos = 0;
    }
    while (0 <= (i = br.read())) {
      c = (char) i;
      if (drp != null) {
        if (dropping) {
          if (c != '\n') 
            continue;
          dropping = false;
          drp.active = true;
        }
        if (checking) {
          if (drp.count > 0) {
            chk = 1;
            drp.count--;
          }
          else {
            chk = drp.check(c, pos);
          }
          if (chk > 0) { // drop the line
            dropping = true;
            checking = false;
            string = false;
            sb.setLength(0);
            drp.active = false;
            continue;
          }
          else if (chk < 0) { // don't drop
            checking = false;
            drp.active = false;
          }
          else { // maybe
            pos++;
          }
        }
        if (c == '\n') {
          drp.active = true;
          checking = true;
          pos = 0;
        }
      }
      if (string) {
        if (c == '\\') {
          i = br.read();
          if (i < 0) 
            break;
          c = (char) i;
          if (c != '"') 
            throw new Exception("invalid escape sequence");
        }
        else if (c == '"') 
          break;
      }
      else {
        if (c == '\'') {                                  // skip up to end of the line 
          while (0 <= (i = br.read())) {                  //-2007-02-19
            c = (char) i;
            if (c == '\n') 
              break;
          }
          continue;
        }
        if (Character.isWhitespace(c)) {
          if (n > 0) 
            break;
          else continue;
        }
        if (c == '"') {
          if (n > 0) 
            throw new Exception("invalid quotation");
          string = true;
          continue;
        }
      }
      sb.append(c);
      n++;
    }
    if (i < 0) 
      throw new Exception("end of file found");
    return string;
  }

  private void putDmn(IBJarr arr, String fn) throws Exception {
    IBJarr brr = null;
    PrintWriter pw = null;
    BufferedOutputStream bo = null;
    String gn = fn + ".dmna";
    try {
      IBJhdr hdr = arr.getHeader();
      header = hdr.getCopy();
      checkHeader(arr, fn);
      if (fn.equals("stdout")) {
        mode = "text";
        data = "*";
        pw = new PrintWriter(System.out);
      }
      else pw = getWriter(gn, chsn, 0);
      //          
      writeHeader(pw, chsn, header, artm);                        //-2012-02-03
      //
      // set consistent values in header and array before writing   -2013-07-27
      float hdrfact = header.getFloat("fact");
      arr.setFactor(Float.isNaN(hdrfact) ? 1 : hdrfact);
      //
      if (sequ == null) {
        brr = arr;
      }
      else {
        if (CHECK) System.err.println("writing mapped array");
        brr = arr.getSelected("mapped", sequ);        
      }
      brr.setCharset(chsn);     // for StringArray
      
      if (mode.equals("binary")) {
        pw.close();
        pw = null;
        File f = new File(fn);
        File g = f.getParentFile();
        f = new File(g, data);
        bo = getOutputStream(f.getPath(), cmpr);
        writeBinary(bo, brr);
        bo.close();
        bo = null;
      }
      else {
        if (!data.equals("*")) {
          pw.close();
          pw = null;
          File f = new File(fn);
          File g = f.getParentFile();
          f = new File(g, data);
          pw = getWriter(f.getPath(), chsn, cmpr);
        }
        writeText(pw, brr);
        pw.printf("***%n");
        pw.flush();
        if (!fn.equals("stdout")) pw.close();
        pw = null;
      }
    }
    catch (Exception e) {
      String emsg = e.toString();
      if (CHECK) e.printStackTrace(System.out);
      if (pw != null) {
        try {
          pw.close();
        }
        catch (Exception x) {}
      }
      if (bo != null) {
        try {
          bo.close();
        }
        catch (Exception x) {}
      }
      throw new Exception(String.format("Can't write file \"%s\"%n%s", fn, emsg));
    }
  }

  private void writeText(PrintWriter pw, IBJarr arr) throws Exception {
    arr.printAll(null, pw);
  }

  private void writeBinary(BufferedOutputStream bo, IBJarr arr) throws Exception {
    try {
      arr.setLSBF(!lsbf.equals("0"));
      int na = arr.arrays.size();
      if (na == 0) 
        return;
      AbstractArray aa = arr.arrays.get(0);
      IBJarr.Structure strc = arr.getStructure();
      int[] ff = strc.getFirstIndex();
      int[] ll = strc.getLastIndex();
      int[] ii = (int[]) ff.clone();
      int ns = strc.getDims();
      int i = 0;
      while (i >= 0) {
        for (int k = 0; k < na; k++) {
          if (na > 1) aa = arr.arrays.get(k);
          aa.toBuffer(ii);
          bo.write(aa.bytes, 0, aa.bytes.length);
        }
        for (i = ns - 1; i >= 0; i--) {
          ii[i]++;
          if (ii[i] <= ll[i]) 
            break;
          else {
            ii[i] = ff[i];
          }
        }
      }
    }
    catch (Exception e) {
      String emsg = e.toString();
      if (CHECK) e.printStackTrace(System.out);
      throw new Exception(String.format("Can't write binary data%n%s", emsg));
    }
  }

  private void checkHeader(IBJarr arr, String fn) throws Exception {
    int size = 0;
    IBJhdr hdr = header;
    IBJarr.Structure struc = arr.getStructure();
    hdr.putInteger("dims", struc.getDims(), "%d");
    hdr.putIntegers("lowb", struc.getFirstIndex(), "%d");
    hdr.putIntegers("hghb", struc.getLastIndex(), "%d");
    hdr.putString("file", new File(fn).getName(), true);          //-2008-11-14
    int na = arr.getSize();
    for (int i = 0; i < na; i++) {
      String ni = arr.arrays.get(i).getName();
      if (ni == null) throw new Exception("undefined name");
    }
    for (int i = 0; i < na; i++) {
      String ni = arr.arrays.get(i).getName();
      for (int j = i + 1; j < na; j++) {
        String nj = arr.arrays.get(j).getName();
        if (ni.equals(nj)) throw new Exception("duplicate name");
      }
    }
    StringBuilder sb = new StringBuilder();
    String[] frms = new String[na];
    for (int i = 0; i < na; i++) {
      AbstractArray aa = arr.arrays.get(i);
      frms[i] = aa.desc.format;
      size += aa.desc.length;
      if (aa.valdef != null) sb.append(aa.valdef);
      if (sb.length() > 0 && sb.length() != i + 1)
        throw new Exception("inconsistent definition of valdef");
    }
    hdr.putInteger("size", size, "%d");
    hdr.putStrings("form", frms, true);
    if (sb.length() > 0) hdr.putString("vldf", sb.toString(), true);
    mode = hdr.getString("mode", true);
    if (mode == null || !mode.equals("binary")) {
      mode = "text";
      hdr.putString("mode", mode, true);
    }
    cmpr = hdr.getInteger("cmpr");
    if (cmpr < 0 || cmpr > 9) {
      cmpr = 0;
      hdr.putInteger("cmpr", cmpr, "%d");
    }
    data = hdr.getString("data", true);
    if (data == null) {
      data = "*";
      hdr.putString("data", data, true);
    }
    if (data.equals("*") && (mode.equals("binary") || cmpr > 0)) {
      String n = new File(fn).getName();
      if (n.endsWith(".dmna")) n = n.substring(0, n.length() - 5);
      data = n + ((mode.equals("binary")) ? ".dmnb" : ".dmnt");
      if (cmpr > 0) data += ".gz";
    }
    sequ = hdr.getString("sequ", true);
    //
    if (chsn == null)                                             //-2012-02-03
      chsn = hdr.getString("cset|chsn", true);
    arr.setCharset(chsn);
    //
    lsbf = hdr.getString("lsbf", true);
    if (lsbf == null) {
      lsbf = (ByteOrder.nativeOrder() == ByteOrder.LITTLE_ENDIAN) ? "1" : "0";
      hdr.putString("lsbf", lsbf, false);
    }
    arr.setLSBF(!lsbf.equals("0"));
  }

  public static void writeHeader(PrintWriter pw, String chsn, IBJhdr hdr,
          boolean artm) throws Exception {                        //-2012-02-03
    if (chsn == null)
      chsn = hdr.getString("cset|chsn", true);
    if (chsn == null)
      chsn = Charset.defaultCharset().name();
    String locl = hdr.getLocale();
    if (!IBJhdr.USENEWLOCALES)
      locl = hdr.getLocale().equalsIgnoreCase("de") ? "german" : "C";  //-2012-09-04
    String format = (artm) ? "%s\t%s%n" : "%-8s \"%s\"%n";
    pw.printf(format, "cset", chsn);
    pw.printf(format, "locl", locl);
    pw.printf(format, "tmzn", hdr.getTimeZone());
    format = (artm) ? "%s\t%s%n" : "%-8s %s%n";
    for (String key: hdr.p.keySet()) {
      if (key.equalsIgnoreCase("cset"))                           //-2012-02-03
        continue;
      if (key.equalsIgnoreCase("chsn"))
        continue;
      if (key.equalsIgnoreCase("locl"))
        continue;
      if (key.equalsIgnoreCase("tmzn"))
        continue;
      String value = hdr.p.get(key);
      pw.printf(format, key, value);
    }
    pw.printf("*%n");
    pw.flush();
  }

  /**
   * Reads data files of type DMN. The data might be binary, formatted and/or
   * compressed. All data types supported by {@link IBJarr IBJarr} are accepted.
   * 
   * @param fn the filename including path and extension {@code .dmna}.
   * @return the array read or null if the file type is not supported.
   * @throws java.lang.Exception
   */
  public static IBJarr readDmn(String fn) throws Exception {
    return readDmn(fn, null, false);
  }
  /**
   * Reads data files of type DMN. The data might be binary, formatted and/or
   * compressed. All data types supported by {@link IBJarr IBJarr} are accepted.
   * 
   * @param fn the filename including path and extension {@code .dmna}.
   * @param chsn
   * @param force
   * @return the array read or null if the file type is not supported.
   * @throws java.lang.Exception
   */
  public static IBJarr readDmn(String fn, String chsn, boolean force) throws Exception {
    IBJarr arr = null;
    if (fn.endsWith(".dmna")) {
      fn = fn.substring(0, fn.length() - 5);
      IBJdmn dmn = new IBJdmn();
      dmn.chsn = (chsn == null) ? Charset.defaultCharset().name() : chsn;
      arr = dmn.getDmn(fn, force);
    }
    return arr;
  }

  /**
   * Reads the header of a file of type DMN. If {@code chsn} is set within
   * the header, an appropriate reader is used. 
   * 
   * @param fn the filename including path and extension {@code .dmna}
   * @return the header or null in case of error
   */
  public static IBJhdr readDmnHeader(String fn) {
    return readDmnHeader(fn, null, false);
  }

  /**
   * Reads the header of a file of type DMN. If {@code chsn} is set within
   * the header, an appropriate reader is used. 
   * 
   * @param fn the filename including path and extension {@code .dmna}
   * @param chsn the default character set name to be used
   * @param force if {@code true} ignore {@code chsn} in file header
   * @return the header or null in case of error
   */
  public static IBJhdr readDmnHeader(String fn, String chsn, boolean force) {
    if (chsn == null)
      chsn = Charset.defaultCharset().name();
    if (fn.endsWith(".dmna")) fn = fn.substring(0, fn.length() - 5);
    String gn = fn + ".dmna";
    File f = new File(gn);
    return readDmnHeader(f, chsn, force);
  }

  /**
   * Reads the header of a file of type DMN. If {@code chsn} is set within
   * the header, an appropriate reader is used. 
   * 
   * @param f the file to be read
   * @param chsn the default character set name to be used
   * @param force if {@code true} ignore {@code chsn} in file header
   * @return the header or null in case of error
   */
  public static IBJhdr readDmnHeader(File f, String chsn, boolean force) {
    IBJdmn dmn = new IBJdmn();
    if (chsn == null)
      chsn = Charset.defaultCharset().name();
    LineNumberReader lr = null;
    try {
      lr = dmn.getReader(f.getPath(), chsn, 0);                   //-2012-02-03
      String cn = dmn.readHeader(lr);
      if (!force && cn != null && !cn.equals(chsn)) {
        lr.close();
        dmn.header.p.clear();
        lr = dmn.getReader(f.getPath(), cn, 0);
        dmn.readHeader(lr);
      }
      else
        dmn.header.putString("cset", chsn, true);                 //-2012-09-04
    }
    catch (Exception e) {
      if (CHECK) e.printStackTrace(System.out);
      return null;
    }
    return dmn.header;
  }

  /**
   * Writes data files of type DMN. The data might be binary, formatted and/or
   * compressed. All data types supported by {@link IBJarr IBJarr} can be
   * written. The output is written according to the parameter values in the
   * header of the array (see {@link IBJdmn IBJdmn}).
   *
   * @param arr the array to be written.
   * @param fn the filename including path and extension {@code .dmna}.
   * @param cset
   * @throws java.lang.Exception
   */
  public static void writeDmn(IBJarr arr, String fn, String cset) 
          throws Exception {
    if (fn.endsWith(".dmna")) {
      fn = fn.substring(0, fn.length() - 5);
    }
    IBJdmn dmn = new IBJdmn(cset);                                //-2012-02-03
    dmn.putDmn(arr, fn);
  }

  public static void writeDmn(IBJarr arr, String fn) 
        throws Exception {
    writeDmn(arr, fn, null); 
  }

  /**
   * Writes data files of type DMN. The data might be binary, formatted and/or
   * compressed. All data types supported by {@link IBJarr IBJarr} can be
   * written. The output is written according to the parameter values in the
   * header of the array (see {@link IBJdmn IBJdmn}).
   *
   * @param arr the array to be written.
   * @param fn the filename including path and extension {@code .dmna}.
   * @param cset
   * @param artm use tab as separator in printed header
   * @throws java.lang.Exception
   */
  public static void writeDmn(IBJarr arr, String fn, String cset, boolean artm)
    throws Exception {
    if (fn.endsWith(".dmna")) {
      fn = fn.substring(0, fn.length() - 5);
    }
    IBJdmn dmn = new IBJdmn(cset);                                //-2012-02-03
    dmn.setArtm(artm);
    dmn.putDmn(arr, fn);
  }
  
  /**
   * Adds the dry deposition to the field of wet deposition and 
   * resets the dry deposition field to zero. 
   * 
   * This routine requires the parameter "axes" in the array header,
   * the selection "k=-2," and original LASAT output files.
   * The header entry artp is set to "X".
   * 
   * @param arr  The array
   * @param sel  The selection string
   * @return true if successful, false otherwise
   * @throws Exception
   */
  public static boolean addDry2Wet(IBJarr arr, String sel) 
  throws Exception {
    IBJhdr hdr = arr.getHeader();
    String artp = hdr.getString("artp", true);
    String axes = hdr.getString("axes", true);
    int[] first = arr.getStructure().getFirstIndex();
    int[] last = arr.getStructure().getLastIndex();
    if (sel.contains("k=-2,") &&
        artp != null && artp.equalsIgnoreCase("C") && 
        axes != null && axes.equalsIgnoreCase("xyzs") &&           
        arr.getStructure().getDims() == 4 &&              
        first[2] == -1 && last[2] >= 0 &&
        arr.getSize() > 0 &&
        arr.getSize() < 3) {
      AbstractArray ca = arr.getArray(0);
      AbstractArray da = (arr.getSize() > 1) ? arr.getArray(1) : null;
      int i1 = first[0];
      int i2 = last[0];
      int j1 = first[1];
      int j2 = last[1];
      int l1 = first[3];
      int l2 = last[3];
      for (int i=i1; i<=i2; i++) {
        for (int j=j1; j<=j2; j++) {
          for (int l=l1; l<=l2; l++) {
            float cw = ((IBJarr.FloatArray)ca).get(i, j, -1, l);
            float cd = ((IBJarr.FloatArray)ca).get(i, j, 0, l);
            ((IBJarr.FloatArray)ca).set(cw+cd, i, j, -1, l);
            ((IBJarr.FloatArray)ca).set(0, i, j, 0, l);
            if (da != null) {
              float dw = ((IBJarr.FloatArray)da).get(i, j, -1, l);
              float dd = ((IBJarr.FloatArray)da).get(i, j, 0, l);
              float d = 0;
              if ((cw+cd) != 0)
                d = (float)(Math.sqrt(cw*cw*dw*dw + cd*cd*dd*dd)/(cw+cd));
              ((IBJarr.FloatArray)da).set(d, i, j, -1, l);
              ((IBJarr.FloatArray)da).set(0, i, j, 0, l);
            }                   
          }
        }
      }
      hdr.putString("artp", "X", true);
      return true;
    }
    else
      return false;
  }
  

  /**
   * Adjust the header information according to the 
   * selected index ranges. This routine requires the original axis information.
   * 
   * @param arr The array
   * @param org_axes The original axis information
   * @throws Exception
   */
  public static void adjustHeader(IBJarr arr, String org_axes) throws Exception {
    if (org_axes == null || arr == null || arr == null || !arr.isMapped())
      return;
    IBJarr.Mapping mapping = arr.getMapping();
    IBJhdr hdr = arr.getHeader();
    int dims = arr.getStructure().getDims();
    int[][] mat = mapping.mat;
    int[] first_org = mapping.dst.getFirstIndex();
    int[] lngth = arr.getStructure().getLength();
    //
    // try to handle axes strings that are too long
    if (org_axes.length() > mapping.dst.getLength().length)
      org_axes = org_axes.substring(0,dims);
    //
    // adjust substance-specific header entries according to the
    // selected substances (original axes entry 's')
    if (org_axes.toLowerCase().contains("s")) {
      String[] entries = { "name", "unit|einheit", "refc", "refd", "refv", "frac1mev" };
      boolean[] unquote = { true, true, false, false, false, false };
      int is = org_axes.toLowerCase().indexOf("s");
      int nc = mapping.dst.getLength()[is];
      //
      // get the original indices of the selected substances
      int il;
      for (il=0; il<dims; il++) {
        if (mat[is][il] != 0)
          break;
      }      
      int nl = (il < dims) ? lngth[il] : 1;
      int[] ll = new int[nl];
      int[] first = arr.getStructure().getFirstIndex();
      for (int l=0; l<nl; l++) {
        int[] ii = mapping.map(first);
        ll[l] = ii[is];
        if (il >= dims)
          break;
        first[il]++;
      }
      //
      // adjust the substance-specific header entries
      for (int i=0; i<entries.length; i++) {
        String[] sa = hdr.getStrings(entries[i], unquote[i]);
        String[] saa = entries[i].split("[|]");
        String name = (saa.length > 0) ? saa[0] : entries[i];
        if (sa != null && sa.length == nc) {
          String[] new_sa = new String[nl];
          for (int l=0; l<nl; l++) {
            new_sa[l] = sa[ll[l]-first_org[is]];
          }
          hdr.putStrings(name, new_sa, unquote[i]);               //-2013-08-19
        }
      }      
    }
    //
    // adjust artp if a deposition layer has been selected
    if (org_axes.toLowerCase().contains("z")) {
      int iz = org_axes.toLowerCase().indexOf("z");
      //
      // get the original indices of the selected substances
      int il;
      for (il=0; il<dims; il++) {
        if (mat[iz][il] != 0)
          break;
      }      
      int nl = (il < dims) ? lngth[il] : 1;
      int[] ll = new int[nl];
      int[] first = arr.getStructure().getFirstIndex();
      for (int l=0; l<nl; l++) {
        int[] ii = mapping.map(first);
        ll[l] = ii[iz];
        if (il >= dims)
          break;
        first[il]++;
      }      
      //
      // adjust the artp if necessary
      boolean ddep = false;
      boolean wdep = false;
      boolean conc = false;
      for (int l=0; l<nl; l++) {
        if (ll[l] == -1)
          wdep = true;
        else if (ll[l] == 0)
          ddep = true;
        else if (ll[l] > 0)
          conc = true;
      }
      String artp = hdr.getString("artp", true);
      if (artp != null && artp.equalsIgnoreCase("C") && !conc && nl == 1) {
        if (wdep)
          hdr.putString("artp", "XW", true);
        else if (ddep)
          hdr.putString("artp", "XD", true);       
      }          
    }    
    //
    // adjust axis entry according to the selected index ranges    
    String new_axes = "";
    for (int i=0; i<dims; i++) {
      for (int j=0; j<org_axes.length(); j++) {
        if (mat[j][i] != 0) {
          new_axes += org_axes.substring(j, j+1);
          break;
        }
      }
    }
    hdr.putString("axes", new_axes, true);
  }

  // =========================================================================
  private static class Drop {
    int count; // line count
    boolean active; // use this
    boolean equal; // check on equality
    boolean any; // check on any of the letters;
    String letters; // letters triggering a line drop

    private Drop(String s) throws Exception {
      count = 0;
      active = false;
      equal = false;
      any = false;
      int i = 0;
      if (s == null || s.length() == 0)
        return;
      String t = s;
      if (Character.isDigit(s.charAt(0))) {
        i = s.indexOf(';');
        if (i >= 0) t = s.substring(0, i);
        count = Integer.valueOf(t);
        if (i < 0) return;
        t = s.substring(i + 1);
      }
      if (t.charAt(0) == '=') equal = true;
      else if (t.charAt(0) != '!')
        return;
      if (t.charAt(1) == '[') {
        any = true;
        letters = t.substring(2, t.length() - 1);
      }
      else letters = t.substring(1);
    }

    /**
     * Check whether the input line must be dropped.
     * 
     * @param c
     *          the character read
     * @param pos
     *          position in the line
     * @return 1: drop; 0 : maybe; -1: don't drop
     */
    private int check(char c, int pos) {
      if (!active || letters == null || letters.length() == 0)
        return -1;
      if (any) {
        for (int i = 0; i < letters.length(); i++) {
          char l = letters.charAt(i);
          if (l == '@') {
            if (Character.isLetter(c) == equal)
              return 1;
          }
          if ((letters.indexOf(c) >= 0) == equal)
            return 1;
        }
      }
      else {
        if (pos < 0 || pos >= letters.length())
          return -1;
        if (equal) {
          if (c == letters.charAt(pos)) {
            if (pos == letters.length() - 1)
              return 1;
            else
              return 0;
          }
          else
            return -1;
        }
        else {
          if (c == letters.charAt(pos)) {
            if (pos == letters.length() - 1)
              return -1;
            else
              return 0;
          }
          else
            return 1;
        }
      }
      return -1;
    }
  }

  // =========================================================================
//  private static void test01() {
//    try {
//      String s = "x %[3]( *100)6.1f  n%03x";
//      String[] ss = IBJarr.Descriptor.expandFormat(s);
//      for (int i = 0; i < ss.length; i++)
//        System.out.printf("%d: %s\n", i, ss[i]);
//    }
//    catch (Exception e) {
//      e.printStackTrace();
//    }
//  }
//
//  private static void test02() {
//    System.out.printf("%tF.%<tT", new Date());
//  }
//
//  private static void test03() {
//    try {
//      IBJarr arr = readDmn("/d/test/test-a01.dmna");
//      arr.header.print(out);
//      arr.printAll("data:", out);
//      //
//      arr.header.putString("mode", "text", true);
//      writeDmn(arr, "/d/test/test-a01a.dmna");
//    }
//    catch (Exception e) {
//      e.printStackTrace();
//    }
//  }
//
//  private static void test04() {
//    try {
//      IBJarr arr = readDmn("/d/test/s01.dmna");
//      arr.printInfo(out);
//      arr.header.print(out);
//      arr.printAll("data:", out);
//      //
//      arr.header.putString("mode", "binary", true);
//      arr.header.putInteger("cmpr", 6, "%d");
//      writeDmn(arr, "/d/test/s01a.dmna");
//      IBJarr brr = readDmn("/d/test/s01a.dmna");
//      brr.printInfo(out);
//      brr.header.print(out);
//      brr.printAll("data:", out);
//    }
//    catch (Exception e) {
//      errlog.printf("%s%n", e.toString());
//      System.err.println("error in test04");
//      e.printStackTrace();
//    }
//  }
//
//  private static void test05() {
//    try {
//      IBJarr arr = readDmn("/d/lasat/2.15/gcl/test/j01.dmna");
//      arr.printInfo(out);
//      arr.header.print(out);
//      arr.printAll("data:", out);
//      //
//      arr.header.putString("mode", "text", true);
//      arr.header.putInteger("cmpr", 0, "%d");
//      writeDmn(arr, "/d/lasat/2.15/gcl/test/j01b.dmna");
//      IBJarr brr = readDmn("/d/lasat/2.15/gcl/test/j01b.dmna");
//      brr.printInfo(out);
//      brr.header.print(out);
//      brr.printAll("data:", out);
//    }
//    catch (Exception e) {
//      errlog.printf("%s%n", e.toString());
//      System.err.println("error in test05");
//      e.printStackTrace();
//    }
//  }
//  
//  private static void test06(Object o) {
//    Object i = Array.get(o,0);
//    System.out.println("class="+i.getClass().getCanonicalName());
//    System.out.println("class="+o.getClass().getCanonicalName());
//    System.out.println("class="+(float.class).getCanonicalName());
//    Object x = (float[])o;
//    System.out.println("done");
//    System.exit(0);
//  }
//
//  public static void test01() {
//    try {
//      String fn = "/data/test-IBJdmn.dmna";
//      int nx = 3;
//      int ny = 4;
//      IBJarr arr = new IBJarr("test xyz", nx, ny);
//      arr.setFirstIndex(1, 1);
//      arr.createArray("x%(*10)5.1f");
//      FloatArray fa = (FloatArray)arr.getArray(0);
//      float[][] ff = (float[][])fa.getData();
//      for (int i=0; i<nx; i++)
//        for (int j=0; j<ny; j++)
//          ff[i][j] = (i+1) + 0.1f*(j+1);
//      IBJhdr hdr = arr.getHeader();
//      hdr.putString("sequ", "j-,i+", true);
//      PrintWriter pw = new PrintWriter(new OutputStreamWriter(System.out));
//      arr.printAll("Daten", pw);
//      IBJdmn.writeDmn(arr, fn);
//    }
//    catch (Exception e) {
//      e.printStackTrace();
//    }
//    System.exit(0);
//  }
//
//  private static void test07() {
//    try {
//      String fn = "/data/test/gras/c0006a00.dmna";
//      prn.printf("--- reading file \"%s\"\n", fn);
//      IBJarr org = readDmn(fn);
//      org.printInfo(prn);
//      IBJhdr hdr = org.getHeader();
//      hdr.print(prn);
//      //
//      String selection = "i,j,k=1,l=0";
//      prn.printf("--- selecting \"%s\"\n", selection);
//      IBJarr sel = org.getSelected("Selected Array", selection);
//      sel.printInfo(prn);
//      sel.printMapping("Mapping sel:", prn);
//      sel.getHeader().print(prn);
//      FloatArray fa = ((FloatArray)sel.getArray(0));
//      prn.printf("value(16, 1) = %e\n", fa.get(16,1));
//      //
//      String sequence = "j-,i+";
//      prn.printf("--- sequence \"%s\"\n", sequence);
//      IBJarr out = sel.getSelected("Output Array", sequence);
//      out.printInfo(prn);
//      out.printMapping("Mapping out:", prn);
//      out.getHeader().print(prn);
//      fa = ((FloatArray)out.getArray(0));
//      prn.printf("value(32,16) = %e\n", fa.get(32,16));
//      //
//    }
//    catch (Exception e) {
//      errlog.printf("%s%n", e.toString());
//      System.err.println("error in test07");
//      e.printStackTrace();
//    }
//    System.exit(4);
//  }


  private static void help() {
    prn.printf("IBJdmn:   Input and Output of data arrays, version %s\n",
        version);
    prn.printf("usage:    IBJdmn <path> <option ...>\n");
    prn.printf("<path>    working directory\n");
    prn.printf("<option>  -a<factor>    : scaling factor\n");
    prn.printf("          -C<chsn>      : character set name\n");
    prn.printf("          -c<level>     : compression level (0..9)\n");
    prn.printf("          -d<data>      : name of output data file\n");
    prn.printf("          -f<format>    : format of output data\n");
    prn.printf("          -g<nodata>    : output as \"grid ascii\" with NODATA_value=nodata\n");
    prn.printf("          -i<input>     : name of input file\n");
    prn.printf("          -l<locale>    : \"C\" or \"german\"\n");
    prn.printf("          -m<mode>      : \"binary\" or \"text\"\n");
    prn.printf("          -o<name>      : name of ouput file\n");
    prn.printf("          -p            : print output to screen\n");
    prn.printf("          -S<selection> : selection of data for output\n");
    prn.printf("          -s<sequence>  : index order in formatted output\n");
    prn.printf("          -t<timezone>  : time zone as GMT[+|-]hh:mm\n");
    prn.printf("          -u<options>   : use these options [acmst]" +
            " from input for output\n");
    System.exit(0);
  }

  private static PrintWriter msg, prn;

  private static void vMsg(String s, Object... p) {
    prn.printf(s, p);
    prn.println();
    prn.flush();
    if (msg != null) {
      msg.printf("%tF.%<tT ", new Date());
      msg.printf(s, p);
      msg.println();
      msg.flush();
    }
  }
    
  public static IBJarr readASC(String fn) throws Exception {
    File f = new File(fn);
    if (!f.canRead())
      throw new Exception("Missing input file");
    BufferedReader br = null;
    IBJarr arr = null;
    String line;
    int nrows, ncols;
    double xllcorner, yllcorner, cellsize;
    float nodata_value;
    String nodata_string;
    try {
      br = new BufferedReader(new FileReader(f));
      line = br.readLine();
      if (line.startsWith("ncols"))
        ncols = Integer.parseInt(line.trim().split("[\\s]+")[1]);
      else
        throw new Exception("Parameter \"ncols\" missing or not readable");
      line = br.readLine();
      if (line.startsWith("nrows"))
        nrows = Integer.parseInt(line.trim().split("[\\s]+")[1]);
      else
        throw new Exception("Parameter \"nrows\" missing or not readable");
      line = br.readLine();
      if (line.startsWith("xllcorner"))
        xllcorner = IBJhdr.parseDoubleOf(line.trim().split("[\\s]+")[1]);
      else
        throw new Exception("Parameter \"xllcorner\" missing or not readable");
      line = br.readLine();
      if (line.startsWith("yllcorner"))
        yllcorner = IBJhdr.parseDoubleOf(line.trim().split("[\\s]+")[1]);
      else
        throw new Exception("Parameter \"yllcorner\" missing or not readable");
      line = br.readLine();
      if (line.startsWith("cellsize"))
        cellsize = IBJhdr.parseDoubleOf(line.trim().split("[\\s]+")[1]);
      else
        throw new Exception("Parameter \"cellsize\" missing or not readable");
      line = br.readLine();
      if (line.startsWith("NODATA_value")) {
        nodata_string = line.trim().split("[\\s]+")[1];
        nodata_value = IBJhdr.parseFloatOf(nodata_string);
      }
      else
        throw new Exception("Parameter \"NODATA_value\" missing or not readable");
      //
      arr = new IBJarr("asc", ncols, nrows);
      arr.setFirstIndex(1, 1);
      IBJhdr hdr = arr.getHeader();
      hdr.putString("xmin", shortFFormat(xllcorner), false);
      hdr.putString("ymin", shortFFormat(yllcorner), false);
      hdr.putString("dd", shortFFormat(cellsize), false);
      hdr.putString("NODATA_value", nodata_string, false);
      hdr.putString("file", f.getName(), true);
      //
      arr.createArray("val%12.6e");
      FloatArray fa = (FloatArray)arr.getArray("val");
      float[][] ff = (float[][])fa.getData();
      int nl = 6;
      for (int j=nrows-1; j>=0; j--) {
        line = br.readLine();
        if (line == null) 
          throw new Exception("Unexpected end of file after line " + nl);
        nl++;
        String[] ss = line.trim().split("[\\s]+");
        if (ss.length != ncols)
          throw new Exception("Improper number of items in line " + nl);
        for (int i=0; i<ncols; i++) {
          if (ss[i].equals(nodata_string))
            ff[i][j] = nodata_value;
          else
            ff[i][j] = IBJhdr.parseFloatOf(ss[i]);
        }
      }
    } 
    catch (Exception e) {
      e.printStackTrace(System.out);
      throw new Exception("Can't read file "+fn, e);
    }
    finally {
      try { br.close(); } catch (Exception e) {}
    }
    return arr;
  }

  public void putASC(IBJarr arr, String fn, double noda) throws Exception {
    IBJarr brr = null;
    PrintWriter pw = null;
    double xmin, ymin, delt, xref, yref;
    IBJarr.Structure struct;
    if (arr == null || fn == null)
      throw new Exception("Missing argument in putASC()");
    if (fn.endsWith(".asc"))
      fn = fn.substring(0, fn.length()-4);
    String gn = fn + ".asc";
    IBJhdr hdr = arr.getHeader();
    Locale locale = new Locale(hdr.getLocale());
    header = hdr.getCopy();
    checkHeader(arr, fn);
    //
    // set consistent values in header and array before writing  -2013-07-27
    float hdrfact = header.getFloat("fact");
    arr.setFactor(Float.isNaN(hdrfact) ? 1 : hdrfact);
    //
    if (fn.equals("stdout")) {
      mode = "text";
      data = "*";
      pw = prn;
    }
    else pw = getWriter(gn, chsn, 0);
    if (Double.isNaN(noda))
      noda = -9999;
    //
    if (sequ == null) {
      brr = arr;
    }
    else {
      if (CHECK) System.err.println("writing mapped array");
      brr = arr.getSelected("mapped", sequ);
    }
    struct = brr.getStructure();
    if (struct.getDims() != 2)
      throw new Exception("GRID ASCII requires 2-dimensional array");
    int[] ll = struct.getLength();
    xmin = hdr.getDouble("xmin|x0");
    ymin = hdr.getDouble("ymin|y0");
    delt = hdr.getDouble("delta|delt|dd");
    xref = hdr.getDouble("gakrx|refx");
    if (!Double.isNaN(xref))
      xmin += xref;
    yref = hdr.getDouble("gakry|refy");
    if (!Double.isNaN(yref))
      ymin += yref;
    //
    pw.printf("nrows %10d%n", ll[0]);                             //-2012-02-03
    pw.printf("ncols %10d%n", ll[1]);                             //-2012-02-03
    pw.printf(locale, "xllcorner %s%n", shortFFormat(xmin));
    pw.printf(locale, "yllcorner %s%n", shortFFormat(ymin));
    pw.printf(locale, "cellsize  %s%n", shortFFormat(delt));
    pw.printf("NODATA_value  %s%n", shortFFormat(noda));
    //
    brr.printAll(null, pw);
    //
    pw.flush();
    if (pw != prn)  pw.close();
  }
  
  public void putXYZ(IBJarr arr, String fn) throws Exception {
    IBJarr brr = null;
    PrintWriter pw = null;
    double xmin, ymin, delt, xref, yref;
    IBJarr.Structure struct;
    if (arr == null || fn == null)
      throw new Exception("Missing argument in putXYZ()");
    if (fn.endsWith(".xyz"))
      fn = fn.substring(0, fn.length()-4);
    String gn = fn + ".xyz";
    IBJhdr hdr = arr.getHeader();
    Locale locale = new Locale(hdr.getLocale());
    header = hdr.getCopy();
    checkHeader(arr, fn);
    //
    // set consistent values in header and array before writing  -2013-07-27
    float hdrfact = header.getFloat("fact");
    arr.setFactor(Float.isNaN(hdrfact) ? 1 : hdrfact);
    //
    if (fn.equals("stdout")) {
      mode = "text";
      data = "*";
      pw = prn;
    }
    else pw = getWriter(gn, chsn, 0);
    writeHeader(pw, chsn, header, artm);
    //
    if (sequ == null) {
      brr = arr;
    }
    else {
      if (CHECK) System.err.println("writing mapped array");
      brr = arr.getSelected("mapped", sequ);
    }
    struct = brr.getStructure();
    if (struct.getDims() > 3)
      throw new Exception("XYZ requires at most 3-dimensional array");
    
    brr.printAllXYZ(null, pw, ' ');
    pw.flush();
    if (pw != prn)  pw.close();
  }

  private static String shortFFormat(double f) {
    String s = String.format(Locale.ENGLISH, "%f", f);
    int id = s.indexOf('.');
    if (id < 0)
      return s;
    int i = 0;
    for (i=s.length()-1; i>=id; i--)
      if (s.charAt(i) != '0' && s.charAt(i) != '.')
        break;
    s = s.substring(0, i+1);
    return s;
  }

  public void putCSV(IBJarr arr, String fn) throws Exception {
    IBJarr brr = null;
    PrintWriter pw = null;
    IBJarr.Structure structure;
    if (arr == null || fn == null)
      throw new Exception("Missing argument in putCSV()");
    if (fn.endsWith(".csv"))
      fn = fn.substring(0, fn.length()-4);
    String gn = fn + ".csv";
    IBJhdr hdr = arr.getHeader();
    header = hdr.getCopy();
    checkHeader(arr, fn);
    if (fn.equals("stdout")) {
      mode = "text";
      data = "*";
      pw = prn;
    }
    else pw = getWriter(gn, chsn, 0);
    //
    if (sequ == null) {
      brr = arr;
    }
    else {
      if (CHECK) System.err.println("writing mapped array");
      brr = arr.getSelected("mapped", sequ);
    }
    structure = brr.getStructure();
    int ns = structure.getDims();
    if (ns != 2)
      throw new Exception("CSV requires 2-dimensional array");
    int na = brr.getSize();
    if (na != 1)
      throw new Exception("CSV requires simple data elements");    
    Map<String, String> map = header.getMap();
    String locl = header.getLocale();
    if (!IBJhdr.USENEWLOCALES)
      locl = hdr.getLocale().equalsIgnoreCase("de") ? "german" : "C";  //-2012-09-04
    pw.printf("cset;\"%s\"%n", chsn);
    pw.printf("locl;\"%s\"%n", locl);
    pw.printf("tmzn;\"%s\"%n", header.getTimeZone());
    for (String key: map.keySet()) {
      if (key.equalsIgnoreCase("cset"))                           //-2012-02-03
        continue;
      if (key.equalsIgnoreCase("chsn"))
        continue;
      if (key.equalsIgnoreCase("locl"))
        continue;
      if (key.equalsIgnoreCase("tmzn"))
        continue;
      String[] ss = header.getStrings(key, false);
      pw.printf("%s", key);
      for (String s : ss) {
        pw.printf(";%s", s);
      }
      pw.println();
    }
    pw.printf("*%n");
    //
    // set consistent values in header and array before writing  -2013-07-27
    float hdrfact = header.getFloat("fact");
    brr.setFactor(Float.isNaN(hdrfact) ? 1 : hdrfact);            //-2013-08-19
    //
    //
    if (sequ == null)
      sequ = "i+,j+";
    pw.printf("\"%s\"", sequ);
    brr.printAll(null, pw, ';');
    //
    pw.flush();
    if (pw != prn)  pw.close();
  }

  /**
   * The main method simulates the behavior of the C-program {@code IBJdata}.
   * It can be used to read and write/print DMN-files with change of 
   * representation (binary/formatted/compressed) and order/selection of 
   * elements. The usage is:<br>
   * <pre>
    usage:    IBJdmn <i>path</i> <i>option ...</i>
    <i>path</i>      working directory
    <i>option</i>    -a<i>factor</i>    : scaling factor
              -C<i>chsn</i>      : character set name
              -c<i>level</i>     : compression level (0..9)
              -d<i>data</i>      : name of output data file
              -f<i>format</i>    : format of output data
              -g<i>nodata</i>    : output as "grid ascii" with NODATA_value=nodata
              -i<i>input</i>     : name of input file
              -l<i>locale</i>    : "C" or "german"
              -m<i>mode</i>      : "binary" or "text"
              -o<i>name</i>      : name of output file
              -p          : print output to screen
              -S<i>selection</i> : selection of data for output
              -s<i>sequence</i>  : index order in formatted output
              -t<i>timezone</i>  : time zone as GMT[+|-]hh:mm
              -u<i>options</i>   : use these options [cmst] from input for output
   </pre>
   * 
   * @param args the calling parameter and options.
   */
  public static void main(String[] args) {
    prn = new PrintWriter(new OutputStreamWriter(System.out), true);
    int n;
    String Path = null;
    IBJhdr hdr;
    IBJarr arr, out;
    try {
      boolean list_format, arcinfo, printing;
      int verbose, cmpr;
      double fact;
      String s;
      String onam = null, data = null, inam = null, fn = null;
      String form = null, sequ = null, mode = null, locl = null, use = null;
      String tmzn = null, chsn = null;
      String gras = null;
      String selection = null;
      double noda = Double.NaN;
      if (args.length == 0) help();
      onam = "test";
      cmpr = -1;
      printing = false;
      verbose = 1;
      list_format = false;
      arcinfo = false;
      fact = Double.NaN;
      for (n = 0; n < args.length; n++) {
        s = args[n];
        if (s.charAt(0) == '-' && s.length() > 1) {
          String t = s.substring(2).trim();
          switch (s.charAt(1)) {
            case 'A':
              arcinfo = true;
              break;
            case 'a':
              fact = IBJhdr.parseDoubleOf(t);  //-2012-09-04
              break;
            case 'C':
              chsn = t;
              break;
            case 'c':
              cmpr = Integer.valueOf(t);
              break;
            case 'd':
              data = t;
              break;
            case 'f':
              form = t;
              break;
            case 'g':
              gras = t;
              break;
            case 'h':
              help();
              break;
            case 'i':
              inam = t;
              break;
            case 'L':
              list_format = true;
              break;
            case 'l':
              locl = t;
              break;
            case 'm':
              mode = t;
              break;
            case 'o':
              onam = t;
              break;
            case 'p':
              printing = true;
              break;
            case 'S':
              selection = t;
              break;
            case 's':
              sequ = t;
              break;
            case 't':
              tmzn = t;
              break;
            case 'u':
              use = t;
              break;
            case 'v':
              if (t.length() > 0)
                verbose = Integer.valueOf(t);
              break;
            default:
              ;
          }
        }
        else Path = s;
      }
      if (Path != null) {
        Path = Path.replace('\\', '/');
      }
      else Path = "./";
      File f = new File(Path, "IBJdmn.log");
      msg = new PrintWriter(new FileOutputStream(f), true);
      vMsg("IBJdmn: Input and Output of data arrays, version %s of %s",
        version, last_change);
      vMsg("Copyright (C) Janicke Consulting, Überlingen, Germany, 1998-2018");
      for (int i = 0; i < n; i++)
        msg.printf(">%s\n", args[i]);
      if (inam != null) {
        if (!inam.endsWith(".dmna")) inam += ".dmna";
        f = new File(Path, inam);
        arr = readDmn(f.getPath());
        if (verbose > 0)
          vMsg("file \"%s\" read", f);
        hdr = arr.getHeader();
        if (verbose > 1)
          vMsg("IN  header:\n%s\n", hdr.getList());
        if (use != null) {
          if (cmpr < 0 && use.indexOf('c') >= 0) cmpr = hdr.getInteger("cmpr");
          if (mode == null && use.indexOf('m') >= 0)
            mode = hdr.getString("mode", true);
          if (sequ == null && use.indexOf('s') >= 0)
            sequ = hdr.getString("sequ", true);
          if (tmzn == null && use.indexOf('t') >= 0)
            tmzn = hdr.getString("tmzn", true);
          /*                                          commented, uj -2012-08-03
          if (Double.isNaN(fact) && use.indexOf('a') >= 0)
            hdr.getDouble("fact");
            */
        }
        hdr.rename("cmpr", null);
        hdr.rename("mode", null);
        hdr.rename("sequ", null);
        hdr.rename("tmzn", null);
        //hdr.rename("fact", null);                   commented, uj -2012-08-03
        if (list_format)
          vMsg("IN structure:\n%s\n", arr.listInfo());
      }
      else {
        arr = new IBJarr("test-array", 3, 3, 5);
        arr.setFirstIndex(1, 2, 0);
        hdr = arr.getHeader();
        FloatArray fa = (FloatArray) arr.createArray("ff%8.2f");
        for (int i = 1; i <= 3; i++)
          for (int j = 2; j <= 4; j++)
            for (int k = 0; k <= 4; k++)
              fa.set(100 * i + 10 * j + k, i, j, k);
        form = "%8.2f";
      }
      if (printing) {
        mode = "text";
        fn = "stdout";
      }
      else {
        if (onam == null) {
          vMsg("*** error: No output file name given.\n");
          return;
        }
        fn = new File(Path, onam).getPath();
      }
      if (locl == null) locl = "C";
      hdr.setLocale(locl);
      //
      if (form != null) {
        out = arr.reform(form, true);
        hdr = arr.getHeader();
      }
      else
        out = arr;
      arr = null;
      if (selection != null) {
        out = out.getSelected(out.getName(), selection);
        IBJarr.Mapping mapping = out.getMapping();
//        out = out.getCopy(out.getName());
        hdr = out.getHeader();
        String artp = hdr.getString("artp", true);
        //
        // adjust new header for selected substances                -2010-01-29
        //
        if (artp != null
          && (artp.equalsIgnoreCase("C") || artp.toUpperCase().startsWith("X")) //-2012-10-22 
          && mapping.dst.getDims() == 4
          && mapping.dst.getFirstIndex()[3] == 0) {
          String axes = hdr.getString("axes", true);
          String[] name = hdr.getStrings("name", true);
          String[] unit = hdr.getStrings("unit", true);
          String[] refc = hdr.getStrings("refc", false);
          String[] refd = hdr.getStrings("refd", false);
          int nc = mapping.dst.getLength()[3];
          int dims = out.getStructure().getDims();
          int il;
          int[][] mat = mapping.mat;
          for (il=0; il<dims; il++) {
            if (mat[3][il] != 0)
              break;
          }
          int[] first = out.getStructure().getFirstIndex();
          int[] lngth = out.getStructure().getLength();
          int nl = (il < dims) ? lngth[il] : 1;
          int[] ll = new int[nl];
          for (int l=0; l<nl; l++) {
            int[] ii = mapping.map(first);
            ll[l] = ii[3];
            if (il >= dims)
              break;
            first[il]++;
          }
          if (name != null && name.length == nc) {
            String[] new_name = new String[nl];
            for (int l=0; l<nl; l++)
              new_name[l] = name[ll[l]];
            hdr.putStrings("name", new_name, true);
          }
          if (unit != null && unit.length == nc) {
            String[] new_unit = new String[nl];
            for (int l=0; l<nl; l++)
              new_unit[l] = unit[ll[l]];
            hdr.putStrings("unit", new_unit, true);
          }
          if (refc != null && refc.length == nc) {
            String[] new_refc = new String[nl];
            for (int l=0; l<nl; l++)
              new_refc[l] = refc[ll[l]];
            hdr.putStrings("refc", new_refc, false);
          }
          if (refd != null && refd.length == nc) {
            String[] new_refd = new String[nl];
            for (int l=0; l<nl; l++)
              new_refd[l] = refd[ll[l]];
            hdr.putStrings("refd", new_refd, false);
          }
          //
          if (axes != null && axes.length() == 4) {
            String new_axes = "";
            for (int i=0; i<dims; i++) {
              for (int j=0; j<4; j++) {
                if (mat[j][i] != 0) {
                  new_axes += axes.substring(j, j+1);
                  break;
                }
              }
            }
            hdr.putString("axes", new_axes, true);
          }
        }
        //------------------------------------------------------------------
      }
      //
      if (locl != null) {
        hdr.putString("locl", locl, true);
      }
      if (mode != null) {
        hdr.putString("mode", mode, true);
      }
      if (data != null) {
        hdr.putString("data", data, true);
      }
      if (sequ != null) {
        hdr.putString("sequ", sequ, true);
      }
      if (form != null) {
        hdr.putString("form", form, true);
      }
      if (cmpr >= 0) {
        hdr.putInteger("cmpr", cmpr, "%d");
      }
      if (tmzn != null) {
        hdr.putString("tmzn", tmzn, true);
      }
      if (chsn != null) {
        hdr.putString("chsn", chsn, true);
      }
      if (!Double.isNaN(fact)) {
        hdr.putDouble("fact", fact, "%12.4e");
        out.setFactor((float)fact);                               //-2012-08-03
      }
      if (list_format) vMsg("OUT structure:\n%s", out.listInfo());
      if (verbose > 1) vMsg("OUT header:\n%s", hdr.getList());
      if (gras != null) {
        noda = IBJhdr.parseDoubleOf(gras);
        IBJdmn dmn = new IBJdmn();
        dmn.putASC(out, fn, noda);
      }
      else {
        writeDmn(out, fn);
      }
      vMsg("file \"%s\" written", fn);
      vMsg("program finished");
    }
    catch (Exception e) {
      e.printStackTrace(System.out);
      vMsg("*** error:\n%s\n", e.toString());
    }
  }
}

/**
 * 2012-02-12 lj  Note on handling encoding information.
 * 
 * The encoding is only used for input/output. The binary data are in unicode
 * independent of the encoding used in a file.
 * 
 * A DMN file contains the information about the character set that has been 
 * used for encoding text data in the
 * header key "cset" (the use of "chsn" should be avoided). This information
 * is returned by readHeader() and not put into the map of key/value pairs. An
 * alternative encoding is provided, which is used if no encoding is specified
 * in the header file or the method is forced to ignore the specified
 * encoding.
 * 
 * After reading the header the calling method getDmn() or readDmnHeader()
 * saves the encoding actually used in the header map with key "cset". In
 * addition the encoding is saved in IBJdmn.chsn if present.
 * 
 * The header is written by the static method writeHeader() which gets the
 * encoding as a separate parameter. Actually this method cannot influence the
 * encoding because the PrintWriter to be used is given. Unfortunately the 
 * encoding used by the PrintWriter cannot be determined. Therefore, an
 * additional parameter is necessary to provide this information.
 * 
 * If the parameter "chsn" given to writeHeader() is null, this method tries
 * to get it from the header itself (key "cset"). If this key is not provided
 * the default character set is used. writeHeader() writes the used encoding
 * into the file and skips a definition of "cset" or "chsn" within the header
 * (if present).
 * 
 */

/**
 * 
 * 2013-07-27 uj  Note on handling factors.
 * 
 * In IBJarr, there is a container factor and a component factor (fact). 
 * At init() and reform(), the component factor is reset to the format factor 
 * of the component, or, if a format factor is not defined, to the container
 * factor.
 * 
 * The container factor is set by setFactor(), called by IBJdmn.getDmn():
 * 
 *    analyseHeader()   : extract a factor f from the header (default 1)
 *    createContainer() : create the container and setFactor(f)
 *    
 * The value of the container factor may be reset to unity in course of
 * array conversions (on purpose or by chance). As a consequence, an entry
 * "fact" in the DMN header and the component factor that has been reset to 
 * the container factor may become inconsistent.
 * 
 * Therefore just before writing (DMN/CSV/ASC), setFactor() is used with 
 * the current "fact" entry in the header to update those component factors for
 * which no format factor is defined.
 * 
 */
