/**
 * Plume Rise Model PLURIS
 *
 * For theory see: 
 * Janicke, U., Janicke, L.: A three-dimensional plume rise model for
 * dry and wet plumes, Atmospheric Environment 35 (2000), 877-890.
 *
 * Copyright (C) Janicke Consulting, Überlingen, Germany, 2015-2020
 * email: info@janicke.de
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 * 
 */

package de.janicke.pluris;

public class IBJplurisHeader {

  public static String  ID;   // Kurzinformation
  public static double  HQ;   // Höhe der Quelle (m)
  public static double  DQ;   // Durchmesser der Quelle (m)
  public static double  UQ;   // Austrittsgeschwindigkeit (m/s)
  public static double  TQ;   // Temperatur der Abluft (C)
  public static double  QQ;   // Wärmestrom der Quelle (MW)
  public static double  ZQ;   // Wasserbeladung (überschreibt sq, rq, lq) (kg/kg trocken)
  public static double  SQ;   // Spezifischer Wasserdampfgehalt (überschreibt rq) (kg/kg)
  public static double  RQ;   // Relative Feuchte (1)
  public static double  LQ;   // Spezifischer Flüssigwassergehalt (kg/kg)
  public static double  A1;   // Austrittswinkel gegen die Horizontale (Grad)
  public static double  A2;   // Austrittswinkel gegen x-Achse (West-Ost) gegen Uhrzeigersinn (Grad)
  public static double  CQ;   // Anfangskonzentration (ME/m3)
  public static double  FQ;   // Quadrat der Froudezahl beim Quellaustritt (überschreibt tq) (1)
  public static double  IQ;   // Turbulenzintensität (1)
  public static double  XQ;   // X-Koordinate der Quelle/Mittelpunkt (m)
  public static double  YQ;   // Y-Koordinate der Quelle/Mittelpunkt (m)
  public static double  RF;   // Expliziter Reduktionsfaktor für die Überhöhung (überschreibt stacktip downwash)
  public static double  FC;   // Coriolis-Parameter (1/s)
  public static double  CA;   // Konzentration in der Umgebung (ME/m3)
  public static double  DA;   // Windrichtung gegen Nord im Uhrzeigersinn (Grad)
  public static double  EF;   // Reduktionsfaktor für die Entrainment-Funktion (1)
  public static double  TA;   // Temperatur in Anemometerhöhe (C)
  public static double  UA;   // Windgeschwindigkeit in Anemometerhöhe (m/s)
  public static double  LM;   // Obukhov-Länge (m)
  public static int     KI;   // Klug/Manier-Klasse
  public static double  HA;   // Anemometerhöhe (nur für lm!=0) (m)
  public static double  HM;   // Mischungsschichthöhe (nur für lm!=0) (m)
  public static double  Z0;   // Rauhigkeitslänge (nur für lm!=0) (m)
  public static double  D0;   // Verdrängungshöhe (nur für lm!=0) (m)
  public static double  TX;   // Temperaturgradient (nur für lm=0)
  public static double  TH;   // Anemometerhöhe der Temperatur (m)
  public static double  UX;   // Geschwindigkeitsexponent (nur für lm=0)
  public static double  UH;   // Anemometerhöhe der Geschwindigkeit (nur für lm=0) (m)
  public static double  IA;   // Turbulenzintensität (1)
  public static int     NZ;   // Anzahl der vertikalen Stützpunkte
  public static double  PA;   // Standarddruck in der untersten Meßhöhe (Pa)
  public static double  RA;   // Relative Feuchte in Höhe th (1)
  public static double  SA;   // Spezifische Feuchte in Höhe th (überschreibt ra) (kg/kg)
  public static double  VS;   // Sedimentationsgeschwindigkeit (m/s)
  public static double  ZA;   // Vertikale Schrittweite zur Erzeugung der Profile (m)
  public static double  US;   // Schubspannungsgeschwindigkeit (nur für lm!=0) (m/s)
  public static int     OO;   // Ausgabemodus (0: rise, 1: std, 2: std/sc, 3: outline, 4: outline/sc)
  public static double  SC;   // Skalierung (m)
  public static double  SD;   // Skalierte Rechen-Schrittweite (sc)
  public static double  SE;   // Skalierter Endpunkt (sc)
  public static int     SN;   // Anzahl der Ausgabeschritte

  public static String[] PrmSections = { "q", "a", "c" };

  public static String[][] PrmParams = {
    { "id", "q" },
    { "hq", "q" },
    { "dq", "q" },
    { "uq", "q" },
    { "tq", "q" },
    { "qq", "q" },
    { "zq", "q" },
    { "sq", "q" },
    { "rq", "q" },
    { "lq", "q" },
    { "a1", "q" },
    { "a2", "q" },
    { "cq", "q" },
    { "fq", "q" },
    { "iq", "q" },
    { "xq", "q" },
    { "yq", "q" },
    { "rf", "q" },
    { "fc", "a" },
    { "ca", "a" },
    { "da", "a" },
    { "ef", "a" },
    { "ta", "a" },
    { "ua", "a" },
    { "lm", "a" },
    { "ki", "a" },
    { "ha", "a" },
    { "hm", "a" },
    { "z0", "a" },
    { "d0", "a" },
    { "tx", "a" },
    { "th", "a" },
    { "ux", "a" },
    { "uh", "a" },
    { "ia", "a" },
    { "nz", "a" },
    { "pa", "a" },
    { "ra", "a" },
    { "sa", "a" },
    { "vs", "a" },
    { "za", "a" },
    { "us", "a" },
    { "oo", "c" },
    { "sc", "c" },
    { "sd", "c" },
    { "se", "c" },
    { "sn", "c" }
  };

  public static void PrmSetDefaults() {
    ID    = "Test";
    HQ    = 10.00;
    DQ    = 1.0000;
    UQ    = 10.00;
    TQ    = 10.00;
    QQ    = -1.0000;
    ZQ    = 0.000000;
    SQ    = 0.000000;
    RQ    = 0.00;
    LQ    = 0.000000;
    A1    = 90.00;
    A2    = 0.00;
    CQ    = 1.0000e+00;
    FQ    = -1.0000e+00;
    IQ    = 0.1000;
    XQ    = 0.00;
    YQ    = 0.00;
    RF    = 1.00;
    FC    = 1.1000e-04;
    CA    = 1.0000e+00;
    DA    = 270.00;
    EF    = 1.00;
    TA    = 10.00;
    UA    = 3.00;
    LM    = 0.0;
    KI    = 0;
    HA    = 13.00;
    HM    = 0.00;
    Z0    = 0.5000;
    D0    = 3.0000;
    TX    = -0.00978;
    TH    = 2.00;
    UX    = 0.0000;
    UH    = 10.00;
    IA    = 0.100;
    NZ    = 0;
    PA    = 101300.0;
    RA    = 0.70;
    SA    = 0.0000;
    VS    = 0.0000;
    ZA    = 10.00;
    US    = 0.00000;
    OO    = 1;
    SC    = 1.0000;
    SD    = 0.0100;
    SE    = 1000.0000;
    SN    = 100;
  }

  public static Object PrmSetValue(String name, String value) throws Exception {
    Object ov = null;
    if (name.equalsIgnoreCase("id")) {
      ID = value;
      ov = value;
    }
    else if (name.equalsIgnoreCase("hq")) {
      HQ = Double.parseDouble(value.trim().replace(',', '.'));
      ov = HQ;
    }
    else if (name.equalsIgnoreCase("dq")) {
      DQ = Double.parseDouble(value.trim().replace(',', '.'));
      ov = DQ;
    }
    else if (name.equalsIgnoreCase("uq")) {
      UQ = Double.parseDouble(value.trim().replace(',', '.'));
      ov = UQ;
    }
    else if (name.equalsIgnoreCase("tq")) {
      TQ = Double.parseDouble(value.trim().replace(',', '.'));
      ov = TQ;
    }
    else if (name.equalsIgnoreCase("qq")) {
      QQ = Double.parseDouble(value.trim().replace(',', '.'));
      ov = QQ;
    }
    else if (name.equalsIgnoreCase("zq")) {
      ZQ = Double.parseDouble(value.trim().replace(',', '.'));
      ov = ZQ;
    }
    else if (name.equalsIgnoreCase("sq")) {
      SQ = Double.parseDouble(value.trim().replace(',', '.'));
      ov = SQ;
    }
    else if (name.equalsIgnoreCase("rq")) {
      RQ = Double.parseDouble(value.trim().replace(',', '.'));
      ov = RQ;
    }
    else if (name.equalsIgnoreCase("lq")) {
      LQ = Double.parseDouble(value.trim().replace(',', '.'));
      ov = LQ;
    }
    else if (name.equalsIgnoreCase("a1")) {
      A1 = Double.parseDouble(value.trim().replace(',', '.'));
      ov = A1;
    }
    else if (name.equalsIgnoreCase("a2")) {
      A2 = Double.parseDouble(value.trim().replace(',', '.'));
      ov = A2;
    }
    else if (name.equalsIgnoreCase("cq")) {
      CQ = Double.parseDouble(value.trim().replace(',', '.'));
      ov = CQ;
    }
    else if (name.equalsIgnoreCase("fq")) {
      FQ = Double.parseDouble(value.trim().replace(',', '.'));
      ov = FQ;
    }
    else if (name.equalsIgnoreCase("iq")) {
      IQ = Double.parseDouble(value.trim().replace(',', '.'));
      ov = IQ;
    }
    else if (name.equalsIgnoreCase("xq")) {
      XQ = Double.parseDouble(value.trim().replace(',', '.'));
      ov = XQ;
    }
    else if (name.equalsIgnoreCase("yq")) {
      YQ = Double.parseDouble(value.trim().replace(',', '.'));
      ov = YQ;
    }
    else if (name.equalsIgnoreCase("rf")) {
      RF = Double.parseDouble(value.trim().replace(',', '.'));
      ov = RF;
    }
    else if (name.equalsIgnoreCase("fc")) {
      FC = Double.parseDouble(value.trim().replace(',', '.'));
      ov = FC;
    }
    else if (name.equalsIgnoreCase("ca")) {
      CA = Double.parseDouble(value.trim().replace(',', '.'));
      ov = CA;
    }
    else if (name.equalsIgnoreCase("da")) {
      DA = Double.parseDouble(value.trim().replace(',', '.'));
      ov = DA;
    }
    else if (name.equalsIgnoreCase("ef")) {
      EF = Double.parseDouble(value.trim().replace(',', '.'));
      ov = EF;
    }
    else if (name.equalsIgnoreCase("ta")) {
      TA = Double.parseDouble(value.trim().replace(',', '.'));
      ov = TA;
    }
    else if (name.equalsIgnoreCase("ua")) {
      UA = Double.parseDouble(value.trim().replace(',', '.'));
      ov = UA;
    }
    else if (name.equalsIgnoreCase("lm")) {
      LM = Double.parseDouble(value.trim().replace(',', '.'));
      ov = LM;
    }
    else if (name.equalsIgnoreCase("ki")) {
      KI = Integer.parseInt(value.trim());
      ov = KI;
    }
    else if (name.equalsIgnoreCase("ha")) {
      HA = Double.parseDouble(value.trim().replace(',', '.'));
      ov = HA;
    }
    else if (name.equalsIgnoreCase("hm")) {
      HM = Double.parseDouble(value.trim().replace(',', '.'));
      ov = HM;
    }
    else if (name.equalsIgnoreCase("z0")) {
      Z0 = Double.parseDouble(value.trim().replace(',', '.'));
      ov = Z0;
    }
    else if (name.equalsIgnoreCase("d0")) {
      D0 = Double.parseDouble(value.trim().replace(',', '.'));
      ov = D0;
    }
    else if (name.equalsIgnoreCase("tx")) {
      TX = Double.parseDouble(value.trim().replace(',', '.'));
      ov = TX;
    }
    else if (name.equalsIgnoreCase("th")) {
      TH = Double.parseDouble(value.trim().replace(',', '.'));
      ov = TH;
    }
    else if (name.equalsIgnoreCase("ux")) {
      UX = Double.parseDouble(value.trim().replace(',', '.'));
      ov = UX;
    }
    else if (name.equalsIgnoreCase("uh")) {
      UH = Double.parseDouble(value.trim().replace(',', '.'));
      ov = UH;
    }
    else if (name.equalsIgnoreCase("ia")) {
      IA = Double.parseDouble(value.trim().replace(',', '.'));
      ov = IA;
    }
    else if (name.equalsIgnoreCase("nz")) {
      NZ = Integer.parseInt(value.trim());
      ov = NZ;
    }
    else if (name.equalsIgnoreCase("pa")) {
      PA = Double.parseDouble(value.trim().replace(',', '.'));
      ov = PA;
    }
    else if (name.equalsIgnoreCase("ra")) {
      RA = Double.parseDouble(value.trim().replace(',', '.'));
      ov = RA;
    }
    else if (name.equalsIgnoreCase("sa")) {
      SA = Double.parseDouble(value.trim().replace(',', '.'));
      ov = SA;
    }
    else if (name.equalsIgnoreCase("vs")) {
      VS = Double.parseDouble(value.trim().replace(',', '.'));
      ov = VS;
    }
    else if (name.equalsIgnoreCase("za")) {
      ZA = Double.parseDouble(value.trim().replace(',', '.'));
      ov = ZA;
    }
    else if (name.equalsIgnoreCase("us")) {
      US = Double.parseDouble(value.trim().replace(',', '.'));
      ov = US;
    }
    else if (name.equalsIgnoreCase("oo")) {
      OO = Integer.parseInt(value.trim());
      ov = OO;
    }
    else if (name.equalsIgnoreCase("sc")) {
      SC = Double.parseDouble(value.trim().replace(',', '.'));
      ov = SC;
    }
    else if (name.equalsIgnoreCase("sd")) {
      SD = Double.parseDouble(value.trim().replace(',', '.'));
      ov = SD;
    }
    else if (name.equalsIgnoreCase("se")) {
      SE = Double.parseDouble(value.trim().replace(',', '.'));
      ov = SE;
    }
    else if (name.equalsIgnoreCase("sn")) {
      SN = Integer.parseInt(value.trim());
      ov = SN;
    }
    return ov;
  }

  public static String Unquote(String s) {
    if (s == null) return null;
    int l = s.length();
    if (l < 1 || s.charAt(0) != '\"') return s;
    if (s.charAt(l - 1) == '\"') l--;
    String t = (l > 1) ? s.substring(1, l) : "";
    return t;
  }

  public static String PrmGetValue(String name) throws Exception {
    String s = null;
    if (name.equalsIgnoreCase("id"))
      s = "\""+String.format("%-20s", Unquote(ID)).trim()+"\"";
    else if (name.equalsIgnoreCase("hq"))
      s = String.format("%6.2f", HQ);
    else if (name.equalsIgnoreCase("dq"))
      s = String.format("%8.4f", DQ);
    else if (name.equalsIgnoreCase("uq"))
      s = String.format("%6.2f", UQ);
    else if (name.equalsIgnoreCase("tq"))
      s = String.format("%6.2f", TQ);
    else if (name.equalsIgnoreCase("qq"))
      s = String.format("%8.4f", QQ);
    else if (name.equalsIgnoreCase("zq"))
      s = String.format("%8.6f", ZQ);
    else if (name.equalsIgnoreCase("sq"))
      s = String.format("%8.6f", SQ);
    else if (name.equalsIgnoreCase("rq"))
      s = String.format("%6.2f", RQ);
    else if (name.equalsIgnoreCase("lq"))
      s = String.format("%8.6f", LQ);
    else if (name.equalsIgnoreCase("a1"))
      s = String.format("%6.2f", A1);
    else if (name.equalsIgnoreCase("a2"))
      s = String.format("%6.2f", A2);
    else if (name.equalsIgnoreCase("cq"))
      s = String.format("%12.4e", CQ);
    else if (name.equalsIgnoreCase("fq"))
      s = String.format("%12.4e", FQ);
    else if (name.equalsIgnoreCase("iq"))
      s = String.format("%8.4f", IQ);
    else if (name.equalsIgnoreCase("xq"))
      s = String.format("%10.2f", XQ);
    else if (name.equalsIgnoreCase("yq"))
      s = String.format("%10.2f", YQ);
    else if (name.equalsIgnoreCase("rf"))
      s = String.format("%6.2f", RF);
    else if (name.equalsIgnoreCase("fc"))
      s = String.format("%12.4e", FC);
    else if (name.equalsIgnoreCase("ca"))
      s = String.format("%12.4e", CA);
    else if (name.equalsIgnoreCase("da"))
      s = String.format("%6.2f", DA);
    else if (name.equalsIgnoreCase("ef"))
      s = String.format("%6.2f", EF);
    else if (name.equalsIgnoreCase("ta"))
      s = String.format("%6.2f", TA);
    else if (name.equalsIgnoreCase("ua"))
      s = String.format("%6.2f", UA);
    else if (name.equalsIgnoreCase("lm"))
      s = String.format("%8.1f", LM);
    else if (name.equalsIgnoreCase("ki"))
      s = String.format("%1d", KI);
    else if (name.equalsIgnoreCase("ha"))
      s = String.format("%6.2f", HA);
    else if (name.equalsIgnoreCase("hm"))
      s = String.format("%6.2f", HM);
    else if (name.equalsIgnoreCase("z0"))
      s = String.format("%6.4f", Z0);
    else if (name.equalsIgnoreCase("d0"))
      s = String.format("%6.4f", D0);
    else if (name.equalsIgnoreCase("tx"))
      s = String.format("%8.5f", TX);
    else if (name.equalsIgnoreCase("th"))
      s = String.format("%6.2f", TH);
    else if (name.equalsIgnoreCase("ux"))
      s = String.format("%6.4f", UX);
    else if (name.equalsIgnoreCase("uh"))
      s = String.format("%6.2f", UH);
    else if (name.equalsIgnoreCase("ia"))
      s = String.format("%8.3f", IA);
    else if (name.equalsIgnoreCase("nz"))
      s = String.format("%2d", NZ);
    else if (name.equalsIgnoreCase("pa"))
      s = String.format("%8.1f", PA);
    else if (name.equalsIgnoreCase("ra"))
      s = String.format("%6.2f", RA);
    else if (name.equalsIgnoreCase("sa"))
      s = String.format("%8.4f", SA);
    else if (name.equalsIgnoreCase("vs"))
      s = String.format("%8.4f", VS);
    else if (name.equalsIgnoreCase("za"))
      s = String.format("%6.2f", ZA);
    else if (name.equalsIgnoreCase("us"))
      s = String.format("%8.5f", US);
    else if (name.equalsIgnoreCase("oo"))
      s = String.format("%1d", OO);
    else if (name.equalsIgnoreCase("sc"))
      s = String.format("%12.4f", SC);
    else if (name.equalsIgnoreCase("sd"))
      s = String.format("%12.4f", SD);
    else if (name.equalsIgnoreCase("se"))
      s = String.format("%12.4f", SE);
    else if (name.equalsIgnoreCase("sn"))
      s = String.format("%12d", SN);
    return s;
  }

}