package de.janicke.graph;

import com.lowagie.text.Document;
import com.lowagie.text.Rectangle;
import com.lowagie.text.pdf.DefaultFontMapper;
import com.lowagie.text.pdf.PdfContentByte;
import com.lowagie.text.pdf.PdfTemplate;
import com.lowagie.text.pdf.PdfWriter;

import java.awt.AlphaComposite;
import java.awt.BasicStroke;
import java.awt.Color;
import java.awt.Composite;
import java.awt.Container;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.Image;
import java.awt.RenderingHints;
import java.awt.Shape;
import java.awt.Stroke;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.awt.event.MouseListener;
import java.awt.geom.AffineTransform;
import java.awt.geom.GeneralPath;
import java.awt.geom.Path2D;
import java.awt.geom.PathIterator;
import java.awt.geom.Point2D;
import java.awt.geom.Rectangle2D;
import java.awt.image.BufferedImage;
import java.io.File;
import java.io.FileOutputStream;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Comparator;
import java.util.Locale;
import javax.imageio.ImageIO;
import javax.swing.JFileChooser;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JMenuItem;
import javax.swing.JPanel;
import javax.swing.JPopupMenu;

/**
 * <p>Copyright 2005-2007 Ing.-B�ro Janicke, 26427 Dunum</p>
 * 
 * <p>Sub-routine Graph for creating pdf graphs</p>
 * 
 * @author Janicke Consulting, Ulf Janicke
 */

//====================================================================

public class Graph extends JLabel {
  
  private static final long serialVersionUID = 1L;
  
  //=======================================================================
  
  public interface Paintable {
    
    public void paintGraph(Graphics2D g2, AffineTransform transform, String action);
    
  }
  
  //=======================================================================

  public static class Diagram {

    public Font fntStandard = new Font("SansSerif", Font.PLAIN, 10);
    public Locale locale = Locale.ENGLISH;

    public double[] sizes = { 
        10, // font size
        6, // left border
        2, // right border
        2.5, // top border
        3.5, // bottom border
        3, // color scale
        140, // standard width
        1.0, // stroke width
        1.4  // header relative size
    };

    public class Axis {
      String format;
      String label;
      double tick;
      int type;
      Color color;
    }

    private class ColorScale {
      Color[] colors;
      String format;
      String label;
      double length;
      int type;
      double[] values;
      double xr;
      double yr;
    }
    
    private double width, height;
    private ArrayList<Axis> axes = new ArrayList<Axis>();
    private ArrayList<Element> ground = new ArrayList<Element>();
    private ArrayList<Element> overlay = new ArrayList<Element>();
    private ArrayList<Symbol> symbols = new ArrayList<Symbol>();
    private ArrayList <Bar> bars = new ArrayList<Bar>();
    public boolean bTitleAdjust = false;
    private ColorScale cscale;
    private Graph.Map map;
    private Rectangle2D scale;
    private String title;
    private int unit = UNIT_MM;
    private Rectangle2D view;
    private Graph graph;

    public Diagram(double width, double height) {
      this.width = width;
      this.height = height;
    }
    
    public Axis addAxis(int type, double tick, String format, String label) {
      return addAxis(type, tick, format, label, Color.black);
    }
    
    public Axis addAxis(int type, double tick, String format, String label, Color clr) {
      Axis a = new Axis();
      a.type = type;
      a.tick = tick;
      a.format = format;
      a.label = label;
      a.color = clr;
      axes.add(a);
      return a;
    }

    public void addGround(Graph.Element g) {
      if (ground == null) ground = new ArrayList<Element>();
      ground.add(g);
    }

    public void addOverlay(Graph.Element g) {
      if (overlay == null) overlay = new ArrayList<Element>();
      overlay.add(g);
    }

    public void addSymbol(Graph.Symbol s) {
      if (symbols == null) symbols = new ArrayList<Symbol>();
      symbols.add(s);
    }

   public void addBar(Graph.Bar b) {
      if (bars == null) bars = new ArrayList<Bar>();
      bars.add(b);
    }
   
    public void addPath(Path2D path, Color color, Stroke stroke) {
      Element ge = new Element();
      ge.color = color;
      ge.stroke = stroke;
      ge.add(path);
      addOverlay(ge);
    }


    public Graph createGraph(Graph g) {
      double szFont = sizes[0];
      double sz = convert(szFont, UNIT_PT, UNIT_MM);
      double br = sizes[2]*sz;
      double bl = sizes[1]*sz;
      double bt = sizes[3]*sz;
      double bb = sizes[4]*sz;
      double lw = sizes[6];
      double lh = lw;
      double db = sizes[5]*sz;
      double vx = 0;
      double vy = 0;
      double vw = 0;
      double vh = 0;
      if (cscale == null)
        db = 0;
      else if (cscale.label != null) db += sz;
      if (view != null) {
        vx = view.getX();
        vy = view.getY();
        vw = view.getWidth();
        vh = view.getHeight();
      }
      if (width <= 0) {
        if (vw <= 0) {
          width = lw;
          vw = width - br - bl;
          vx = bl;
        } else {
          if (vx > 0)
            bl = vx;
          else
            vx = bl;
          width = bl + vw + br;
        }
      } else {
        if (vw <= 0) {
          vw = width - br - bl;
          vx = bl;
        } else {
          if (vx > 0)
            bl = vx;
          else
            vx = bl;
        }
      }
      if (scale != null) {
        if (vh <= 0 && height <= 0 && scale.getWidth() != 0
            && scale.getHeight() != 0) {
          vh = vw * scale.getHeight() / scale.getWidth();
          if (vh <= 0 || vh > 3 * vw) vh = 0;
        }
      }
      if (height <= 0) {
        if (vh <= 0) {
          height = lh + db;
          vy = bb + db;
          vh = height - bt - bb - db;
        } else {
          if (vy > 0)
            bb = vy - db;
          else
            vy = bb + db;
          height = db + bb + vh + bt;
        }
      }
      if (vh <= 0) vh = height - bt - bb - db;
      if (vy <= 0) vy = bb + db;
      view = new Rectangle2D.Double(vx, vy, vw, vh);
      //
      if (g != null) {
        graph = g;
        graph.clear();
        graph.init(width, height, unit);
      } 
      else
        graph = new Graph(width, height, unit);
      graph.font = fntStandard.deriveFont((float)szFont);
      graph.locale = locale;
      Graph.Element root = graph.getRoot();
      root.color = Color.BLACK;
      root.stroke = new BasicStroke(1f);
      Graph.Element ge = graph.createElement();
      ge.setView(view);
      ge.setScale(scale);
      ge.setClip();
      if (ground != null) {
        for (Element ground1 : ground) {
          ge.add(ground1);
        }
      }
      if (bars != null) {
        for (Bar bar : bars) {
          ge.add(bar);
        }
      }
      if (symbols != null) {
        for (Symbol symbol : symbols) {
          ge.add(symbol);
        }
      }
      if (map != null) ge.add(map);
      root.add(ge);
      if (title != null) {
        Font f;  
        Graph.Label lbl;
        if (bTitleAdjust) {
          f = graph.font;
          lbl = new Graph.Label(vx, vy + vh,
            title, f);
          lbl.alignment = ALIGN_LEFT + ALIGN_BOTTOM;
        }
        else {
          f = graph.font.deriveFont((float)sizes[8] * graph.font.getSize2D());
          lbl = new Graph.Label(vx + 0.5 * vw, vy + vh + sz / 2,
            title, f);
          lbl.alignment = ALIGN_XCENTER + ALIGN_BOTTOM;
        }
        root.add(lbl);
      }
      for (Axis a : axes) {
        Graph.Element axis = graph.createAxis(ge, a.type, a.tick, a.format,
                a.label);
        axis.color = a.color;
        axis.font = graph.font;
        axis.stroke = new BasicStroke((float)sizes[7]);
        root.add(axis);
      }
      
      if (overlay != null) {
        ge = new Graph.Element(ge);
        for (Element overlay1 : overlay) {
          ge.add(overlay1);
        }
        root.add(ge);
      }
      if (cscale != null) {
        if (cscale.label != null) {
          db -= sz;
          Graph.Label lbl = new Graph.Label(vx, db, cscale.label);
          lbl.alignment = ALIGN_LEFT + ALIGN_BOTTOM;
          root.add(lbl);
        }
        Graph.Element skala = graph.makeColorScale(HORIZONTAL, vx, db, vw,
            cscale.colors, cscale.values, cscale.format);
        skala.color = Color.BLACK;
        skala.font = graph.font;
        skala.stroke = new BasicStroke((float)sizes[7]);
        root.add(skala);
      }
      return graph;
    }

    /**
     * @return
     */
    public double getHeight() {
      return height;
    }

    /**
     * @return
     */
    public Rectangle2D getScale() {
      return scale;
    }

    /**
     * @return
     */
    public String getTitle() {
      return title;
    }

    /**
     * @return
     */
    public int getUnit() {
      return unit;
    }

    /**
     * @return
     */
    public Rectangle2D getView() {
      return view;
    }

    /**
     * @return
     */
    public double getWidth() {
      return width;
    }
    
    private ColorScale getColorScale() {
      return cscale;
    }

    public void setColorScale(Color[] colors, double[] values, String format,
        String label) {
      if (values == null) {
        cscale = null;
        return;
      }
      if (colors == null) colors = Graph.colorSet10;
      ColorScale s = new ColorScale();
      s.colors = colors;
      s.values = values;
      s.format = format;
      s.label = label;
      cscale = s;
    }

    /**
     * @param d
     */
    public void setHeight(double d) {
      height = d;
    }

    public void setMap(Graph.Map map) {
      this.map = map;
    }

    public void setScale(double x, double y, double width, double height) {
      scale = new Rectangle2D.Double(x, y, width, height);
    }

    public void setScale(Rectangle2D r) {
      scale = r;
    }

    public void setTitle(String s) {
      title = s;
    }
    
    public void setTitle(String s, boolean b) {
      title = s;
      bTitleAdjust = b;
    }

    /**
     * @param i
     */
    public void setUnit(int i) {
      unit = i;
    }

    public void setView(double x, double y, double width, double height) {
      view = new Rectangle2D.Double(x, y, width, height);
    }

    public void setView(Rectangle2D r) {
      view = r;
    }

    /**
     * @param d
     */
    public void setWidth(double d) {
      width = d;
    }

  }

  //====================================================================

  public static class Element {
    public String action;
    Shape clip;
    public Color color;
    ArrayList<Object> elements;
    public Font font;
    public String name;
    public int rule;
    Rectangle2D.Double scale;
    public Stroke stroke;
    AffineTransform transform;
    Rectangle2D.Double view;

    public Element() {
    }

    public Element(String name) {
      this.name = name;
    }

    public Element(Element e) {
      this.transform = (AffineTransform) e.transform.clone();
      this.color = e.color;
      this.stroke = e.stroke;
      this.action = e.action;
      this.font = e.font;
      if (e.view != null) this.view = (Rectangle2D.Double) e.view.clone();
      if (e.scale != null) this.scale = (Rectangle2D.Double) e.scale.clone();
      if (e.clip != null) this.clip = e.clip;
    }

    public void add(Object o) {
      if (elements == null) 
        elements = new ArrayList<Object>();
      elements.add(o);
    }
    
    public void insert(Object o, int index) {
      if (elements == null) 
        elements = new ArrayList<Object>();
      int ne = elements.size();
      if (index < 0)
        index = 0;
      else if (index > ne)
        index = ne;
      elements.add(index, o);
    }

    public void clear() {
      elements = null;
    }

    public ArrayList<Object> getElements() {
      return elements;
    }

    public Rectangle2D.Double getView() {
      return view;
    }

    public Rectangle2D.Double getScale() {
      return scale;
    }

    public void paint(Graphics g) {
      paint(g, null);
    }

    public void paint(Graphics g, AffineTransform at) {
      if (elements == null) return;
      if (at == null) at = new AffineTransform();
      Graphics2D g2 = (Graphics2D) g;
//      System.out.printf("Element.paint()\n");
      //
      // saved objects
      //
      AffineTransform transform2 = g2.getTransform();
      Font font2 = g2.getFont();
      Color color2 = g2.getColor();
      Stroke stroke2 = g2.getStroke();
      Composite composite2 = g2.getComposite();
      Shape clip2 = g2.getClip();
      //
      if (font != null) {
        g2.setFont(font);
      }
      if (color != null) {
        g2.setColor(color);
      }
      if (stroke != null) {
        g2.setStroke(stroke);
      }
      if (rule > 0) {
        try {
          g2.setComposite(AlphaComposite.getInstance(rule));
        } catch (Exception e) {}
      }
      String act = this.action;
      if (act == null) act = "draw";
      AffineTransform trf = new AffineTransform(at);
      if (clip != null) {
        g2.clip(trf.createTransformedShape(clip));
      }
      if (this.transform != null) trf.concatenate(this.transform);
      AffineTransform atm = new AffineTransform(trf);
      //
      AffineTransform transform3 = g2.getTransform();
      Font font3 = g2.getFont();
      Color color3 = g2.getColor();
      Stroke stroke3 = g2.getStroke();
      Composite composite3 = g2.getComposite();
      Shape clip3 = g2.getClip();
      int n = elements.size();
      for (int i = 0; i < n; i++) {
        Object o = elements.get(i);
        if (o == null) 
          continue;
        if (o instanceof Paintable) { //---------------------------- Paintable
//          System.out.printf("%d: painting Paintable\n", i);
          ((Paintable) o).paintGraph(g2, trf, act);
        }
        else if (o instanceof Element) { //--------------------------- Element
//          System.out.printf("%d: painting Element\n", i);
          ((Element) o).paint(g, trf);
        }
        else if (o instanceof Shape) { //--------------------------- Shape
//          System.out.printf("%d: painting Shape\n", i);
          Shape shape = (Shape)o;
//          System.out.println("Bounds="+shape.getBounds2D());
          Shape s = trf.createTransformedShape((Shape) o);
//          System.out.println("Bounds="+s.getBounds2D());
         if (act.equals("fill"))
            g2.fill(s);
          else
            g2.draw(s); 
        } 
        else {
          System.out.println("unknown graphics element: " + o.getClass());
        }
        trf = new AffineTransform(atm);
        g2.setTransform(transform3);
        g2.setColor(color3);
        g2.setFont(font3);
        g2.setStroke(stroke3);
        g2.setClip(clip3);
        g2.setComposite(composite3);
      }
      g2.setTransform(transform2);
      g2.setClip(clip2);
      g2.setColor(color2);
      g2.setFont(font2);
      g2.setStroke(stroke2);
      g2.setComposite(composite2);
    }

    public void setClip() {
      clip = view;
    }

    public void setClip(Rectangle2D r) {
      if (r == null)
        clip = null;
      else {
        if (transform == null)
          clip = (Shape) r.clone();
        else
          clip = transform.createTransformedShape(r);
      }
    }

    public void setScale(Rectangle2D r) {
      if (r == null)
        scale = null;
      else {
        scale = new Rectangle2D.Double();
        scale.setRect(r);
      }
      setTransform();
    }

    public void setTransform() {
      if (view == null || scale == null) return;
      transform = new AffineTransform();
      transform.translate(view.x, view.y);
      transform.scale(view.width / scale.width, view.height / scale.height);
      transform.translate(-scale.x, -scale.y);
    }

    public void setTransform(AffineTransform at) {
      if (at != null) {
        transform = new AffineTransform(at);
        return;
      }
      transform = null;
      setTransform();
    }

    public void setView(Rectangle2D r) {
      boolean setc = (clip != null) ? clip.equals(view) : false;
      if (r == null)
        view = null;
      else {
        view = new Rectangle2D.Double();
        view.setRect(r);
      }
      if (setc) clip = view;
      setTransform();
    }

    public boolean writePDF(File f, double factor) {
      boolean success = false;
      float w = (float)Math.round(view.width*factor);
      float h = (float)Math.round(view.height*factor);

      // step 1: creation of a document-object
      Document document = new Document(new Rectangle(w, h), 0, 0, 0, 0);

      try {

        // step 2:
        // we create a writer that listens to the document
        // and directs a PDF-stream to a file
        PdfWriter writer = PdfWriter.getInstance(document,
            new FileOutputStream(f));
        writer.setPdfVersion(PdfWriter.VERSION_1_4);

        // step 3: we open the document
        document.open();

        // step 4: we grab the ContentByte and do some stuff with it
        // we create a fontMapper and read all the fonts in the font directory
        DefaultFontMapper mapper = new DefaultFontMapper();

        // we create a template and a Graphics2D object that corresponds with it
        PdfContentByte cb = writer.getDirectContent();
        PdfTemplate tp = cb.createTemplate(w, h);
        Graphics2D g2 = tp.createGraphics(w, h, mapper);
        tp.setWidth(w);
        tp.setHeight(h);
        g2.transform(
          createTransform(view, new Rectangle2D.Double(0, h, w, -h)));
        paint(g2);
        g2.dispose();
        cb.addTemplate(tp, 0, 0);
        success = true;
      } catch (Exception exc) {
        exc.printStackTrace(System.out);
      } finally {
        document.close();
      }
      return success;
    }
    
    public boolean writePNG(File file, double factor) {
      boolean success = false;
      try {
        int w = (int) Math.round(view.width*factor);
        int h = (int) Math.round(view.height*factor);
        BufferedImage bi = new BufferedImage(w, h, BufferedImage.TYPE_INT_ARGB);
        Graphics2D g2 = bi.createGraphics();
        g2.setPaint(Color.WHITE);
        g2.fillRect(0, 0, w, h);
        g2.transform(
            createTransform(view, new Rectangle2D.Double(0, h, w, -h)));
        paint(g2);
        ImageIO.write(bi, "png", file);
        success = true;
      }
      catch (Exception e) {
        e.printStackTrace(System.out);
      }
      return success;
    }
    
    public boolean write(File f, double factor) {
      String nm = f.getName();
      if (nm.endsWith(".pdf"))
        return writePDF(f, factor);
      else if (nm.endsWith(".png"))
        return writePNG(f, factor);
      else
        return false;
    }
    
  }

  //=====================================================================================

  public static class IsoLine {

    //-------------------------------------------------------------------

    private final class IsoCompare implements Comparator<Object> {

      @Override
      public int compare(Object o1, Object o2) {
        IsoLocation l1 = null, l2 = null;
        if (o1 instanceof IsoConnection)
          l1 = ((IsoConnection) o1).from;
        else
          l1 = (IsoLocation) o1;
        if (o2 instanceof IsoConnection)
          l2 = ((IsoConnection) o2).from;
        else
          l2 = (IsoLocation) o2;
        if (l1.ident < l2.ident) return -1;
        if (l1.ident > l2.ident) return 1;
        return 0;
      }
    } //---------------------------------------------------------------

    //---------------------------------------------------------------

    private final class IsoConnection {

      private IsoLocation from, to;
      private boolean isVirtual;

      public IsoConnection(IsoLocation from, IsoLocation to, boolean isVirtual) {
        this.from = from;
        this.to = to;
        this.isVirtual = isVirtual;
      }

      @Override
      public String toString() {
        return from.toString() + ((isVirtual) ? " move to " : " line to ")
            + to.toString();
      }
    }

    //--------------------------------------------------------------

    private final class IsoLocation {

      int ident;

      public IsoLocation(int i, boolean onHor, int j, boolean onVer) {
        ident = 2 * i;
        if (onHor) ident++;
        ident = (ident << 16) + 2 * j;
        if (onVer) ident++;
      }

      @Override
      public String toString() {
        int i = ident >> 16;
        boolean onHor = ((i & 0x01) != 0);
        i >>= 1;
        int j = ident & 0x0000ffff;
        boolean onVer = ((j & 0x01) != 0);
        j >>= 1;
        return "(" + i + ((onHor) ? "+, " : "., ") + j
            + ((onVer) ? "+)" : ".)");
      }

    }

    public static boolean DEBUG = false;

    public static void printPath(Path2D gp) {
      float[] pc = new float[6];
      String form = " %7.2f %7.2f\n";
      PathIterator pi = gp.getPathIterator(new AffineTransform());
      while (!pi.isDone()) {
        int type = pi.currentSegment(pc);
        if (type == PathIterator.SEG_CLOSE)
          System.out.println("close");
        else {
          if (type == PathIterator.SEG_MOVETO) System.out.print("move to ");
          if (type == PathIterator.SEG_LINETO) System.out.print("line to ");
          System.out.printf(form, pc[0], pc[1]);
        }
        pi.next();
      }
    }
    boolean[][] isOut;

    int ni, nj;
    double[] xp, yp;
    double[][] xx, yy, vv;

    public IsoLine(double[] xp, double[] yp, double[][] vv) {
      this.xp = xp;
      this.yp = yp;
      this.vv = vv;
      init();
    }

    public IsoLine(double[][] vv) {
      this.vv = vv;
      init();
    }

    public Path2D getBorder() {
      Path2D gp = new Path2D.Double();
      gp.moveTo(xx[0][0], yy[0][0]);
      for (int i = 1; i <= ni; i++)
        gp.lineTo(xx[i][0], yy[i][0]);
      for (int j = 1; j <= nj; j++)
        gp.lineTo(xx[ni][j], yy[ni][j]);
      for (int i = ni - 1; i >= 0; i--)
        gp.lineTo(xx[i][nj], yy[i][nj]);
      for (int j = nj - 1; j >= 0; j--)
        gp.lineTo(xx[0][j], yy[0][j]);
      gp.closePath();
      return gp;
    }

    public Path2D getLine(double v) {
      return getLine(v, null);
    }

    public Path2D getLine(double v, Path2D go) {
      ArrayList<IsoConnection> connections = new ArrayList<IsoConnection>();
      IsoLocation from = null;
      IsoLocation to = null;
      //
      // check all valid cells
      //
      for (int ip = 1; ip <= ni; ip++) {
        int im = ip - 1;
        for (int jp = 1; jp <= nj; jp++) {
          int jm = jp - 1;
          if (isOut[ip][jp]) continue;
          double v00 = vv[im][jm] - v;
          double v01 = vv[im][jp] - v;
          double v10 = vv[ip][jm] - v;
          double v11 = vv[ip][jp] - v;
          if (v00 > 0) { //  . .
            //  + .
            if (v10 <= 0) { //  . .
              //  + -
              from = new IsoLocation(im, true, jm, false);
              if (v01 <= 0) { //  - .
                //  + -
                to = new IsoLocation(im, false, jm, true);
                connections.add(new IsoConnection(from, to, false));
                if (v11 > 0) { //  - +
                  //  + -
                  from = new IsoLocation(im, true, jp, false);
                  to = new IsoLocation(ip, false, jm, true);
                  connections.add(new IsoConnection(from, to, false));
                }
              } else { //  + .
                //  + -
                if (v11 > 0)
                  to = new IsoLocation(ip, false, jm, true);
                else
                  to = new IsoLocation(im, true, jp, false);
                connections.add(new IsoConnection(from, to, false));
              }
            } else { //  . .
              //  + +
              if (v01 > 0) { // + .
                // + +
                if (v11 <= 0) {
                  from = new IsoLocation(ip, false, jm, true);
                  to = new IsoLocation(im, true, jp, false);
                  connections.add(new IsoConnection(from, to, false));
                }
              } else { //  - .
                //  + +
                to = new IsoLocation(im, false, jm, true);
                if (v11 > 0)
                  from = new IsoLocation(im, true, jp, false);
                else
                  from = new IsoLocation(ip, false, jm, true);
                connections.add(new IsoConnection(from, to, false));
              }
            }
          } else { //  . .
            //  - .
            if (v10 > 0) { //  . .
              //  - +
              to = new IsoLocation(im, true, jm, false);
              if (v11 <= 0) { //  . -
                //  - +
                from = new IsoLocation(ip, false, jm, true);
                connections.add(new IsoConnection(from, to, false));
                if (v01 > 0) {
                  from = new IsoLocation(im, false, jm, true);
                  to = new IsoLocation(im, true, jp, false);
                  connections.add(new IsoConnection(from, to, false));
                }
              } else { //  . +
                //  - +
                if (v01 <= 0)
                  from = new IsoLocation(im, true, jp, false);
                else
                  from = new IsoLocation(im, false, jm, true);
                connections.add(new IsoConnection(from, to, false));
              }
            } else { //  . .
              //  - -
              if (v01 > 0) { //  + .
                //  - -
                from = new IsoLocation(im, false, jm, true);
                if (v11 > 0)
                  to = new IsoLocation(ip, false, jm, true);
                else
                  to = new IsoLocation(im, true, jp, false);
                connections.add(new IsoConnection(from, to, false));
              } else { //  - .
                //  - -
                if (v11 > 0) {
                  from = new IsoLocation(im, true, jp, false);
                  to = new IsoLocation(ip, false, jm, true);
                  connections.add(new IsoConnection(from, to, false));
                }
              }
            }
          }
        }
      }
      //
      // check vertical lines for borders
      //
      double v0, v1;
      IsoLocation il0, il1;
      for (int i = 0; i <= ni; i++) {
        for (int jp = 1; jp <= nj; jp++) {
          int jm = jp - 1;
          if (isOut[i][jp] == isOut[i + 1][jp]) continue;
          v0 = vv[i][jm] - v;
          v1 = vv[i][jp] - v;
          if (v0 > 0) {
            il0 = new IsoLocation(i, false, jm, false);
            if (v1 > 0)
              il1 = new IsoLocation(i, false, jp, false);
            else
              il1 = new IsoLocation(i, false, jm, true);
          } else {
            if (v1 <= 0) continue;
            il0 = new IsoLocation(i, false, jm, true);
            il1 = new IsoLocation(i, false, jp, false);
          }
          if (isOut[i][jp])
            connections.add(new IsoConnection(il1, il0, true));
          else
            connections.add(new IsoConnection(il0, il1, true));
        }
      }
      //
      // check horizontal lines for borders
      //
      for (int j = 0; j <= nj; j++) {
        for (int ip = 1; ip <= ni; ip++) {
          int im = ip - 1;
          if (isOut[ip][j] == isOut[ip][j + 1]) continue;
          v0 = vv[im][j] - v;
          v1 = vv[ip][j] - v;
          if (v0 > 0) {
            il0 = new IsoLocation(im, false, j, false);
            if (v1 > 0)
              il1 = new IsoLocation(ip, false, j, false);
            else
              il1 = new IsoLocation(im, true, j, false);
          } else {
            if (v1 <= 0) continue;
            il0 = new IsoLocation(im, true, j, false);
            il1 = new IsoLocation(ip, false, j, false);
          }
          if (isOut[ip][j])
            connections.add(new IsoConnection(il0, il1, true));
          else
            connections.add(new IsoConnection(il1, il0, true));
        }
      }
      //
      //  connect the pieces
      //
      if (connections.isEmpty()) return null;
      Path2D gp = new Path2D.Double();
      IsoConnection[] ica = new IsoConnection[1];
      ica = (IsoConnection[]) connections.toArray(ica);
      IsoCompare icm = new IsoCompare();
      Arrays.sort(ica, icm);

      String form = " %5.1f";
      if (DEBUG) {
        for (int j = nj; j >= 0; j--) {
          for (int i = 0; i <= ni; i++)
            System.out.printf(form, vv[i][j]);
          System.out.println();
        }
        for (int i = 0; i < ica.length; i++)
          System.out.println(i + " : " + ica[i].toString());
      }

      boolean start = true;
//      IsoLocation il = null;
      IsoConnection ic = null;
      Point2D.Float p;
      if (DEBUG) System.out.println("ica.length=" + ica.length);
      for (IsoConnection ica1 : ica) {
        if (start) {
          ic = null;
          for (IsoConnection ica2 : ica) {
            ic = ica2;
            if (ic.to != null) break;
            ic = null;
          }
          if (ic == null) {
            System.out.println("start not found");
            return null;
          }
          p = getPoint(ic.from, v);
          if (DEBUG)
            System.out.printf("start @ (" + form + ", " + form + ")\n",
                    p.x, p.y);
          gp.moveTo(p.x, p.y);
          if (go != null) go.moveTo(p.x, p.y);
          start = false;
        }
        p = getPoint(ic.to, v);
        if (ic.isVirtual) {
          if (DEBUG)
            System.out.printf("move to (" + form + ", " + form + ")\n",
                    p.x, p.y);
          gp.moveTo(p.x, p.y);
        } else {
          if (DEBUG)
            System.out.printf("line to (" + form + ", " + form + ")\n",
                    p.x, p.y);
          gp.lineTo(p.x, p.y);
        }
        if (go != null) go.lineTo(p.x, p.y);
        int m = Arrays.binarySearch(ica, ic.to, icm);
        ic.to = null;
        if (m < 0 || ica[m].to == null) {
          start = true;
          if (DEBUG) System.out.println("close");
          if (go != null) go.closePath();
        } else
          ic = ica[m];
      }
      return gp;
    }

    private Point2D.Float getPoint(IsoLocation l, double v) {
      if (vv == null || xx == null || yy == null) return null;
      int i = l.ident >> 16;
      boolean onHor = ((i & 0x01) != 0);
      i >>= 1;
      int j = l.ident & 0x0000ffff;
      boolean onVer = ((j & 0x01) != 0);
      j >>= 1;
      double x = xx[i][j];
      double y = yy[i][j];
      if (onHor) {
        double a = (v - vv[i][j]) / (vv[i + 1][j] - vv[i][j]);
        x += a * (xx[i + 1][j] - xx[i][j]);
        y += a * (yy[i + 1][j] - yy[i][j]);
      } else if (onVer) {
        double b = (v - vv[i][j]) / (vv[i][j + 1] - vv[i][j]);
        x += b * (xx[i][j + 1] - xx[i][j]);
        y += b * (yy[i][j + 1] - yy[i][j]);
      }
      return new Point2D.Float((float) x, (float) y);
    }

    private void init() {
      if (vv == null) return;
      ni = vv.length - 1;
      nj = vv[0].length - 1;
      if (xx == null) {
        xx = new double[ni + 1][nj + 1];
        if (xp == null) {
          xp = new double[ni + 1];
          for (int i = 0; i <= ni; i++)
            xp[i] = i;
        }
        for (int i = 0; i <= ni; i++) {
          for (int j = 0; j <= nj; j++) {
            xx[i][j] = xp[i];
          }
        }
      }
      if (yy == null) {
        yy = new double[ni + 1][nj + 1];
        if (yp == null) {
          yp = new double[nj + 1];
          for (int j = 0; j <= nj; j++)
            yp[j] = j;
        }
        for (int i = 0; i <= ni; i++) {
          for (int j = 0; j <= nj; j++) {
            yy[i][j] = yp[j];
          }
        }
      }
      if (isOut == null) {
        isOut = new boolean[ni + 2][nj + 2];
        for (int i = 0; i <= ni + 1; i++) {
          isOut[i][0] = true;
          isOut[i][nj + 1] = true;
        }
        for (int j = 1; j <= nj; j++) {
          isOut[0][j] = true;
          isOut[ni + 1][j] = true;
        }
      }
    }
  }

  //====================================================================

  public static class Label implements Paintable {
    public int alignment = 1;
    // 0..3: bottom, base, center, top
    // 0, 4, 8: left, center, right
    public double angle;
    public Font f;
    public Point2D p;
    public String t;
    
    int inset = 0;
    Color bgColor = null;
    boolean framed = false;
    boolean shifted = false;
    Rectangle2D region;

    public Label(double x, double y, String t) {
      this(x, y, t, null);
    }

    public Label(double x, double y, String t, Font f) {
      p = new Point2D.Double(x, y);
      this.t = t;
      this.f = f;
    }
    
    public void setBorder(int inset, Color bgColor, boolean framed) {
      this.inset = inset;
      this.bgColor = bgColor;
      this.framed = framed;
    }
    
    public void setShifted(boolean shifted, Rectangle2D region) {
      this.shifted = shifted;
      this.region = region;
    }

    @Override
    public void paintGraph(Graphics2D g2, AffineTransform transform, String action) {
      if (t == null) 
        return;
      if (region != null && !region.contains(p))
        return;
      Point2D q = transform.transform(p, null);
      double xq = q.getX();
      double yq = q.getY();
      if (this.f != null) {
        g2.setFont(this.f);
      }
      Font fnt = g2.getFont();
      Rectangle2D r = fnt.getStringBounds(t, g2.getFontRenderContext());
      double x = r.getX();
      double y = r.getY();
      double w = r.getWidth();
      double h = r.getHeight();
      int xalign = (alignment / 4) % 3;
      int yalign = alignment % 4;
      double rx = w + 2*inset;
      double ry = h + 2*inset;
      double dsp = 0.5*ry;
      if (shifted) {  // restricted positioning
        if (yalign == 1 || yalign == 2 || xalign == 1)
          shifted = false;
        else if (region != null) {
          Rectangle2D rr = transform.createTransformedShape(region).getBounds2D();
          if (xalign == 0 && xq + rx > rr.getMaxX()) 
            xalign = 2;
          else if (xalign == 2 && xq - rx < rr.getMinX())
            xalign = 0;
          if (yalign == 0 && yq - ry - dsp < rr.getMinY())
            yalign = 3;
          else if (yalign == 3 && yq + ry + dsp > rr.getMaxY())
            yalign = 0;
        }
      }
      double dx = 0;
      switch (xalign) {
        case 1 :
          dx = 0.5 * w;
          break;
        case 2 :
          dx = w;
          break;
        default :
          ;
      }
      double dy = 0;
      switch (yalign) {
        case 0 :
          dy = y + h;
          break;
        case 1 :
          dy = 0;
          break;
        case 2 :
          dy = y + 0.5 * h;
          break;
        case 3 :
          dy = y;
        default :
          ;
      }
      g2.translate(xq, yq);
      if (angle != 0) 
        g2.rotate(-Math.toRadians(angle));
      //
      double xs = -dx;  // string origin
      double ys = -dy;
      if (framed || bgColor!=null || inset>0) {
        Color c2 = null;
        if (bgColor != null) {
          c2 = g2.getColor();
          g2.setColor(bgColor);
        }
        double x0 = xs - inset;   // enclosing rectangle
        double y0 = ys + y - inset;
        //
        // shift by inset
        //
        if (yalign == 0) {
          ys -= y0 + ry;
          y0 = -ry;
        }
        else if (yalign == 3) {
          ys -= y0;
          y0 = 0;
        }
        if (xalign == 0) {
          xs -= x0;
          x0 = 0;
        }
        else if (xalign == 2) {
          xs -= x0 + rx;
          x0 = -rx;
        }
        //
        if (shifted) {  // additional displacement by height of rectangle
          double ya=0, yb=0, xa=0, xb=0;
          if (yalign == 0) {
            ya = -dsp;
            yb = -ry - dsp;
          }
          else {
            ya = dsp;
            yb = ry + dsp;
          }
          ys += ya;
          if (xalign == 0) {
            xa = 0.5*dsp;
            xb = rx;
          }
          else {
            xa = -0.5*dsp;
            xb = -rx;
          }
          Path2D border = new Path2D.Double();
          border.moveTo(0, 0);
          border.lineTo(0, yb);
          border.lineTo(xb, yb);
          border.lineTo(xb, ya);
          border.lineTo(xa, ya);
          border.closePath();
          if (bgColor != null) {
            g2.fill(border);
            g2.setColor(c2);
          }
          if (framed) {
            g2.draw(border);
          }
        }
        else {
          Rectangle2D.Double lr = new Rectangle2D.Double(x0, y0, rx, ry);
          if (bgColor != null) {
            g2.fill(lr);
            g2.setColor(c2);
          }
          if (framed) {
            g2.draw(lr);
          }
        }
      }
      g2.drawString(t, (float) xs, (float) ys);
    }

  }
  
  public static class Bar implements Paintable {

    public Color drawColor;
    public Color fillColor;
    double size;
    double[] xpos;
    double[] ypos;
    
    public Bar(double size, double[] xx, double[] yy) {      
      this.size = size;
      if (xx != null && yy != null) {
        int n = xx.length;
        if (yy.length < n) n = yy.length;
        xpos = new double[n];
        ypos = new double[n];
        for (int i = 0; i < n; i++) {
          xpos[i] = xx[i];
          ypos[i] = yy[i];
        }
      }
    }
   
     @Override
    public void paintGraph(Graphics2D g2, AffineTransform transform,
        String action) {
      //float dd = (float) (0.5 * size);  
      Point2D p0 = new Point2D.Double(0, 0);
      Point2D q0 = transform.transform(p0, null);
      Point2D p1 = new Point2D.Double(0.5*size, 0);
      Point2D q1 = transform.transform(p1, null);
      float dd = (float)(q1.getX() - q0.getX());
      for (int j = 0; j < xpos.length; j++) {
        AffineTransform t = g2.getTransform();
        if (Double.isNaN(xpos[j]) || Double.isNaN(ypos[j])) continue;
        Point2D p = new Point2D.Double(xpos[j], ypos[j]);           
        Point2D q = transform.transform(p, null);               
        GeneralPath gg = new GeneralPath();  
        gg.moveTo(dd, 0f);
        gg.lineTo(dd,  (float)(q0.getY()-q.getY()));
        gg.lineTo(-dd, (float)(q0.getY()-q.getY()));
        gg.lineTo(-dd, 0f);
        gg.lineTo(dd, 0f);
        g2.translate((float)q.getX(), (float)q.getY());
        if (fillColor != null) {
          g2.setColor(fillColor);
          g2.fill(gg);
        }
        if (drawColor != null) {
          g2.setColor(drawColor);
          g2.draw(gg);
        }
        g2.setTransform(t);
      }
    }
  }

  //====================================================================

  public static class Map implements Paintable {

    //-------------------------------------------------------------- readMap

//    public static Graph.Map readMap(File map) {
//      LasatDef ldf = new LasatDef(map.getParent(), map.getName(), null);
//      if (!ldf.valid) return null;
//      double dd[];
//      double xtrn[], ytrn[];
//      int i, n;
//      int nparts = ldf.getPartCount();
//      if (nparts != 1) return null;
//      String fname[] = ldf.getStrings(0, "file");
//      if (fname == null || fname.length < 1) return null;
//      xtrn = ldf.getDoubles(0, "xtrn");
//      if (xtrn == null || xtrn.length != 3) return null;
//      ytrn = ldf.getDoubles(0, "ytrn");
//      if (ytrn == null || ytrn.length != 3) return null;
//      AffineTransform aft = new AffineTransform(xtrn[0], ytrn[0], xtrn[1],
//          ytrn[1], xtrn[2], ytrn[2]);
//      File pic;
//      if (fname[0].charAt(0) == '/')
//        pic = new File(fname[0]);
//      else
//        pic = new File(map.getParent(), fname[0]);
//      if (!pic.canRead()) return null;
//      Graph.Map image = null;
//      try {
//        java.awt.Image img = Toolkit.getDefaultToolkit().createImage(
//            pic.getPath());
//        try {
//          Button dummy = new Button();
//          MediaTracker tracker = new MediaTracker(dummy);
//          tracker.addImage(img, 0);
//          tracker.waitForID(0);
//        } catch (Exception e) {
//          e.printStackTrace();
//        }
//        int width = img.getWidth(null);
//        int height = img.getHeight(null);
//        BufferedImage bi = new BufferedImage(width, height,
//            BufferedImage.TYPE_4BYTE_ABGR);
//        Graphics2D g2 = bi.createGraphics();
//        g2.drawImage(img, 0, 0, null);
//        aft.concatenate(new AffineTransform(1, 0, 0, -1, 0, bi.getHeight()));
//        image = new Graph.Map(bi, aft);
//      } catch (Exception e) {
//        e.printStackTrace();
//      }
//      return image;
//    }
    
    BufferedImage image;
    AffineTransform transform;

    public Map(BufferedImage image, AffineTransform transform) {
      this.transform = transform;
      this.image = image;
    }

    public Map(BufferedImage image, Rectangle2D rectangle) throws Exception {
      if (image == null || rectangle == null)
        throw new Exception("invalid argument");
      this.image = image;
      double wImage = image.getWidth();
      double hImage = image.getHeight();
      double wRectangle = rectangle.getWidth();
      double hRectangle = rectangle.getHeight();
      transform = new AffineTransform(wRectangle/wImage, 0, 0, 
          hRectangle/hImage, rectangle.getX(), rectangle.getY());
    }
    
    public Map(File f, Point2D ll, double pxsize) throws Exception {
      image = ImageIO.read(f);
      transform = new AffineTransform(pxsize, 0, 0, pxsize, ll.getX(), ll.getY());
    }

    public void makeTransparent(double intensity) {
      makeTransparent(intensity, 255);
    }
    
    public void makeTransparent(double intensity, int mul) {
      if (image == null) return;
      int w = image.getWidth();
      int h = image.getHeight();
      int m = 3*mul;
      BufferedImage bim = new BufferedImage(w, h, BufferedImage.TYPE_INT_ARGB);
      for (int i = 0; i < w; i++) {
        for (int j = 0; j < h; j++) {
          int rgb = image.getRGB(i, j);
          int alpha = (rgb & 0xff000000) >>> 24;
          int red = (rgb & 0x00ff0000) >>> 16;
          int green = (rgb & 0x0000ff00) >>> 8;
          int blue = (rgb & 0x000000ff);
          alpha = (int) (intensity*(alpha*(m - red - green - blue))/m);
          if (alpha > 255) 
            alpha = 255;
          if (alpha < 0)  
            alpha = 0;
          rgb = (rgb & 0x00ffffff) | (alpha << 24);
          bim.setRGB(i, j, rgb);
        }
      }
      image = bim;
    }

    @Override
    public void paintGraph(Graphics2D g2, AffineTransform transform,
        String action) {
      AffineTransform a = new AffineTransform(transform);
      a.concatenate(this.transform);
      a.concatenate(new AffineTransform(1, 0, 0, -1, 0, image.getHeight()));
      if (image != null && transform != null)
        g2.drawImage(image, a, null);
    }
  }
  
  //=========================================================================

  class PopupListener extends MouseAdapter {

    private void maybeShowPopup(MouseEvent e) {
      if (e.isPopupTrigger()) {
        popup.show(e.getComponent(), e.getX(), e.getY());
      }
    }

    @Override
    public void mousePressed(MouseEvent e) {
      maybeShowPopup(e);
    }

    @Override
    public void mouseReleased(MouseEvent e) {
      maybeShowPopup(e);
    }
  }

  //================================================================== Symbol

  public static class Symbol implements Paintable {

    public Color drawColor;
    public Color fillColor;
    Shape shape;
    double[] xpos;
    double[] ypos;

    public Symbol(Shape shape, int unit, double x, double y) {
      this(shape, unit, new double[]{ x }, new double[]{ y });
    }

    public Symbol(Shape shape, int unit, double x, double[] yy) {
      this(shape, unit, yy, yy);
      for (int i = 0; i < yy.length; i++)
        xpos[i] = x;
    }

    public Symbol(Shape shape, int unit, double[] xx, double y) {
      this(shape, unit, xx, xx);
      for (int i = 0; i < xx.length; i++)
        ypos[i] = y;
    }

    public Symbol(Shape shape, int unit, double[] xx, double[] yy) {
      if (unit < 0 || unit > 4) unit = UNIT_MM;
      double fac = convFactors[unit] / convFactors[UNIT_PT];
      AffineTransform aff = AffineTransform.getScaleInstance(fac, -fac);
      this.shape = aff.createTransformedShape(shape);
      if (xx != null && yy != null) {
        int n = xx.length;
        if (yy.length < n) n = yy.length;
        xpos = new double[n];
        ypos = new double[n];
        for (int i = 0; i < n; i++) {
          xpos[i] = xx[i];
          ypos[i] = yy[i];
        }
      }
    }

    @Override
    public void paintGraph(Graphics2D g2, AffineTransform transform,
        String action) {
      Point2D p;
      for (int j = 0; j < xpos.length; j++) {
        AffineTransform t = g2.getTransform();
        if (Double.isNaN(xpos[j]) || Double.isNaN(ypos[j])) 
          continue;
        p = new Point2D.Double(xpos[j], ypos[j]);
        Point2D q = transform.transform(p, null);
        g2.translate(q.getX(), q.getY());
        if (fillColor != null) {
          g2.setColor(fillColor);
          g2.fill(shape);
        }
        if (drawColor != null) {
          g2.setColor(drawColor);
          g2.draw(shape);
        }
        g2.setTransform(t);
      }
    }

  }

  //============================================================= Shapes

  public static class Shapes implements Paintable {

    Shape[] shapes;
    Color[] colors;
    double[] xx;
    double[] yy;
    int cntShapes;

    public Shapes(Shape shape, Color[] colors, double[] xx, double[] yy) {
      this(new Shape[]{ shape }, colors, xx, yy);
    }

    public Shapes(Shape[] shapes, Color color, double[] xx, double[] yy) {
      this(shapes, new Color[]{ color }, xx, yy);
    }

    public Shapes(Shape shape, Color color, double[] xx, double[] yy) {
      this(new Shape[]{ shape }, new Color[]{ color }, xx, yy);
    }

    public Shapes(Shape[] shapes, Color[] colors, double[] xx, double[] yy) {
      int n = xx.length;
      if (yy.length < n) n = yy.length;
      if (shapes.length == 1 && n > 1) {
        Shape[] ss = new Shape[n];
        for (int i = 0; i < n; i++)
          ss[i] = shapes[0];
        shapes = ss;
      }
      if (colors.length == 1 && n > 1) {
        Color[] cc = new Color[n];
        for (int i = 0; i < n; i++)
          cc[i] = colors[0];
        colors = cc;
      }
      this.shapes = shapes;
      this.colors = colors;
      this.xx = xx;
      this.yy = yy;
      cntShapes = shapes.length;
      if (cntShapes > colors.length) cntShapes = colors.length;
      if (cntShapes > xx.length) cntShapes = xx.length;
      if (cntShapes > yy.length) cntShapes = yy.length;
    }

    @Override
    public void paintGraph(Graphics2D g2, AffineTransform transform, String action) {
      Shape s, st;
      for (int k = 0; k < cntShapes; k++) {
        st = AffineTransform.getTranslateInstance(xx[k], yy[k])
            .createTransformedShape(shapes[k]);
        s = transform.createTransformedShape(st);
        g2.setColor(colors[k]);
        if (action.equals("fill"))
          g2.fill(s);
        else
          g2.draw(s);
      }
    }

  }
  
  //=================================================================== GPanel
  
  public static class GPanel extends JPanel {
    
    Graph.Element ge;
    Rectangle2D view;
    Rectangle2D dest;
    AffineTransform t;
    
    @SuppressWarnings("OverridableMethodCallInConstructor")
    public GPanel(Graph.Element ge, Rectangle2D dst) {
      this.ge = ge;
      view = ge.getView();
      if (view == null) {
        t = new AffineTransform();
        return;
      }
      if (dst == null)  dst = view;
      dest = new Rectangle2D.Double(
          dst.getX(), dst.getY() + dst.getHeight(), 
          dst.getWidth(), -dst.getHeight());
      t = createTransform(view, dest);
      int xmax = (int)Math.max(dest.getMinX(), dest.getMaxX());
      int ymax = (int)Math.max(dest.getMinY(), dest.getMaxY());
      setPreferredSize(new Dimension(xmax, ymax));
    }
    
    @Override
    public void paintComponent(Graphics g) {
      super.paintComponent(g);
      ge.paint(g, t);
    }
    
  }

  //====================================================================

  public static final int ALIGN_BASE = 1;
  public static final int ALIGN_BOTTOM = 0;
  public static final int ALIGN_LEFT = 0;
  public static final int ALIGN_RIGHT = 8;
  public static final int ALIGN_TOP = 3;
  public static final int ALIGN_XCENTER = 4;
  public static final int ALIGN_YCENTER = 2;

  public static final int AXIS_GRID = 8;
  public static final int AXIS_INSIDE = 4;
  public static final int AXIS_LOG = 16;
  public static final int AXIS_NOGRID = 0;
  public static final int AXIS_NORMAL = 0; // left or bottom
  public static final int AXIS_OPPOSITE = 2;
  public static final int AXIS_OUTSIDE = 0;
  public static final int AXIS_X = 0;
  public static final int AXIS_Y = 1;

  public static Color[] colorSet10 = { new Color(0.9f, 0.9f, 1.0f),
      new Color(0.6f, 1.0f, 1.0f), new Color(0.4f, 0.9f, 0.7f),
      new Color(0.3f, 0.8f, 0.5f), new Color(0.2f, 0.9f, 0.0f),
      new Color(0.6f, 1.0f, 0.0f), new Color(1.0f, 1.0f, 0.0f),
      new Color(1.0f, 0.8f, 0.1f), new Color(1.0f, 0.5f, 0.0f),
      new Color(1.0f, 0.0f, 0.0f) };

  private static final double[] convFactors = { 1, 0.1, 25.4, 25.4/72.27,
      25.4/72.0 };

  public static final int DIAGRAM_SYMBOLS = 1;
  static File fMAP = new File("/data/test/plan100.png");
  static File fPDF = new File("/d/test/test.pdf");
  static File fPNG = new File("/d/test/test.png");

  //static JFrame frame;

  //=====================================================================

  static Graph graph;

//  public static final int HORIZONTAL = 0;
//  public static final int VERTICAL = 1;

  public static String pathSave;
  public static final int SHAPE_ASTERISK = 8;
  public static final int SHAPE_CIRCLE = 4;
  public static final int SHAPE_CROSS = 5;
  public static final int SHAPE_DIAMOND = 1;
  public static final int SHAPE_PYRAMIDE = 3;
  public static final int SHAPE_BAR = 11;

  public static final int SHAPE_SQUARE = 0;
  public static final int SHAPE_STAR = 7;
  public static final int SHAPE_TIMES = 6;
  public static final int SHAPE_TRIANGLE = 2;
  
  public static final int UNIT_BP = 4;
  public static final int UNIT_CM = 1;
  public static final int UNIT_INCH = 2;

  public static final int UNIT_MM = 0;
  public static final int UNIT_PT = 3;
  

  public static double convert(double value, int from, int to) {
    if (from < 0 || from >= convFactors.length) from = 0;
    if (to < 0 || to >= convFactors.length) to = 0;
    return value * convFactors[from] / convFactors[to];
  }

  public static Shape createShape(int type, double size) {
    Path2D g;
    double s;
    switch (type) {
      case SHAPE_DIAMOND :
        g = new Path2D.Double();
        s = size/2;
        g.moveTo(s, 0);
        g.lineTo(0, s);
        g.lineTo(-s, 0);
        g.lineTo(0, -s);
        g.lineTo(s, 0);
        return g;
      case SHAPE_PYRAMIDE :
        g = new Path2D.Double();
        s = size/6;
        g.moveTo(-3 * s, -2 * s);
        g.lineTo(3 * s, -2 * s);
        g.lineTo(0, 4 * s);
        g.lineTo(-3 * s, -2 * s);
        return g;
      case SHAPE_TRIANGLE :
        g = new Path2D.Double();
        s = size/6;
        g.moveTo(3 * s, 2 * s);
        g.lineTo(-3 * s, 2 * s);
        g.lineTo(0, -4 * s);
        g.lineTo(3 * s, 2 * s);
        return g;
      case SHAPE_CROSS :
        g = new Path2D.Double();
        s = size/8;
        g.moveTo(-4 * s, 0);
        g.lineTo(-s, -s);
        g.lineTo(0, -4 * s);
        g.lineTo(s, -s);
        g.lineTo(4 * s, 0);
        g.lineTo(s, s);
        g.lineTo(0, 4 * s);
        g.lineTo(-s, s);
        g.lineTo(-4 * s, 0);
        return g;
      case SHAPE_TIMES :
        g = new Path2D.Double();
        s = size/4;
        g.moveTo(-2 * s, -2 * s);
        g.lineTo(0, -s);
        g.lineTo(2 * s, -2 * s);
        g.lineTo(s, 0);
        g.lineTo(2 * s, 2 * s);
        g.lineTo(0, s);
        g.lineTo(-2 * s, 2 * s);
        g.lineTo(-s, 0);
        g.lineTo(-2 * s, -2 * s);
        return g;
      case SHAPE_STAR :
        g = new Path2D.Double();
        s = size/16;
        g.moveTo(-8 * s, 0);
        g.lineTo(-2 * s, -s);
        g.lineTo(-4 * s, -6 * s);
        g.lineTo(0, -2 * s);
        g.lineTo(4 * s, -6 * s);
        g.lineTo(2 * s, -s);
        g.lineTo(8 * s, 0);
        g.lineTo(2 * s, s);
        g.lineTo(4 * s, 6 * s);
        g.lineTo(0, 2 * s);
        g.lineTo(-4 * s, 6 * s);
        g.lineTo(-2 * s, s);
        g.lineTo(-8 * s, 0);
        return g;
      case SHAPE_ASTERISK :
        g = new Path2D.Double();
        s = size/8;
        double sy = size/6;
        g.moveTo(-4 * s, 0);
        g.lineTo(-2 * s, -sy);
        g.lineTo(-2 * s, -3 * sy);
        g.lineTo(0, -2 * sy);
        g.lineTo(2 * s, -3 * sy);
        g.lineTo(2 * s, -sy);
        g.lineTo(4 * s, 0);
        g.lineTo(2 * s, sy);
        g.lineTo(2 * s, 3 * sy);
        g.lineTo(0, 2 * sy);
        g.lineTo(-2 * s, 3 * sy);
        g.lineTo(-2 * s, sy);
        g.lineTo(-4 * s, 0);
        return g;
      case SHAPE_CIRCLE :
        g = new Path2D.Double();
        double s2 = size/2;
        double s1 = size/4;
        g.moveTo(-s2, 0);
        g.curveTo(-s2, -s1, -s1, -s2, 0, -s2);
        g.curveTo(s1, -s2, s2, -s1, s2, 0);
        g.curveTo(s2, s1, s1, s2, 0, s2);
        g.curveTo(-s1, s2, -s2, s1, -s2, 0);
        return g;
      default :
        return new Rectangle2D.Double(-0.5*size, -0.5*size, size, size);
    }
  }
  
  public static AffineTransform createTransform(Rectangle2D src, Rectangle2D dst) {
    AffineTransform t = new AffineTransform();
    if (src == null || dst == null)
      return t;
    t.translate(dst.getX(), dst.getY());
    t.scale(dst.getWidth()/src.getWidth(), dst.getHeight()/src.getHeight());
    t.translate(-src.getX(), -src.getY());
    return t;
  }

  public static void main(String[] args) {
    graph = new Graph();
  }

  public static Image makeIconImage(String t) {
    BufferedImage bi = new BufferedImage(48, 48, BufferedImage.TYPE_3BYTE_BGR);
    Graphics g2 = bi.createGraphics();
    g2.setColor(new Color(0, 0, 128));
    g2.fillRect(0, 0, 48, 48);
    Font f = new Font("SansSerif", Font.BOLD, 16);
    g2.setFont(f);
    g2.setColor(new Color(255, 0, 0));
    g2.drawString(t, 6, 24);
    return bi;
  }

  // ---------------------------------------------------- class IsoLine

  public static Graph.Element makeIsoPicture(
      Graph.IsoLine g, // the data
      double[] values, // the values of the iso-lines
      Color[] colors // the fill colors, [0]: background
  ) {
    return makeIsoPicture(g, values, colors, Color.gray, 1);
  }
  
  public static Graph.Element makeIsoPicture(
      Graph.IsoLine g, // the data
      double[] values, // the values of the iso-lines
      Color[] colors,  // the fill colors, [0]: background
      Color lcolor,    // line color
      double lwidth    // line width
    ) {
    if (g == null || values == null) return null;
    int nv = values.length;
    int nc;
    if (colors == null) {
      colors = new Color[nv + 1];
      for (int i = 0; i <= nv; i++) {
        int j = i % 10;
        colors[i] = Graph.colorSet10[j];
      }
      nc = nv + 1;
    } 
    else {
      nc = colors.length;
      if (nv > nc - 1) nv = nc - 1;
    }
    Graph.Element base = new Graph.Element();
    base.color = new Color(0.3f, 0.3f, 0.3f);
    base.stroke = new BasicStroke(1f);
    Graph.Element area;
    Path2D gp;
    gp = g.getBorder();
    area = new Graph.Element();
    area.color = colors[0];
    area.action = "fill";
    area.add(gp);
    base.add(area);
    Graph.Element lines = new Graph.Element();
    lines.action = "draw";
    lines.color = lcolor;
    lines.stroke = new BasicStroke((float)lwidth);
    ArrayList<Path2D> areas = new ArrayList<Path2D>();
    for (int n = 0; n < nv; n++) {
      Path2D go = new Path2D.Double();
      gp = g.getLine(values[n], go);
      if (gp == null) 
        break;
      areas.add(go);
      lines.add(gp);
    }
    int n1 = areas.size() - 1;
    for (int n=0; n<=n1; n++) {
      Path2D go = areas.get(n);
      if (n < n1) {
        Path2D gn = areas.get(n+1);
        go.append(gn, false);
      }
      go.setWindingRule(Path2D.WIND_EVEN_ODD);
      area = new Graph.Element();
      area.color = colors[n + 1];
      area.action = "fill";
      area.add(go);
      base.add(area);
      lines.add(gp);
    }
    if (lwidth > 0) base.add(lines);
    return base;
  }

  public static Path2D makePath(double[] xx, double[] yy) {
    Path2D gp = new Path2D.Double();
    boolean started = false;
    for (int i=0; i<xx.length; i++) {
      if (Double.isNaN(xx[i]) || Double.isNaN(yy[i]))
        continue;
      if (started)
        gp.lineTo(xx[i], yy[i]);
      else {
        gp.moveTo(xx[i], yy[i]);
        started = true;
      }
    }
    return gp;
  }

  public static String test_iso() {
    IsoLine.DEBUG = true;
    double v = 1.5;
    double[][] vv = new double[][]{ { 2, 2, 2, 1 }, { 2, 2, 1, 1 },
        { 2, 1, 1, 1 }, { 1, 1, 1, 1 }, };
    IsoLine iso = new IsoLine(vv);
    Path2D go = new Path2D.Double();
    Path2D gp = iso.getLine(v, go);
    Graph.Diagram dgm = new Graph.Diagram(120, 0);
    dgm.setScale(0, -1, 3, 4);
    dgm.addAxis(8, 1, "%1.0f", "x-Achse");
    dgm.addAxis(9, 1, "%1.0f", "y-Achse");
    Graph.Element ge;
    //
    ge = new Graph.Element();
    ge.color = Color.GREEN;
    ge.action = "fill";
    ge.add(go);
    dgm.addOverlay(ge);
    //
    ge = new Graph.Element();
    ge.color = Color.RED;
    ge.stroke = new BasicStroke(3f);
    ge.action = "draw";
    ge.add(gp);
    dgm.addOverlay(ge);
    //
    Graph grp = dgm.createGraph(null);
    grp.show("test", null);
    return null;
  }

  public Locale locale = Locale.ENGLISH;
  public double zoom = 1.0;
  
  private JFileChooser fchSave = new JFileChooser(System
      .getProperty("user.home"));
  private Font font;
  private double height;
  private JPopupMenu popup;
  private Element root;
  private int unit;
  private double width;
  private JFrame frame;

  public Graph() {
    this(140, 140, UNIT_MM);
  }

  public Graph(double width, double height, int unit) {
    init(width, height, unit);
    //Create the popup menu.
    popup = new JPopupMenu();
    JMenuItem menuItem = new JMenuItem("Als PDF-Datei speichern");
    menuItem.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent a) {
        writePDF(null);
      }
    });
    popup.add(menuItem);
    menuItem = new JMenuItem("Als PNG-Datei (150 dpi) speichern");
    menuItem.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent a) {
        writePNG(null, 150);
      }
    });
    popup.add(menuItem);
    menuItem = new JMenuItem("Als PNG-Datei (300 dpi) speichern");
    menuItem.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent a) {
        writePNG(null, 300);
      }
    });
    popup.add(menuItem);

    //Add listener to components that can bring up popup menus.
    MouseListener popupListener = new PopupListener();
    this.addMouseListener(popupListener);

  }

  public void clear() {
    root.clear();
  }

  public Element createAxis(Element pe, int type, double tick, String format,
      String label) {
    if (pe == null) return null;
    if (pe.view == null) return null;
    if (pe.scale == null) return null;
    Font fnt = getGraphFont();
    Element ge = new Element();
    ge.color = pe.color;
    ge.stroke = pe.stroke;
    ge.font = fnt;
//    SimpleCFormat scf = null;
    int field_length = 0;
//    if (format != null) {
//      scf = new SimpleCFormat(format);
//      scf.setLocale(locale);
//    }
    double fontSize = convert(fnt.getSize2D(), UNIT_PT, unit);
    double tickLength = 0.3 * fontSize;
    boolean yaxis = ((type & Graph.AXIS_Y) != 0);
    boolean xaxis = !yaxis;
    boolean opposite = ((type & Graph.AXIS_OPPOSITE) != 0);
    boolean normal = !opposite;
//    boolean inside = ((type & Graph.AXIS_INSIDE) != 0);
//    boolean outside = !inside;
    boolean grid = ((type & Graph.AXIS_GRID) != 0);
//    boolean nogrid = !grid;
    boolean logarithmic = ((type & Graph.AXIS_LOG) != 0);
    boolean linear = !logarithmic;
    boolean labels = (format != null);
    Label l = null;
    double dl = 0, da = 0, db = 0, a = 0, a1 = 0, a2 = 0, b = 0, b1 = 0, b2 = 0, d = 0;
    double v = 0, vmin = 0, vmax = 0, v1 = 0, v2 = 0;
    double[] vv;
    double[] logTick10 = { 1, 10 };
    double[] logTick3 = { 1, 3, 10 };
    double[] logTick2 = { 1, 2, 5, 10 };
    double[] logTick1 = { 1, 1.5, 2, 3, 5, 7, 10 };
    double[] logTick = null;
    int frmlen = 0;
    if (format != null) {
      int ip1 = -1, ip2 = -1;
      ip1 = format.indexOf("%");
      if (ip1 >= 0) {
        ip2 = format.substring(ip1+1).indexOf(".");
        if (ip2 < 0)
          ip2 = format.substring(ip1+1).indexOf("d");
        if (ip2 < 0)
          ip2 = format.substring(ip1+1).indexOf("s");
        if (ip2 >= 0)
          ip2 += ip1+1;
      }
      if (ip1 >= 0 && ip2 >= 0) {
        try { frmlen = Integer.parseInt(format.substring(ip1+1, ip2)); }
        catch (Exception ee) {}
      }
    }    
    int i, nt = 0;
    Path2D ticks = new Path2D.Double();
    if (logarithmic) {
      if (tick <= 1.5)
        logTick = logTick1;
      else if (tick <= 2)
        logTick = logTick2;
      else if (tick <= 3)
        logTick = logTick3;
      else
        logTick = logTick10;
      tick = Math.log(10);
    }
    if (xaxis) {
      d = 1.e-4 * Math.abs(pe.scale.width);
      v1 = pe.scale.x;
      v2 = v1 + pe.scale.width;
      if (format != null)
        field_length = String.format(format, pe.scale.x).length();
    } 
    else {
      d = 1.e-4 * Math.abs(pe.scale.height);
      v1 = pe.scale.y;
      v2 = v1 + pe.scale.height;
      if (format != null)
        field_length = String.format(format, pe.scale.y).length();
    }
    if (v2 < v1) {
      v = v1;
      v1 = v2;
      v2 = v;
    }
    vmin = tick * Math.floor(v1 / tick);
    vmax = tick * Math.ceil(v2 / tick);
    if (linear) {
      if (vmin < v1 - d) vmin += tick;
      if (vmax > v2 + d) vmax -= tick;
      nt = (int) (1.5 + (vmax - vmin) / tick);
      vv = new double[nt];
      for (i = 0; i < nt; i++)
        vv[i] = vmin + i * tick;
    } else {
      nt = (int) (1.5 + (vmax - vmin) * (logTick.length - 1) / tick);
      vv = new double[nt];
      nt = 0;
      int ii = 0;
      for (v = vmin; v <= vmax + d;) {
        double vi = v + Math.log(logTick[ii]);
        if (vi > v2 + d) break;
        if (vi > v1 - d) {
          vv[nt] = vi;
          nt++;
        }
        ii++;
        if (ii >= logTick.length - 1) {
          v += tick;
          ii = 0;
        }
      }
    }
    if (xaxis) { // x-axis
      if (pe.view.width == 0) return null;
      x_ticks : {
        if (normal) {
          if (grid) {
            b1 = pe.view.y + pe.view.height;
            b2 = pe.view.y;
            db = -tickLength;
          } else {
            b1 = pe.view.y;
            b2 = pe.view.y;
            db = -tickLength;
          }
        } else {
          if (grid) {
            b1 = pe.view.y;
            b2 = pe.view.y + pe.view.height;
            db = tickLength;
          } else {
            b1 = pe.view.y + pe.view.height;
            b2 = pe.view.y + pe.view.height;
            db = tickLength;
          }
        }
        if ((format == null) && grid) db = 0;
        ticks.moveTo((float) pe.view.x, (float) b1);
        ticks.lineTo((float) (pe.view.x + pe.view.width), (float) b1);
        if (b1 != b2) {
          ticks.moveTo((float) pe.view.x, (float) b2);
          ticks.lineTo((float) (pe.view.x + pe.view.width), (float) b2);
        }
        if (pe.scale == null) break x_ticks;
        if (tick <= 0) break x_ticks;
        dl += tickLength;
        if (labels) dl +=  1.5 * fontSize;
        for (i = 0; i < nt; i++) {
          v = vv[i];
          a = pe.view.x + (v - pe.scale.x) * pe.view.width / pe.scale.width;
          ticks.moveTo((float) a, (float) b1);
          ticks.lineTo((float) a, (float) (b2 + db));
          if (labels) {
            if (logarithmic) v = Math.exp(v);
            l = new Label(a, b2 + db, String.format(locale, format, v), fnt);
            l.alignment = (normal) ? 7 : 4;
            ge.add(l);
          }
        }
      }
      if (label != null) {
        b = b2 + ((normal) ? -dl : dl);
        l = new Label(pe.view.x + 0.5 * pe.view.width, b, label, fnt);
        l.alignment = (normal) ? 7 : 4;
      }
    } else { // y-axis
      if (pe.view.height == 0) return null;
      y_ticks : {
        if (normal) {
          if (grid) {
            a1 = pe.view.x + pe.view.width;
            a2 = pe.view.x;
            da = -tickLength;
          } else {
            a1 = pe.view.x;
            a2 = pe.view.x;
            da = -tickLength;
          }
        } else {
          if (grid) {
            a1 = pe.view.x;
            a2 = pe.view.x + pe.view.width;
            da = tickLength;
          } else {
            a1 = pe.view.x + pe.view.width;
            a2 = pe.view.x + pe.view.width;
            da = tickLength;
          }
        }
        if ((format == null) && grid) da = 0;
        ticks.moveTo((float) a1, (float) pe.view.y);
        ticks.lineTo((float) a1, (float) (pe.view.y + pe.view.height));
        if (a1 != a2) {
          ticks.moveTo((float) a2, (float) pe.view.y);
          ticks.lineTo((float) a2, (float) (pe.view.y + pe.view.height));
        }
        if (pe.scale == null) break y_ticks;
        if (tick <= 0) break y_ticks;
        dl += 2 * tickLength;
        if (labels) dl += 0.6*fontSize*(1 + field_length);
        for (i = 0; i < nt; i++) {
          v = vv[i];
          b = pe.view.y + (v - pe.scale.y) * pe.view.height / pe.scale.height;
          ticks.moveTo((float) a1, (float) b);
          ticks.lineTo((float) (a2 + da), (float) b);
          if (labels) {
            if (logarithmic) v = Math.exp(v);
            l = new Label(a2 + 2*da, b, String.format(locale, format, v), fnt);
            l.alignment = (normal) ? 10 : 2;
            ge.add(l);
          }
        }
      }
      if (label != null) {
        a = a2 + ((normal) ? -dl : dl);
        l = new Label(a, pe.view.y + 0.5 * pe.view.height, label, fnt);
        l.alignment = (normal) ? 4 : 7;
        l.angle = 90;
      }
    }

    ge.add(ticks);
    if (l != null) ge.add(l);

    return ge;
  }

  public Graph.Element createElement() {
    return new Element();
  }

  public Font getGraphFont() {
    return this.font;
  }

  public double getGraphHeight() {
    return height;
  }

  public int getGraphUnit() {
    return unit;
  }

  public double getGraphWidth() {
    return width;
  }

  public Element getRoot() {
    return root;
  }
  
  public JPopupMenu getPopup() {
    return popup;
  }
  
  public JFrame getFrame() {
    return frame;
  }

  public final void init(double width, double height, int unit) {
    this.unit = unit;
    this.width = width;
    this.height = height;
    int w = (int) Math.round(convert(width, unit, UNIT_BP));
    int h = (int) Math.round(convert(height, unit, UNIT_BP));
    root = new Element();
    root.setView(new Rectangle2D.Double(0, 0, w, h));
    root.setScale(new Rectangle2D.Double(0, 0, width, height));
    root.color = Color.BLACK;
    Dimension d = new Dimension(w, h);
    this.setPreferredSize(d);
    this.font = new Font("SansSerif", Font.PLAIN, 10);
  }

  public Graph.Element makeColorScale(int type, double xr, double yr, double length,
      Color[] colors, double[] values, String format) {
     return makeScale(type, xr, yr, length, colors, values, format, false);
  }

  public Graph.Element makeScale(int type, double xr, double yr, double length,
      Color[] colors, double[] values, String format, boolean extended) {
    if (length <= 0) return null;
    if (colors == null) return null;
    if (values == null) return null;
    if (format == null) return null;
    Graph.Element scale = createElement();
    Graph.Element ge = null;
    int nc = colors.length;
    if (nc > values.length + 1) nc = values.length + 1;
    double d1 = convert(font.getSize2D(), UNIT_BP, unit);
    double d2 = (nc > 0) ? length / nc : 0;
    for (int i = 0; i < nc; i++) {
      ge = createElement();
      ge.action = "fill";
      ge.color = colors[i];
      if (type == HORIZONTAL) { // horizontal
        ge.add(new Rectangle2D.Double(xr + i * d2, yr - d1, d2, d1));
      } else {
        ge.add(new Rectangle2D.Double(xr, yr + i * d2, d1, d2));
      }
      scale.add(ge);
    }
    ge = createElement();
    ge.action = "draw";
    Path2D gp = new Path2D.Double();
    Graph.Label l = null;
    if (type == HORIZONTAL) {
      gp.moveTo((float) xr, (float) yr);
      gp.lineTo((float) (xr + length), (float) yr);
      gp.lineTo((float) (xr + length), (float) (yr - d1));
      gp.lineTo((float) xr, (float) (yr - d1));
      gp.closePath();
      for (int i = 1; i < nc; i++) {
        gp.moveTo((float) (xr + i * d2), (float) yr);
        gp.lineTo((float) (xr + i * d2), (float) (yr - 1.3 * d1));
        l = new Graph.Label(xr + i * d2, yr - 1.3 * d1, 
            String.format(locale, format, values[i - 1]), font);
        l.alignment = 7;
        scale.add(l);
      }
    } else {
      gp.moveTo((float) xr, (float) yr);
      gp.lineTo((float) (xr + d1), (float) yr);
      gp.lineTo((float) (xr + d1), (float) (yr + length));
      gp.lineTo((float) xr, (float) (yr + length));
      gp.closePath();
      for (int i = 1; i < nc; i++) {
        gp.moveTo((float) xr, (float) (yr + i * d2));
        gp.lineTo((float) (xr + 1.3 * d1), (float) (yr + i * d2));
        l = new Graph.Label(xr + 1.6 * d1, yr + i * d2,
            String.format(locale, format, values[i - 1]), font);
        l.alignment = 2;
        scale.add(l);
      }
    }
    ge.add(gp);
    scale.add(ge);
    return scale;
  }

  @Override
  public void paintComponent(Graphics g) {
    // System.out.println("paint component start: " + counter);
    Graphics2D g2 = (Graphics2D) g;
    Font oldFont = g2.getFont();
    if (font != null) g2.setFont(font);
    g2.setRenderingHint(RenderingHints.KEY_ANTIALIASING,
        RenderingHints.VALUE_ANTIALIAS_ON);
    AffineTransform tp = new AffineTransform(1, 0, 0, -1, 0, root.view.height);
    super.paintComponent(g);
    g2.scale(zoom, zoom);
    root.paint(g, tp);
    if (oldFont != null) g2.setFont(oldFont);
    // System.out.println("paint component finished");
  }

  public void setGraphFont(Font font) {
    if (font != null) this.font = font;
  }

  public void setRoot(Element root) {
    this.root = (root != null) ? root : new Element();
  }

  public JFrame show(String title, Image icon) {
    JFrame frm = new JFrame();
    if (icon != null)
      frm.setIconImage(icon);
    frm.setDefaultCloseOperation(JFrame.DISPOSE_ON_CLOSE);
    frm.setTitle(title);
    frm.setIconImage(icon);
    Container content = frm.getContentPane();
    content.setBackground(Color.WHITE);
    content.add(this);
    Dimension dim = this.getPreferredSize();
    this.setPreferredSize(new Dimension((int)(zoom*dim.width), (int)(zoom*dim.height)));
    frm.pack();
    frm.setVisible(true);
    frame = frm;
    return frm;
  }

  //------------------------------------------------------------------------
  // writePDF

  public boolean writePDF(File f) {
    boolean success = false;
    float w = (float) convert(width, unit, UNIT_BP);
    float h = (float) convert(height, unit, UNIT_BP);
    double zm = 1.0;

    if (f == null) {
      fchSave.setDialogTitle("Save as PDF file");
      if (pathSave != null) 
        fchSave.setCurrentDirectory(new File(pathSave));
      else
        fchSave.setCurrentDirectory(new File(System.getProperty("user.dir")));
      fchSave.rescanCurrentDirectory();
      int r = fchSave.showSaveDialog(this);
      if (r != JFileChooser.APPROVE_OPTION) return false;
      pathSave = fchSave.getCurrentDirectory().getPath();
      f = fchSave.getSelectedFile();
      String fn = f.getPath();
      if (!fn.endsWith(".pdf") && !fn.endsWith(".PDF")) {
        f = new File(fn + ".pdf");
      }
    }

    // step 1: creation of a document-object
    Document document = new Document(new Rectangle(w, h), 0, 0, 0, 0);

    try {

      // step 2:
      // we create a writer that listens to the document
      // and directs a PDF-stream to a file
      PdfWriter writer = PdfWriter.getInstance(document,
          new FileOutputStream(f));
      writer.setPdfVersion(PdfWriter.VERSION_1_4);

      // step 3: we open the document
      document.open();

      // step 4: we grab the ContentByte and do some stuff with it

      // we create a fontMapper and read all the fonts in the font directory
      DefaultFontMapper mapper = new DefaultFontMapper();

      // we create a template and a Graphics2D object that corresponds with it
      PdfContentByte cb = writer.getDirectContent();
      PdfTemplate tp = cb.createTemplate(w, h);
      Graphics2D g2 = tp.createGraphics(w, h, mapper);
      tp.setWidth(w);
      tp.setHeight(h);
      zm = this.zoom;
      this.zoom = 1;
      paintComponent(g2);
      this.zoom = zm;
      g2.dispose();
      cb.addTemplate(tp, 0, 0);
      success = true;
    } catch (Exception exc) {
      exc.printStackTrace(System.out);
    } finally {
      document.close();
    }
    return success;
  }

  public boolean writePNGold(File file, double resolution) {
    boolean success = false;
    if (root == null)
      return false;
    if (file == null) {
      fchSave.setDialogTitle("Save as PNG file (" + (int) resolution + " dpi)");
      int r = fchSave.showSaveDialog(this);
      if (r != JFileChooser.APPROVE_OPTION)
        return false;
      file = fchSave.getSelectedFile();
      String fn = file.getPath();
      if (!fn.endsWith(".png") && !fn.endsWith(".PNG")) {
        file = new File(fn + ".png");
      }
    }
    int w = (int) Math.round(convert(width, unit, UNIT_INCH) * resolution);
    int h = (int) Math.round(convert(height, unit, UNIT_INCH) * resolution);
    BufferedImage bi = new BufferedImage(w, h, BufferedImage.TYPE_4BYTE_ABGR);
    Graphics2D g2 = bi.createGraphics();
    g2.setPaint(Color.WHITE);
    g2.fillRect(0, 0, w, h);
    Dimension dim = getSize();                                    //-2014-01-24
    g2.scale(w/(double)dim.width, h/(double)dim.height);          //-2014-01-24
    paintComponent(g2);
    try {
      ImageIO.write(bi, "png", file);
      success = true;
    } catch (Exception e) {
    }
    return success;
  }
  
  public boolean writePNG(File file, double resolution) {
      boolean success = false;
      int w = (int) Math.round(convert(width, unit, UNIT_INCH) * resolution);
      int h = (int) Math.round(convert(height, unit, UNIT_INCH) * resolution);
      BufferedImage bi = new BufferedImage(w, h, BufferedImage.TYPE_4BYTE_ABGR);
      Graphics2D g2 = bi.createGraphics();
      g2.setRenderingHint(RenderingHints.KEY_ANTIALIASING,
              RenderingHints.VALUE_ANTIALIAS_ON);
      g2.setPaint(Color.WHITE);
      g2.fillRect(0, 0, w, h); 
      g2.scale(2, 2);   
      paintComponent(g2);
      try {
        ImageIO.write(bi, "png", file);
        success = true;
      } catch (Exception e) {
        e.printStackTrace(System.out);
      }
      return success;
    }

}