/**
 * Plume Rise Model PLURIS
 *
 * For theory see: 
 * Janicke, U., Janicke, L.: A three-dimensional plume rise model for
 * dry and wet plumes, Atmospheric Environment 35 (2000), 877-890.
 *
 * Copyright (C) Janicke Consulting, Überlingen, Germany, 2015-2019
 * email: info@janicke.de
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 * 
 */

package de.janicke.pluris;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileWriter;
import java.io.InputStreamReader;
import java.net.URL;
import java.net.URLConnection;
import java.util.LinkedHashMap;
import java.util.Locale;


/**
 *
 * @author ibju
 */
public class IBJparamSet {
  
  public static boolean DEBUG = false;
  
  public static enum USAGE { OPTIONAL, REQUIRED, EXPERT }
  public static enum PTYPE { STRING, CHAR, INT,  FLOAT }
  
  public class Param {
    public String Section;
    public String Name;
    public String Format;
    public String Note;
    public int Size;
    public USAGE Usage;    
    public PTYPE Type;
    public String[] Default;
    public String[] Allowed;
    public double Gmin, Gmax, Pmin, Pmax;
  }
  
  private LinkedHashMap<String, Param> hmparam;
  private LinkedHashMap<String, String> hmsec;
  
  
  public Param getParam(String name) {
    return (hmparam != null) ? hmparam.get(name.toLowerCase()) : null;
  }
  
  public String getSection(String name) {
    return (hmsec != null) ? hmsec.get(name.toLowerCase()) : null;
  }
  
  private String unquote(String s) {
    if (s == null)
      return null;
    int l = s.length();
    if (l < 1 || s.charAt(0) != '\"')
      return s;
    if (s.charAt(l - 1) == '\"')
      l--;
    String t = (l > 1) ? s.substring(1, l) : "";
    return t;
  }
  
  private String quote(String s) {
    if (s == null)
      return null;
    else
      return "\"" + unquote(s) + "\"";
  }
  
  private URL getURL(Object obj, String name) {
    URL u = null;
    if (obj == null) {     
      try { u = new File(name).toURI().toURL(); }
      catch (Exception e) { 
        if (DEBUG) System.out.printf("URL: no url via name.\n");
        u=null; 
      }
    }
    if (u == null) {
      try { u = obj.getClass().getResource(name); }
      catch (Exception e) {
        if (DEBUG) System.out.printf("URL: no url via given object.\n");
        u=null; 
      }
    }
    if (u == null) {     
      try { u = IBJparamSet.class.getResource("/"+name); }
      catch (Exception e) { 
        if (DEBUG) System.out.printf("URL: no url via IBJparamSet object.\n");
        u=null; 
      }   
    }  
    return u;
  }
  
  public void readParamSet(Object obj, String name) throws Exception {
    BufferedReader br = null;
    String line;
    if (hmparam == null)
      hmparam = new LinkedHashMap<String, Param>();
    else
      hmparam.clear();
    if (hmsec == null)
      hmsec = new LinkedHashMap<String, String>();
    else
      hmsec.clear();
    //
    // read parameter template
    URL url = getURL(obj, name);
    if (url == null)
      throw new Exception("can't get URL for parameter set "+name);
    URLConnection urlc = url.openConnection();
    br = new BufferedReader(new InputStreamReader(urlc.getInputStream()));
    String sec = null;
    while ((line = br.readLine()) != null) {
      if (line.trim().length() == 0 || line.startsWith("#"))
        continue;
      if (line.startsWith("[")) {
        sec = line.substring(1,line.indexOf("]"));
        if (hmsec.get(sec.toLowerCase()) != null)
          throw new Exception("duplicate section: "+sec);
        hmsec.put(sec.toLowerCase(), sec.toLowerCase());
        continue;
      }
      String[] sa = (line.split("[;]"));
      if (sa.length != 9)
        throw new Exception("invalid number of items in line: "+line);
      for (int i=0; i<sa.length; i++)
        sa[i] = sa[i].trim();
      Param p = new Param();
      p.Section = sec;
      int col = 0;
      //
      // name
      p.Name = sa[col++].toLowerCase();
      if (hmparam.get(p.Name.toLowerCase()) != null)
        throw new Exception("duplicate parameter: "+p.Name);
      hmparam.put(p.Name, p);
      //
      // format, type
      p.Format = sa[col++].trim();
      if (p.Format.endsWith("f") || p.Format.endsWith("e"))
        p.Type = PTYPE.FLOAT;
      else if (p.Format.endsWith("d"))
        p.Type = PTYPE.INT;
      else if (p.Format.endsWith("c"))
        p.Type = PTYPE.CHAR;
      else 
        p.Type = PTYPE.STRING;       
      //
      // size
      p.Size = Integer.parseInt(sa[col++].trim());
      //
      // usage
      if (sa[col].equals("O"))
        p.Usage = USAGE.OPTIONAL;
      else if (sa[col].startsWith("R"))
        p.Usage = USAGE.REQUIRED;
      else
        p.Usage = USAGE.EXPERT;
      col++;
      //
      // default
      if (sa[col].length() != 0) {
        p.Default = new String[p.Size];
        String[] saa = sa[col].split("[,]");
        if (saa.length != p.Size)
          throw new Exception("invalid number of default values in line: "+line);
        System.arraycopy(saa, 0, p.Default, 0, p.Size);         
      }
      col++;
      //
      // allowed values
      if (sa[col].length() != 0) {
        String[] saa = sa[col].split("[,]");
        p.Allowed = new String[saa.length];        
        System.arraycopy(saa, 0, p.Allowed, 0, saa.length);           
      }
      col++;
      //
      // general and plausible value range
      p.Gmin = -Double.MAX_VALUE;
      p.Gmax = Double.MAX_VALUE;
      p.Pmin = -Double.MAX_VALUE;
      p.Pmax = Double.MAX_VALUE;
      for (int i=0; i<2; i++) {
        if (sa[col].trim().length() != 0) {
          String[] saa = sa[col].split("[,]");
          for (int j=0; j<2; j++) {
            double d = 0;
            if (saa[j].equalsIgnoreCase("-inf"))
              d = -Double.MAX_VALUE;
            else if (saa[j].equalsIgnoreCase("+inf"))
              d = Double.MAX_VALUE;
            else
              d = Double.parseDouble(saa[j].trim().replace(',', '.'));
            if (i == 0) {
              if (j == 0) p.Gmin = d; else p.Gmax = d;
            }
            else {
              if (j == 0) p.Pmin = d; else p.Pmax = d;
            }
          }
        }
        col++;
      }
      //
      // note
      p.Note = sa[col];
      
      sa = null;
      line = null;   
    }   
    br.close();
    if (DEBUG) System.out.printf("%s: %d parameters read.\n", name, hmparam.size());
  }
  
  private void makeHeaderFile(String fnin, String name) throws Exception {
    StringBuilder sb = new StringBuilder();
    File fin = new File(fnin);
    File fout = new File(fin.getParent(), name);
    //
    // read param set file
    readParamSet(null, fnin);
    if (DEBUG) System.out.printf("makeHeaderFile: %d parameters read.\n", hmparam.size());
    BufferedWriter bw = null;
    bw = new BufferedWriter(new FileWriter(fout));
    //
    // package information
    String s = fin.getParentFile().getCanonicalPath();
    int ii = s.indexOf("de" + File.separator + "janicke");
    if (ii < 0)
      throw new Exception("can't locate package information from path");
    s = s.substring(ii).replace(File.separatorChar, '.');
    bw.write("package "+s+";");
    bw.newLine();
    bw.newLine();
    //
    // class name
    ii = name.indexOf(".java");
    if (ii < 0)
      throw new Exception("can't locate class name from name");
    s = name.substring(0,ii);
    bw.write("public class "+s+" {");
    bw.newLine();
    bw.newLine();
    //
    // parameters
    for ( String key : hmparam.keySet() ) {
      Param p = hmparam.get(key);
      String ss = (p.Size == 1) ? "" : "[]";
      String nn = p.Name.toUpperCase();
      String tt = "double";
      if (p.Type == PTYPE.CHAR)
        tt = "char";
      else if (p.Type == PTYPE.INT)
        tt = "int";
      else if (p.Type == PTYPE.STRING)
        tt = "String";
      bw.write(String.format("  public static %-6s%s  %s;   // %s",
        tt, ss, nn, p.Note));
      bw.newLine();
    }
    bw.newLine();
    //
    // lists
    sb.setLength(0);
    sb.append("  public static String[] PrmSections = {");
    for (String key : hmsec.keySet())
      sb.append(sb.toString().endsWith("{") ? " " : ", ").append(quote(hmsec.get(key).trim()));
    sb.append(" };");
    bw.write(sb.toString());
    bw.newLine();
    bw.newLine();  
    sb.setLength(0);
    sb.append("  public static String[][] PrmParams = {");
    for (String key : hmparam.keySet()) { 
      bw.write(sb.toString());
      bw.newLine();
      sb.setLength(0);
      sb.append("    { ").append(quote(hmparam.get(key).Name.trim()));
      sb.append(", ").append(quote(hmparam.get(key).Section.trim()));
      sb.append(" },");
    }
    bw.write(sb.toString().substring(0, sb.toString().length()-1));
    bw.newLine();
    bw.write("  };");
    bw.newLine();
    bw.newLine();
    
    //
    // defaults
    bw.write("  public static void PrmSetDefaults() {");
    bw.newLine();
    for ( String key : hmparam.keySet() ) {
      Param p = hmparam.get(key);
      String nn = p.Name.toUpperCase();
      String tt = "double";
      if (p.Type == PTYPE.CHAR)
        tt = "char";
      else if (p.Type == PTYPE.INT)
        tt = "int";
      else if (p.Type == PTYPE.STRING)
        tt = "String";
      sb.setLength(0);
      sb.append("    ").append(String.format("%-5s = ", p.Name.toUpperCase()));
      if (p.Size > 1)
        sb.append("{ ");
      for (int i=0; i<p.Size; i++) {
        if (i > 0)
          sb.append(" , ");
        if (p.Type == PTYPE.CHAR)
          sb.append(String.format("%"+p.Format, p.Default[i].charAt(0)).trim());
        else if (p.Type == PTYPE.STRING)
          sb.append(String.format("%"+p.Format, "\""+unquote(p.Default[i])+"\"").trim());
        else if (p.Type == PTYPE.INT)
          sb.append(String.format("%"+p.Format, Integer.parseInt(p.Default[i].trim())).trim());
        else
          sb.append(String.format("%"+p.Format, Double.parseDouble(p.Default[i].trim().replace(',', '.'))).trim());
      }
      if (p.Size > 1)
        sb.append(" }");
      sb.append(";");
      bw.write(sb.toString());
      bw.newLine();
    }
    bw.write("  }");
    bw.newLine();
    bw.newLine();
    //
    // set a value
    bw.write("  public static Object PrmSetValue(String name, String value) throws Exception {"); 
    bw.newLine();
    bw.write("    Object ov = null;");
    bw.newLine();
    boolean first = true;
    for ( String key : hmparam.keySet() ) {
      Param p = getParam(key); 
      if (first) {
        bw.write("    if (name.equalsIgnoreCase(\""+key+"\")) {"); 
        first = false;
      }
      else
        bw.write("    else if (name.equalsIgnoreCase(\""+key+"\")) {"); 
      bw.newLine();
      if (p.Type == PTYPE.CHAR) {
        bw.write("      "+p.Name.toUpperCase()+" = value.charAt(0);"); bw.newLine();
        bw.write("      ov = value.charAt(0);"); bw.newLine();
      }
      else if (p.Type == PTYPE.STRING) {
        bw.write("      "+p.Name.toUpperCase()+" = value;"); bw.newLine();
        bw.write("      ov = value;"); bw.newLine();
      }
      else if (p.Type == PTYPE.INT) {
        bw.write("      "+p.Name.toUpperCase()+" = Integer.parseInt(value.trim());"); bw.newLine();
        bw.write("      ov = "+p.Name.toUpperCase()+";"); bw.newLine();
      }
      else {
        bw.write("      "+p.Name.toUpperCase()+" = Double.parseDouble(value.trim().replace(',', '.'));"); bw.newLine();
        bw.write("      ov = "+p.Name.toUpperCase()+";"); bw.newLine();
      }    
      bw.write("    }");
      bw.newLine();  
    }
    bw.write("    return ov;");
     bw.newLine();
    bw.write("  }");
    bw.newLine();
    bw.newLine();
    //
    // unquote
    bw.write("  public static String Unquote(String s) {"); 
    bw.newLine();
    bw.write("    if (s == null) return null;");
    bw.newLine();
    bw.write("    int l = s.length();");
    bw.newLine();
    bw.write("    if (l < 1 || s.charAt(0) != '\\\"') return s;");
    bw.newLine();
    bw.write("    if (s.charAt(l - 1) == '\\\"') l--;");
    bw.newLine();
    bw.write("    String t = (l > 1) ? s.substring(1, l) : \"\";");
    bw.newLine();
    bw.write("    return t;");
    bw.newLine();
    bw.write("  }");
    bw.newLine();
    bw.newLine();
    //
    // get a value
    bw.write("  public static String PrmGetValue(String name) throws Exception {"); 
    bw.newLine();
    bw.write("    String s = null;");
    bw.newLine();
    first = true;
    for ( String key : hmparam.keySet() ) {
      Param p = getParam(key);
      if (first) {
        bw.write("    if (name.equalsIgnoreCase(\""+key+"\"))"); 
        first = false;
      }
      else
        bw.write("    else if (name.equalsIgnoreCase(\""+key+"\"))"); 
      bw.newLine();
      if (p.Type == PTYPE.STRING)
        bw.write("      s = \"\\\"\"+String.format(\"%"+p.Format+"\", Unquote("+p.Name.toUpperCase()+")).trim()+\"\\\"\";");
      else
        bw.write("      s = String.format(\"%"+p.Format+"\", "+p.Name.toUpperCase()+");");
      bw.newLine();
    }
    bw.write("    return s;");
    bw.newLine();
    bw.write("  }");
    bw.newLine();
    bw.newLine();
    //
    bw.write("}");
    bw.close();
    if (DEBUG) System.out.printf("%s written.\n", fout.getAbsolutePath());
  }
  

  /**
   * @param args the command line arguments
   */
  public static void main(String[] args) {
    Locale.setDefault(Locale.ENGLISH);
    IBJparamSet ps = new IBJparamSet();
    try {
      ps.makeHeaderFile(args[0], args[1]);
    }
    catch (Exception e) {
      e.printStackTrace(System.out);
    }
  }
  
}
