// ================================================================ IBJhdr.java
//
// Utility for managing DMN headers
// ================================
//
// Copyright 2005-2022 Janicke Consulting, 88662 Überlingen
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation; either version 2 of
// the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
//
// History:
//
// 2007-01-04 format and parse date as well as long
// 2007-02-19 keys not case sensitive
// 2008-01-14  uj  long extensions; getDayOfWeek()
// 2008-12-02  lj  handling of timezone
// 2008-12-05  lj  merged with uj
// 2009-04-21  lj  Properties replaced by LinkedHashMap
// 2010-01-20  lj  parseTime(), parseDate() and parseDateLong() revised
// 2010-01-22  lj  class ZonedDate
// 2010-02-02  lj  contains() added
// 2010-02-16  uj  new function formatValidDate()
// 2010-06-08  uj  catch zero time zone in parseDateLong()
// 2011-05-24  lj  new function parseTimeDouble
//                 extended formatTime(double), formatDate(long)
// 2011-08-18  uj  parseTime: Allow constructs of the form .:: 
// 2011-10-31  lj  ZonedDate corrected for new ISO date format
// 2011-11-01  lj  merged with uj
// 2011-11-25  uj  getReferenceDate() passes Exception
// 2012-07-29  uj  some returns adjusted
// 2012-09-03  uj  date separator "T/." extracted to be used for LASAT 3.2
// 2012-09-14  uj  provide static routines for parsing floats
// 2012-10-15  uj  formatDate: with and without time zone
// 2013-03-11  uj  split("[:]) replaced by split("[:]", -1)
// 2013-08-19  uj  test modifications from beginning of 2013-08 deleted
// 2022-09-27  uj  getReferenceDate(): time zone extraction corrected
//
// =============================================================================

package de.janicke.ibjutil;

import java.io.OutputStream;
import java.io.PrintWriter;
import java.text.DecimalFormatSymbols;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.LinkedHashMap;
import java.util.Locale;
import java.util.Map;
import java.util.Set;
import java.util.TimeZone;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.regex.Pattern;

/**
 * 
 *  <p>Copyright 2005-2013 Ing.-Büro Janicke, 88662 Überlingen</p>
 * 
 * A header for {@link IBJarr IBJarr} containing key/value pairs, a locale
 * and a time zone. Both the keys
 * and the values are strings. Binary data are formatted according to format 
 * definitions supplied to the appropriate methods.  
 * See {@link IBJarr.Descriptor IBJarr.Descriptor} for data of type 
 * <code><i>DATE</i></code> or <code><i>TIME</i></code>.
 *
 * The parameters locale and time zone should be handeled by the specific
 * getter and setter only, not by getString() or setString().
 *
 * @author Janicke Consulting, Überlingen
 * @version 2013-08-19
 */

public class IBJhdr {
  
  /**
   * compatibility mode required to apply the current JAVA files for
   * the JAR packages of LASAT 3.2.
   * 
   * Effected issues:
   *
   * time format: 3.2: ".", 3.3: "T"
   * locale signature in DMNA files ("locl"): 3.2: "C"/"german", 3.3: "en"/"de"
   *
   **/  
  public static final boolean COMPAT0302 = false;
  public static final boolean USENEWLOCALES = false;
  
  /** Print check output to {@code System.err} */
  public static boolean CHECK = false;
  public static final long LHOUR = 3600000;
  public static final long LDAY = 86400000;
  public static final String date_separator = (COMPAT0302) ? "." : "T";
  public static final String date_format = "yyyy-MM-dd'"+date_separator+"'HH:mm:ss";
  public static final String date_formatz = "yyyy-MM-dd'"+date_separator+"'HH:mm:ssZ";
  public static final String default_date = "2006-01-01"+date_separator+"00:00:00";
  public static final String default_tmzn = "GMT+01:00";
  public static final String default_locl = "en";
  public Map<String, String> p;
  public TimeZone time_zone;
  public Locale locale;
  public char decimal_separator;  
  
  private static final double time_0 = 88609161600000.0;  // 1970-01-01.00:00:00 in ms
  private static boolean ParseChecked = false;
  
  public static String pointSep(String s) {
    if (s == null || s.length() <= 10)
      return s;
    if (s.charAt(10) == 'T')
      return s.substring(0,10) + "." + s.substring(11);
    else
      return s;
  }
  
  private static void checkParse() {
    double d = 0;
    String s = null;
    ParseChecked = false;
    try {
      d = Double.valueOf("3.5");         
      if (d != 3.5)
        throw new Exception();
      d = Double.parseDouble("3.5");
      if (d != 3.5)
        throw new Exception();     
      ParseChecked = true;
    }
    catch (Exception e) {
      System.out.printf("*** FATAL ERROR ENCOUNTERED BY IBJhdr: invalid float parsing (d=%1.3f)!\n    Program aborted.\n\n", d, s);
      System.exit(0);
    }
  }


  /**
   * Create a void header.
   * @param locl the locale to be used for representing numbers
   * @param tmzn the time zone to be used for representing dates
   */
  public IBJhdr(String locl, String tmzn) {
    p = new LinkedHashMap<String, String>();
    if (locl == null)
      locl = default_locl;
    if (tmzn == null)
      tmzn = default_tmzn;
    setLocale(locl);
    setTimeZone(tmzn);
}
  
  /**
   * Create a void header.
   */
  public IBJhdr() {
    this(null, null);
  }
  
  /**
   * Create a copy of this header.
   * @return the copy
   */
  public IBJhdr getCopy() {
    IBJhdr hdr = new IBJhdr(locale.getLanguage(), time_zone.getID());
    hdr.p = new LinkedHashMap<String, String>(p);
    return hdr;
  }
  
  /**
   * Create a copy of this header with changed values of locale or time_zone.
   * @param locl the new locale
   * @param tmzn the new time zone
   * @return the copy
   */
  public IBJhdr getCopy(String locl, String tmzn) {
    boolean change_dsep = false;
    char old_dsep = decimal_separator;
    char new_dsep = old_dsep;
    if (locl != null) {
      new_dsep = new DecimalFormatSymbols(new Locale(locl)).getDecimalSeparator();
      change_dsep = new_dsep != old_dsep;
    }
    boolean change_tmzn = (tmzn != null);
    if (change_tmzn && tmzn.equals(time_zone.getID()))
      change_tmzn = false;
    if (!change_dsep && !change_tmzn)
      return getCopy();
    //
    String s_float = null;
    String s_flvec = null;
    if (old_dsep == '.') 
      s_float = "[+-]?[0-9]+(\\.[0-9]*)?([eE][+-]?[0-9]+)?";
    else 
      s_float = "[+-]?[0-9]+(,[0-9]*)?([eE][+-]?[0-9]+)?";
    s_flvec = s_float + "([\\s]+" + s_float + ")*";
    Pattern p_flvec = Pattern.compile(s_flvec);
    Pattern p_date0 = Pattern.compile("[0-9]{4}-[0-9]{2}-[0-9]{2}.*");
    Pattern p_date1 = Pattern.compile("[0-9]{4}-[0-9]{2}-[0-9]{2}");
    Pattern p_date2 = Pattern.compile(
          "[0-9]{4}-[0-9]{2}-[0-9]{2}.?[0-9]{2}:[0-9]{2}:[0-9]{2}");
    SimpleDateFormat old_sdf = new SimpleDateFormat(date_format);
    old_sdf.setTimeZone(time_zone);
    SimpleDateFormat new_sdf = new SimpleDateFormat(date_format);
    if (change_tmzn)
      new_sdf.setTimeZone(TimeZone.getTimeZone(tmzn));
    IBJhdr hdr = new IBJhdr(locl, tmzn);
    hdr.p = new LinkedHashMap<String, String>();
    for (String key: p.keySet()) {
      String val = p.get(key).trim();
      if (!val.startsWith("\"")) {
        if (change_dsep && p_flvec.matcher(val).matches()) {
          val = val.replace(old_dsep, new_dsep);
        }
        else if (change_tmzn && p_date0.matcher(val).matches()) {
          if (p_date1.matcher(val).matches()) 
            val += date_separator+"00:00:00";
          if (p_date2.matcher(val).matches()) {
            val = val.substring(0, 10) + date_separator + val.substring(11);
            try {
              Date date = old_sdf.parse(val);
              val = new_sdf.format(date);
            } catch (ParseException ex) {
              Logger.getLogger(IBJhdr.class.getName()).log(Level.SEVERE, null, ex);
            }
          }
        } // change time zone
      }
      hdr.p.put(key, val);
    }
    return hdr;
  }
  
  /**
   * Get the map containing the header data
   * @return the map
   */
  public Map<String, String> getMap() {
    return p;
  }
  
  /**
   * Parse an array of strings for doubles.
   * @param ss array of strings to be parsed.
   * @return array of double values or {@code null} in case of parsing error.
   */
  public static double[] parseDoubles(String[] ss) {
    if (!ParseChecked) checkParse();
    double[] dd = null;
    try {
      int n = ss.length;
      dd = new double[n];
      for (int i = 0; i < n; i++)
        dd[i] = Double.valueOf(ss[i].replace(',', '.').trim());
    } catch (Exception e) {
      return null;
    }
    return dd;
  }
  
  /**
   * Parse a string for a double value.
   * @param s string to be parsed.
   * @return the double value or {@code Double.NaN} in case of parsing error.
   */
  public static double parseDouble(String s) {
    if (!ParseChecked) checkParse();
    double d = Double.NaN;
    try {
      d = Double.valueOf(s.replace(',', '.').trim());
    } 
    catch (Exception e) {}
    return d;
  }
  
  /**
   * Parse a string for a double value.
   * @param s string to be parsed.
   * @return the double value or an Exception in case of parsing error.
   * @throws java.lang.Exception
   */
  public static double parseDoubleOf(String s) throws Exception {
    if (!ParseChecked) checkParse();
    double d = Double.valueOf(s.replace(',', '.').trim());
    return d;
  }
  
  
  /**
   * Parse an array of strings for float values.
   * @param ss array of strings to be parsed.
   * @return array of float values or {@code null} in case of parsing error.
   */
  public static float[] parseFloats(String[] ss) {
    if (!ParseChecked) checkParse();
    float[] ff = null;
    try {
      int n = ss.length;
      ff = new float[n];
      for (int i = 0; i < n; i++)
        ff[i] = Float.valueOf(ss[i].replace(',', '.').trim());
    } catch (Exception e) {
      return null;
    }
    return ff;
  }
  
  /**
   * Parse a string for a float value.
   * @param s string to be parsed.
   * @return the float value or {@code Float.NaN} in case of parsing error.
   */
  public static float parseFloat(String s) {
    if (!ParseChecked) checkParse();
    float f = Float.NaN;
    try {
      f = Float.valueOf(s.replace(',', '.').trim());
    } catch (Exception e) {}
    return f;
  }
  
  /**
   * Parse a string for a float value.
   * @param s string to be parsed.
   * @return the float value or an Exception in case of parsing error.
   * @throws java.lang.Exception
   */
  public static float parseFloatOf(String s) throws Exception {
    if (!ParseChecked) checkParse();
    float f = Float.valueOf(s.replace(',', '.').trim());
    return f;
  }
  
  /**
   * Parse an array of strings for integer values.
   * @param ss array of strings to be parsed.
   * @return array of integer values or {@code null} in case of parsing error.
   */
  public static int[] parseIntegers(String[] ss) {
    int[] ii;
    try {
      int n = ss.length;
      ii = new int[n];
      for (int i = 0; i < n; i++)
        ii[i] = Integer.valueOf(ss[i].trim());
    } catch (Exception e) {
      return null;
    }
    return ii;
  }
  
  /**
   * Parse a string for an integer value.
   * @param s string to be parsed.
   * @return the integer value or {@code 0} in case of parsing error.
   */
  public static int parseInteger(String s) {
    int i = 0;
    try {
      i = Integer.valueOf(s.trim());
    }
    catch (Exception e) {}
    return i;
  }

  /**
   * Parse an array of strings for long values.
   * @param ss array of strings to be parsed.
   * @return array of long values or {@code null} in case of parsing error.
   */
  public static long[] parseLongs(String[] ss) {
    long[] ll = null;
    try {
      int n = ss.length;
      ll = new long[n];
      for (int i = 0; i < n; i++)
        ll[i] = Long.valueOf(ss[i].trim());
    } catch (Exception e) {
      return null;
    }
    return ll;
  }
  
  /**
   * Parse a string for a long value.
   * @param s string to be parsed.
   * @return the long value or {@code 0} in case of parsing error.
   */
  public static long parseLong(String s) {
    long l = 0;
    try {
      l = Long.valueOf(s.trim());
    } catch (Exception e) {}
    return l;
  }

  /**
   * Parse an array of strings for time values formatted as DDD.HH:mm:ss.
   * @param ss array of strings to be parsed.
   * @return array of int values (times) or {@code null} in case of parsing error.
   */
  public static int[] parseTimes(String[] ss) {
    int[] ii;
    try {
      int n = ss.length;
      ii = new int[n];
      for (int i = 0; i < n; i++)
        ii[i] = parseTime(ss[i]);
    } catch (Exception e) {
      return null;
    }
    return ii;
  }
  
  /**
   * Parse a string for a time value formatted as DDD.HH:mm:ss.
   * @param s string to be parsed.
   * @return the int value or {@code -1} in case of parsing error.
   */
  public static int parseTime(String s) {
    int time=0, days=0, hours=0, minutes=0, seconds=0;
    if (s == null)                                                //-2010-01-20
      return -1;
    s = s.toLowerCase().trim();
    if (s.startsWith("-inf"))
      return Integer.MIN_VALUE;
    if (s.startsWith("+inf"))
      return Integer.MAX_VALUE;
    s = s.replace(",", ".");                                      //-2011-05-25
    try {
      String[] ss = s.split("[.]");
      if (ss.length == 2) {
        days = Integer.valueOf(ss[0]);
        ss[0] = ss[1];
      }
      String[] tt = ss[0].split("[:]", -1);                       //-2013-03-11
      if (tt.length == 3) {
        hours = (tt[0].length() == 0) ? 0 : Integer.valueOf(tt[0]); //-2011-08-18
        tt[0] = tt[1];
        tt[1] = tt[2];
      }
      if (tt.length >= 2) {
        minutes = (tt[0].length() == 0) ? 0 : Integer.valueOf(tt[0]); //-2011-08-18
        tt[0] = tt[1];
      }
      seconds = (tt[0].length() == 0) ? 0 : Integer.valueOf(tt[0]); //-2011-08-18
      if (days < 0 || hours < 0 || minutes < 0 || seconds < 0)    //-2010-01-20
        time = -1;
      else
        time = seconds + 60*(minutes + 60*(hours + 24*days));
    }
    catch (Exception e) {
      if (CHECK) e.printStackTrace(System.out);
      time = -1;                                                  //-2010-01-20
    }
    return time;
  }
  
  /**
   * Parse a string for a time value formatted as [-][{D}.]HH:mm:ss[.{S}].
   * @param s string to be parsed.
   * @return the double value or {@code NaN} in case of parsing error.
   */
  public static double parseTimeDouble(String s) {
    int days=0, hours=0, minutes=0, sign=1;
    double time=0, seconds=0;
    if (s == null)                                           
      return Double.NaN;
    s = s.toLowerCase().trim();
    if (s.startsWith("-inf"))
      return Double.NEGATIVE_INFINITY;
    if (s.startsWith("+inf"))
      return Double.POSITIVE_INFINITY;
    s = s.replace(",", ".");
    try {
      String[] tt = s.split("[:]", -1);                         //-2013-03-11
      if (tt.length != 3)
        return Double.NaN;
      String t = tt[0].trim();
      if (t.startsWith("-")) {
        sign = -1;
        t = t.substring(1);
      }
      String[] ss = t.split("[.]");
      if (ss.length > 1) {
        days = Integer.parseInt(ss[0].trim());
        hours = Integer.parseInt(ss[1].trim());
      }
      else {
        hours = Integer.parseInt(ss[0].trim());
      }
      minutes = Integer.parseInt(tt[1].trim());
      seconds = parseDoubleOf(tt[2]);
      if (days < 0 || hours < 0 || minutes < 0 || seconds < 0)  
        time = Double.NaN;
      else
        time = sign*(seconds + 60*(minutes + 60*(hours + 24*days)));
    }
    catch (Exception e) {
      time = Double.NaN;                                     
    }
//    
//    System.out.printf(
//            "### parseTimeDouble(%s): sec=%f, min=%d, hour=%d, day=%d, time=%f\n", 
//            s, seconds, minutes, hours, days, time);
//    
    return time;
  }

  
  /**
   * Parse an array of strings for date values formatted as yyyy-MM-dd.HH:mm:ss.
   * @param ss array of strings to be parsed.
   * @param zone the name of the time zone to be used.
   * @return array of double values (dates) or {@code null} in case of parsing error.
   */
  public static double[] parseDates(String[] ss, String zone) {
    if (zone == null)                                             //-2010-01-22
      zone = default_tmzn;
    TimeZone tz = TimeZone.getTimeZone(zone);
    return parseDates(ss, tz);
  }
        
  /**
   * Parse an array of strings for date values formatted as yyyy-MM-dd.HH:mm:ss.
   * @param ss array of strings to be parsed.
   * @param tz the time zone to be used.
   * @return array of double values (dates) or {@code null} in case of parsing error.
   */
  public static double[] parseDates(String[] ss, TimeZone tz) {
    double[] dd;
    try {
      int n = ss.length;
      dd = new double[n];
      for (int i = 0; i < n; i++)
        dd[i] = parseDate(ss[i], tz);
    } catch (Exception e) {
      return null;
    }
    return dd;
  }
  
  /**
   * Parse a string for a date value formatted as yyyy-MM-dd.HH:mm:ss.
   * @param s the strings to be parsed.
   * @param zone the name of the time zone to be used.
   * @return the double value (date) or {@code Double.NaN} in case of parsing error.
   */
  public static double parseDate(String s, String zone) {
    if (zone == null)                                             //-2010-01-22
      zone = default_tmzn;
    TimeZone tz = TimeZone.getTimeZone(zone);
    return parseDate(s, tz);
  }
  
  /**
   * Parse a string for a date value formatted as yyyy-MM-dd.HH:mm:ss.
   * @param s the strings to be parsed.
   * @param tz the time zone to be used.
   * @return the double value (date) or {@code Double.NaN} in case of parsing error.
   */
  public static double parseDate(String s, TimeZone tz) {
    s = s.toLowerCase();
    if (s.startsWith("-inf"))
      return Double.NEGATIVE_INFINITY;
    if (s.startsWith("+inf"))
      return Double.POSITIVE_INFINITY;
    long ms = parseDateLong(s, tz);
    if (ms < 0)
      return Double.NaN;
    double d = (ms + time_0)/(3600*24*1000);
    return d;
  }
  
  /**
   * Parse a string for a date value formatted as yyyy-MM-dd.HH:mm:ssZ.
   * @param s the strings to be parsed.
   * @return the long value (date) or -1 in case of parsing error.
   */
  public static long parseDateZ(String s) {                    //-2017-03-28
    if (s == null)
      return -1;
    s = s.toUpperCase().trim();
    if (s.contains(" ") || s.contains("GMT") || s.contains("UTC"))
      return -1;
    else if (s.startsWith("-INF"))
      return Long.MIN_VALUE;
    else if (s.startsWith("+INF"))
      return Long.MAX_VALUE;
    long ms = -1;
    try {
      SimpleDateFormat sdf = new SimpleDateFormat(date_formatz); 
      Date dt = null;
      int l = s.length();
      if (l < 10)
        return -1;
      else if (l == 10) 
        s += date_separator+"00:00:00";
      else {
        if (s.charAt(10) != date_separator.charAt(0))
          s = s.substring(0, 10) + date_separator + s.substring(11);
      }
      if (s.length() == 19)
        s += "+0100";
      else if (s.length() == 21) {
        s = s.substring(0,20)+"0"+s.substring(20,21);
      }
      else if (s.length() == 22);
        s += "00";
      /*
      if (s.length() > 19) {
        System.out.printf("### s=%s, len=%d\n", s, s.length());
        if (s.length() == 22) {
          int hh = Integer.parseInt(s.substring(20,22));
        }
        else if (s.length() == 24) {
          int hh = Integer.parseInt(s.substring(20,22));
          int mm = Integer.parseInt(s.substring(22,24));
          System.out.printf("### hh=%d, mm=%d\n", hh, mm);
        }
        else
          throw new Exception();
      }
      */
      dt = sdf.parse(s);
      ms = dt.getTime();
    }
    catch (Exception e) {
      //e.printStackTrace();
      ms = -1;
    }
    return ms;
  }
  
  /**
   * Parse a string for a date value formatted as yyyy-MM-dd.HH:mm:ss.
   * @param s the strings to be parsed.
   * @param tz the time zone to be used.
   * @return the long value (date) or -1 in case of parsing error.
   */
  public static long parseDateLong(String s, TimeZone tz) {
    s = s.toUpperCase();                                          //-2011-05-23
    if (s.startsWith("-INF"))
      return Long.MIN_VALUE;
    else if (s.startsWith("+INF"))
      return Long.MAX_VALUE;
    long ms = -1;
    try {
      SimpleDateFormat sdf = new SimpleDateFormat(date_format + ".SSS"); 
      Date dt = null;
      int i = s.indexOf("GMT");
      int l = s.length();
      if (i > 0) {
        tz = TimeZone.getTimeZone(s.substring(i));
        s = s.substring(0, i).trim();
      }
      else if (l > 19) {
        if ((s.charAt(l-3) == '-') || (s.charAt(l-3) == '+')) {
          tz = TimeZone.getTimeZone("GMT" + s.substring(l-3));
          s = s.substring(0, l-3).trim();
        }
        else if ((s.charAt(l-5) == '-') || (s.charAt(l-5) == '+')) {
          tz = TimeZone.getTimeZone("GMT" + s.substring(l-5));
          s = s.substring(0, l-5).trim();
        }
      }
      l = s.length();
      if (l < 10)
        return -1;
      else if (l == 10) 
        s += date_separator+"00:00:00.000";
      else {
        if (s.charAt(10) != date_separator.charAt(0))
          s = s.substring(0, 10) + date_separator + s.substring(11);
      }
      l = s.length();
      i = s.lastIndexOf('.');                            //-2012-09-03 uj (with date_separator "." the "." can appear twice)
      if (i < 0 || (i == 10 && l > 11))
        s += ".000";
      else if (l-4 < i)
        s += "000".substring(0, i-l+4);
      else 
        s = s.substring(0, i+4);
      sdf.setTimeZone((tz != null) ? tz : TimeZone.getTimeZone(default_tmzn));
      dt = sdf.parse(s);
      ms = dt.getTime();
    }
    catch (Exception e) {
      //e.printStackTrace();
      ms = -1;
    }
    return ms;
  }


  /**
   * Unquote a string.
   * 
   * @param s
   *          the original string
   * @return the unquoted string
   */
  public static String unquote(String s) {
    if (s == null)
      return null;
    int l = s.length();
    if (l < 1 || s.charAt(0) != '\"')
      return s;
    if (s.charAt(l - 1) == '\"')
      l--;
    String t = (l > 1) ? s.substring(1, l) : "";
    return t;
  }
  
  /**
   * Unquote an array of strings.
   * @param ss array of quoted strings
   * @return array of unquoted strings
   */
  public static String[] unquote(String[] ss) {
    if (ss == null)  return null;
    int n = ss.length;
    String[] tt = new String[n];
    for (int i=0; i<n; i++)
      tt[i] = unquote(ss[i]);
    return tt;
  }
  
  /**
   * Format a time value as DDD.HH:mm:ss.
   * @param f time (milliseconds).
   * @return the formatted time.
   */
  public static String formatTime(long f) {
    if (f == Long.MIN_VALUE) return "          -inf";
    if (f == Long.MAX_VALUE) return "          +inf";
    return IBJhdr.formatTime((int)(f/1000));
  }
  
  /**
   * Format a time value as DDD.HH:mm:ss.
   * @param f time (seconds).
   * @return the formatted time.
   */
  public static String formatTime(int f) {
    int sec, min, hour, day;
    char sign;
    if (f == Integer.MIN_VALUE)                                   //-2012-07-29
      return "          -inf";
    if (f == Integer.MAX_VALUE)                                   //-2012-07-29
      return "          +inf"; 
    if (f < 0) {
      sign = '-';
      f = -f;
    }
    else sign = ' ';
    sec = f % 60;
    f /= 60;
    min = f % 60;
    f /= 60;
    hour = f % 24;
    f /= 24;
    day = f;
    String s = String.format("    %c%d.%02d:%02d:%02d", sign, day, hour, min, sec);
    s = s.substring(s.length()-14);    
    return s;
  }
  
  /**
   * Format a time value as [-][{D}.]HH:mm:ss[.{S}].
   * @param f time (seconds).
   * @param format the time format as %width[.precision]T
   * @return the formatted time.
   */
  public static String formatTime(double f, String format) {
    int min, hour, day, g;
    double sec;
    char sign;
    String fs = "%02d:%02d:";
    int width = 8;
    int prec = 0;
    String s = null;
    if (f == Double.POSITIVE_INFINITY)
      return "+inf";
    if (f == Double.NEGATIVE_INFINITY)
      return "-inf";
    if (Double.isNaN(f))
      return "nan";
    if (f < 0) {
      sign = '-';
      f = -f;
    }
    else sign = ' ';
    sec = f % 60;
    g = (int)(f/60);
    min = g % 60;
    g /= 60;
    hour = g % 24;
    g /= 24;
    day = g;
    //
    if (format != null && format.startsWith("%") && format.endsWith(date_separator)) {
      String[] ss = format.substring(1, format.length()-1).split("[.]");
      try {
        if (ss.length >= 1)
          width = Integer.parseInt(ss[0]);
        if (ss.length >= 2)
          prec = Integer.parseInt(ss[1]);
      } catch (Exception e) {}
      if (width < 8)
        width = 8;
      if (prec > 0) 
        fs += String.format("%s%d.%df", "%0", prec+3, prec);
      else
        fs += "%02.0f";
      s = String.format(Locale.ENGLISH, fs, hour, min, sec);
      if (day > 0 || width > s.length()+1)
        s = String.format("%d.", day) + s;
    }
    else 
      s = String.format(Locale.ENGLISH, "%d.%02d:%02d:%02.0f", 
              day, hour, min, sec);
    if (sign == '-')
      s = "-" + s;
    int n = width - s.length();
    if (n > 0) {
      StringBuilder sb = new StringBuilder();
      for (int i=0; i<n; i++)
        sb.append(' ');
      sb.append(s);
      s = sb.toString();
    }
//    
//    System.out.printf(
//            "### formatTime(%f, %s): sec=%f, min=%d, hour=%d, day=%d, width=%d, n=%d, fs=%s, s=%s\n", 
//            f, format, sec, min, hour, day, width, n, fs, s);
//    
    return s;
  }
  
  /**
   * Format a date value as yyyy-MM-dd.HH:mm:ss.
   * @param d the date as double.
   * @param tz the time zone to be used.
   * @return the formatted time.
   */
  public static String formatDate(double d, TimeZone tz) {
    if (d < -1.e-5)
      return "-inf";
    if (d < 0)  d = 0;                
    if (d >= 9999999)
      return "+inf";
    long ms = Math.round(d*3600*24*1000 - time_0);
    Date dt = new Date(ms);
    SimpleDateFormat sdf = new SimpleDateFormat(date_format);
    sdf.setTimeZone(tz);
    String dstr = sdf.format(dt);
    return dstr;
  }
  
  /**
   * Format a date value as yyyy-MM-dd'T'HH:mm:ssZ.
   * @param l the date as long.
   * @param tz the time zone to be used.
   * @return the formatted time.
   */
  public static String formatDateZ(long l, TimeZone tz) {        //-2017-03-28
    if (l == Long.MIN_VALUE)
      return "-inf";
    if (l == Long.MAX_VALUE)
      return "+inf";
    if (l < 0)  l = Calendar.getInstance().getTimeInMillis();
    if (tz == null)
      tz = TimeZone.getTimeZone(default_tmzn);
    SimpleDateFormat sdf = new SimpleDateFormat(date_formatz);
    sdf.setTimeZone(tz);
    String dstr = sdf.format(new Date(l));
    return dstr;
  }
  
  /**
   * Format a date value as yyyy-MM-dd'T'HH:mm:ss.
   * @param l the date as long.
   * @param tz the time zone to be used.
   * @return the formatted time.
   */
  public static String formatDate(long l, TimeZone tz) {
    if (l == Long.MIN_VALUE)
      return "-inf";
    if (l == Long.MAX_VALUE)
      return "+inf";
    if (l < 0)  l = Calendar.getInstance().getTimeInMillis();
    if (tz == null)                                               //-2010-01-22
      tz = TimeZone.getTimeZone(default_tmzn);
    SimpleDateFormat sdf = new SimpleDateFormat(date_format);
    sdf.setTimeZone(tz);
    String dstr = sdf.format(new Date(l));
    return dstr;
  }
  
  /**
   * Format a date value as yyyy-MM-dd'T'HH:mm:ss.
   * @param l the date as long.
   * @param tz the time zone to be used.
   * @param format the format as %width[.precision]D
   * @return the formatted time.
   */
  public static String formatDate(long l, TimeZone tz, String format) {
    int width = 28;
    int prec = 3;
    String fs = "yyyy-MM-dd'"+date_separator+"'HH:mm:ss.SSSZ";
    String fs_short = "yyyy-MM-dd'"+date_separator+"'HH:mm:ss";
    if (l == Long.MIN_VALUE)
      return "-inf";
    if (l == Long.MAX_VALUE)
      return "+inf";
    if (l < 0)  l = Calendar.getInstance().getTimeInMillis();
    if (tz == null)                                             
      tz = TimeZone.getTimeZone(default_tmzn);
    if (format != null && format.startsWith("%") && format.endsWith("lt")) { //-2012-10-15
      try {
        int w = Integer.parseInt(format.substring(1, format.length()-2));
        if (w < 23)
          fs = fs_short;
      }
      catch (Exception e) {}
    }
    else if (format != null && format.startsWith("%") && format.endsWith("D")) {
      String[] ss = format.substring(1, format.length()-1).split("[.]");
      try {
        if (ss.length >= 1)
          width = Integer.parseInt(ss[0]);
        if (ss.length >= 2)
          prec = Integer.parseInt(ss[1]);
      } catch (Exception e) {}
      if (width < 10)
        width = 10;
      if (prec < 0)
        prec = 0;
      else if (prec > 0)
        prec = 3;
      fs = "yyyy-MM-dd";
      if (width >= 19)
        fs += "'"+date_separator+"'HH:mm:ss";
      if (prec > 0 && width >= 23)
        fs += ".SSS";
      if (width >= fs.length() + 3)
        fs += "Z";
    }
    SimpleDateFormat sdf = new SimpleDateFormat(fs);
    sdf.setTimeZone(tz);
    String dstr = sdf.format(new Date(l));
    if (width > dstr.length()) {
      StringBuilder sb = new StringBuilder();
      for (int i=0; i<width-dstr.length(); i++)
        sb.append(' ');
      sb.append(dstr);
      dstr = sb.toString();
    }
    return dstr;
  }
  
  /**
   * Format a date value as yyyy-MM-dd'T'HH:mm:ss.
   * @param l the date as long.
   * @param tz the time zone to be used.
   * @return the formatted time.
   */
  public static String formatValidDate(long l, TimeZone tz) {
    if (l <= 0 && l != Long.MIN_VALUE)
      return "                   ";
    if (l == Long.MIN_VALUE)  
      return "-inf";   
    if (l == Long.MAX_VALUE)  
      return "+inf";  
    if (tz == null)
      tz = TimeZone.getTimeZone(default_tmzn);
    SimpleDateFormat sdf = new SimpleDateFormat(date_format);
    sdf.setTimeZone(tz);
    String dstr = sdf.format(new Date(l));
    return dstr;
  }
  
  /**
   * Returns the day of the week (1: Monday to 7: Sunday; or -1)
   * @param t  the date
   * @param tz the time zone
   * @return day-of-week index
   */
  public static int getDayOfWeek(long t, TimeZone tz) {
    Calendar cl = Calendar.getInstance(tz);
    cl.setTimeInMillis(t);
    int d = cl.get(Calendar.DAY_OF_WEEK);
    if (d == Calendar.MONDAY)
      d = 1;
    else if (d == Calendar.TUESDAY)
      d = 2;
    else if (d == Calendar.WEDNESDAY)
      d = 3;
    else if (d == Calendar.THURSDAY)
      d = 4;
    else if (d == Calendar.FRIDAY)
      d = 5;
    else if (d == Calendar.SATURDAY)
      d = 6;
    else if (d == Calendar.SUNDAY)
      d = 7;
    else
      d = -1;
    return d;   
  }
  
  /**
   * Convert date from double to long
   * @param d the date as double.
   * @return the date as long.
   */
  public static long longDate(double d) {  
    return Math.round(d*LDAY - time_0);
  }
  
  public static double doubleDate(long l) {
    return (l + time_0)/LDAY;
  }

  
  /**
   * Convert a <code><i>DATE</i></code> into a {@code java.util.Date}.
   * @param d the date as double.
   * @return the Date.
   */
  public static Date getJDate(double d) {
    long ms = Math.round(d*LDAY - time_0);
    return new Date(ms);   
  }
  
  /**
   * Convert a {@code java.util.Date} into a <code><i>DATE</i></code>.
   * @param dt the Date.
   * @return the date value as double.
   */
  public static double getDate(Date dt) {
    return (dt.getTime() + time_0)/LDAY;    
  }
  
  /**
   * Checks a date value.
   * @param d the date value.
   * @return is plus infinity.
   */
  public static boolean isPlusInfTime(double d) {
    return (d >= 9999999);
  }
  
  /**
   * Checks a date value.
   * @param d the date value.
   * @return is minus infinity.
   */
  public static boolean isNegInfTime(double d) {
    return (d < -1.0e-5);
  }

  /**
   * Tokenize a line with quoted tokens.
   * 
   * @param line
   *          the line to be tokenized.
   * @return the tokens found.
   */
  static public String[] tokenize(String line) {
    if (line == null)
      return null;
    ArrayList<String> vv = new ArrayList<String>();
    String[] ss = null;
    int l = line.length();
    for (int i=0; i<l; i++) {
      for (; i < l; i++)
        if (!Character.isWhitespace(line.charAt(i)))
          break;
      if (i >= l)
        break;
      int i1 = i;
      if (line.charAt(i) == '\"') {
        i++;
        for (; i < l; i++) {
          if (line.charAt(i) == '\"') {
            if (line.charAt(i-1) == '\\') continue;
            i++;
            break;
          }
        }
      } 
      else {
        for (; i < l; i++)
          if (Character.isWhitespace(line.charAt(i)))
            break;
      }
      vv.add(line.substring(i1, i));
    }
    ss = vv.toArray(new String[] {});
    return ss;
  }
  
  //-----------------------------------------------------------------------

  public boolean contains(String key) {
    return p.containsKey(key);
  }
  
  /**
   * Set the time zone for formatted dates.
   * @param zone the name of the time zone.
   */
  final void setTimeZone(String zone) {
    if (zone == null)
      zone = default_tmzn;
    time_zone = TimeZone.getTimeZone(zone);
  }
  
  /**
   * Get the time zone currently used for formatted dates.
   * @return the time zone.
   */
  public String getTimeZone() {
    return time_zone.getID();
  }
  
  /**
   * Set the locale to be used for formatting floating point numbers. 
   * @param loc the new locale.
   */
  final void setLocale(String loc) {
    if (loc == null)
      loc = default_locl;
    else {
      loc = unquote(loc);
      if (loc.equalsIgnoreCase("C"))
        loc = "en";
      else if (loc.equalsIgnoreCase("german"))
        loc = "de";
    }
    locale = new Locale(loc);
    decimal_separator = new DecimalFormatSymbols(locale).getDecimalSeparator();
  }
    
  /**
   * Get the locale currently used for formatting floating point numbers.
   * @return the locale.
   */
  public String getLocale() {
    return locale.getLanguage();
  }
  
  public char getDecimalSeparator() {
    return decimal_separator;
  }
  
  /**
   * Get the double values stored with this key.
   * 
   * @param key
   *          the key to be used.
   * @return the values or <code>null</code>, if the key is not found or a
   *         parsing error occurs.
   */
  public double[] getDoubles(String key) {
    if (key == null)
      return null;
    double[] dd = parseDoubles(getStrings(key));
    return dd;
  }
  
  /**
   * Get the double value stored with this key.
   * 
   * @param key
   *          the key to be used.
   * @return the value or <code>NaN</code>, if the key is not found or a
   *         parsing error occurs.
   */
  public double getDouble(String key) {
    double d = Double.NaN;
    if (key == null)
      return d;
    String s = getString(key);
    d = parseDouble(s);
    return d;
  }
  
  /**
   * Get the float values stored with this key.
   * 
   * @param key
   *          the key to be used.
   * @return the values or <code>null</code>, if the key is not found or a
   *         parsing error occurs.
   */
  public float[] getFloats(String key) {
    if (key == null)
      return null;
    String[] ss = getStrings(key);
    float[] ff = parseFloats(ss);
    return ff;
  }
  
  /**
   * Get the float value stored with this key.
   * 
   * @param key
   *          the key to be used.
   * @return the value or <code>Float.NaN</code>, if the key is not found or a
   *         parsing error occurs.
   */
  public float getFloat(String key) {
    float f = Float.NaN;
    if (key == null)
      return f;
    String s = getString(key);
    f = parseFloat(s);
    return f;
  }
  
  /**
   * Get the integer values stored with this key.
   * 
   * @param key
   *          the key to be used.
   * @return the values or <code>null</code>, if the key is not found or a
   *         parsing error occurs.
   */
  public int[] getIntegers(String key) {
    if (key == null)
      return null;
    String[] ss = getStrings(key);
    int[] ii = parseIntegers(ss);
    return ii;
  }
  
  /**
   * Get the integer value stored with this key.
   * 
   * @param key
   *          the key to be used.
   * @return the value or <code>0</code>, if the key is not found or a
   *         parsing error occurs.
   */
  public int getInteger(String key) {
    if (key == null)
      return 0;
    String s = getString(key);
    int i = parseInteger(s);
    return i;
  }
  
  /**
   * Get the long values stored with this key.
   * 
   * @param key
   *          the key to be used.
   * @return the values or <code>null</code>, if the key is not found or a
   *         parsing error occurs.
   */
  public long[] getLongs(String key) {
    if (key == null)
      return null;
    String[] ss = getStrings(key);
    long[] ll = parseLongs(ss);
    return ll;
  }
  
  /**
   * Get the long value stored with this key.
   * 
   * @param key
   *          the key to be used.
   * @return the value or <code>0</code>, if the key is not found or a
   *         parsing error occurs.
   */
  public long getLong(String key) {
    if (key == null)
      return 0;
    String s = getString(key);
    long l = parseLong(s);
    return l;
  }
  
  /**
   * Get the string values stored with this key.
   * 
   * @param key
   *          the key to be used.
   * @return the values or <code>null</code>, if the key is not found.
   */
  public String[] getStrings(String key) {
    return getStrings(key, false);
  }
  
  /**
   * Get the string values stored with this key.
   * 
   * @param key
   *          the key to be used.
   * @param unquote unquote strings.
   * @return the values or <code>null</code>, if the key is not found.
   */
  public String[] getStrings(String key, boolean unquote) {
    if (key == null)
      return null;
    String[] ss = null;
    String s = null;
    int i1 = 0, i2 = 0, l = key.length();
    while (i2 < l) {
      i2 = key.indexOf('|', i1);
      if (i2 < 0)
        i2 = l;
      s = p.get(key.substring(i1, i2).toLowerCase());
      if (s != null)
        break;
      i1 = i2 + 1;
    }
    if (s == null)  return null;
    ss = tokenize(s);
    if (unquote) {
      for (int i=0; i<ss.length; i++)
        ss[i] = unquote(ss[i]);
    }
    return ss;
  }
  
  /**
   * Get the string value stored with this key.
   * 
   * @param key
   *          the key to be used.
   * @return the value or <code>null</code>, if the key is not found.
   */ 
  public String getString(String key) {
    return getString(key, false);
  }
  
  /**
   * Get the string value stored with this key.
   * 
   * @param key
   *          the key to be used.
   * @param unquote unquote the string.
   * @return the value or <code>null</code>, if the key is not found.
   */
  public String getString(String key, boolean unquote) {
    if (key == null)
      return null;
    String s = null;
    int i1 = 0, i2 = 0, l = key.length();
    while (i2 < l) {
      i2 = key.indexOf('|', i1);
      if (i2 < 0)
        i2 = l;
      s = p.get(key.substring(i1, i2).toLowerCase());
      if (s != null)
        break;
      i1 = i2 + 1;
    }
    if (unquote && s != null) s = unquote(s);
    return s;
  }

  /**
   * Get the time values stored with this key.
   * 
   * @param key
   *          the key to be used.
   * @return the values or <code>null</code>, if the key is not found or a
   *         parsing error occurs.
   */
  public int[] getTimes(String key) {
    if (key == null)
      return null;
    String[] ss = getStrings(key, true);
    int[] ii = parseTimes(ss);
    return ii;
  }
  
  /**
   * Get the time value stored with this key.
   * 
   * @param key
   *          the key to be used.
   * @return the value or <code>0</code>, if the key is not found or a
   *         parsing error occurs.
   */
  public int getTime(String key) {
    if (key == null)
      return 0;
    String s = getString(key, true);
    int i = parseTime(s);
    return i;
  }

  /**
   * Get the date values stored with this key.
   * 
   * @param key
   *          the key to be used.
   * @return the values or <code>null</code>, if the key is not found or a
   *         parsing error occurs.
   */
  public double[] getDates(String key) {
    if (key == null)
      return null;
    String[] ss = getStrings(key, true);
    if (ss == null)                                      //-2012-07-29
      return null;
    double[] dd = parseDates(ss, time_zone);
    return dd;
  }
  
  /**
   * Get the date value stored with this key.
   * 
   * @param key
   *          the key to be used.
   * @return the value or <code>Double.NaN</code>, if the key is not found or a
   *         parsing error occurs.
   */
  public double getDate(String key) {
    if (key == null)
      return 0;
    String s = getString(key, true);
    if (s == null)                                         //-2012-07-29
      return Double.NaN;
    double d = parseDate(s, time_zone);
    return d;
  }
  
  /**
   * Change the key of an entry.
   * @param oldKey the old key.
   * @param newKey the new key.
   * @return the old key or {@code null} if the key doesn't exist.
   */
  public String rename(String oldKey, String newKey) {
    if (oldKey == null)
      return null;
    String s = null;
    String key = null;
    int i1 = 0, i2 = 0, l = oldKey.length();
    while (i2 < l) {
      i2 = oldKey.indexOf('|', i1);
      if (i2 < 0)
        i2 = l;
      key = oldKey.substring(i1, i2).toLowerCase();
      s = p.get(key);
      if (s != null)
        break;
      i1 = i2 + 1;
    }
    if (s == null)  key = null;
    else p.remove(key);
    if (newKey != null)  p.put(newKey.toLowerCase(), s);
    return key;
  }
  
  /**
   * Format the double value and store it under the given key. 
   * @param key the key to be used.
   * @param d the double value.
   * @param format the JAVA format for formatting.
   */
  public void putDouble(String key, double d, String format) {
    String s = String.format(locale, format, d);
    if (decimal_separator != '.')
      s = s.replace('.', decimal_separator);
    p.put(key.toLowerCase(), s);
  }

  /**
   * Format the double values and store them under the given key. 
   * @param key the key to be used.
   * @param dd the double values.
   * @param format the JAVA format for formatting.
   */
  public void putDoubles(String key, double[] dd, String format) {
    int n = dd.length;
    StringBuilder sb = new StringBuilder();
    for (int i=0; i<n; i++) {
      if (i > 0) sb.append(' ');
      String s = String.format(locale, format, dd[i]);
      if (decimal_separator != '.')
        s = s.replace('.', decimal_separator);
      sb.append(s);
    }
    p.put(key.toLowerCase(), sb.toString());
  }

  /**
   * Format the float value and store it under the given key. 
   * @param key the key to be used.
   * @param f the float value.
   * @param format the JAVA format for formatting.
   */
  public void putFloat(String key, float f, String format) {
    String s = String.format(locale, format, f);
    if (decimal_separator != '.')
      s = s.replace('.', decimal_separator);
    p.put(key.toLowerCase(), s);
  }

  /**
   * Format the float values and store them under the given key. 
   * @param key the key to be used.
   * @param ff the float values.
   * @param format the JAVA format for formatting.
   */
  public void putFloats(String key, float[] ff, String format) {
    int n = ff.length;
    StringBuilder sb = new StringBuilder();
    for (int i=0; i<n; i++) {
      if (i > 0) sb.append(' ');
      String s = String.format(locale, format, ff[i]);
      if (decimal_separator != '.')
        s = s.replace('.', decimal_separator);
      sb.append(s);
    }
    p.put(key.toLowerCase(), sb.toString());
  }

  /**
   * Format the integer value and store it under the given key. 
   * @param key the key to be used.
   * @param i the integer value.
   * @param format the JAVA format for formatting.
   */
  public void putInteger(String key, int i, String format) {
    String s = String.format(locale, format, i);
    p.put(key.toLowerCase(), s);
  }

  /**
   * Format the integer values and store them under the given key. 
   * @param key the key to be used.
   * @param ii the integer values.
   * @param format the JAVA format for formatting.
   */
  public void putIntegers(String key, int[] ii, String format) {
    int n = ii.length;
    StringBuilder sb = new StringBuilder();
    for (int i=0; i<n; i++) {
      if (i > 0) sb.append(' ');
      sb.append(String.format(locale, format, ii[i]));
    }
    p.put(key.toLowerCase(), sb.toString());
  }

  /**
   * Format the long value and store it under the given key. 
   * @param key the key to be used.
   * @param l the long value.
   * @param format the JAVA format for formatting.
   */
  public void putLong(String key, long l, String format) {
    String s = String.format(locale, format, l);
    p.put(key.toLowerCase(), s);
  }

  /**
   * Format the long values and store them under the given key. 
   * @param key the key to be used.
   * @param ll the long values.
   * @param format the JAVA format for formatting.
   */
  public void putLongs(String key, long[] ll, String format) {
    int n = ll.length;
    StringBuilder sb = new StringBuilder();
    for (int i=0; i<n; i++) {
      if (i > 0) sb.append(' ');
      sb.append(String.format(locale, format, ll[i]));
    }
    p.put(key.toLowerCase(), sb.toString());
  }

  /**
   * Format the time value and store it under the given key. 
   * @param key the key to be used.
   * @param i the time value.
   */
  public void putTime(String key, int i) {
    String s = formatTime(i).trim();
    putString(key, s, true);
  }

  /**
   * Format the time values and store them under the given key. 
   * @param key the key to be used.
   * @param ii the time values.
   */
  public void putTimes(String key, int[] ii) {
    int n = ii.length;
    StringBuilder sb = new StringBuilder();
    for (int i=0; i<n; i++) {
      if (i > 0) sb.append(' ');
      sb.append("\"").append(formatTime(ii[i]).trim()).append("\"");
    }
    p.put(key.toLowerCase(), sb.toString());
  }

  /**
   * Store the string under the given key.
   * @param key the key to be used.
   * @param s the string.
   * @param quoted quote the string.
   */
  public void putString(String key, String s, boolean quoted) {
    if (quoted && s.length()>0 && s.charAt(0) != '"')  s = "\"" + s + "\"";
    p.put(key.toLowerCase(), s);
  }

  /**
   * Store the strings under the given key.
   * @param key the key to be used.
   * @param ss the strings.
   * @param quoted quote the strings.
   */
  public void putStrings(String key, String[] ss, boolean quoted) {
    int n = ss.length;
    StringBuilder sb = new StringBuilder();
    for (int i=0; i<n; i++) {
      if (i > 0) sb.append(' ');
      boolean q = quoted && ss[i].length()>0 && ss[i].charAt(0) != '"';
      if (q) sb.append('"');
      sb.append(ss[i]);
      if (q) sb.append('"');
    }
    p.put(key.toLowerCase(), sb.toString());
  }

  /**
   * Format the date value and store it under the given key. 
   * @param key the key to be used.
   * @param d the date value.
   */
  public void putDate(String key, double d) {
    String s = formatDate(d, time_zone);
    p.put(key.toLowerCase(), s);
  }

  /**
   * Format the date values and store them under the given key. 
   * @param key the key to be used.
   * @param dd the date values.
   */
  public void putDates(String key, double[] dd) {
    int n = dd.length;
    StringBuilder sb = new StringBuilder();
    for (int i=0; i<n; i++) {
      if (i > 0) sb.append(' ');
      sb.append(formatDate(dd[i], time_zone));
    }
    p.put(key.toLowerCase(), sb.toString());
  }
  
  /**
   * Print a list of all key/value pairs.
   * @param pw the print writer to be used.
   */
  public void print(PrintWriter pw) {
    pw.print(getList());
    pw.flush();
  }
  
  /**
   * Print a list of all key/value pairs.
   * @param title an optional title
   * @param ps the output stream to be used.
   */
  public void print(String title, OutputStream ps) {
    PrintWriter pw = new PrintWriter(ps);
    if (title != null)
      pw.println(title);
    pw.print(getList());
    pw.flush();
  }
  
  /**
   * Get a list of all key/value pairs.
   * @return the list.
   */
  public String getList() {
    StringBuilder sb = new StringBuilder();
    Set<String> keys = p.keySet();
    sb.append("locl").append(" \"").append(locale.getLanguage()).append("\"\n");
    sb.append("tmzn").append(" \"").append(time_zone.getID()).append("\"\n");
    for (String key: keys) {
      String value = p.get(key);
      sb.append(key).append(' ').append(value).append('\n');
    }
    return sb.toString();
  }
  
  /**
   * Put a list of key/value pairs into the internal map. The keys "locl" and 
   * "tmzn|zone" are evaluated and then discarded.
   * @param list the list of key/value pairs
   */
  public void putList(String list) {
    if (list == null)
      return;
    int n = list.length();
    if (n == 0)
      return;
    String key = null;
    String val = null;
    int i0=0, i1=0, i2=-1;
    while (i2 < n) {
      i0 = i2 + 1;
      i1 = list.indexOf(' ', i0);
      if (i1 < 0)
        break;
      i2 = list.indexOf('\n', i1+1);
      if (i2 < 0)
        i2 = n+1;
      key = list.substring(i0, i1).trim();
      if (key.length() == 0)
        continue;
      val = list.substring(i1, i2).trim();
      // -2022-09-27 for consistency expanded to "zone"; this routine putList()
      // has not been used so far
      if (key.equals("tmzn") || key.equals("zone")) {
        setTimeZone(val);
        continue;
      }
      if (key.equals("locl")) {
        setLocale(val);
        continue;
      }
      p.put(key, val);
    }
  }

  public ZonedDate getReferenceDate() throws Exception {          //-2011-11-25
    //String tmzn =  getString("tmzn|zone", true);
    // "tmzn" is consumed and deleted when the header is read;
    // this function has been used so far only by Lopxtr which
    // applies LASAT/AUSTAL files as input and these files do not
    // use "tmzn" in the header but specify the time zone (if) in rdat
    String tmzn = getTimeZone();                                  //-2022-09-27
    String rdat = getString("rdat|refdate|refdatum", true);
    ZonedDate zd = null;
    if (rdat == null)                                             //-2011-10-31
      return null;
    else
      zd = ZonedDate.getZonedDate(tmzn, rdat);
    if (zd == null)
      throw new Exception();                                      //-2011-11-25
    return zd;
  }
  
  //=========================================================================

  /**
   * For testing only.
   * @param args (not used).
   */
  public static void main(String[] args) {
  }
  
  //--------------------------------------------------------------//-2010-01-22

  public static class ZonedDate {

//    private SimpleDateFormat sdf = new SimpleDateFormat(date_format);
    private long seconds;
    private TimeZone tz;
    private Date date;

    public ZonedDate(TimeZone tz, long seconds) {
      if (tz == null)
        tz = TimeZone.getTimeZone(default_tmzn);
      this.tz = tz;
      this.seconds = seconds;
      date = new Date(seconds*1000);
    }

    public ZonedDate(String zone, String datum) throws Exception {
      if (zone == null)
        tz = TimeZone.getTimeZone(default_tmzn);                  //-2011-10-31
      else
        tz = TimeZone.getTimeZone(zone);
      long ms = parseDateLong(datum, tz);                         //-2011-10-31
      date = new Date(ms);                                        //-2011-10-31
      seconds = ms/1000;
    }

    public long getSeconds() {
      return seconds;
    }

    public TimeZone getZone() {
      return tz;
    }

    public Date getDate() {
      return date;
    }

    @Override
    public String toString() {
      return toString(0);
    }

    public String toString(long t) {
      long ms = (seconds + t)*1000;
      Date d = new Date(ms);
      SimpleDateFormat sdf = new SimpleDateFormat(date_format);
      sdf.setTimeZone(tz);
      String s = sdf.format(d) + " " + tz.getID();
      return s;
    }

    public String getDateString(boolean cut) {
      SimpleDateFormat sdf = new SimpleDateFormat(date_format);
      sdf.setTimeZone(tz);
      String s = sdf.format(date);
      if (cut && s.endsWith(date_separator+"00:00:00"))                         //-2011-10-31
        s = s.substring(0, s.length()-9);
      return s;
    }
    
    public String getDateZoneString() {
      SimpleDateFormat sdf = new SimpleDateFormat(date_format+"XX");
      sdf.setTimeZone(tz);
      String s = sdf.format(date);
      return s;
    }

    public String getZoneString() {
      return tz.getID();
    }

    public static ZonedDate getZonedDate(String tmzn, String rdat) {
      ZonedDate zd = null;
      if (rdat == null || rdat.length() == 0) {
        try {
          zd = new ZonedDate(default_tmzn, default_date);
        }
        catch (Exception e) {}
        return zd;
      }
      if (tmzn == null || !tmzn.startsWith("GMT"))
        tmzn = default_tmzn;
      int l = rdat.indexOf("GMT");
      if (l > 0) {
        tmzn = rdat.substring(l);
        rdat = rdat.substring(0, l).trim();
      }
      else {
        l = rdat.length();
        if (l > 5 && ((rdat.charAt(l-5) == '-') || (rdat.charAt(l-5) == '+'))) {
          tmzn = "GMT" + rdat.substring(l-5);
          rdat = rdat.substring(0, l-5).trim();
        }
      }
      try {
        zd = new ZonedDate(tmzn, rdat);
      }
      catch (Exception e) {}
      return zd;
    }

  }
  //-------------------------------------------------------------------------
  
}

