/**
 * Meteorological boundary layer model according to
 * standard VDI 3783 Part 8 (2017)
 *
 * Copyright (C) Janicke Consulting, Überlingen, Germany, 2013-2019
 * email: info@janicke.de
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 * 
 */

////////////////////////////////////////////////////////////////////////////////
//
// history
//
// 2013-05-23  uj  0.3    optionally hm calculated with Lm=99999
// 2014-08-06  uj  1.0    adjustments, interpolation revised
// 2014-08-29  uj  1.1    interpolation and additional rotation for h1 > ha
// 2015-01-16  uj  1.2    moved to NetBeans, internal adjustments
// 2015-12-21  uj  1.3    adjusted Kw factor (15 --> 9)
// 2016-01-22  uj  1.4    new Lm classification, no T=const. for z>hm
// 2016-04-05  uj  1.5    new Lm classification for III/1 (draft)
// 2016-06-30  uj  1.6    temperature interpolation adjustment
// 2016-08-02  uj  1.7    optional new temp gradient gu above height hu
// 2018-09-08  uj  1.8    adjust Hm(neutral) to Fcor
//
////////////////////////////////////////////////////////////////////////////////
package de.janicke.meteo;
import java.util.Locale;

/**
 * Class that provides the wind profile for a 2 layer model.
 * 
 * @author Ulf Janicke
 *
 */
public class BLProfile2Layer extends BLProfile {
  
  public static final double[] Z0VEC =
     { 0.01, 0.02, 0.05, 0.10, 0.20, 0.50,  1.0,  1.5,  2.0 };
  
 public static final double[][] LMTAL = {
                    {       5,     7,    9,   13,   17,   28,   44,   60,   77 }, 
                    {      25,    31,   44,   59,   81,  133,  207,  280,  358 },
                    {     354,   448,  631,  842, 1160, 1893, 2951, 4000, 5107 },
                    {     -37,   -47,  -66,  -88, -122, -199, -310, -420, -536 },
                    {     -15,   -19,  -27,  -36,  -49,  -80, -125, -170, -217 },
                    {      -6,    -8,  -11,  -15,  -20,  -33,  -52,  -70,  -89 } };

  public static final double[][] DTDZ_KTA = {
    { -1.13, -1.03, -0.91, -0.37, 0.78 },
    { -1.18, -1.05, -0.91, -0.22, 1.12 },
    { -1.39, -1.18, -0.97, -0.16, 1.25 },
    { -1.61, -1.33, -1.00, -0.10, 1.32 },
    { -1.82, -1.48, -1.04, -0.04, 1.39 },
    { -2.03, -1.62, -1.08,  0.02, 1.46 },
    { -2.24, -1.77, -1.16,  0.08, 1.53 },
    { -2.45, -1.92, -1.25,  0.14, 1.60 },
    { -2.66, -2.07, -1.40,  0.20, 1.67 } };
  
  public static final String VERSION = "1.8";
  public static final double FA_DEFAULT = 0.2;
  public static final double FCOR_DEFAULT = 1.1e-4;
  public static final double GRAV = 9.8066;
  public static final double CPRESS = 1004.1;
  public static final double GAMDRYAD = -(GRAV/CPRESS);
  
  public static final double BLMDEFHM = 800.;
  public static final double BLMFCOR = 1.1e-4;
  public static final double BLMFCORMIN = 6.1618e-5;
  
  public static final double VKK = 0.40;
  public static final double ALPHA = 0.3;
  public static final double HMMIN = 20;
  public static final int ITERMAX = 30;
  public static final double ITERDEV = 0.005;
  public static final double ITERSTEP = 0.1;
  public static final String name = "Profile(2Layer)";
  public double Suf = 2.4;
  public double Svf = 2.0;
  public double Swf = 1.3;
  public double Tuf = 0.9;
  public double Tvf = 0.9;
  public double Twf = 1.0;
  public double TlMin = 0;
  public double TlMax = 3600;
  public double AvExp = 0.18;
  private double Radd;
  private double Fa;
  private double Fcor;
  private double Ugeo; 
  private double H1;
  private double K2;
  private double A;
  private double Ust;
  private double Z0, D0, Lm, Hm, Ha, Aa, Ra, Ua, Up, Upd, Alpha;
  private double HmNeutral;
  private double Avf;
  private int Interval;
  private boolean Interpolation;  
  private String Info;
  private String Comment;
  private int uInit;
  private double lastHh, lastUu, lastRr, lastUx, lastUy;
  
  public static void main(String[] args) {
    Locale.setDefault(Locale.ENGLISH);
    double _ha = 10.;
    double _ua = 3.0;
    double _z0 = 0.2;
    double _d0 = -1;
    double _ra = 225.;
    double _us = -1;
    double _lm = 99999.;
    double _hm = -1;
    double _fc = Double.NaN;
    double _lt = Double.NaN;  
    double _it = 3600.;
    double _ta = 10.;
    double _ht = 2.;
    boolean divus = false;
    boolean interpolation = true;    
    boolean all = true;
    boolean all2 = false;
    double[] za = { 0, 5, 10, 20, 30, 40, 50, 60, 70, 80, 90, 100, 120, 150, 200, 250, 300, 
        400, 500, 600, 800, 1000, 1500, 2000 };
    try {
      if (args.length == 0 || args[0].equalsIgnoreCase("-h") || args[0].equalsIgnoreCase("--help")) {
        System.out.printf("\nWind (2-layer) and turbulence profiles, Version %s, (c) 2013-2019 Janicke Cons.\n", VERSION);
        System.out.printf("Call: java -jar BLProfile2Layer.jar [options]\n\n");
        System.out.printf("Options (at least one option required, defaults in square brackets):\n");
        System.out.printf("--ha=<ha>     anemometer height ha (m) [10]\n");
        System.out.printf("--ua=<ua>     wind speed at ha (m/s), alternative to u* [3.0]\n");
        System.out.printf("--us=<u*>     friction velocity (m/s), alternative to ua [calculated]\n");
        System.out.printf("--ra=<ra>     wind direction at ha (deg) [225]\n");
        System.out.printf("--z0=<z0>     roughness length (m) [0.2]\n");
        System.out.printf("--d0=<d0>     displacement height (m) [1.2]\n");
        System.out.printf("--lm=<lm>     Obukhov length (m) [99999]\n");
        System.out.printf("--ht=<ht>     temperature reference height (m) [2]\n");
        System.out.printf("--ta=<ta>     temperature (Celsius) at ht [10]\n");
        System.out.printf("--lt=<lat>    latitude <lat> (deg) from which fc is derived [44]\n");
        System.out.printf("--hm=<hm>     mixing layer height (m) [calculated]\n");
        System.out.printf("--it=<int>    averaging interval (s) [3600]\n");
        System.out.printf("--za=<z1>,<z2>,... vertical heights [default]\n"); 
        System.out.printf("--fc=<fc>     Coriolis parameter (1/s) [1.1e-4]\n"); 
        System.out.printf("--divus       divide result by us\n"); 
        System.out.printf("--all[2]      all: show all new profiles, all2: show also LST and A2K profiles\n"); 
        System.out.printf("--nointerpol  no displacement height, u profile down to z=0\n"); 
        System.out.printf("--help        this help text\n");
        System.exit(0);
      }
      for (String arg : args) {
        if (arg.startsWith("--ha=")) {
          _ha = Double.parseDouble(arg.substring(5));
        }
        else if (arg.startsWith("--ua=")) {
          _ua = Double.parseDouble(arg.substring(5));
        }
        else if (arg.startsWith("--us=")) {
          _us = Double.parseDouble(arg.substring(5));
        }
        else if (arg.startsWith("--z0=")) {
          _z0 = Double.parseDouble(arg.substring(5));
        }
        else if (arg.startsWith("--d0=")) {
          _d0 = Double.parseDouble(arg.substring(5));
        }
        else if (arg.startsWith("--ra=")) {
          _ra = Double.parseDouble(arg.substring(5));
        }
        else if (arg.startsWith("--it=")) {
          _it = Double.parseDouble(arg.substring(5));
        }
        else if (arg.startsWith("--lm=")) {
          _lm = Double.parseDouble(arg.substring(5));
        }
        else if (arg.startsWith("--lt=")) {
          _lt = Double.parseDouble(arg.substring(5));
        }
        else if (arg.startsWith("--hm=")) {
          _hm = Double.parseDouble(arg.substring(5));
        }
        else if (arg.startsWith("--fc=")) {
          _fc = Double.parseDouble(arg.substring(5));
        }
        else if (arg.startsWith("--divus")) {
          divus = true;
        }
        else if (arg.startsWith("--nointerpol")) {
          interpolation = false;
        }
        else if (arg.startsWith("--za=")) {
          String[] sa = arg.substring(5).split("[,]");
          za = new double[sa.length];
          for (int j=0; j<sa.length; j++)
            za[j] = Double.parseDouble(sa[j]);
        }
      }
      if (_d0 < 0)
        _d0 = 6*_z0;
      if (!interpolation)
        _d0 = 0;
      //
      double uu = (_us > 0) ? _us : _ua;
      if (Double.isNaN(_fc))
        _fc = FCOR_DEFAULT;
      if (!Double.isNaN(_lt)) {
        if (_lt >= 0 && _lt < 15)
          _lt = 15;
        if (_lt < 0 && _lt > -15)
          _lt = -15;
        _fc = 2 * 2*Math.PI/(24*3600.) * Math.sin(Math.toRadians((double)_lt));
      }
      BLProfile2Layer prf = new BLProfile2Layer(_fc, _z0, _d0, _ha, uu, _ra, _lm, _hm, (_us > 0), interpolation);
      prf.init();
      prf.setInterval((int)_it);  
      if (true) {
        System.out.printf("\n- Analytical 2-layer wind profile, Version %s, (c) 2013-2019 Janicke Cons.\n", VERSION);
        System.out.printf(prf.getInfo());
        System.out.printf("- ust(2Layer)=%1.3f\n", prf.getUst());
        System.out.printf("- profile (last 2 columns: A2K):\n");
        if (!divus) {
          System.out.printf("%6s %8s %8s   %8s %8s\n", 
            "- z(m)", "u(m/s)", "v(m/s)", "U(m/s)", "R(deg)");
          for (int i=0; i<za.length; i++) {
            prf.calculateU(za[i]);
            System.out.printf("%6.1f %8.2f %8.2f   %8.2f %8.1f\n", 
              prf.lastHh, prf.lastUx, prf.lastUy, prf.lastUu, prf.lastRr);
          }
        }
        else {
         System.out.printf("%6s %8s %8s   \n", 
            "- z(m)", "u(m/s)", "v(m/s)");
          for (int i=0; i<za.length; i++) {
            prf.calculateU(za[i]);
            System.out.printf("%6.1f %8.2f %8.2f   %8.1f %8.1f\n", 
              prf.lastHh, prf.lastUx, prf.lastUy, prf.lastUu/prf.getUst(), prf.lastRr);
          }       
        }
      }
    }
    catch (Exception e) {
      e.printStackTrace(System.out);
    }  
  }
  
  
  public BLProfile2Layer() {
    this(FCOR_DEFAULT, 0.5, 3.0, 10.0, 3.0, 270.0, 99999.0, 800.0, false, true);
  }

  public BLProfile2Layer(
    double fc, 
    double z0,
    double d0,
    double ha,
    double ua,
    double ra,
    double lm,
    double hm) {
    this(fc, z0, d0, ha, ua, ra, lm, hm, false, true);
  }

  public BLProfile2Layer(
    double fc,
    double z0,
    double d0,
    double ha,
    double ua,  // ua or us
    double ra,
    double lm,
    double hm,
    boolean is_us,
    boolean interpolation
    ) {
    Fcor = fc;
    Z0 = z0;
    D0 = d0;
    Ha = ha;
    Ra = ra;
    Lm = lm;
    if (Math.abs(Lm) > 999999.)
      Lm = (Lm > 0) ? 999999. : -999999.;
    Hm = hm;
    HmNeutral = BLMDEFHM*BLMFCOR/Math.max(BLMFCORMIN, Math.abs(Fcor));  //-2018-09-08
    Interpolation = interpolation;
    Ust = (is_us) ? ua : -1;
    Ua = (is_us) ? -1 : ua;
    uInit = -1;
    Interval = 3600;
    Avf = Math.pow(Interval/3600.0, AvExp);
  }
  
  @Override
  public void init() {
    if (uInit > 0) 
      return;
    uInit = 0;
    Comment = "";
    lastHh = Double.NaN;
    double hmdef = Hm;
    if (Ua < 0 && Ust > 0) {
      setModelParameters(false);
      Ua = getU(Ha);
    }
    else if (Ua > 0 && Ust < 0) {
      Ust = 1;
      double u = getU1(Ha, true);
      Ust = Ua/u;
      setModelParameters(false);
    }
    //
    // if inconsistent, try to derive us iteratively
    if (H1 < Ha) {
      double diff = Double.MAX_VALUE;
      double diffold = diff;
      double step = ITERSTEP;     
      int n = 0;
      Ust = 0;
      while (Math.abs(diff) > ITERDEV) {          
        if (diff*diffold < 0 || Math.abs(diff) > Math.abs(diffold))
          step *= -0.5;                      
        Ust += step;
        if (Ust <= 0.) {
          Ust -= step;
          step *= 0.5;
        }
        if (hmdef <= 0)
          Hm = -1;
        setModelParameters(false);
        diffold = diff;        
        diff = Ua - getU(Ha);
        //System.out.printf("### Iter %2d: H1=%4.1f, Hm=%3.0f, Lm=%5.0f, Ust=%6.3f, Ua=%4.1f, Ua-U=%6.3f\n", n, H1, Hm, Lm, Ust, Ua, diff);
        if (++n > ITERMAX)
          break; 
      }
      //
      // if iteration did not succeed, adjust H1 to ha
      if (n > ITERMAX) {
        Comment = Comment + "- iterative setting of u* did not succeed, adjusting H1.\n";
        System.out.printf(">>> iterative setting of u* did not succeed, adjusting H1.\n");
        Ust = 1;
        Ust = Ua/getU1(Ha, true);
        if (hmdef <= 0)                                            //-2015-12-21
          Hm = -1;
        setModelParameters(true);
      }
      else
        Comment = Comment + String.format("- h1 < ha: %d steps for interpolating Ua (diff=%1.1e)\n", n, diff);        
    }   
    if (H1 < Ha) {
      Radd = getR(Ha) - Ra;
      Comment = Comment + String.format("- h1 < ha: result rotation by %1.1f deg\n", Radd);
    }
    Ugeo = Math.sqrt(Up*Up + (Upd*Upd + Aa*Aa*Up*Up)/(2*A*A) + Up*(Upd - Math.abs(Aa)*Up)/A);  
    setInfo();
    lastHh = Double.NaN;
    uInit = 1;
  }
  
  private void setModelParameters(boolean adjust) {
    if (uInit != 0) {
      System.out.printf("attemt to call setModelParameters() outside initialization!");
      return;
    }
    if (Hm <= 0)  
      Hm = getHm(Ust, Lm);   
    H1 = (Lm < 0) ? Hm/(6*ALPHA) : 0.1*Lm*(-1 + Math.sqrt(1. + 10.*Hm/(3.*ALPHA*Lm)));
    H1 /= 2.0; 
    Fa = FA_DEFAULT;
    if (adjust) {
      Fa *= H1/Ha;
      H1 = Ha;
    }
    K2 = getKwm(H1, false); 
    A = Math.sqrt(0.5*Math.abs(Fcor)/K2);
    Aa = -Math.signum(Fcor) * A * Fa;
    Alpha = Math.toRadians(270 - Ra);      
    Up = getU1(H1, true);
    Upd = Ust*Ust/getKwm(H1, true);
    TlMin = (Ust > 0) ? Z0/Ust : 0;
  }
  
  public final void clear() {
    Fa = FA_DEFAULT;
    Fcor = FCOR_DEFAULT;
    A = -1;
    H1 = -1;
    K2 = -1;
    Ugeo = -1;
    Hm = -1;
    Interpolation = true;    
    Z0 = 0.5;
    D0 = 6*Z0;
    Lm = 99999;
    Avf = 1.0;
    Interval = 3600;
    Radd = 0.0;
    Comment = "";
    uInit = -1;
  }
  
  /**
   * 
   * @param z
   * @param h0
   * @param t0
   * @param u10
   * @param ki
   * @return 
   */
  public static double GetTmpKTA(double z, double h0, double t0, double u10, int ki) {
    double u10round, g;
    int iu, kta, ktai;
    u10round = (int)(u10*10 + 0.5)/10.0;
    iu = (int)u10round;
    if (iu > 8)
      iu = 8;
    kta = 7 - ki;
    ktai = kta - 1;
    if (kta == 6)
      g = DTDZ_KTA[iu][ktai-1] + 0.5*(DTDZ_KTA[iu][ktai-1] - DTDZ_KTA[iu][ktai-2]);
    else if (kta == 1)
      g = DTDZ_KTA[iu][ktai] + 0.5*(DTDZ_KTA[iu][ktai] - DTDZ_KTA[iu][ktai+1]);
    else
      g = 0.5*(DTDZ_KTA[iu][ktai-1] + DTDZ_KTA[iu][ktai]);  
    return t0 + 0.01*g*(z - h0);
  }
  
  /**
   * 
   * @param h
   * @param hb
   * @param tb
   * @return 
   */
  public double getTmp(double h, double hb, double tb) {
    return getTmp(h, hb, tb, -1, -1);
  }
  
  /**
   * 
   * @param h
   * @param hb
   * @param tb
   * @param zu
   * @param gu
   * @return 
   */ 
  public double getTmp(double h, double hb, double tb, double zu, double gu) {
    double z, zb; 
    if (Interpolation) {     
      z = (h < D0 + 6*Z0) ? 6*Z0 : h - D0;
      //
      // this settings yields temperature tb at height hb+d0 or,
      // if hb+d0 < d0+6*z0, temperature at d0+6*z0
      zb = hb;
      //
      // this setting would yield temperature tb at height hb
      //// zb = (hb < D0 + 6*Z0) ? 6*Z0 : hb - D0;
    }
    else {
      z = h;
      zb = hb;
    }   
    double tst = (283.15/(VKK*GRAV))*Ust*Ust/Lm;
    double t = tb + (tst/VKK)*(getTheta(z) - getTheta(zb));
    if (Double.isNaN(zu) || zu < 0) {
      t += GAMDRYAD*(z - zb);
      //
      // testing
      double theta_z = getTheta(z, zb); 
      double theta_zb = getTheta(zb, zb); 
      double t2 = tb + (tst/VKK)*(theta_z - theta_zb) + GAMDRYAD*(z-zb);  
      if (Math.abs(t-t2) > 0.0001)
        System.out.printf("### check T: t=%f, t2=%f (h=%f, hb=%f, tb=%f, gam=%f)\n", t, t2, h, hb, tb, GAMDRYAD);
      //
    }
    else {
      double gamma1 = GAMDRYAD;
      double gamma2 = gu;
      if (zb < zu) {
        if (z < zu)
          t += gamma1*(z - zb);
        else {
          t += gamma2*(z - zb) + (gamma1 - gamma2)*(zu - zb);
        }
      }
      else {
        if (z < zu)
          t += gamma1*(z - zb) + (gamma2 - gamma1)*(zu - zb);
        else {
          t += gamma2*(z - zb);
        }
      }
    }
    double tbase = (tb > 200) ? 273.15 : 0.;
    if (t > tbase+40)
      t = tbase+40;
    else if (t < tbase-30)
      t = tbase-30; 
    return t;
  }
  
  
  private double getTheta(double z, double zb) {
    double theta;
    double zeta = z/Lm;
    double zeta0 = zb/Lm;    
    if (Lm > 0.) {
      if (zeta < 0.5)
        theta = Math.log(zeta/zeta0) + 5*(zeta - zeta0);
      else if (zeta < 10.)
        theta = 8.*Math.log(2.*zeta) + 4.25/zeta - 0.5/(zeta*zeta) - Math.log(2.*zeta0) -5*zeta0 - 4.;
      else
        theta = 0.7585*zeta + 8.*Math.log(20.) - 11.165 - Math.log(2.*zeta0) -5.*zeta0;   
    }
    else {
      double y = Math.sqrt(1 - 15*zeta);
      double y0 = Math.sqrt(1 - 15*zeta0);
      theta = Math.log(zeta/zeta0);
      //if (y > y0)
        theta -= 2.*Math.log((1+y)/(1+y0));  
    }
    return theta;
  }
  
  
  private double getTheta(double z) {
    double theta;
    double zeta = z/Lm; 
    if (Lm > 0.) {
      if (zeta < 0.5)
        theta = Math.log(zeta) + 5*zeta -7*Math.log(2.) + 4;
      else if (zeta < 10.)
        theta = 8.*Math.log(zeta) + 4.25/zeta - 0.5/(zeta*zeta);
      else
        theta = 0.7585*zeta + 8.*Math.log(10.) - 7.165;   
    }
    else {
      double y = Math.sqrt(1 - 15*zeta);
      theta = Math.log(Math.abs(zeta)) - 2.*Math.log(1+y);  
    }
    return theta;
  }
  
  
  /**
   * 
   * @param z0
   * @param lm
   * @param km
   * @return 
   */
  public static double[] GetStability(double z0, double lm, int km) {
    double lz0, a1, a2, lz1, lz2;
    double[] rr = new double[6];
    double[] ll = new double[6];
    int k, i1, i2;   
    int n0 = Z0VEC.length;
    if (z0 < Z0VEC[0])     z0 = Z0VEC[0];
    if (z0 > Z0VEC[n0-1])  z0 = Z0VEC[n0-1];
    for (i2=1; i2<n0; i2++) {
      if (z0 <= Z0VEC[i2])  break;
    }
    i1 = i2-1;
    lz0 = Math.log(z0);
    lz1 = Math.log(Z0VEC[i1]);
    lz2 = Math.log(Z0VEC[i2]);
    a1 = (lz2 - lz0)/(lz2 - lz1);
    a2 = (lz0 - lz1)/(lz2 - lz1);
    for (k=0; k<6; k++) {
      rr[k] = a1/LMTAL[k][i1] + a2/LMTAL[k][i2];
      ll[k] = 1/rr[k];
    }
    if (lm == 0) {
      if (km <= 0) 
        return new double[] { 0., 0. };
      if (km > 6)  
        lm = (km == 7) ? ll[2] : 0;      
      else
        lm = ll[km-1];
    }
    else {
      for (k=0; k<5; k++)  
        if (1./lm > 0.5*(rr[k] + rr[k+1]))  
          break;
      km = k+1;
    }
    return new double[] { lm, (double)km };
  }

  
  /**
   * @param h
   * @return wind speed
   */
  @Override
  public double getU(double h) {
    if (uInit != 1 || h != lastHh) {
      try {
        calculateU(h);
      }
      catch (Exception e) {
        e.printStackTrace(System.out);
        return Double.NaN;
      }
    }  
    return lastUu; 
  }

  /**
   * @param h
   * @return meteorological wind direction in deg
   */
  @Override
  public double getR(double h) {
    if (uInit != 1 || h != lastHh) {
      try {
        calculateU(h);
      }
      catch (Exception e) {
        e.printStackTrace(System.out);
        return Double.NaN;
      }
    }  
    return lastRr; 
  }

  /**
   * @return wind velocity at anemometer
   */
  public double getUa() {
    return Ua;
  }

  /**
   * @return friction velocity
   */
  @Override
  public double getUst() {
    return Ust;
  }
  
  @Override
  public String getName() { 
    return name; 
  }
  
  @Override
  public String getInfo() {
    return Info;
  } 

  /**
   * @return zero plane displacement
   */
  public double getD0() {
    return D0;
  }

  /**
   * @return anemometer height
   */
  public double getHa() {
    return Ha;
  }

  /**
   * @return mixing height
   */
  @Override
  public double getHm() {
    return Hm;
  }
    
  public double getFa() {
    return Fa;
  }
  
  public double getH1() {
    return H1;
  }
  
  public double getK2() {
    return K2;
  }
  
  public double getUgeo() {
    return Ugeo;
  }

  public double getFcor() {
    return Fcor;
  }
  
  /**
   * @return Monin Obukhov length
   */
  public double getLm() {
    return Lm;
  }
  
  /**
   * @return wind direction at anemometer
   */
  public double getRa() {
    return Ra;
  }
  
  @Override
  public double getSu(double h) {
    double z, a, su;    
    if (h > 3*Hm + D0) h = 3*Hm + D0;
    if (Interpolation)
      z = (h < D0 + 6*Z0) ? 6*Z0 : h - D0;
    else
      z = h;   
    a = (Hm > 0) ? z/Hm : 0;   
    if (Lm >= 0) 
      su = 1.0;
    else
      su = Math.pow(1.0 - 0.0371*(Hm/Lm)*Math.exp(-3*ALPHA*a), 1./3.);
    su *= Avf*Suf*Ust;  
    return su;
  }
  
  @Override
  public double getSv(double h) {
    double z, a, sv;   
    if (h > 3*Hm + D0) h = 3*Hm + D0;
    if (Interpolation)
      z = (h < D0 + 6*Z0) ? 6*Z0 : h - D0;
    else
      z = h;   
    a = (Hm > 0) ? z/Hm : 0;   
    if (Lm >= 0) 
      sv = 1.0;
    else
      sv = Math.pow(1.0 - 0.0642*(Hm/Lm)*Math.exp(-3*ALPHA*a), 1./3.);
    sv *= Avf*Svf*Ust;   
    return sv;
  }
  
  @Override
  public double getSw(double h) {
    double z, e, s, a, f, sw;
    if (h > 3*Hm + D0) h = 3*Hm + D0;
    if (Interpolation)
      z = (h < D0 + 6*Z0) ? 6*Z0 : h - D0;
    else
      z = h;
    a = (Hm > 0) ? z/Hm : 0.0;
    s = (Lm != 0) ? z/Lm : 0.0;
    e = (a > 0) ? Math.exp(-3*ALPHA*a) : 1.0;
    f = ((1 - 0.8*a) > 0) ? (1 - 0.8*a) : 0;
    if (Lm >= 0) 
      sw = e;
    else          
      sw = Math.pow(e*e*e - 2.5*s*f*f*f, 1./3.);
    sw *= Swf*Ust;
    return sw;
  }

  @Override
  public double getKv(double h) {
    double z, uu, sv, kv, hm;  
    if (h > 3*Hm + D0) h = 3*Hm + D0;
    if (Interpolation)
      z = (h < D0 + 6*Z0) ?  D0 + 6*Z0 : h;
    else
      z = h;
    uu = getU(z);
    sv = getSv(z);
    hm = (Hm > 800.) ? 800. : Hm;
    kv = Avf * Tvf * (uu/Ust) * (hm/100.) * sv;
    if (kv > TlMax*sv*sv)
      kv = TlMax*sv*sv;
    if (kv < TlMin*sv*sv)
      kv = TlMin*sv*sv;
    return kv;
  }
  
  @Override
  public double getKu(double h) {
    double z, uu, su, ku, hm;  
    if (h > 3*Hm + D0) h = 3*Hm + D0;
    if (Interpolation)
      z = (h < D0 + 6*Z0) ? D0 + 6*Z0 : h;
    else
      z = h;
    uu = getU(z);
    su = getSu(z);
    hm = (Hm > 800.) ? 800. : Hm;
    ku = Avf * Tuf * (uu/Ust) * (hm/100.) * su;
    if (ku > TlMax*su*su)
      ku = TlMax*su*su;
    if (ku < TlMin*su*su)
      ku = TlMin*su*su;
    return ku;
  }
  
/**
   * 
   * @param h
   * @return vertical exchange coefficient (heat) of the full profile (with hm corrections)
   */
  @Override
  public double getKw(double h) {
    double z, s, a, kw, b, dc, dm;   
    if (h > 3*Hm + D0) h = 3*Hm + D0;
    if (Interpolation)
      z = (h < D0 + 6*Z0) ? 6*Z0 : h - D0;
    else
      z = h;
    a = (Hm > 0) ? z/Hm : 0.;
    s = (Lm != 0) ? z/Lm : 0.;    
    b = 1 - 0.8*a;   
    dc = (b > 0) ? b*b : 0;
    dm = (a > 0) ? Math.exp(-6*ALPHA*a) : 1.0;
    if (Lm >= 0)
      kw = dm/(1 + 5*s);
    else {
      //kw = Math.pow(dm*dm*dm - 15*s*dc*dc*dc, 1./3.);
      //kw = Math.pow(dm*dm - 15*s*dc*dc, 1./2.);
      kw = Math.pow(dm*dm - 9*s*dc*dc, 1./2.); 
    }
    kw *= Twf * VKK * Ust * z;  
    double sw = getSw(h);
    if (kw > TlMax*sw*sw)
      kw = TlMax*sw*sw;
    if (kw < TlMin*sw*sw)
      kw = TlMin*sw*sw;
    return kw;
  }
  
  @Override
  public double getTu(double h) {
    double su = getSu(h);
    double ku = getKu(h);
    double tu = (su > 0) ? ku/(su*su) : TlMax;    
    if (tu < TlMin) 
      tu = TlMin;
    if (tu > TlMax)
      tu = TlMax;
    return tu;   
  }
  
  @Override
  public double getTv(double h) {
    double sv = getSv(h);
    double kv = getKv(h);
    double tv = (sv > 0) ? kv/(sv*sv) : TlMax;    
    if (tv < TlMin) 
      tv = TlMin;
    if (tv > TlMax)
      tv = TlMax;
    return tv;   
  }
  
  @Override
  public double getTw(double h) {
    double sw = getSw(h);
    double kw = getKw(h);
    double tw = (sw > 0) ? kw/(sw*sw) : TlMax;    
    if (tw < TlMin) 
      tw = TlMin;
    if (tw > TlMax)
      tw = TlMax;
    return tw;   
  }
  
  public void setInterval(int i) {
    if (i < 0)
      i = 3600;
    if (i < 60)
      i = 60;
    if (i > 3600)
      i = 3600;
    Interval = i;
    Avf = Math.pow(Interval/3600.0, AvExp);
    TlMax = Interval;
  }
   
  private void setInfo() {
    Info = String.format("- %s %s, z0=%1.3f, d0=%1.3f, ha=%1.1f, ua=%1.3f, ra=%1.1f\n",
        name, VERSION, Z0, D0, Ha, Ua, Ra);
    Info += String.format("- fcor=%1.1e, lm=%1.1f, hm=%1.1f, us=%1.5f\n",
        Fcor, Lm, Hm, Ust);
    Info += String.format("- interpolation=%b, h1=%1.1f, K2=%1.4f, fa=%1.2f, ugeo=%1.2f\n",
        Interpolation, H1, K2, Fa, Ugeo); 
    if (Comment != null && Comment.length() != 0)
      Info += Comment;
  }
  
  public double[][] getProfile(double[] za) throws Exception {       
    double[][] daa = new double[6][za.length];
    for (int i=0; i<za.length; i++) {
      calculateU(za[i]);
      daa[0][i] = za[i];
      daa[1][i] = getKwm(za[i], false);
      daa[2][i] = lastUu;
      daa[3][i] = lastRr;
      if (daa[3][i] > 360) daa[3][i] -= 360;
      if (daa[3][i] < 0) daa[3][i] += 360;
    }
    calculateU(10000.);
    for (int i=0; i<za.length; i++) {
      daa[3][i] = daa[3][i] - lastRr;
      if (daa[3][i] > 180) daa[3][i] -= 360;     
    }
    return daa;
  }

  private double getHm(double us, double lm) {                     //-2018-09-08
    double alpha = 0.3;
    double hm = HmNeutral;
    if (lm < 0)
      hm = Math.max(HmNeutral, 1100.);
    else {
      double fc = Math.max(BLMFCORMIN, Math.abs(Fcor));
      hm = 0.3*us/fc;
      double a = Math.sqrt(fc*lm/us);
      if (a < 1)  
        hm *= a;
      if (hm > HmNeutral) 
        hm = HmNeutral; 
    }
    if (hm < HMMIN) 
      hm = HMMIN;
    return hm;
  }
  
  /**
   * 
   * @param h
   * @param simple
   * @return vertical exchange coefficient (momentum) of the full profile (with hm corrections)
   */
  public double getKwm(double h, boolean simple) {
    double z, s, a, kw, b, dc, dm;    
    if (Interpolation)
      z = (h < D0 + 6*Z0) ? 6*Z0 : h - D0;
    else
      z = h;  
    if (Lm == 0)
      s = 0;
    else
      s = (Interpolation) ? z/Lm : (z+Z0)/Lm;
    a = (Hm > 0) ? z/Hm : 0.;
    b = 1 - 0.8*a;  
    if (simple) {
      dc = 1;
      dm = 1;
    }
    else {
      dc = (b > 0) ? b*b : 0;
      dm = (a > 0) ? Math.exp(-6*ALPHA*a) : 1.0;
    }
    if (Lm >= 0)
      kw = dm/(1 + 5*s);
    else    
      kw = Math.pow(dm*dm*dm*dm - 15*s*dc*dc*dc*dc, 1./4.);  
    kw *= Twf * VKK * Ust; 
    kw *= (Interpolation) ? z : (z+Z0); 
    return kw;
  }
  
  public double getKwmMax() {
    double zz = 0;
    double kk2=0, kk1=0;
    while (zz < 2*Hm) {
      kk2 = getKwm(zz, false);
      if (kk2 < kk1)
        break;
      kk1 = kk2;
      zz += 0.5;
    }
    return kk1;
  }
   
  public double getU1(double h, boolean simple) {
    double u = 0, z, zm, zm0, psi, psi0, u1, hh;  
    if (Interpolation) {
      hh = D0 + 6*Z0;
      z = (h < hh) ? hh - D0 : h - D0;
    }
    else {
      hh = h;
      z = h+Z0;
    }
    if (Lm >= 0) {
      if (Lm > 0) {
        zm = z/Lm;
        zm0 = Z0/Lm;
      } 
      else {
        zm = 0;
        zm0 = 0;
      }
      if (simple) {
        psi = (zm > zm0) ? Math.log(zm/zm0) + 5*(zm - zm0) : 0.;
      }
      else {              
        if (zm < 0.5)
          psi = Math.log(zm/zm0) + 5*(zm - zm0);
        else if (zm < 10)
          psi = 8*Math.log(2*zm) + 4.25/zm - 1./(2*zm*zm) - Math.log(2*zm0) - 5*zm0 - 4;
        else
          psi = 0.7585*zm + 8*Math.log(20) - 11.165 - Math.log(2*zm0) - 5*zm0;
      }
      u = Ust * psi / VKK;
    }
    else {
      zm = z/Lm;
      zm0 = Z0/Lm;
      psi = Math.pow(1. - 15.*zm, 0.25);
      psi0 = Math.pow(1. - 15.*zm0, 0.25);
      u1 = (psi0+1)*(psi0+1)*(psi0*psi0+1)*zm/zm0;
      u1 = Math.log(u1/( (psi+1)*(psi+1)*(psi*psi+1) ));
      u = Ust * (2*(Math.atan(psi) - Math.atan(psi0)) + u1) / VKK;
    }    
    if (h < hh)
      u *= h / hh;
    return u;    
  }

  /**
   * 
   * @param h
   */
  public void calculateU(double h) {
    double hh, z, u, v, za, z1;
    lastHh = Double.NaN;
    lastUx = Double.NaN;
    lastUy = Double.NaN;
    lastUu = Double.NaN;
    lastRr = Double.NaN;
    if (uInit < 0)
      return;
    if (Z0 <= 0 || Ust <= 0) {
      System.out.printf("invalid z0 or ust!");
      return;
    }
    //
    // displacement zone treatment
    if (Interpolation) {
      hh = D0 + 6 * Z0;
      z = (h < hh) ? hh - D0 : h - D0;
      za = (Ha < hh) ? hh - D0 : Ha - D0;
      z1 = (H1 < hh) ? hh - D0 : H1 - D0;
    }
    else {
      hh = h;
      z = h;
      za = Ha;
      z1 = H1;
    }
    if (z < 0) {
      System.out.printf("invalid z!");
      return;
    }
    //
    //  profile for the first layer
    if (h < H1) {
      double uu = getU1(h, true);
      if (uu == 0)
        uu = 1.e-6;
      double cz = Math.cos(Aa*(z-za) + Alpha);
      double sz = Math.sin(Aa*(z-za) + Alpha);
      u = uu * cz;
      v = uu * sz;
    }
    //
    // profile for the second layer
    else {      
      double cp = Math.cos(Aa*(z1-za) + Alpha);
      double sp = Math.sin(Aa*(z1-za) + Alpha);      
      double Az = A * (z-z1);
      double wp = cp + sp;
      double wm = cp - sp;     
      double cz = Math.exp(-Az) * Math.cos(Az);
      double sz = Math.exp(-Az) * Math.sin(Az);
      double p = Upd*wp + Aa*Up*wm;
      double q = Upd*wm - Aa*Up*wp;
      if (Fcor > 0) {
        u = Up*cp + (0.5/A)*((1-cz)*p + sz*q);
        v = Up*sp + (0.5/A)*((cz-1)*q + sz*p);
      }
      else {
        u = Up*cp + (0.5/A)*((1-cz)*q + sz*p);
        v = Up*sp - (0.5/A)*((cz-1)*p + sz*q);
      }
      //
      // linear extension in displacement zone
      if (h < hh) {
        if (h == 0)
          h = 1.e-6;
        u *= h/hh;
        v *= h/hh;
      }
    }
    //
    // store results
    if (Radd != 0.0) {
      double rr = Math.toRadians(Radd);
      double uu = u*Math.cos(rr) - v*Math.sin(rr);
      double vv = u*Math.sin(rr) + v*Math.cos(rr);
      u = uu;
      v = vv;
    }
    double uu = Math.sqrt(u*u + v*v);
    double rr = Math.toDegrees((Math.atan2(-u, -v)));
    if (rr < 0)
      rr += 360.;
    lastHh = h;
    lastUx = u;
    lastUy = v;
    lastUu = uu;
    lastRr = rr;
  }

}
