////////////////////////////////////////////////////////////////////////////////
//
// history
//
// 2015-10-16 lj 0.1    batch mode
// 2015-10-24 lj 0.2    interactive mode
// 2015-10-26 lj 0.3    new call of IBJpluris, listing error corrected
// 2015-10-29 lj 0.4    save history
// 2015-11-10 lj 0.5    logging pluris version
// 2015-11-20 lj 0.6    vq=0 caught, parameter rq and lq
// 2015-11-23 lj 0.7    rq in percent, direct call of IBJpluris
// 2015-12-28 uj 0.8    main call adjusted, stacktip-downwash
// 2016-03-10 uj 0.9    break factor, eq from external in kg/h, internal in g/s
// 2016-05-02 lj 0.10   start from highest value when checking stack height
// 2016-05-04 uj 0.11   graph bits adjusted
// 2016-06-27 uj 0.12   optionally user-defined Gamma (K/m)
// 2016-06-28 uj 0.13   user-defined gamma --> ra=0.7 = const. (ra=0 else)
// 2016-07-05 uj 0.14   imported modifications by lj and redefined gamma
// 2016-08-02 uj 0.15   redefined gamu option for IBJPluris
// 2017-02-10 uj 0.20   break factor transmitted to IBJluris with value 1.7
// 2018-09-02 uj 0.30   extended to hq=6m
// 2018-09-12 lj 0.31   TabPane for 'history' and 'detail'
// 2018-09-13 lj 0.32   ProgressButton
// 2018-09-14 uj 0.33   adjust names, set root directory
// 2018-09-14 lj 0.34   'detail' restored if 'history' item selected
// 2018-09-15 uj 0.35   minor adjustments
// 2018-09-15 lj 0.36   ProgressButton revised
// 2019-01-16 uj 0.37   test option --var-ra
// 2019-02-12 uj 0.38   IBJpluris 3.1.3
// 2019-03-04 uj 0.4.0  IBJpluris 3.1.4
// 2021-09-16 uj 1.0.0  version update, rename to Besmin
// 2021-10-07 uj 1.0.1  data check
//
////////////////////////////////////////////////////////////////////////////////
 
 /**
 * BESMIN
 * 
 * Determination of stack height according to TA Luft (2021).
 *
 * Copyright (C) Umweltbundesamt Dessau-Roßlau, Germany, 2016-2021
 * Copyright (C) Janicke Consulting, Überlingen, Germany, 2016-2021
 * email: info@austal.de
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 * 
 */

package de.janicke.tal;

import de.janicke.fxutil.CheckedTextField;
import de.janicke.fxutil.ProgressButton;
import de.janicke.pluris.IBJpluris;
import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.PipedReader;
import java.io.PipedWriter;
import java.io.PrintWriter;
import java.net.URI;
import java.net.URL;
import java.nio.file.Files;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.zip.CRC32;
import javafx.application.Platform;
import javafx.beans.property.StringProperty;
import javafx.beans.value.ChangeListener;
import javafx.beans.value.ObservableValue;
import javafx.concurrent.Task;
import javafx.embed.swing.JFXPanel;
import javafx.event.ActionEvent;
import javafx.geometry.HPos;
import javafx.geometry.Insets;
import javafx.geometry.Orientation;
import javafx.geometry.Pos;
import javafx.scene.Scene;
import javafx.scene.control.Button;
import javafx.scene.control.ChoiceBox;
import javafx.scene.control.Label;
import javafx.scene.control.ListView;
import javafx.scene.control.Separator;
import javafx.scene.control.Tab;
import javafx.scene.control.TabPane;
import javafx.scene.control.TextArea;
import javafx.scene.control.TextField;
import javafx.scene.image.Image;
import javafx.scene.layout.BorderPane;
import javafx.scene.layout.ColumnConstraints;
import javafx.scene.layout.GridPane;
import javafx.scene.layout.HBox;
import javafx.scene.layout.Priority;
import javafx.scene.layout.VBox;
import javafx.scene.text.Font;
import javafx.scene.text.FontPosture;
import javafx.scene.text.FontWeight;
import javafx.scene.text.Text;
import javafx.stage.Stage;


public class Besmin {
  
  private static String my_name = "BESMIN";
  private static String my_version = "1.0.1";
  private static String base = "./";
  private static String PLURIS_VERSION = null;
  private static boolean batch = false;

  private static final Locale LOC = Locale.GERMAN;
  private static final String UNKNOWN = "Unbekannt";
  private static final SimpleDateFormat SDF = 
          new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
  private static final double BREAKFACTOR = 1.7;
  private static final int NH = 27+2;   // for heights 6 and 8 m
  private static final int NB = 39+4;   // additional levels
  private static final int NS = 43; // maximum number of situations
  private static final int NN = 25; // used number of situations
  private static final int NR = 9;  // maximum number of z0 values
  private static final double HEMAX = 800;
  private static final double HQMAX = 250;
  private static final double HQMIN = 6;
  private static final double Z0 = 0.5;
  private static final int Z0i = 5;
  private static final double metvers = 5.3;
  private static final double[] ua9 = 
          new double[] { 1.0, 1.5, 2.0, 3.0, 4.5, 6.0, 7.5, 9.0, 12.0 };
  private static final double[] kl6 =
          new double[] { 1.0, 2.0, 3.1, 3.2, 4.0, 5.0 }; 
  private static final double[] vz0 = new double[] 
    { 0.01, 0.02, 0.05, 0.1, 0.2, 0.5, 1.0, 1.5, 2.0 }; // meter
  private static final double[] vhq = new double[NH];
  private static final double[] vhb = new double[NB];
  static {
    vhq[0] = 6;                                                   //-2018-09-02
    vhq[1] = 8;
    for (int ih=0; ih<vhq.length-2; ih++)                         //-2018-09-02
      vhq[ih+2] = 10*Math.pow(2, ih*0.25);                        //-2018-09-02
    vhb[0] = 6;                                                   //-2018-09-02
    vhb[1] = 7;
    vhb[2] = 8;
    vhb[3] = 9;
    for (int ib=0; ib<vhb.length-4; ib++)                         //-2018-09-02
      vhb[ib+4] = 10*Math.pow(2, ib*0.125);                       //-2018-09-02
  }
  private static String names = "eq,dq,tq,vq,rq,lq,zq,sv,";
  private static Map<String, Double> s_values;
  private final List<String> infos = new ArrayList<>();
  private final Result[] results = new Result[NN];
  private static boolean first_pluris_call;
  private static boolean print_pluris_log = false;
  private static boolean skip_stacktip_downwash = false;
  private static double break_factor = BREAKFACTOR;
  private static double zgamu = -1;                               //-2016-08-02
  private static double gamu = Double.NaN;                        //-2016-06-27

  private static boolean chk = false;
  private static boolean prn = true;
  private double[][][] maxval;
  private double[][][] maxdev; 
  
  public Besmin(String base_dir) {
    if (base_dir != null)
      base = base_dir;
    maxval = new double[NR][NH][NS];
    maxdev = new double[NR][NH][NS];
    
  }
  
  public String readMaxima() {
    String msg = null;
    boolean empty = true;
    try {
      checkPlumeData(base);                                       //-2021-10-07
      for (int ir=0; ir<NR; ir++) {
        double z0 = vz0[ir];
        double hq = vhq[NH-1];                                    //-2016-06-08
        String dn = String.format("mv%02.0frl%03.0fhq%03.0f",     //-2016-06-08
                  10*metvers, 100*z0, hq);        
        File g = new File(base, "maxima/maxima-" + dn + ".txt");  //-2016-06-08
        if (!g.exists()) {                                        //-2016-06-08
          maxval[ir] = null;                                      //-2016-06-08
          continue;
        }
        for (int ih=0; ih<NH; ih++) {
          hq = vhq[ih];
          dn = String.format("mv%02.0frl%03.0fhq%03.0f", 10*metvers, 100*z0, hq);
          g = new File(base, "maxima/maxima-" + dn + ".txt");
          List<String> list = Files.readAllLines(g.toPath());
          int is = 0;
          for (String line: list) {
            if (line.startsWith("-"))
              continue;
            line = line.trim();
            if (line.length() == 0)
              continue;
            String[] ss = line.split("[\\s,]+");
            maxval[ir][ih][is] = Double.parseDouble(ss[0].trim().replace(',', '.'));
            maxdev[ir][ih][is] = Double.parseDouble(ss[1].trim().replace(',', '.'));
            is++;
            if (is == NN) {                                       //-2016-06-08
              empty = false;
              break;
            }
          }
          if (is != NN)                                           //-2016-06-08
            return "*** missing values in " + dn;
        }
      }
    } 
    catch (Exception e) {
      e.printStackTrace(System.out);
      msg = "*** " + e.toString();
    }
    if (empty)
      msg = "*** no data";
    return msg;
  }
  
  public static void initSvalues() {
    s_values = new LinkedHashMap<>();
    s_values.put(UNKNOWN, 1.0);
    s_values.put("Arsen", 0.00016);
    s_values.put("Benzo(a)pyren", 0.000026);
    s_values.put("Benzol", 0.005);  
    s_values.put("Blei", 0.0025);
    s_values.put("Cadmium", 0.00013);
    s_values.put("Chlor", 0.09);
    s_values.put("Chlorwasserstoff", 0.1);
    s_values.put("Fluorwasserstoff", 0.0018);
    s_values.put("Formaldehyd", 0.025);
    s_values.put("Kohlenmonoxid", 7.5);
    s_values.put("Nickel", 0.00052);
    s_values.put("Partikel", 0.08);
    s_values.put("Quarz-Feinstaub", 0.005);
    s_values.put("Quecksilber", 0.00013);
    s_values.put("Schwefeldioxid", 0.14);
    s_values.put("Schwefelwasserstoff", 0.003);
    s_values.put("Stickstoffdioxid", 0.1);
    s_values.put("Thallium", 0.00026);
    s_values.put("Stoffe-5.2.2-I", 0.005);
    s_values.put("Stoffe-5.2.2-II", 0.05);
    s_values.put("Stoffe-5.2.2-III", 0.1);
    s_values.put("Stoffe-5.2.5-C", 0.1);
    s_values.put("Stoffe-5.2.5-I", 0.05);
    s_values.put("Stoffe-5.2.5-II", 0.1);
    s_values.put("Stoffe-5.2.7-I", 0.00005);
    s_values.put("Stoffe-5.2.7-II", 0.0005);
    s_values.put("Stoffe-5.2.7-III", 0.005);
  }
  
  public static double ua(int seq0) {
    if (seq0 < 3)
      return ua9[seq0];
    else if (seq0 < 7)
      return ua9[seq0 - 3];
    else 
      return ua9[(seq0-7)%9];
  }
  
  public static double kl(int seq0) {
    if (seq0 < 3)
      return kl6[0];
    else if (seq0 < 7)
      return kl6[1];
    else
      return kl6[2 + (seq0-7)/9];
  }
  
  public static double[] pluris(File wrk, double z0, double d0, double ha, 
          double ua, double kl, double hq, double dq, double vq, double tq,
          double zq, double rq, double lq) {
    double[] plr = null;
    List<String> args = new ArrayList<>();
    try {  
      args.add(wrk.toString());
      args.add("--z0=" + z0);
      args.add("--d0=" + d0);
      args.add("--ha=" + ha);
      args.add("--ua=" + ua);
      args.add("--ki=" + ((int)kl + ((kl > 3.11) ? 1 : 0)));
      args.add("--hq=" + hq);
      args.add("--dq=" + dq);
      args.add("--uq=" + vq);
      args.add("--tq=" + tq);
      args.add("--zq=" + zq);
      args.add("--rq=" + rq/100);
      args.add("--lq=" + lq);
      args.add("--sc=" + dq);                                     //-2015-12-30
      args.add("--se=" + 50000);                                  //-2015-12-30
      args.add("--sd=" + 0.01);                                   //-2015-12-30
      args.add("--verbose=1");
      args.add("--stop-at-finalrise");
      args.add("--skip-graph");
      if (skip_stacktip_downwash)
        args.add("--skip-stacktip-downwash");                     //-2015-12-28
      if (break_factor > 0) {                                     //-2016-03-10
        args.add("--break-factor="+ break_factor);
        args.add("--const-ust");                                  //-2019-02-12
      }
      if (zgamu >= 0) {                                           //-2016-06-28
        args.add("--gamu="+String.format(Locale.ENGLISH, "%1.1f;%1.5f", zgamu, gamu));
      }
      if (!print_pluris_log || !first_pluris_call) {
        args.add("--skip-log");
        args.add("--skip-dmn");
      }
      if (batch) {
        System.out.print(".");
        System.out.flush();
      }
      if (first_pluris_call)
        first_pluris_call = false;
      //                                                            -2015-11-23
      PipedWriter piw = new PipedWriter();
      PrintWriter pw = new PrintWriter(piw);
      IBJpluris.setPrinter(pw);
      Thread thread = new Thread(() -> {
        IBJpluris.main(args.toArray(new String[0]));
      });
      PipedReader pir = new PipedReader(piw);
      BufferedReader br = new BufferedReader(pir);
      thread.start();
      //
      String line = null;
      while (null != (line = br.readLine())) {
       if (line.startsWith("IBJpluris") && line.toLowerCase().contains("version"))
          PLURIS_VERSION = line;
        if (line.startsWith("Vq=")) {
          String[] ss = line.trim().split(";\\s");
          if (ss.length != 4)
            throw new Exception("unexpected PLURIS output string: "+line);
          if (!ss[1].startsWith("Ts="))
            throw new Exception("unexpected element 2 (Ts=) in PLURIS output string: "+line);
          if (!ss[2].startsWith("Dh="))
            throw new Exception("unexpected element 3 (Dh=) in PLURIS output string: "+line);
          plr = new double[3];
          plr[0] = Double.parseDouble(ss[0].substring(3).trim().replace(',', '.'));
          plr[1] = Double.parseDouble(ss[1].substring(3).trim().replace(',', '.'));
          plr[2] = Double.parseDouble(ss[2].substring(3).trim().replace(',', '.'));
        }
      }
    } 
    catch (Exception e) {
      e.printStackTrace(System.out);
      plr = null;
    }    
    finally {
    }
    return plr;
  }
  
  /**
   * Get minimum stack height for given roughness.
   * @param ha  anemometer height (m)
   * @param eq  emission (g/s)
   * @param dq  stack diameter (m)
   * @param tq  emission temperature (C)
   * @param vq  exit velocity
   * @param zq  water load (kg/kg dry)
   * @param rq  relative humidity (%)
   * @param lq  liquid water content (kg/kg)
   * @param sv  S-value (mg/m³)
   * @param sn  substance name or 'unknown'
   * @param ir  index of roughness class
   * @return    required minimum stack height
   */
  public Result getHeight(
      double ha,  // anemometer height (m)
      double eq,  // emission (g/s)
      double dq,  // stack diameter (m)
      double tq,  // gas temperature (C) 
      double vq,  // exit velocity (m/s)
      double zq,  // water load (kg/kg dry)
      double rq,  // relative humidity (%)
      double lq,  // liquid water content (kg/kg)
      double sv,  // S-value (mg/m³)
      String sn,  // substance name
      int ir)     // index of roughness length
  {
    first_pluris_call = true;
    for (int is=0; is<NN; is++) 
      results[is] = null;
    for (int is=0; is<NN; is++) {
      Result r = getHeight(ha, eq, dq, tq, vq, zq, rq, lq, sv, sn, ir, is);
      if (r == null)
        return null;
      if (r.hs > HQMAX)                                           //-2015-10-29
        r.hs = 999.9;
      results[is] = r;
    }
    //
    double hmin = 0;
    int imin = 0;
    Result rmin = null;
    for (int is=0; is<NN; is++) {
      Result r = results[is];
      double h = r.hs;
      if (h > hmin) {
        hmin = h;
        imin = is;
        rmin = r;
      }
    }
    //
    infos.clear();
    for (int is=0; is<NN; is++) {
      Result r = results[is];
      infos.add(String.format(Locale.ENGLISH, 
              "%skl=%1.1f, ua=%4.1f: hb=%5.1f, he=%5.1f (%4.1f%%)", 
              ((is==imin) ? "* " : "  "), kl(is), ua(is), r.hs, r.he, 100*r.de));
    }
    return rmin;
  }
  
  private double interpol(double h0, double c0, double h1, double c1, double cs) {
    double z0 = c0/cs;
    double z1 = c1/cs;
    double d = h1 - h0;
    double p = (2*h0*d - z1*h1*h1 + z0*h0*h0)/(d*d);
    double q = h0*h0*(1-z0)/(d*d);
    double a = -p/2 + Math.sqrt(0.25*p*p - q);
    double hs = h0 + a*d;
    return hs;
  }
  
  public double[] requiredHeff(double sq, int ir, int is) {
    if (chk) System.out.printf(LOC, "requiredHeff(%9.3e, %d, %d)%n", sq, ir, is);
    int k0 = 0;
    double c0 = maxval[ir][k0][is];
    double d0 = maxdev[ir][k0][is];
    if (sq > c0)
      return new double[] { 0, d0 };
    int k1 = NH-1;
    double c1 = maxval[ir][k1][is];
    double d1 = maxdev[ir][k1][is];
    if (sq < c1)
      return new double[] { vhq[k1], d1 };
    while(k1 > k0+1) {
      int k = (k0 + k1)/2;
      double c = maxval[ir][k][is];
      if (c <= sq) {
        k1 = k;
        c1 = c;
      }
      else {
        k0 = k;
        c0 = c;
      }
    }
    double h0 = vhq[k0];
    double h1 = vhq[k1];
    double he = interpol(h0, c0, h1, c1, sq);
    d0 = maxdev[ir][k0][is];
    d1 = maxdev[ir][k1][is];
    double de = d0 + (d1-d0)*(sq - c0)/(c1 - c0);
    if (chk) System.out.printf(LOC,
            "requiredHeff: k0=%d, h0=%1.1f, k1=%d, h1=%1.1f, he=%1.1f (%04.1f%%), c0=%9.3e, c1=%9.3e, sq=%9.3e%n", 
            k0, h0, k1, h1, he, 100*de, c0, c1, sq);
    return new double[] { he, de };
  }
  
  /**
   * Get minimum stack height for given roughness and meteorological situation.
   * With this stack height the near ground concentration equals the S-value.
   * @param ha  anemometer height (m)
   * @param eq  emission (g/s)
   * @param dq  stack diameter (m)
   * @param tq  emission temperature (C)
   * @param vq  exit velocity (m/s)
   * @param zq  water load (kg/kg dry)
   * @param rq  relative humidity (%)
   * @param lq  liquid water content (kg/kg)
   * @param sv  S-value (mg/m³)
   * @param sn  substance name or 'unknown'
   * @param ir  index of roughness length
   * @param is  index of meteorological situation
   * @return    minimum stack height 
   */
  public Result getHeight(
      double ha,  // anemometer height (m)
      double eq,  // emission (g/s)
      double dq,  // stack diameter (m)
      double tq,  // gas temperature (C)
      double vq,  // exit velocity (m/s)
      double zq,  // water load (kg/kg dry)
      double rq,  // relative humidity (%)
      double lq,  // liquid water content (kg/kg)
      double sv,  // S-value (mg/m³)
      String sn,  // substance name
      int ir,     // index of roughness length
      int is)     // index of meteorological situation
  {
    double sq = 0.001*sv/eq;
    Result result = new Result(eq, dq, tq, vq, zq, rq, lq, sv, sn, ir, is);
    double hs;
    double hreq;
    double dreq;
    double h0, h1, hm;
    //
    try {
      double[] hd = requiredHeff(sq, ir, is);
      hreq = hd[0]; // required effective plume height
      dreq = hd[1];
      if (hreq < vhq[0])  // minimum stack height is sufficient
        return result.insert(vhq[0], hreq, dreq);
      if (hreq >= HEMAX)  // maximum possible plume height is not sufficient
        return result.insert(HEMAX, hreq, dreq);
      //
      // try the lowest stack height and the next level
      //
      int k0 = 0;
      int k1;
      h0 = getHeff(ha, dq, tq, vq, zq, rq, lq, ir, is, k0);
      h1 = getHeff(ha, dq, tq, vq, zq, rq, lq, ir, is, k0+1);
      if (chk) System.out.printf(LOC, "getHeight: h[0]=%1.1f,  h[1]=%1.1f%n", h0, h1);
      if (h1 > h0) {  // effective plume height increases
        if (h0 >= hreq) // lowest stack height is sufficient
          return result.insert(vhb[k0], hreq, dreq);
      }
      //
      // try the highest used stack height
      //
      int km = NB-1;  
      hm = getHeff(ha, dq, tq, vq, zq, rq, lq, ir, is, km);
      if (chk) System.out.printf(LOC, "getHeight: hm=%1.1f%n", hm);
      if (hm < hreq)  // calculated plume height is not sufficient
        return result.insert(vhb[km], hreq, dreq);
      //
      if (h1 > h0) {
        //
        // determine hs by interval nesting
        //
        k1 = km;
        h1 = hm;
        while (k1 > k0+1) {
          int k = (k0 + k1)/2;
          double h = getHeff(ha, dq, tq, vq, zq, rq, lq, ir, is, k);
          if (h > hreq) {
            k1 = k;
            h1 = h;
          }
          else {
            k0 = k;
            h0 = h;
          }
        } // while
      }
      else {
        //
        // determine hs by stepping down from the highest level
        //
        k1 = km;
        k0 = k1-1;
        for ( ; k0>=0; k0--) {
          h0 = getHeff(ha, dq, tq, vq, zq, rq, lq, ir, is, k0);
          if (h0 <= hreq)
            break;
          h1 = h0;
          k1 = k0;
        }
      }
      //
      //  interpolate between h0 and h1
      //
      if (k0 < 0)
        hs = vhb[0];
      else {
        double a1 = (hreq - h0)/(h1 - h0);
        double a0 = 1 - a1;
        hs = a0*vhb[k0] + a1*vhb[k1];
      }
      if (chk) System.out.printf(LOC,
              "getHeight: k0=%d, h0=%1.1f, k1=%d, h1=%1.1f, hreq=%1.1f, hs=%1.1f%n", 
              k0, h0, k1, h1, hreq, hs);
    } 
    catch (Exception e) {
      e.printStackTrace(System.out);
      return null;
    }
    return result.insert(hs, hreq, dreq);
  }
  
  /**
   * Get maximum near ground concentration from active source with eq=1.
   * Plume rise is calculated by PLURIS.
   * @param ha  anemometer height (m). if <=0: 10+d0 used (m)
   * @param dq  stack diameter (m)
   * @param tq  emission temperature (C)
   * @param vq  exit velocity (m/s)
   * @param zq  water load (kg/kg dry)
   * @param rq  relative humidity (%)
   * @param lq  liquid water content (kg/kg)
   * @param ir  index of roughness class
   * @param is  index of meteorological situation
   * @param ib  index of stack height value
   * @return    effective source height
   */
  public double getHeff(
      double ha,  // anemometer height. if <=0: 10+d0 used (m)
      double dq,  // stack diameter (m)
      double tq,  // gas temperature (C)
      double vq,  // exit velocity (m/s)
      double zq,  // water load (kg/kg dry)
      double rq,  // relative humidity (%)
      double lq,  // liquid water content (kg/kg)
      int ir,     // index of roughness length
      int is,     // index of meteorological situation
      int ib)     // index of source height
  {
    if (chk) System.out.printf(LOC, 
            "getHeff(%4.2f, %1.0f, %1.2f, %d, %d, %d)%n", 
            dq, tq, vq, ir, is, ib);
    File wrk = new File(base);
    double z0 = vz0[ir];
    double d0 = 6*z0;
    if (ha <= 0)
      ha = 10.0 + d0;
    double ua = ua(is);
    double kl = kl(is);
    double hq = vhb[ib];
    double[] plr = new double[] { 0.0, 0.0, 0.0 };
    if (vq > 0 && dq > 0)                                //-2015-11-20 -2018-09-14
      plr = pluris(wrk, z0, d0, ha, ua, kl, hq, dq, vq, tq, zq, rq, lq);
    if (plr == null)
      throw new RuntimeException("PLURIS failed");
    double heff = hq + plr[2];
    if (chk) System.out.printf(LOC, "getHeff: vq=%1.1f, plr2=%1.1f, hq=%1.1f, heff=%1.1f%n", vq, plr[2], hq, heff);
    return heff;
  }

  //==========================================================================
  
  public static class Result {
    
    public final double eq;  // emission (g/s)
    public final double dq;  // stack diameter (m)
    public final double tq;  // gas temperature (C)
    public final double vq;  // exit velocity (m/s)
    public final double zq;  // water load (kg/kg dry)
    public final double rq;  // relative humidity (%)
    public final double lq;  // liquid water content (kg/kg)
    public final double sv;  // S-value (mg/m³)
    public final String sn;  // substance name
    public final int ir;     // index of roughness length
    public final int is;     // index of meteorological situation
    private double hs;  // stack height
    private double he;  // effective source height
    private double de;  // deviation of used concentration values
    
    public Result() {
      eq = 0;
      dq = 0;
      tq = 0;
      vq = 0;
      zq = 0;
      rq = 0;
      lq = 0;
      sv = 0;
      ir = 0;
      is = 0;
      sn = null;
    }
    
    public Result(double eq, double dq, double tq, double vq, 
            double zq, double rq, double lq, double sv, int ir, int is) {
      this(eq, dq, tq, vq, zq, rq, lq, sv, "unknown", ir, is);
    }
    
    public Result(double eq, double dq, double tq, double vq, 
            double zq, double rq, double lq, double sv, String sn, int ir, int is) {
      this.eq = eq;
      this.dq = dq;
      this.tq = tq;
      this.vq = vq;
      this.zq = zq;
      this.rq = rq;
      this.lq = lq;
      this.sv = sv;
      this.sn = sn;
      this.ir = ir;
      this.is = is;
    }
    
    public Result insert(double hs, double he, double de) {
      this.hs = hs;
      this.he = he;
      this.de = de;
      return this;
    }
    
    public double getHs() {
      return hs;
    }
    
    @Override
    public String toString() {
      String ss = ""+sv;
      String s = String.format(LOC, View.hform, sn, ss.replace('.', ','), 3.6*eq, dq, vq, tq, zq, hs);      
      return s;
    }
  }
  
  //==========================================================================
  
  public static class View {
    
    private static String hform = 
            "%-20s %8s %8.2E %4.1f %4.1f %4.0f %6.4f %5.1f";

    private Stage stage;
    private Scene scene;
    private BorderPane content;
    private Text title;
    private GridPane input;
    private ListView<Result> history;
    private TextArea detail;
    //
    private ChoiceBox<String> chcSubstance;
    private CheckedTextField ctfSubstance;
    private CheckedTextField ctfEmission;
    private CheckedTextField ctfDiameter;
    private CheckedTextField ctfTemperature;
    private CheckedTextField ctfVelocity;
    private CheckedTextField ctfWaterLoad;
    private CheckedTextField ctfHumidity;
    private CheckedTextField ctfLiquid;
    private TextField txtResult;
    private ProgressButton btnCalculate;                          //-2018-09-13
    private Button btnSave;
    private ChangeListener<String> listener;
    private Image logo0, logo1;
    //
    private Set<String> invalid_fields;
    private boolean init_detail = false;
    private final Map<Result, Result[]> runs = new LinkedHashMap<>();
    private final Besmin bmin;
    
    private View(Besmin sh) {
      this.bmin = sh;
    }
    
    private void create() {
      listener = (ObservableValue<? extends String> obs, String old, String val) 
              -> {
        Object bean = ((StringProperty)obs).getBean();
        CheckedTextField field = (CheckedTextField)bean;
        checkField(field, old, val);
      };
      invalid_fields = new HashSet<>();
      //
      content = new BorderPane();
      title = new Text("Schornsteinhöhe nach Nr. 5.5.2.2 TA Luft (2021)");
      title.setFont(Font.font("SansSerif", FontWeight.SEMI_BOLD, 
              FontPosture.REGULAR, 16.0));
      BorderPane.setMargin(title, new Insets(6, 0, 0, 0));
      //
      input = new GridPane();
      input.setHgap(4);
      input.setVgap(2);
      ColumnConstraints col0 = new ColumnConstraints();
      ColumnConstraints col1 = new ColumnConstraints();
      ColumnConstraints col2 = new ColumnConstraints(60, 60, 60,
              Priority.ALWAYS, HPos.LEFT, true);
      input.getColumnConstraints().addAll(col0, col1, col2);
      HBox boxSubstance = new HBox(4);
      boxSubstance.setAlignment(Pos.BASELINE_LEFT);
      chcSubstance = new ChoiceBox<>();
      chcSubstance.getItems().addAll(s_values.keySet());
      chcSubstance.setMaxWidth(Double.MAX_VALUE);
      ctfSubstance = new CheckedTextField("0.140", "sv", listener);
      ctfSubstance.setAlignment(Pos.BASELINE_RIGHT);
      ctfEmission = new CheckedTextField("100", "eq", listener);
      ctfEmission.setAlignment(Pos.BASELINE_RIGHT);
      ctfDiameter = new CheckedTextField("1", "dq", listener);
      ctfDiameter.setAlignment(Pos.BASELINE_RIGHT);
      ctfTemperature = new CheckedTextField("40", "tq", listener);
      ctfTemperature.setAlignment(Pos.BASELINE_RIGHT);
      ctfVelocity = new CheckedTextField("10", "vq", listener);
      ctfVelocity.setAlignment(Pos.BASELINE_RIGHT);
      ctfWaterLoad = new CheckedTextField("0", "zq", listener);
      ctfWaterLoad.setAlignment(Pos.BASELINE_RIGHT);
      ctfHumidity = new CheckedTextField("0", "rq", listener);
      ctfHumidity.setAlignment(Pos.BASELINE_RIGHT);
      ctfLiquid = new CheckedTextField("0", "lq", listener);
      ctfLiquid.setAlignment(Pos.BASELINE_RIGHT);
      btnCalculate = new ProgressButton();                        //-2018-09-13
      btnCalculate.setText("Schornsteinhöhe berechnen");
      btnCalculate.setMaxWidth(Double.MAX_VALUE);
      txtResult = new CheckedTextField();
      txtResult.setEditable(false);
      logo0 = new Image(btnCalculate.getClass().getResource("besmin.gif").toExternalForm());
      logo1 = new Image(btnCalculate.getClass().getResource("besmin-active.gif").toExternalForm());
      //
      VBox center = new VBox();
      int row = 0;
      Label ll;
      Label label = new Label("Stoff");
      label.setLabelFor(chcSubstance);
      boxSubstance.getChildren().addAll(label, chcSubstance);
      ll = new Label("S"); ll.getStyleClass().add("mono-text");
      input.addRow(row++, boxSubstance, 
              ll, ctfSubstance, new Label("mg/m³"));
      ll = new Label("eq"); ll.getStyleClass().add("mono-text");
      input.addRow(row++, new Label("Emissionsmassenstrom"), 
              ll, ctfEmission, new Label("kg/h"));   //-2016-03-10
      ll = new Label("dq"); ll.getStyleClass().add("mono-text");
      input.addRow(row++, new Label("Innendurchmesser"), 
              ll, ctfDiameter, new Label("m"));
      ll = new Label("vq"); ll.getStyleClass().add("mono-text");
      input.addRow(row++, new Label("Austrittsgeschwindigkeit"), 
              ll, ctfVelocity, new Label("m/s"));
      ll = new Label("tq"); ll.getStyleClass().add("mono-text");
      input.addRow(row++, new Label("Austrittstemperatur"), 
              ll, ctfTemperature, new Label("°C")); 
      ll = new Label("zq"); ll.getStyleClass().add("mono-text");
      input.addRow(row++, new Label("Wasserbeladung"), 
              ll, ctfWaterLoad, new Label("kg/(kg tr)"));
      input.add(btnCalculate, 0, row++, 4, 1);
      ll = new Label("hb"); ll.getStyleClass().add("mono-text");
      input.addRow(row++, new Label("Berechnete Schornsteinhöhe"), 
              ll, txtResult, new Label("m"));
      //
      Separator h_sep = new Separator(Orientation.HORIZONTAL);
      Label h_title = new Label("Durchgeführte Berechnungen");
      TextField h_header = new TextField(
              "Stoff                       S       eq   dq   vq   tq     zq    hb");
      h_header.setEditable(false);
      h_header.setPrefColumnCount(70);
      history = new ListView<Result>();
      history.setEditable(false);
      history.setPrefHeight(100);
      history.setMaxHeight(Double.MAX_VALUE);
      VBox.setVgrow(history, Priority.ALWAYS);
      btnSave = new Button("Rechenergebnisse speichern");         //-2015-10-29
      btnSave.setMaxWidth(Double.MAX_VALUE);
      //
      TabPane output = new TabPane();
      output.setMaxHeight(Double.MAX_VALUE);
      VBox.setVgrow(output, Priority.ALWAYS);
//      output.setBackground(new Background(new BackgroundFill(
//          Color.TRANSPARENT, null,null)));
      Tab tabHistory = new Tab("Durchgeführte Berechnungen");
      tabHistory.setClosable(false);
      VBox boxHistory = new VBox(h_header, history, btnSave);
      boxHistory.setMaxHeight(Double.MAX_VALUE);
      tabHistory.setContent(boxHistory);
      output.getTabs().add(tabHistory);
      //
      center.getChildren().addAll(input, h_sep, output);
      center.setPadding(new Insets(10, 10, 10, 10));
      VBox.setMargin(h_sep, new Insets(4, 0, 10, 0));
      //
      VBox east = new VBox();
      east.setPadding(new Insets(10, 10, 10, 0));
      Label i_title = new Label("Zwischenergebnisse");
      TextField i_header = new TextField("  kl    ua   heff   dev     hb  ");
      i_header.setEditable(false);
      i_header.setPrefColumnCount(32);
      detail = new TextArea();
      detail.setEditable(false);
      detail.setPrefRowCount(NN+2);
      detail.setPrefColumnCount(32);
      initDetail(true);
      east.getChildren().addAll(i_header, detail);
      //
      Tab tabDetail = new Tab("Zwischenergebnisse");
      tabDetail.setClosable(false);
      tabDetail.setContent(east);
      output.getTabs().add(tabDetail);      
      //
      BorderPane.setAlignment(title, Pos.CENTER);
      content.setTop(title);
      content.setCenter(center);
      scene = new Scene(content);
      stage = new Stage();
      stage.getIcons().add(logo0);
      stage.setScene(scene);
    }
    
    private void init() {
      chcSubstance.getSelectionModel().selectedItemProperty().addListener(
        (ObservableValue<? extends String> obs, String old, String val) -> {
          ctfSubstance.setEditable(val.equals(UNKNOWN));
          String s = ""+s_values.get(val);
          ctfSubstance.setText(s.replace('.', ','));
      });
      chcSubstance.getSelectionModel().select(UNKNOWN);
      history.getSelectionModel().selectedItemProperty().addListener(
        (ObservableValue<? extends Result> obs, Result old, Result val) -> {
          if (val == null)
            return;
          Result[] rr = runs.get(val);
          restoreInput(val, rr);
      });
      btnCalculate.setOnAction((ActionEvent e) -> {
        run();
      });
      btnSave.setOnAction((ActionEvent e) -> {                    //-2015-10-29
        btnSave.setDisable(true);
        saveHistory();
      });
      btnSave.setDisable(true);
      scene.getStylesheets().add("Besmin.css");
      stage.setTitle(my_name + " - Version " + my_version);
      stage.show();
    }
    
    private void checkField(CheckedTextField field, String old, String val) {
      initDetail(true);
      String id = field.getId();
      try {
        boolean rejected = false;
        double v = Double.parseDouble(val.trim().replace(',', '.'));
        switch(id) {
          case "sv": rejected = (v <= 0);
            break;
          case "eq": rejected = (v <= 0);
            break;
          case "dq": rejected = (v<0 || v>200);
            break;
          case "tq": rejected = (v<10 || v>600);
            break;
          case "vq": rejected = (v<0 || v>50);                    //-2021-10-07
            break;
          case "zq": rejected = (v<0 || v>2);                    //-2021-10-07
            break;
          case "rq": rejected = (v<0 || v>100);
            break;
          case "lq": rejected = (v<0 || v>0.1);
            break;
        }
        if (rejected) {
          field.setRejected(true);
          field.setInvalid(false);                                //-2021-10-04
          invalid_fields.add(id);
        }
        else {
          field.setInvalid(false);
          field.setRejected(false);
          invalid_fields.remove(id);
        }
      } 
      catch (Exception e) {
        invalid_fields.add(id);
        field.setRejected(false);
        field.setInvalid(true);
      }
      btnCalculate.setDisable(!invalid_fields.isEmpty());
    }

    private void appendHistory(Result r) {
      history.getItems().add(r);
      btnSave.setDisable(false);
    }
    
    private void saveHistory() {                                  //-2015-10-29
      File flog = new File(base);
      flog = new File(flog.getParentFile(), "log");
      if (!flog.exists())
        flog.mkdir();
      String[] files = flog.list();
      int nf = 0;
      for (String file: files) {
        String[] ss = file.split("[()]");
        if (ss.length==3 && ss[0].equals("besmin") && ss[2].equals(".log")) {
          try {
            int n = Integer.parseInt(ss[1].trim());
            if (n > nf)
              nf = n;
          } 
          catch (Exception e) {}
        }
      }
      File f = new File(flog, String.format("besmin(%d).log", nf+1));
      //
      List<String> list = new ArrayList<>();
      list.add(String.format("%s %s Version %s", 
              SDF.format(new Date()), my_name, my_version));
      if (PLURIS_VERSION != null)
        list.add(PLURIS_VERSION);
      list.add("Berechnete Schornsteinhöhen hb (in m):");
      list.add("");
      list.add("Stoff                       S       eq   dq   vq   tq     zq    hb");
      for (Result r: history.getItems())
        list.add(r.toString());
      try {
        Files.write(f.toPath(), list);
        if (chk) 
          System.out.printf("file '%s' written%n", f.getAbsolutePath());
      } 
      catch (Exception e) {
        e.printStackTrace(System.out);
      }
    }
    
    private void initDetail(boolean init) {
      if (!init) {
        init_detail = false;
        return;
      }
      if (init_detail)
        return;
      StringBuilder sb = new StringBuilder();
      for (int i=0; i<NN; i++)
        sb.append(String.format(LOC,
                " %3.1f  %4.1f                      \n", kl(i), ua(i)));
      detail.setText(sb.toString());
      txtResult.clear();
      init_detail = true;
    }
    
    private void replaceDetail(int is, double heff, double dev,
            double hb, boolean star) {
      int pos = is*33 + 12;
      String s = String.format(LOC, 
              "%5.1f %4.1f%%  %5.1f%2s", heff, 100*dev, hb, (star) ? " *" : " ");
      detail.replaceText(pos, pos+s.length(), s);
      init_detail = false;
    }
    
    private void restoreInput(Result r, Result[] rr) {
      if (r == null || rr == null)
        return;
      chcSubstance.getSelectionModel().select(r.sn);
      if (r.sn.equals(UNKNOWN))
        ctfSubstance.setText(("" + r.sv).replace('.', ','));
      ctfEmission.setText(("" + 3.6*r.eq).replace('.', ','));      //-2016-03-10
      ctfDiameter.setText(("" + r.dq).replace('.', ','));
      ctfTemperature.setText(("" + r.tq).replace('.', ','));
      ctfVelocity.setText(("" + r.vq).replace('.', ','));
      ctfWaterLoad.setText(("" + r.zq).replace('.', ','));
      txtResult.setText(String.format(LOC, "%5.1f", r.hs));
      
      for (Result s: rr)
        replaceDetail(s.is, s.he, s.de, s.hs, r==s);
    }
    
    private void disableInput(boolean doit) {
      chcSubstance.setDisable(doit);
      ctfSubstance.setDisable(doit);
      ctfEmission.setDisable(doit);
      ctfDiameter.setDisable(doit);
      ctfTemperature.setDisable(doit);
      ctfVelocity.setDisable(doit);
      ctfWaterLoad.setDisable(doit);
      btnCalculate.setDisable(doit);
    }
    
    private void initRun() {
      if (prn) System.out.printf("%s calculation started%n", 
          SDF.format(new Date()));
      disableInput(true);
      btnCalculate.setProgress(0);
      history.getSelectionModel().clearSelection();               //-2018-09-14
      stage.getIcons().remove(logo0);
      stage.getIcons().add(logo1);
    }
    
    private void finishRun(Besmin sh) {
      if (prn) System.out.printf("%s calculation finished%n", 
          SDF.format(new Date()));
      btnCalculate.clearProgress();                               //-2018-09-13
      stage.getIcons().remove(logo1);
      stage.getIcons().add(logo0);
      double hmin = 0;
      Result rmin = null;
      Result[] results = new Result[NN];                          //-2018-09-14
      for (int is=0; is<NN; is++) {
        Result r = sh.results[is];
        results[is] = r;                                          //-2018-08-14
        if (r == null) {
          rmin = null;
          break;
        }
        double h = r.hs;
        if (h <= 0) {
          rmin = null;
          break;
        }
        if (h > hmin) {
          hmin = h;
          rmin = r;
        }
      }
      if (rmin == null)
        txtResult.setText("?");
      else {
        txtResult.setText(String.format(LOC, "%5.1f",  rmin.hs));
        replaceDetail(rmin.is, rmin.he, rmin.de, rmin.hs, true);
        runs.put(rmin, results);                                  //-2018-09-14
        appendHistory(rmin);
      }
      disableInput(false);
    }

    private void run() {
      try {
        btnCalculate.setProgress(0);                              //-2018-09-13
        initDetail(true);
        initRun();
        double sv = Double.parseDouble(
                ctfSubstance.getText().trim().replace(',', '.'));
        double eq = (1/3.6)*Double.parseDouble(                   //-2016-03-10
                ctfEmission.getText().trim().replace(',', '.'));
        double dq = Double.parseDouble(
                ctfDiameter.getText().trim().replace(',', '.'));
        double tq = Double.parseDouble(
                ctfTemperature.getText().trim().replace(',', '.'));
        double vq = Double.parseDouble(
                ctfVelocity.getText().trim().replace(',', '.'));
        double zq = Double.parseDouble(
                ctfWaterLoad.getText().trim().replace(',', '.'));
        double rq = 0;
        double lq = 0;
        String sn = chcSubstance.getValue();
        for (int is=0; is<NN; is++)
          bmin.results[is] = new Result(eq, dq, tq, vq, zq, rq, lq, sv, sn, Z0i, is);
        LinkedBlockingQueue<String> commands = new LinkedBlockingQueue<>();
        Work work = bmin.new Work(commands);
        //
        work.valueProperty().addListener((ObservableValue<? extends Result> obs,
                Result old, Result val) -> {
          try {
            if (val == null) {  // Work finished
              commands.clear();
              finishRun(bmin);
              return;
            }
            btnCalculate.setProgress((val.is+1)/(double)NN);      //-2018-09-14
            if (val.hs > HQMAX)
              val.hs = 999.9;
            bmin.results[val.is] = val;
            replaceDetail(val.is, val.he, val.de, val.hs, false);
            if (val.hs <= 0) {  // error
              finishRun(bmin);
              return;
            }
            commands.put("continue");
          } 
          catch (InterruptedException ex) {
            ex.printStackTrace(System.out);
          }
        });
        // 
        new Thread(work).start();
        commands.put("continue");
      } 
      catch (Exception e) {
        e.printStackTrace(System.out);
      }
    }
        
  } //------------------View
  
  //==========================================================================
  
  public class Work extends Task<Result> {
//    private final Result[] results;
    private final LinkedBlockingQueue<String> cmds;
    
    public Work(LinkedBlockingQueue<String> cmds) {
//      this.results = results;
      this.cmds = cmds;
    }

    @Override
    protected Result call() throws Exception {
      try {
        for (Result p: results) {
          String cmd = cmds.take();
          if (!cmd.equals("continue"))
            break;
          Result r = getHeight(-1., p.eq, p.dq, p.tq, p.vq, p.zq, p.rq, p.lq, p.sv, 
                  p.sn, p.ir, p.is);
          if (r == null) {  // error
            updateValue(p);
            return p;
          }
          updateValue(r);
        }
      } 
      catch (Exception e) {}
      cmds.take();
      updateValue(null);
      return null;
    }
    
  } 
   
  //==========================================================================
  
  private static void test1() {
    chk = false;
    try {
      double eq = 100;    // emission (g/s)
      double dq = 5;      // stack diameter (m)
      double tq;          // gas temperature (C)
      double vq = 10;     // exit velocity (m/s)
      double zq = 0;      // water load (kg/kg dry)
      double rq = 0;      // relative humidity (1)
      double lq = 0;      // liquid water content (kg/kg)
      double sv = 0.14;   // S-value (mg/m³)
      int iz0 = 5;
      Besmin sh = new Besmin(null);
      for (int i=0; i<=10; i++) {
        tq = 30 + i;
        Result r = sh.getHeight(-1., eq, dq, tq, vq, zq, rq, lq, sv, 
                UNKNOWN, iz0);
        System.out.printf(LOC, "tq=%1.0f, hb=%5.1f%n", tq, r.hs);
      }
    } 
    catch (Exception e) {
      e.printStackTrace(System.out);
    }
    System.exit(0);
  }
  
  private static void test2() {
    try {
      PipedWriter piw = new PipedWriter();
      Thread t = new Thread(() -> {
        PrintWriter pw;
        pw = new PrintWriter(piw);
        for (int i=1; i<=10; i++) {
          pw.println("line " + i);  pw.flush();
        }
        pw.close();
      });
      PipedReader pir = new PipedReader(piw);
      BufferedReader br = new BufferedReader(pir);
      t.start();
      String line = null;
      while (null != (line = br.readLine())) {
        System.out.println(line);
      }
    } 
    catch (Exception e) {
      e.printStackTrace(System.out);
    }
    System.out.println("finished");
    System.exit(0);
  }
  
  private static String my_home() {                               //-2018-09-14
    Class<Besmin> cls = Besmin.class;
    String nm = cls.getName();
    URL u2 = cls.getResource(cls.getSimpleName() + ".class"); 
    String s2;
    File root = null;
    try {
      URI uri = u2.toURI();
      String sScheme = uri.getScheme();
      s2 = uri.getSchemeSpecificPart();
      if (sScheme.equals("jar")) { 
        if (s2.startsWith("file:")) s2 = s2.substring(5); 
      } 
      else if (!sScheme.equals("file")) { 
        return null;
      }
      int k = s2.indexOf(nm.replace('.', '/'));
      if (k > 0) s2 = s2.substring(0, k-1);
      if (s2.endsWith(".jar!")) { 
        s2 = s2.substring(0, s2.length() - 1);
        root = (new File(s2)).getParentFile();
      }
      else {
        root = new File(s2);
      }
      if (root.getName().equals("classes") 
          && root.getParentFile().getName().equals("build"))
        root = root.getParentFile().getParentFile();
    } 
    catch (Exception e) {
      e.printStackTrace(System.out);
    }
    return (root != null) ? root.getAbsolutePath() : null;
  }
  
  private static void checkPlumeData(String dir) throws Exception {
    FileInputStream fis;
    CRC32 crc32 = new CRC32();
    byte[] arr;
    String sr, sc;
    boolean found;
    for (int ih=0; ih<NH; ih++) {
      double hq = vhq[ih];
      String dn = String.format("mv%02.0frl%03.0fhq%03.0f", 10*metvers, 100*Z0, hq);    
      String fn = "maxima/maxima-" + dn + ".txt";
      File f = new File(dir, fn);
      fis = new FileInputStream(f);
      arr = new byte[(int)f.length()];
      fis.read(arr, 0, arr.length);
      fis.close();
      crc32.reset();
      crc32.update(arr);
      sc = String.format("%x", crc32.getValue());
      found = false;
      for (String s : PLUME_LIST) {
        if (s.contains(fn)) {
          found = true;
          sr = s.substring(s.lastIndexOf("|")+1);
          if (!sc.equalsIgnoreCase(sr))
            throw new Exception("unregistered content in file "+fn);
        }
      }
      if (!found)
        throw new Exception("file "+fn+" not found in internal file list");
    }
  }
  
  private static final String[] PLUME_LIST = {
    "maxima/maxima-mv53rl050hq006.txt|c124535a", "maxima/maxima-mv53rl050hq008.txt|dc8cc777", 
    "maxima/maxima-mv53rl050hq010.txt|74a1e3ab", "maxima/maxima-mv53rl050hq012.txt|68ba55c7", 
    "maxima/maxima-mv53rl050hq014.txt|9bd46aee", "maxima/maxima-mv53rl050hq017.txt|10dda463", 
    "maxima/maxima-mv53rl050hq020.txt|e28f29b0", "maxima/maxima-mv53rl050hq024.txt|67299d85", 
    "maxima/maxima-mv53rl050hq028.txt|19846a9d", "maxima/maxima-mv53rl050hq034.txt|900fe0c6", 
    "maxima/maxima-mv53rl050hq040.txt|e58022cf", "maxima/maxima-mv53rl050hq048.txt|9bd70178", 
    "maxima/maxima-mv53rl050hq057.txt|af42e351", "maxima/maxima-mv53rl050hq067.txt|869d1b57", 
    "maxima/maxima-mv53rl050hq080.txt|c5c5b2ae", "maxima/maxima-mv53rl050hq095.txt|cf904faa", 
    "maxima/maxima-mv53rl050hq113.txt|397ffe06", "maxima/maxima-mv53rl050hq135.txt|16035724", 
    "maxima/maxima-mv53rl050hq160.txt|5313c46b", "maxima/maxima-mv53rl050hq190.txt|bdaf1ca3", 
    "maxima/maxima-mv53rl050hq226.txt|ab417160", "maxima/maxima-mv53rl050hq269.txt|2f115da7", 
    "maxima/maxima-mv53rl050hq320.txt|d74b76b3", "maxima/maxima-mv53rl050hq381.txt|f755b796", 
    "maxima/maxima-mv53rl050hq453.txt|a183d02a", "maxima/maxima-mv53rl050hq538.txt|4c275004", 
    "maxima/maxima-mv53rl050hq640.txt|bc56aa4c", "maxima/maxima-mv53rl050hq761.txt|83f8a629", 
    "maxima/maxima-mv53rl050hq905.txt|f22cfa45"
  };
  
  
  public static void main(String[] args) {
    //Locale.setDefault(Locale.ENGLISH);    
    String base_dir = my_home();                                  //-2018-09-14
    for (String arg: args) {
      if (arg.startsWith("--"))
        batch = true;
      else
        base_dir = arg;
    }
    //
    try {
      Besmin sh = new Besmin(base_dir);
      String msg = sh.readMaxima();
      if (msg != null) {
        System.out.printf("%s%n", msg);
        return;
      }
      initSvalues();
      //
      if (!batch) {
        JFXPanel jfxp = new JFXPanel();
        Platform.runLater(() -> {
          View view = new View(sh);
          view.create();
          view.init();
        });
        return;
      }
      double ha = -1.;          // anemometer height, default: 10+d0 (m)
      double eq = Double.NaN; // emission (g/s)
      double dq = Double.NaN; // stack diameter (m)
      double tq = Double.NaN; // gas temperature (C)
      double vq = Double.NaN; // exit velocity (m/s)
      double zq = 0;          // water load (kg/kg dry)
      double rq = 0;          // relative humidity (relative to saturation)
      double lq = 0;          // liquid water content (mass ratio)
      double sv = Double.NaN; // S-value (mg/m³)
      //
      boolean check = true;
      boolean non_standard = false;
      for (String arg : args) {
        if (arg.equalsIgnoreCase(base))                            //-2015-12-28
          continue;
        else if (arg.equals("--log-pluris")) {                    //-2018-09-15
          print_pluris_log = true;
          continue;
        }
        else if (arg.equals("--skip-stacktip-downwash")) {
          skip_stacktip_downwash = true;
          non_standard = true;
          continue;
        }
        else if (arg.startsWith("--break-factor=")) {
          break_factor = Double.parseDouble(arg.substring(15).trim().replace(',', '.'));
          if (break_factor != BREAKFACTOR)
            non_standard = true;
          continue;
        }
        /*
        else if (arg.startsWith("--gamu=")) {
          String[] sa = arg.substring(7).split("[;]");
          zgamu = Double.parseDouble(sa[0].trim().replace(',', '.'));
          gamu = Double.parseDouble(sa[1].trim().replace(',', '.'));  
          if (zgamu >= 0)
            non_standard = true;
          continue;
        }
        */
        else if (!arg.startsWith("--") || arg.length()<6 || arg.charAt(4) != '=') {
          System.out.printf("*** invalid argument '%s'%n", arg);
          check = false;
          continue;
        }
        String name = arg.substring(2, 4).toLowerCase();
        if (!names.contains(name+",")) {
          System.out.printf("*** invalid name '%s'%n", name);
          check = false;
          continue;
        }
        double value;
        try {
          value = Double.parseDouble(arg.substring(5).trim().replace(',', '.'));
        } 
        catch (Exception e) {
          System.out.printf("*** can't parse value of '%s'%n", name);
          check = false;
          continue;
        }
        switch(name) {
          /*
          case "ha": 
            ha = value;
            break;
          */
          case "eq":
            if (value <= 0) {
              System.out.printf("*** invalid value of 'eq' (%s)%n", value);
              check = false;
            }
            else 
              eq = value/3.6;                                     //-2016-03-10
            break;
          case "dq":
            if (value < 0 || value > 200) {
              System.out.printf("*** invalid value of 'dq' (%s)%n", value);
              check = false;
            }
            else 
              dq = value;
            break;
          case "tq":
            if (value < 10 || value > 600) {
              System.out.printf("*** invalid value of 'tq' (%s)%n", value);
              check = false;
            }
            else 
              tq = value;
            break;
          case "vq":
            if (value < 0 || value > 50) {
              System.out.printf("*** invalid value of 'vq' (%s)%n", value);
              check = false;
            }
            else 
              vq = value;
            break;
          case "zq":
            if (value < 0 || value > 2) {                         //-2021-10-07
              System.out.printf("*** invalid value of 'zq' (%s)%n", value);
              check = false;
            }
            else
              zq = value;
            break;
          /*
          case "rq":
            if (value < 0 || value > 100) {
              System.out.printf("*** invalid value of 'rq' (%s)%n", value);
              check = false;
            }
            else 
              rq = value;
            break;
          case "lq":
            if (value < 0 || value > 0.1) {
              System.out.printf("*** invalid value of 'lq' (%s)%n", value);
              check = false;
            }
            else 
              lq = value;
            break;
          */
          case "sv":
            if (value <= 0) {
              System.out.printf("*** invalid value of 'sv' (%s)%n", value);
              check = false;
            }
            else 
              sv = value;
            break;
          default:
            ;
        } // switch
      } // for arg;
      if (Double.isNaN(eq)) {
        System.out.println("*** missing value of 'eq'");
        check = false;
      }
      if (Double.isNaN(dq)) {
        System.out.println("*** missing value of 'dq'");
        check = false;
      }
      if (Double.isNaN(tq)) {
        System.out.println("*** missing value of 'tq'");
        check = false;
      }
      if (Double.isNaN(vq)) {
        System.out.println("*** missing value of 'vq'");
        check = false;
      }
      if (Double.isNaN(sv)) {
        System.out.println("*** missing value of 'sv'");
        check = false;
      }
      if (!check) {
        System.out.println("program stopped");
        return;
      }
      //
      System.out.printf("%s, Version %s%n", my_name, my_version);   
      Result result = sh.getHeight(ha, eq, dq, tq, vq, zq, rq, lq, sv, UNKNOWN, Z0i);
      if (result == null) {
        System.out.println("*** can't determine stack height");
      }
      else {
        System.out.println();
        if (PLURIS_VERSION != null)
          System.out.printf("%s%n", PLURIS_VERSION); 
        for (String r: sh.infos)
          System.out.println(r);
        if (non_standard)
          System.out.printf("WARNING: non-standard settings specified on program call!\n");
        System.out.printf(Locale.ENGLISH, "calculated stack height = %1.1f m%n", result.hs);
      }
      System.out.println("program finished");
    } 
    catch (Exception e) {
      e.printStackTrace(System.out);
    }
  }
  
}
