package de.janicke.ibjutil;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.nio.charset.Charset;
import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.Map;

/**
 *
 * @author ibj
 * 
 * Last change: 2012-07-11
 * @param <T>
 */
public class IBJlst<T extends IBJlst.Dump> {
  
  public static boolean DEBUG = false;
  Map<String, String> header;
  ArrayList<T> list;
    
  public IBJlst(Map<String, String> hdr, ArrayList<T> lst) {
    if (hdr == null)
      hdr = new LinkedHashMap<String, String>();
    header = hdr;
    list = lst;
  }
    
  public int write(String fn) throws Exception {
    int n = 0;
    FileOutputStream fos = null;
    BufferedOutputStream bos = null;
    ByteOrder order = ByteOrder.nativeOrder();
    Charset cset = Charset.defaultCharset();
    String s = null;
    try {
      s = header.get("Encoding");
      if (s != null) 
        cset = Charset.forName(s);
      else
        header.put("Encoding", cset.displayName());
      s = header.get("ByteOrder");
      if (s != null) {
        if (s.equalsIgnoreCase("LITTLE_ENDIAN")) 
          order = ByteOrder.LITTLE_ENDIAN;
        else if (s.equalsIgnoreCase("BIG_ENDIAN"))
          order = ByteOrder.BIG_ENDIAN;
      }
      else
        header.put("ByteOrder", order.toString());
      //
      File f = new File(fn);
      fos = new FileOutputStream(f);
      bos = new BufferedOutputStream(fos);
      //
      // write header
      // 
      byte[] bytes;
      int nl = "\n".getBytes(cset)[0];
      for (String key: header.keySet()) {
        bytes = (key + "=" + header.get(key) + "\n").getBytes(cset);
        bos.write(bytes);
        n += bytes.length;
      }
      bos.write(nl);
      n++;
      //
      // write data
      //
      ByteBuffer bb = ByteBuffer.allocate(4);
      bb.order(order);
      for (T t: list) {
        bytes = t.dump(order, cset);
        bb.clear();
        bb.putInt(bytes.length);
        bos.write(bb.array());
        n += 4;
        bos.write(bytes);
        n += bytes.length;
      }
      bb.clear();
      bb.putInt(0);
      bos.write(bb.array());
      n += 4;
      bos.close();
      bos = null;
    }
    catch (Exception e) {
      e.printStackTrace(System.out);
      n = -1;
      if (bos != null)
        try { bos.close(); bos = null; } catch (Exception x) {}
    }
    return n;
  }
  
  public int read(String fn, Build<T> factory ) {
    int n = 0;
    list.clear();
    header.clear();
    Charset cset = Charset.defaultCharset();
    ByteOrder order = ByteOrder.nativeOrder();
    if (DEBUG) System.out.printf("native order = %s\n", order);
    FileInputStream fis = null;
    BufferedInputStream bis = null;
    try {
      File f = new File(fn);
      fis = new FileInputStream(f);
      bis = new BufferedInputStream(fis);
      // 
      // read header
      //
      int nl = "\n".getBytes(cset)[0];
      byte[] bytes = new byte[32000];
      int nb = 0;
      while(true) {
        if (0 == bis.read(bytes, nb, 1))
          throw new Exception("unexpected end of file");
        n++;
        if (bytes[nb] == nl) {
          if (nb == 0)
            break;
          String s = new String(bytes, 0, nb, cset);
          if (DEBUG) System.out.printf("READ HDR: %s\n", s);
          String[] ss = s.split("[=]");
          if (ss.length != 2)
            throw new Exception("invalid header entry \"" + s + "\"");
          header.put(ss[0], ss[1]);
          if (ss[0].equalsIgnoreCase("Encoding"))
            cset = Charset.forName(ss[1]);
          else if (ss[0].equalsIgnoreCase("ByteOrder")) {
            if (ss[1].equalsIgnoreCase("LITTLE_ENDIAN")) 
              order = ByteOrder.LITTLE_ENDIAN;
            else if (ss[1].equalsIgnoreCase("BIG_ENDIAN"))
              order = ByteOrder.BIG_ENDIAN;
          }
          nb = 0;
        }
        else
          nb++;
      }
      //
      // read data
      //
      byte[] ba = new byte[4];
      ByteBuffer bb = ByteBuffer.wrap(ba);
      bb.order(order);
      if (DEBUG) System.out.printf("using order = %s\n", order);
      int nr = 0;
      while (true) {
        bb.clear();
        nr = bis.read(ba);
        if (DEBUG) System.out.printf("A nr=%d\n", nr);
        if (nr != 4)
          throw new Exception("can't read length field");
        n += nr;
        int l = bb.getInt();
        if (DEBUG) System.out.printf("len=%d\n", l);
        if (l == 0)
          break;
        bytes = new byte[l];
        nr = bis.read(bytes);
        if (DEBUG) System.out.printf("B nr=%d\n", nr);
        if (nr != l)
          throw new Exception("can't read record");
        n += nr;
        T t = factory.build(bytes, order, cset);
        list.add(t);
      }
      bis.close();
      bis = null;
    }
    catch (Exception e) {
      e.printStackTrace(System.out);
      n = -1;
      if (bis != null)
        try { bis.close(); bis = null; } catch (Exception x) {}
    }
    return n;
  }
  
  
  //===========================================================================
  
  public interface Dump {
    public byte[] dump(ByteOrder order, Charset cset);
  }
  
  public interface Build<U> {
    public U build(byte[] bytes, ByteOrder order, Charset cset);
  }
  
}
