////////////////////////////////////////////////////////////////////////////////
//
// history
//
// 2015-10-28 lj 0.1 
// 2015-10-31 lj 0.2    columns removable
// 2015-11-03 lj 0.3    graphics
// 2015-11-04 lj 0.4    threshold (5%) in graphics
// 2015-11-05 lj 0.5    save results
// 2015-11-10 lj 0.6    logging pluris version
// 2015-11-23 lj 0.7    parameter rq and lq, direct call of IBJpluris
// 2016-04-05 lj 0.8    batch mode, logging of source parameter q
// 2016-04-05 lj 0.9    source strength in kg/h, pluris parameter from BESMIN
// 2016-04-08 uj 0.10   source strength in kg/h in command line, add. pluris parms
// 2016-07-12 uj 0.11   restricted to z0=0.5m
// 2019-03-04 uj 0.4.0  updated input parameters
// 2019-09-12 uj 0.5.0  height intervals corrected
// 2019-09-30 uj 0.6.0  height check (hq) activated
// 2021-09-16 uj 1.0.0  multi-threading, user-defined number of threads
//                      option --interactive
//                      import command-line source definitions to GUI
// 2021-10-07 uj 1.0.1  data check
// 
////////////////////////////////////////////////////////////////////////////////

/**
 * BESMAX
 * 
 * Determination of maximum concentration for several sources using
 * the plumes calculated for the determination of the minimum stack height 
 * according to TA Luft (2021)
 *
 * Copyright (C) Umweltbundesamt Dessau-Roßlau, Germany, 2016-2021
 * Copyright (C) Janicke Consulting, Überlingen, Germany, 2016-2021
 * email: info@austal.de
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 * 
 */

package de.janicke.tal;

import de.janicke.SVG.Draw; 
import de.janicke.fxutil.CheckedTextField;
import de.janicke.fxutil.ProgressButton;
import javafx.scene.image.Image;
import de.janicke.ibjutil.IBJarr;
import de.janicke.ibjutil.IBJdmn;
import de.janicke.ibjutil.IBJhdr;
import de.janicke.pluris.IBJpluris;
import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.PipedReader;
import java.io.PipedWriter;
import java.io.PrintWriter;
import java.net.URI;
import java.net.URL;
import java.nio.file.Files;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;
import java.util.function.Function;
import java.util.zip.CRC32;
import javafx.application.Platform;
import javafx.beans.property.SimpleDoubleProperty;
import javafx.beans.property.StringProperty;
import javafx.beans.value.ChangeListener;
import javafx.beans.value.ObservableValue;
import javafx.concurrent.Task;
import javafx.embed.swing.JFXPanel;
import javafx.event.ActionEvent;
import javafx.geometry.HPos;
import javafx.geometry.Insets;
import javafx.geometry.Point2D;
import javafx.geometry.Pos;
import javafx.scene.Node;
import javafx.scene.Scene;
import javafx.scene.control.Button;
import javafx.scene.control.ChoiceBox;
import javafx.scene.control.ContentDisplay;
import javafx.scene.control.Label;
import javafx.scene.control.ScrollPane;
import javafx.scene.control.Separator;
import javafx.scene.control.TextField;
import javafx.scene.layout.BorderPane;
import javafx.scene.layout.ColumnConstraints;
import javafx.scene.layout.GridPane;
import javafx.scene.layout.HBox;
import javafx.scene.layout.Priority;
import javafx.scene.layout.StackPane;
import javafx.scene.layout.VBox;
import javafx.scene.paint.Color;
import javafx.scene.text.Font;
import javafx.scene.text.FontPosture;
import javafx.scene.text.FontWeight;
import javafx.scene.text.Text;
import javafx.stage.Stage;
import javafx.stage.WindowEvent;


public class Besmax {
  
  private static String base = "./";
  private static String PLURIS_VERSION = null;

  private static final String my_name = "BESMAX";
  private static final String my_version = "1.0.1";
  private static final Locale LOC = Locale.GERMAN;
  private static final SimpleDateFormat sdf = 
          new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
  private static final int NH = 27+2;
  private static final int NN = 25;
  private static final double HQMAX = 250;
  private static final double HQMIN = 6;
  private static final double BREAKFACTOR = 1.7;
  private static boolean chk = false;
  private static final double metvers = 5.3;
  private static final double[] ua9 = 
          new double[] { 1.0, 1.5, 2.0, 3.0, 4.5, 6.0, 7.5, 9.0, 12.0 };
  private static final double[] kl6 =
          new double[] { 1.0, 2.0, 3.1, 3.2, 4.0, 5.0 }; 
  
  private static final double[] vz0 = new double[]                //-2016-07-12
    { 0.5 }; // meter
  
  private static final double[] vhq = new double[NH];
  static {
    vhq[0] = 6;                                                   //-2018-09-02
    vhq[1] = 8;
    for (int ih=0; ih<vhq.length-2; ih++)
      vhq[ih+2] = 10*Math.pow(2, ih*0.25);                        //-2019-09-12
  }
  private static boolean skip_stacktip_downwash = false;          //-2016-04-08
  private static double break_factor = BREAKFACTOR;               //-2019-02-28
  private static boolean first_pluris_call;
  //
  private double z0 = 0;
  private int nq;
  private Source[] sources;
  private Point2D center;
  //
  private Map<Integer, ConDev> condevs;
  private double steps_maximal, steps_current;
  private SimpleDoubleProperty progress;
  
  public Besmax(String base_dir, List<Source> list, int ir) {
    if (base_dir != null)
      base = base_dir;
    z0 = vz0[ir];
    nq = list.size();
    sources = list.toArray(new Source[nq]);
    condevs = new LinkedHashMap<>();
    progress = new SimpleDoubleProperty(this, "progress", 0.0);
    steps_maximal = 3*nq*NN;
    steps_current = 0;
  }
  
  private void step(int n) {
    steps_current += n;
    progress.set(steps_current/steps_maximal);
  }
  
  private String init() {
    String msg;
    try {
      if (nq == 0) 
        return "no sources";
      checkPlumeData(base);
      double xsum = 0;
      double ysum = 0;
      double wsum = 0;
      first_pluris_call = true;
      for (Source src: sources) {
        msg = setHeff(src); // plume rise calculation
        if (msg != null)
          return msg;
        double w = src.q;
        xsum += w*src.x;
        ysum += w*src.y;
        wsum += w;
      }
      if (wsum <= 0)
        return "no emissions";
      double xc = xsum/wsum;
      double yc = ysum/wsum;
      center = new Point2D(xc, yc);
      for (Source src: sources) {   // emission center point is new origin
        src.x -= xc;
        src.y -= yc;
      }
      //
      log("setting layers...");
      for (Source src: sources) {
        msg = setLayers(src);
        if (msg != null)
          return msg;
      }         
    } 
    catch (Exception e) {
      e.printStackTrace(System.out);
      return "*** " + e.toString();
    }
    return null;
  }

  private ConDev map(int is, int rot, boolean outer, double cmin) {
    ConDev condev = new ConDev();
    condev.sources = new Source[nq];
    double dd = Double.POSITIVE_INFINITY;
    double xmin = Double.POSITIVE_INFINITY;
    double ymin = Double.POSITIVE_INFINITY;
    double xmax = Double.NEGATIVE_INFINITY;
    double ymax = Double.NEGATIVE_INFINITY;
    int ni;
    int nj;
    double sin = Math.sin(Math.toRadians(rot));
    double cos = Math.cos(Math.toRadians(rot));
    //
    try {
      for (int iq=0; iq<sources.length; iq++) {
        Source src = sources[iq];
        ConDev cd0 = src.lower[is];
        ConDev cd1 = src.upper[is];
        double d0 = cd0.dd;
        if (iq == 0) {
          xmin = cd0.xmin;
          xmax = cd0.xmax;
          ymin = cd0.ymin;
          ymax = cd0.ymax;
        }
        double sx = cos*src.x - sin*src.y; 
        double sy = cos*src.y + sin*src.x;
        condev.sources[iq] = src.getShifted(sx, sy);
        if (d0 < dd)  dd = d0;
        //
        double xa = sx + Math.min(cd0.xmin, cd1.xmin);
        if (xa < xmin)  xmin = xa;
        double ya = sy + Math.min(cd0.ymin, cd1.ymin);
        if (ya < ymin)  ymin = ya;
        double yb = sy + Math.max(cd0.ymax, cd1.ymax);
        if (yb > ymax)  ymax = yb;
      //
        if (outer) {  // outer boundary
//          double xa = sx + Math.min(cd0.xmin, cd1.xmin);
          double xb = sx + Math.max(cd0.xmax, cd1.xmax);
//          double ya = sy + Math.min(cd0.ymin, cd1.ymin);
//          double yb = sy + Math.max(cd0.ymax, cd1.ymax);
//          if (xa < xmin)  xmin = xa;
          if (xb > xmax)  xmax = xb;
//          if (ya < ymin)  ymin = ya;
//          if (yb > ymax)  ymax = yb;
        }
        else {  // inner boundary
//          double xa = sx + Math.max(cd0.xmin, cd1.xmin);
          double xb = sx + Math.min(cd0.xmax, cd1.xmax);
//          double ya = sy + Math.max(cd0.ymin, cd1.ymin);
//          double yb = sy + Math.min(cd0.ymax, cd1.ymax);
//          if (xa > xmin)  xmin = xa;
          if (xb < xmax)  xmax = xb;
//          if (ya > ymin)  ymin = ya;
//          if (yb < ymax)  ymax = yb;
        }
      }
      ni = (xmax <= xmin + dd) ? 1 : (int)Math.floor((xmax-xmin)/dd);
      nj = (ymax <= ymin + dd) ? 1 : (int)Math.floor((ymax-ymin)/dd);
      xmax = xmin + ni*dd;
      ymax = ymin + nj*dd;
      double[][] cc = new double[ni][nj];
      double[][] vv = new double[ni][nj];
      condev.area = false;
      condev.convar = new double[][][] { cc, vv };
      condev.is = is;
      condev.key = 256*rot + is;
      condev.nx = ni;
      condev.ny = nj;
      condev.dd = dd;
      condev.xmin = xmin;
      condev.xmax = xmax;
      condev.ymin = ymin;
      condev.ymax = ymax;
      //
      for (Source src: sources) {
        ConDev cd0 = src.lower[is];
        double w0 = src.w_lower[is];
        ConDev cd1 = src.upper[is];
        double w1 = src.w_upper[is];
        double sx = cos*src.x - sin*src.y; 
        double sy = cos*src.y + sin*src.x;
        for (int i=0; i<ni; i++) {
          double x = xmin + (i+0.5)*dd;   // cell center point
          for (int j=0; j<nj; j++) {
            double y = ymin + (j+0.5)*dd;
            int i0 = (int)Math.floor((x - cd0.xmin - sx)/cd0.dd);
            int j0 = (int)Math.floor((y - cd0.ymin - sy)/cd0.dd);
            double c0 = 0;
            double v0 = 0;
            if (i0>=0 && j0>=0 && i0<cd0.nx && j0<cd0.ny) {
              c0 = cd0.convar[0][i0][j0];
              v0 = cd0.convar[1][i0][j0];
            }
            int i1 = (int)Math.floor((x - cd1.xmin - sx)/cd1.dd);
            int j1 = (int)Math.floor((y - cd1.ymin - sy)/cd1.dd);
            double c1= 0;
            double v1 = 0;
            if (i1>=0 && j1>=0 && i1<cd1.nx && j1<cd1.ny) {
              c1 = cd1.convar[0][i1][j1];
              v1 = cd1.convar[1][i1][j1];
            }
            double c = w0*c0 + w1*c1;
            double v = w0*v0 + w1*v1;
            cc[i][j] += src.q*c;
            vv[i][j] += src.q*src.q*v;
          } // for j
        } // for i
      } // for src
      //
      if (cmin > 0) { // reduction to the area with c >= cmin
        int i0 = 0;
        int i1 = ni-1;
        int j0 = 0;
        int j1 = nj-1;
        int i;
        int j;
        for (i=i0; i<=i1; i++) {
          for (j=j0; j<=j1; j++)
            if (cc[i][j] >= cmin)
              break;
          if (j <= j1)
            break;
        }
        i0 = i;
        for (i=i1; i>=i0; i--) {
          for (j=j0; j<=j1; j++)
            if (cc[i][j] >= cmin)
              break;
          if (j <= j1)
            break;
        }
        i1 = i;
        for (j=j0; j<=j1; j++) {
          for (i=i0; i<=i1; i++)
            if (cc[i][j] >= cmin)
              break;
          if (i <= i1)
            break;
        }
        j0 = j;
        for (j=j1; j>=j0; j--) {
          for (i=i0; i<=i1; i++)
            if (cc[i][j] >= cmin)
              break;
          if (i <= i1)
            break;
        }
        j1 = j;
        if (i1 >= i0 && j1 >= j0) {
          xmin += i0*dd;
          ni = i1 - i0 + 1;
          xmax = xmin + ni*dd;
          ymin += j0*dd;
          nj = j1 - j0 + 1;
          ymax = ymin + nj*dd;
          double[][] cc_red = new double[ni][nj];
          double[][] vv_red = new double[ni][nj];
          for (i=0; i<ni; i++) {
            for (j=0; j<nj; j++) {
              cc_red[i][j] = cc[i+i0][j+j0];
              vv_red[i][j] = vv[i+i0][j+j0];
            }
          }
          condev.convar = new double[][][] { cc_red, vv_red };
          condev.nx = ni;
          condev.ny = nj;
          condev.xmin = xmin;
          condev.xmax = xmax;
          condev.ymin = ymin;
          condev.ymax = ymax;
        }
      } // if cmin
    } 
    catch (Exception e) {
      e.printStackTrace(System.out);
      condev = null;
    }
    return condev;
  }
  
  private void testMap() {
    int is = 18;
    for (int rot=0; rot<360; rot+=2) {
      ConDev cd = map(is, rot, true, 0);
      double cmax = 0;
      for (int i=0; i<cd.nx; i++) {
        for (int j=0; j<cd.ny; j++) {
          double c = cd.convar[0][i][j];
          if (c > cmax)
            cmax = c;
        }
      }
      System.out.printf(LOC, "%3d %4.1f %5.1f : %9.3e%n", rot, kl(is), ua(is), cmax);
    }
  }
  
  public Map<String, Double> getMaxCon(int nthreads) { 
    ArrayList<MapRes> vres = new ArrayList<MapRes>();
    if (nthreads < 1)
      nthreads = 1;
    if (nthreads > 64)
      nthreads = 64;
    ExecutorService executor = Executors.newFixedThreadPool(nthreads);
    for (int is=0; is<NN; is++) {
      for (int rot=0; rot<360; rot+=5) {
         Runnable run = new RunMap(is, rot, this, vres);
         executor.execute(run);
      }
    }
    executor.shutdown();
    try {
      executor.awaitTermination(10, TimeUnit.HOURS);
      if (vres.size() != 2*36*NN)
        throw new Exception("unexpected size of result array!");
    }
    catch (Exception e) {
      e.printStackTrace(System.out);
      return null;
    }
    MapRes max = vres.get(0);
    for (MapRes mr : vres) {
      if (mr.cm > max.cm)
        max = mr;
    }
    Map<String, Double> map = new LinkedHashMap<>();
    map.put("con", max.cm);
    map.put("dev", max.dm);
    map.put("xp", max.xm + center.getX());
    map.put("yp", max.ym + center.getY());
    map.put("ra", max.ram);
    map.put("kl", kl(max.ism));
    map.put("ua", ua(max.ism));
    map.put("is", (double)max.ism);
    map.put("dd", max.dd);
    return map;
  }
  
  
  public Map<String, Double> getMaxCon() {
    int ism = 0;
    double ram = 0;
    double cm = 0;
    double dm = 0;
    double xm = 0;
    double ym = 0;
    double dd = 0;
    for (int is=0; is<NN; is++) {
      for (int rot=0; rot<360; rot+=5) {
        double sin = Math.sin(Math.toRadians(rot));
        double cos = Math.cos(Math.toRadians(rot));
        ConDev cd = map(is, rot, true, 0);
        for (int i=0; i<cd.nx; i++) {
          for (int j=0; j<cd.ny; j++) {
            double c = cd.convar[0][i][j];
            double d = cd.convar[1][i][j];
            if (c > cm) {
              cm = c;
              dm = Math.sqrt(d)/c;
              double x = cd.xmin + (i+0.5)*cd.dd;
              double y = cd.ymin + (j+0.5)*cd.dd;
              xm = cos*x + sin*y;
              ym = cos*y - sin*x;
              ism = is;
              ram = (270 + rot) % 360;
              dd = cd.dd;
            }
          } // for j
        } // for i
      } // for rot
      step(nq);
    } // for is
    Map<String, Double> map = new LinkedHashMap<>();
    map.put("con", cm);
    map.put("dev", dm);
    map.put("xp", xm + center.getX());
    map.put("yp", ym + center.getY());
    map.put("ra", ram);
    map.put("kl", kl(ism));
    map.put("ua", ua(ism));
    map.put("is", (double)ism);
    map.put("dd", dd);
    return map;
  }
  
  private void draw(int is, int ra, double cref, double tick) {
    int rot = (ra - 270) % 360;
    if (rot < 0)
      rot += 360;
    int rotation = rot;
    String title = String.format(LOC, 
            "%s: Kl=%3.1f KM, Ua=%3.1f m/s, Ra=%d Grad, Cref=%9.3e g/m³",
            my_name, kl(is), ua(is), ra, cref);
    new JFXPanel();
    Platform.runLater(() -> {
      double fsz = 2; // mm
      try {
        ConDev cd = map(is, rotation, false, 0.05*cref);
        double xmin = cd.xmin;
        double xmax = cd.xmax;
        double ymin = cd.ymin;
        double ymax = cd.ymax;
        double dd = cd.dd;
        for (Source src: cd.sources) {
          if (src.x < xmin)
            xmin = src.x;
        }
        xmin -= 2*dd;
        double[] xx = new double[cd.nx+1];
        for (int i=0; i<=cd.nx; i++)
          xx[i] = cd.xmin + i*cd.dd;
        double[] yy = new double[cd.ny+1];
        for (int j=0; j<=cd.ny; j++)
          yy[j] = cd.ymin + j*cd.dd;
        Function<Double, Color> g = Draw.Gauge.getVioletToLightBlue(cref*1.2);
        Function<Double, String> t = (Double c) -> {
          return String.format("%1.0f", Math.min(100*c/cref, 999));
        };
        double tck = (tick <= 0) ? 500 : tick;
        Draw d = new Draw();
        Draw.setFont(null, 3.0, null, null);
        d.paper(0, 0, Draw.stroke(0.2), Draw.fill("beige"));
        d.view(0, 0, 2*fsz*(xmax-xmin)/dd, 2*fsz*(ymax-ymin)/dd, null, 
                Draw.fill("white"));
        d.scale(xmin, ymin, xmax, ymax);
        d.title(title);
        d.axis("bottom", tck, 0, "%1.0f", "Koordinate in Lee-Richtung (m)", true);
        d.axis("left", tck, 0, "%1.0f", "Koordinate quer zur Windrichtung (m)", true);
        Draw.Mosaic m = Draw.Mosaic.getInstance(xx, yy, cd.convar[0]);
        m.setGauge(g, t, fsz);
        d.mosaic(m);
        for (Source src: cd.sources) {
          d.text(new Point2D(src.x, src.y), 
                  src.name, 
                  Pos.CENTER, 
                  Draw.fill("black"), 
                  Draw.font("bold", 5));
        }
        String svg = d.show(true, null);
      } 
      catch (Exception e) {
        e.printStackTrace(System.out);
      }
    });
  }
  
  private String setLayers(Source src) {
    src.upper = new ConDev[NN];
    src.lower = new ConDev[NN];
    src.w_upper = new double[NN];
    src.w_lower = new double[NN];
    for (int is=0; is<NN; is++) {
      int ih = src.ih[is];
      double h0 = vhq[ih];
      double h1 = vhq[ih+1];
      double he = src.heff[is];
      if (he < h0 || he > h1)
        throw new IllegalStateException("illegal layer setting, src="+src.name+", he="+he+", h0="+h0+", h1="+h1);
      src.w_lower[is] = (h1 - he)/(h1 - h0);
      src.w_upper[is] = 1 - src.w_lower[is];
      src.lower[is] = getConDev(ih, is);
      src.upper[is] = getConDev(ih+1, is);
      step(1);
    }
    return null;
  }
  
  private ConDev getConDev(int ih, int is) {
    int key = 256*ih + is;
    ConDev cd = condevs.get(key);
    if (cd == null) {
      cd = ConDev.getInstance(z0, ih, is);
      if (cd == null)
        throw new IllegalStateException("ConDev not available");
      condevs.put(key, cd);
    }
    return cd;
  }
  
  public static double ua(int seq0) {
    if (seq0 < 3)
      return ua9[seq0];
    else if (seq0 < 7)
      return ua9[seq0 - 3];
    else 
      return ua9[(seq0-7)%9];
  }
  
  public static double kl(int seq0) {
    if (seq0 < 3)
      return kl6[0];
    else if (seq0 < 7)
      return kl6[1];
    else
      return kl6[2 + (seq0-7)/9];
  }
  
  public static double[] pluris(File wrk, double z0, double d0, double ha, 
          double ua, double kl, double hq, double dq, double vq, double tq,
          double zq, double rq, double lq) {                                 //-2015-11-23
    double[] plr = null;
    try {
      List<String> args = new ArrayList<>();
      args.add(wrk.toString());
      args.add("--z0=" + z0);
      args.add("--d0=" + d0);
      args.add("--ha=" + ha);
      args.add("--ua=" + ua);
      args.add("--ki=" + ((int)kl + ((kl > 3.11) ? 1 : 0)));
      args.add("--hq=" + hq);
      args.add("--dq=" + dq);
      args.add("--uq=" + vq);
      args.add("--tq=" + tq);
      args.add("--zq=" + zq);                                     //-2019-02-28
      args.add("--rq=" + rq/100);                                 //-2015-11-23
      args.add("--lq=" + lq);                                     //-2015-11-23
      args.add("--sc=" + dq);                                     //-2016-04-06
      args.add("--se=" + 50000);                                  //-2016-04-06
      args.add("--sd=" + 0.01);                                   //-2016-04-06
      args.add("--verbose=1");
      args.add("--stop-at-finalrise");
      args.add("--skip-graph");
      if (!chk || !first_pluris_call) {
        args.add("--skip-dmn");
        args.add("--skip-log");
      }
      if (skip_stacktip_downwash)
        args.add("--skip-stacktip-downwash");                     //-2016-04-08
      if (break_factor > 0) {                                     //-2016-04-08
        args.add("--break-factor="+String.format("%1.4f", break_factor));
        args.add("--const-ust");                                  //-2019-02-28
      }
      if (first_pluris_call)
        first_pluris_call = false;
      PipedWriter piw = new PipedWriter();                        
      PrintWriter pw = new PrintWriter(piw);
      IBJpluris.setPrinter(pw);
      Thread thread = new Thread(() -> {
        IBJpluris.main(args.toArray(new String[0]));
      });
      PipedReader pir = new PipedReader(piw);
      BufferedReader br = new BufferedReader(pir);
      thread.start();
      //
      String line = null;
      while (null != (line = br.readLine())) {
        if (line.startsWith("IBJpluris") && line.toLowerCase().contains("version"))
          PLURIS_VERSION = line;
        if (line.startsWith("Vq=")) {
          String[] ss = line.trim().split(";\\s");
          if (ss.length != 4)
            throw new Exception("unexpected PLURIS output string: "+line);
          if (!ss[1].startsWith("Ts="))
            throw new Exception("unexpected element 2 (Ts=) in PLURIS output string: "+line);
          if (!ss[2].startsWith("Dh="))
            throw new Exception("unexpected element 3 (Dh=) in PLURIS output string: "+line);
          plr = new double[3];
          plr[0] = Double.parseDouble(ss[0].substring(3).trim().replace(',', '.'));
          plr[1] = Double.parseDouble(ss[1].substring(3).trim().replace(',', '.'));
          plr[2] = Double.parseDouble(ss[2].substring(3).trim().replace(',', '.'));
        }  
      }
    } 
    catch (Exception e) {
      e.printStackTrace(System.out);
      plr = null;
    }    
    return plr;
  }
  
  public String setHeff(Source src) {
    String msg = null;
    try {
      File wrk = new File(base);
      src.ih = new int[NN];
      src.heff = new double[NN];
      double d0 = 6*z0;
      double ha = 10 + d0;
      for (int is=0; is<NN; is++) {
        double kl = kl(is);
        double ua = ua(is);
        double[] plr = new double[] { 0.0, 0.0, 0.0 };
        if (src.v > 0 && src.d > 0)                                //-2015-11-20 -2018-09-14
          plr = pluris(wrk, z0, d0, ha, ua, kl, src.h, src.d, src.v, src.t, src.z, src.r, src.l); 
        if (plr == null)
          return "PLURIS failed";
        double he = src.h + plr[2];
        if (chk) System.out.printf(LOC,
                "&&& %s: kl=%1.1f ua=%4.1f [%5.1f %4.1f %4.1f %3.0f %6.4f] -> %6.2f%n", 
                src.name, kl, ua, src.h, src.d, src.v, src.t, src.z, he);
        int i;
        for (i=1; i<NH; i++) {
          if (vhq[i] >= he)
            break;
        }
        if (i >= NH)
          throw new IllegalStateException(
                  String.format(LOC, "src=%s, he=%s, i=%d%n", src, he, i));
        src.heff[is] = he;
        src.ih[is] = i-1;
        step(1);
      }
    } 
    catch (Exception e) {
      e.printStackTrace(System.out);
      msg = "*** " + e.toString();
    }
    return msg;
  }
  
  //==========================================================================
  
  public static class Source {
    
    public static final String INVALID = "invalid argument";  
    //
    private String name;
    private double x, y, h;
    private double d, t, v, z, r, l;                   //-2019-02-28 2015-11-23
    private double q;
    //
    private double[] heff;    // effective source heights
    private int[] ih;         // index of next lower layer
    private ConDev[] upper;   // upper layer
    private ConDev[] lower;   // lower layer
    private double[] w_upper; // weight of upper layer
    private double[] w_lower; // weight of lower layer
    
    public Source(String name, double x, double y, double h, double d, 
            double v, double t, double z, double r, double l, double q) {   //-2019-02-28 s2015-11-23
      this.name = name;
      this.x = x;
      this.y = y;
      this.h = h;
      this.d = d;
      this.t = t;
      this.v = v;
      this.z = z;
      this.r = r;                                                 //-2015-11-23
      this.l = l;                                                 //-2015-11-23
      this.q = q;
    }
    
    public Source(String sorg) {
      String s;
      if (sorg.startsWith("--source="))
        s = sorg.substring(9);
      else {
        System.out.println("Source(): invalid call");
        name = INVALID;
        return;
      }
      String[] ss = s.trim().split("[,\\s]+");
      if (ss.length != 9) {                                       //-2019-02-28
        System.out.println("Source(): invalid argument:");
        for (int i=0; i<ss.length; i++)
          System.out.printf("%d: '%s'%n", i, ss[i]);
        name = INVALID;
        return;
      }
      try {
        name = ss[0];
        x = Double.parseDouble(ss[1].trim().replace(',', '.'));
        y = Double.parseDouble(ss[2].trim().replace(',', '.'));
        h = Double.parseDouble(ss[3].trim().replace(',', '.'));
        d = Double.parseDouble(ss[4].trim().replace(',', '.'));
        v = Double.parseDouble(ss[5].trim().replace(',', '.'));
        t = Double.parseDouble(ss[6].trim().replace(',', '.'));
        z = Double.parseDouble(ss[7].trim().replace(',', '.'));                            //-2019-02-28
        r = 0;
        l = 0;
        q = Double.parseDouble(ss[8].trim().replace(',', '.')); 
        q /= 3.6;
        boolean rejected = false;
        if (h < HQMIN || h > HQMAX)
          rejected = true;
        if (d < 0)
          rejected = true;
        if (v < 0 || v > 50)
          rejected = true;
        if (t < 10 || t > 600)
          rejected = true;
        if (z < 0 || z > 2)
          rejected = true;
        if (q < 0)
          rejected = true;
      } 
      catch (Exception e) {
        name = INVALID;
        e.printStackTrace(System.out);
      }
    }
    
    public Source getShifted(double xs, double ys) {
      Source src = new Source(name, xs, ys, h, d, v, t, z, r, l, q);
      src.heff = heff;
      src.ih = ih;
      src.lower = lower;
      src.upper = upper;
      src.w_lower = w_lower;
      src.w_upper = w_upper;
      return src;
    }
    
    @Override
    public String toString() {
      return String.format(LOC, 
              "%4s [%5.0f %5.0f %5.1f] %4.1f %4.1f %3.0f %6.4f %9.3e", 
              name, x, y, h, d, v, t, z, q*3.6);
    }
  }
  
  //=========================================================================

  public class MapRes {
    public int ism;
    public double ram;
    public double cm;
    public double dm;
    public double xm;
    public double ym;
    public double dd;
  }

  public class RunMap implements Runnable {
    private int is;
    private int rot;
    private ArrayList<MapRes> vres;
    private Besmax nmc;

    RunMap(int is, int rot, Besmax nmc, ArrayList<MapRes> vres) {
      this.is = is;
      this.rot = rot;
      this.vres = vres;
      this.nmc = nmc;
    }

    @Override
    public void run() {
      int ism = 0;
      double ram = 0;
      double cm = 0;
      double dm = 0;
      double xm = 0;
      double ym = 0;
      double dd = 0;
      double sin = Math.sin(Math.toRadians(rot));
      double cos = Math.cos(Math.toRadians(rot));
      ConDev cd = nmc.map(is, rot, true, 0);
      for (int i=0; i<cd.nx; i++) {
        for (int j=0; j<cd.ny; j++) {
          double c = cd.convar[0][i][j];
          double d = cd.convar[1][i][j];
          if (c > cm) {
            cm = c;
            dm = Math.sqrt(d)/c;
            double x = cd.xmin + (i+0.5)*cd.dd;
            double y = cd.ymin + (j+0.5)*cd.dd;
            xm = cos*x + sin*y;
            ym = cos*y - sin*x;
            ism = is;
            ram = (270 + rot) % 360;
            dd = cd.dd;
          }
        }
      }
      cd = null;
      MapRes mres = new MapRes();
      mres.ism = ism;
      mres.ram = ram;
      mres.cm = cm;
      mres.dm = dm;
      mres.xm  = xm;
      mres.ym = ym;
      mres.dd = dd;
      synchronized ("NomoMaxCon") {
        vres.add(mres);
      }
      if (rot == 355)
        nmc.step(nmc.nq);
    }
  }

  public static class ConDev {
    private int key;
    private boolean area;
    //
    private int is;
    private int ih;
    private int nx;
    private int ny;
    private double hq;
    private double xmin;
    private double ymin;
    private double xmax;
    private double ymax;
    private double dd;
    private double[][][] convar;
    private Source[] sources;
    
    private ConDev() {}
    
    private static ConDev getInstance(double z0, int ih, int is) {
      if (chk) System.out.printf("&&& ConDev.getInstance(%d, %d)%n", ih, is);
      ConDev cd;
      try {
        double hq = vhq[ih];
        File dir = new File(base, "plumes");
        String dn = String.format("mv%2.0frl%03.0fhq%03.0f", 10*metvers, 100*z0, hq);
        File wrk = new File(dir, dn);
        String fn = String.format("c%04da00.dmna", is+1);
        File f = new File(wrk, fn);
        if (chk) System.out.printf("&&& reading '%s'%n", f);
        if (!f.exists())
          return null;
        IBJarr arr = IBJdmn.readDmn(f.getPath());
        IBJhdr hdr = arr.getHeader();
        Map<String, String> map = hdr.getMap();
        IBJarr.AbstractArray aa = arr.getArray("Con");
        IBJarr.AbstractArray ss = arr.getArray("Dev");
        IBJarr.Structure strct = aa.getStructure();
        int[] fi = strct.getFirstIndex();
        int[] li = strct.getLastIndex();
        int nx = li[0];
        int ny = li[1];
        int nz = li[2];
        int k = 1 - fi[2];
        float dd = hdr.getFloat("dd|delta");
        float xmin = hdr.getFloat("xmin");
        float ymin = hdr.getFloat("ymin");
        float[][][][] data = (float[][][][])aa.getData();
        float[][][][] scat = (float[][][][])ss.getData();
        //
        int i0 = 0;
        int j0 = 0;
        int i1 = nx - 1;
        int j1 = ny - 1;
        for (; i0<=i1; i0++) {
          int j = 0;
          for (; j<=j1; j++)
            if (data[i0][j][k][0] > 0)
              break;
          if (j <= j1)
            break;
        }
        for (; i1>=i0; i1--) {
          int j = 0;
          for (; j<=j1; j++)
            if (data[i1][j][k][0] > 0)
              break;
          if (j <= j1)
            break;
        }
        for (; j0<=j1; j0++) {
          int i = 0;
          for (; i<=i1; i++)
            if (data[i][j0][k][0] > 0)
              break;
          if (i <= i1)
            break;
        }
        for(; j1>=j0; j1--) {
          int i = 0;
          for (; i<=i1; i++)
            if (data[i][j1][k][0] > 0)
              break;
          if (i <= i1)
            break;
        }
        int ni = i1 - i0 + 1;
        int nj = j1 - j0 + 1;
        //
        cd = new ConDev();
        cd.area = true;
        cd.key = ih*256 + is;
        cd.is = is;
        cd.ih = ih;
        cd.nx = ni;
        cd.ny = nj;
        cd.hq = hq;
        cd.xmin = xmin + i0*dd;
        cd.ymin = ymin + j0*dd;
        cd.xmax = cd.xmin + ni*dd;
        cd.ymax = cd.ymin + nj*dd;
        
        if (chk) System.out.printf("&&& ih=%2d, is=%2d: [x=%1.0f..%1.0f, y=%1.0f..%1.0f]"
                + " -> [x=%1.0f..%1.0f, y=%1.0f..%1.0f]%n",
                ih, is, xmin, xmin+nx*dd, ymin, ymin+ny*dd, 
                cd.xmin, cd.xmin+ni*dd, cd.ymin, cd.ymin+nj*dd);
        
        cd.dd = dd;
        cd.convar = new double[2][ni][nj];
        for (int i=0; i<ni; i++) {
          for (int j=0; j<nj; j++) {
            double c = data[i+i0][j+j0][k][0];
            double d = scat[i+i0][j+j0][k][0];
            cd.convar[0][i][j] = c;        // concentration
            cd.convar[1][i][j] = c*c*d*d;  // variance
          }
        }
      } 
      catch (Exception e) {
        e.printStackTrace(System.out);
        cd = null;
      }
      return cd;
    }
  }
  
  //==========================================================================
  
  public static class View {
    
    private Stage stage;
    private Scene scene;
    private BorderPane content;
    private Text title;
    private GridPane input;
    private GridPane result;
    private ScrollPane scroll;
    private StackPane stack;
    private Besmax nomaco;
    private int nthreads;
    //
    int idq = 0;
    //
    private Text heading1;
    private Text heading2;
    private Label[] headers;
    private Label[] names;
    private Node[] units;
    private String[] ids;
    private List<Node[]> fields;
    private ColumnConstraints col_header;
    private ColumnConstraints col_name;
    private ColumnConstraints col_label;
    private ColumnConstraints col_field;
    private TextField[] results;
    private Button btnSave;
    private Button btnNewColumn;
    private ChangeListener<String> listener;
    private ProgressButton btnCalculate;                          //-2018-09-13
    private Label[] result_labels;
    private Label[] result_names;
    private Label[] result_units;
    //
    private Set<String> invalid_fields;
    //
    private Button btnPaint;
    private CheckedTextField ctfRa;
    private CheckedTextField ctfCr;
    private CheckedTextField ctfTc;
    private ChoiceBox<String> chcSituations;
    private Image logo0, logo1;
    
    private void create(ArrayList<Source> vsrc, int nthreads) {
      this.nthreads = nthreads;
      fields = new ArrayList<>();
      invalid_fields = new HashSet<>();
      headers = new Label[] {
        new Label("Bezeichnung der Quelle"),
        new Label("Emissionsmassenstrom"), 
        new Label("x-Koordinate"),
        new Label("y-Koordinate"),
        new Label("Schornsteinbauhöhe"),
        new Label("Innendurchmesser"),
        new Label("Austrittsgeschwindigkeit"),
        new Label("Austrittstemperatur"),
        new Label("Wasserbeladung"),                              //-2019-02-28
      };
      names = new Label[] {
        new Label("nq"),
        new Label("eq"),
        new Label("xq"),
        new Label("yq"),
        new Label("hb"),
        new Label("dq"),
        new Label("vq"),
        new Label("tq"),
        new Label("zq"),                                          //-2019-02-28
      };
      for (Label lbl: names) {
        lbl.getStyleClass().add("mono-text");
      }
      btnNewColumn = new Button("+");
      btnNewColumn.setOnAction((ActionEvent e) -> {
        addColumn(1, 0, 0, 100, 5, 0, 0, 0);                 //-2015-11-23
      });
      units = new Node[] {
        btnNewColumn,
        new Label("kg/h"),                                        //-2016-04-06
        new Label("m"),
        new Label("m"),
        new Label("m"),
        new Label("m"),
        new Label("m/s"),                                         //-2015-11-23
        new Label("°C"), 
        new Label("kg/(kg tr)"),                                       //-2019-02-28 2015-11-23
      };
      listener = (ObservableValue<? extends String> obs, String old, String val) 
              -> {
        Object bean = ((StringProperty)obs).getBean();
        CheckedTextField field = (CheckedTextField)bean;
        checkField(field, old, val);
      };
      invalid_fields = new HashSet<>();
      //
      content = new BorderPane();
      title = new Text("      Maximale bodennahe Konzentration nach Nr. 5.5.2.1 TA Luft (2021)      ");
      title.setFont(Font.font("SansSerif", FontWeight.SEMI_BOLD, 
              FontPosture.REGULAR, 16.0));
      BorderPane.setMargin(title, new Insets(6, 10, 4, 10));
      heading1 = new Text("Liste der Emissionsquellen:");
      heading1.setFont(Font.font("SansSerif", FontWeight.SEMI_BOLD, 
              FontPosture.REGULAR, 14.0));
      //
      VBox center = new VBox(4);
      center.setPadding(new Insets(4, 4, 4, 4));
      //
      input = new GridPane();
      scroll = new ScrollPane(input);
      scroll.setVbarPolicy(ScrollPane.ScrollBarPolicy.NEVER);
      scroll.setMaxHeight(Double.MAX_VALUE);
      scroll.setPadding(new Insets(4, 4, 4, 4));
      input.setHgap(4);
      input.setVgap(2);
      col_header = new ColumnConstraints();
      col_name = new ColumnConstraints();
      col_field = new ColumnConstraints();
      col_label = new ColumnConstraints(
              60, 60, 60, Priority.ALWAYS, HPos.LEFT, true);
      input.getColumnConstraints().addAll(col_header, col_name, col_label);
      //
      input.addColumn(0, headers);
      input.addColumn(1, names);
      input.addColumn(2, units);
      //
      if (vsrc == null || vsrc.isEmpty())
        addColumn(100, 0, 0, 21.9, 1, 10, 40, 0);                   //-2019-02-28 2016-04-08 -2015-11-23
      else {
        for (Source src : vsrc)
          addColumn(src.name, src.q*3.6, src.x, src.y, src.h, src.d, src.v, src.t, src.z);
      }
      //
      btnCalculate = new ProgressButton();                        //-2018-09-13
      btnCalculate.setText("Maximale Konzentration berechnen");
      btnCalculate.setMaxWidth(Double.MAX_VALUE);
      //
      logo0 = new Image(btnCalculate.getClass().getResource("besmax.gif").toExternalForm());
      logo1 = new Image(btnCalculate.getClass().getResource("besmax-active.gif").toExternalForm());
      //
      stack = new StackPane();
      stack.getChildren().addAll(btnCalculate);
      center.getChildren().addAll(scroll, stack);
      VBox.setVgrow(scroll, Priority.SOMETIMES);
      //
      heading2 = new Text("Maximale bodennahe Konzentration:");
      heading2.setFont(Font.font("SansSerif", FontWeight.SEMI_BOLD, 
              FontPosture.REGULAR, 14.0));
      VBox.setMargin(heading2, new Insets(10, 0, 0, 0));
      result = new GridPane();
      result.setHgap(4);
      result.setVgap(2);
      result_labels = new Label[] {
        new Label("Maximaler Konzentrationswert"),
        new Label("Unsicherheit des Maximalwertes"),
        new Label("x-Koordinate des Maximalwertes"),
        new Label("y-Koordinate des Maximalwertes"),
        new Label("Stabilitätsklasse"),
        new Label("Windgeschwindigkeit"),
        new Label("Windrichtung"),
      };
      result.addColumn(0, result_labels);
      result_names = new Label[] {
        new Label("cm"),
        new Label("dm"),
        new Label("xm"),
        new Label("ym"),
        new Label("kl"),
        new Label("ua"),
        new Label("ra"),
      };
      for (Label lbl: result_names) {
        lbl.getStyleClass().add("mono-text");
      }
      result.addColumn(1, result_names);
      results = new TextField[7];
      for (int i=0; i<7; i++) {
        results[i] = new TextField("");
        results[i].setPrefWidth(9*9);
        results[i].setAlignment(Pos.BASELINE_RIGHT);
        results[i].setEditable(false);
      }
      result.addColumn(2, results);
      result_units = new Label[] {      
        new Label("g/m³"),
        new Label("%"),
        new Label("m"),
        new Label("m"),
        new Label("KM"),
        new Label("m/s"),
        new Label("Grad"),
      };
      result.addColumn(3, result_units);
      btnSave = new Button(
              "Emissionsquellen und berechnete Konzentration abspeichern");
      btnSave.setContentDisplay(ContentDisplay.CENTER);
      btnSave.setMaxWidth(Double.MAX_VALUE);
      btnSave.setDisable(true);
      center.getChildren().addAll(heading2, result, btnSave);
      //
      // ------------ paint
      //
      btnPaint = new Button("Grafik");
      btnPaint.setOnAction((ActionEvent e) -> {
        try {
          if (nomaco == null)
            return;
          for (int iq=0; iq<nomaco.nq; iq++) {
            TextField tf = (TextField)fields.get(iq)[0];
            String name = tf.getText().trim();
            nomaco.sources[iq].name = name;
          }
          int is = chcSituations.getSelectionModel().selectedIndexProperty().get();
          int ra = (int)Double.parseDouble(ctfRa.getText().trim().replace(',', '.'));
          double cr = Double.parseDouble(ctfCr.getText().trim().replace(',', '.'));
          double tc = Double.parseDouble(ctfTc.getText().trim().replace(',', '.'));
          nomaco.draw(is, ra, cr, tc);
        } 
        catch (Exception ex) {
          ex.printStackTrace(System.out);
        }
      });
      chcSituations = new ChoiceBox<>();
      for (int i=0; i<NN; i++) {
        chcSituations.getItems().add(String.format(LOC, 
                "%3.1f %4.1f", kl(i), ua(i)));
      }
      Set<String> inv_fields = new HashSet<>();
      ChangeListener<String> lst = (ObservableValue<? extends String> obs, 
              String old, String val) -> {
        Object bean = ((StringProperty)obs).getBean();
        CheckedTextField field = (CheckedTextField)bean;
        String id = field.getId();
        boolean rejected = false;
        boolean invalid = false;
        try {
          double d = Double.parseDouble(field.getText().trim().replace(',', '.'));
          switch (id) {
            case "ra": rejected = (d < -360 || d > 360);
              break;
            case "cr": rejected = (d <= 0);
              break;
            case "tc": rejected = (d <= 0);
              break;
          }
        } 
        catch (Exception e) {
          invalid = true;
        }
        if (invalid) {
          field.rejected.set(false);
          field.invalid.set(true);
          inv_fields.add(id);
        } 
        else if (rejected) {
          field.invalid.set(false);
          field.rejected.set(true);
          inv_fields.add(id);
        }
        else {
          field.invalid.set(false);
          field.rejected.set(false);
          inv_fields.remove(id);
        }
        btnPaint.setDisable(!inv_fields.isEmpty());
      };
      ctfRa = new CheckedTextField("0", "ra", lst);
      ctfRa.setText("");
      ctfRa.setPrefColumnCount(6);
      ctfRa.setAlignment(Pos.BASELINE_RIGHT);
      ctfTc = new CheckedTextField("200", "tc", lst);
      ctfTc.setText("");
      ctfTc.setPrefColumnCount(9);
      ctfTc.setAlignment(Pos.BASELINE_RIGHT);
      ctfCr = new CheckedTextField("-1", "cr", lst);
      ctfCr.setText("");
      ctfCr.setAlignment(Pos.BASELINE_RIGHT);
      HBox bottom = new HBox(4);
      bottom.setAlignment(Pos.BASELINE_LEFT);
      bottom.getChildren().addAll(
              new Label("Kl, Ua:"),
              chcSituations,
              new Label("Ra:"),
              ctfRa,
              new Label("Tick:"),
              ctfTc,
              new Label("Cref:"),
              ctfCr,
              btnPaint);
      center.getChildren().addAll(new Separator(), bottom);
      //
      BorderPane.setAlignment(title, Pos.CENTER);
      content.setTop(title);
      content.setCenter(center);
      scene = new Scene(content);
      stage = new Stage();
      stage.setScene(scene);
    }
    
    private void init(String base_dir) {
      btnCalculate.setOnAction((ActionEvent e) -> {
        run(base_dir);
      });
      btnSave.setOnAction((ActionEvent e) -> {
        saveResult();
      });
      scene.getStylesheets().add("Besmax.css");
      stage.setTitle(my_name + " - Version " + my_version);
      stage.setOnCloseRequest((WindowEvent w) -> {
        System.exit(0);
      });
      stage.getIcons().add(logo0);
      stage.show();
    }
    
    private void addColumn(double q, double x, double y, double h, 
      double d, double v, double t, double z) {
      addColumn(null, q, x, y, h, d, v, t, z);
    }      
    
    private void addColumn(String name, double q, double x, double y, double h, 
      double d, double v, double t, double z) {                                 //-2019-02-28 2015-11-23
      setResult(null, null);
      idq++;
      int ic = idq + 1;
      TextField[] col = new TextField[] {
        new TextField((name != null) ? name : "" + idq),
        new CheckedTextField((""+q).replace('.', ','), "eq-"+idq, listener),
        new CheckedTextField((""+x).replace('.', ','), "xq-"+idq, listener),
        new CheckedTextField((""+y).replace('.', ','), "yq-"+idq, listener),
        new CheckedTextField((""+h).replace('.', ','), "hq-"+idq, listener),
        new CheckedTextField((""+d).replace('.', ','), "dq-"+idq, listener),
        new CheckedTextField((""+v).replace('.', ','), "vq-"+idq, listener),
        new CheckedTextField((""+t).replace('.', ','), "tq-"+idq, listener),
        new CheckedTextField((""+z).replace('.', ','), "zq-"+idq, listener),          //-2019-02-28 2015-11-23
      };
      for (TextField f: col) {
        f.setPrefWidth(9*9);
        f.setAlignment(Pos.BASELINE_RIGHT);
      }
      col[0].setId("nq-" + idq);
      col[0].setOnAction((ActionEvent e) -> {
        TextField tf = (TextField)e.getSource();
        if (tf.getText().trim().length() == 0)
          removeColumn(tf);
      });
      for (Node n: units) {
        input.getChildren().remove(n);
      }
      if (ic < input.getColumnConstraints().size())
        input.getColumnConstraints().remove(ic);
      input.addColumn(ic, col);
      input.addColumn(ic+1, units);
      fields.add(col);
    }
    
    private void removeColumn(TextField tf) {
      if (fields.size() == 1) {
        tf.setText("1");
        return;
      }
      setResult(null, null);
      int ic;
      Node[] col = null;
      for (ic=0; ic<fields.size(); ic++) {
        col = fields.get(ic);
        if (col[0] == tf)
          break;
      }
      fields.remove(col);
      input.getChildren().clear();
      input.addColumn(0, headers);
      input.addColumn(1, names);
      for (int i=0; i<fields.size(); i++) {
        input.addColumn(i+2, fields.get(i));
      }
      input.addColumn(2+fields.size(), units);
    }
    
    private void checkField(CheckedTextField field, String old, String val) {
      setResult(null, null);
      String id = field.getId();
      try {
        boolean rejected = false;
        double v = Double.parseDouble(val.trim().replace(',', '.'));
        switch (id.substring(0, 2)) {
          case "eq": rejected = (v < 0);
            break;
          case "hq": rejected = (v<HQMIN || v>HQMAX);             //-2019-09-30
            break;
          case "dq": rejected = (v<0 || v>200);
            break;
          case "tq": rejected = (v<10 || v>600);
            break;
          case "vq": rejected = (v<0 || v>50);
            break;
          case "rq": rejected = (v<0 || v>100);                   //-2015-11-23
            break;
          case "lq": rejected = (v<0 || v>0.1);                   //-2015-11-23
            break;
          case "zq": rejected = (v<0 || v>2);                   //-2021-10-07
            break; 
        }
        field.setInvalid(false);
        if (rejected) {
          field.setRejected(true);
          field.setInvalid(false);                                //-2021-10-04
          invalid_fields.add(id);
        }
        else {
          field.setRejected(false);
          invalid_fields.remove(id);
        }
      } 
      catch (Exception e) {
        invalid_fields.add(id);
        field.setRejected(false);
        field.setInvalid(true);
      }
      btnCalculate.setDisable(!invalid_fields.isEmpty());
    }
    
    private void run(String base_dir) {
      log("running ...");
      int ir = 0;
      List<Source> list = new ArrayList<>();     
      try {
        for (Node[] col : fields) {
          String sn = ((TextField)col[0]).getText().trim();
          double eq = (1./3.6)*Double.parseDouble(((TextField)col[1])  //-2016-04-06
                  .getText().trim().replace(',', '.'));
          double xq = Double.parseDouble(((TextField)col[2])
                  .getText().trim().replace(',', '.'));
          double yq = Double.parseDouble(((TextField)col[3])
                  .getText().trim().replace(',', '.'));
          double hb = Double.parseDouble(((TextField)col[4])
                  .getText().trim().replace(',', '.'));
          double dq = Double.parseDouble(((TextField)col[5])
                  .getText().trim().replace(',', '.'));
          double vq = Double.parseDouble(((TextField)col[6])
                  .getText().trim().replace(',', '.'));
          double tq = Double.parseDouble(((TextField)col[7])
                  .getText().trim().replace(',', '.'));
          double zq = Double.parseDouble(((TextField)col[8])      //-2019-02-28
                  .getText().trim().replace(',', '.'));
          Source src = new Source(sn, xq, yq, hb, dq, vq, tq, zq, 0, 0, eq);  //-2019-02-28 2015-11-23
          list.add(src);
        }
        //
        log("creating ...");
        Besmax nmc = new Besmax(base_dir, list, ir);
        Work work = new Work(nmc, nthreads);
        btnCalculate.setProgress(0);
        btnCalculate.setDisable(true);
        stage.getIcons().remove(logo0);
        stage.getIcons().add(logo1);
        work.progressProperty().addListener(
                (ObservableValue<? extends Number> obs, Number old, Number val)
                -> {
                  double p = val.doubleValue();
                  btnCalculate.setProgress(p);
                  if (p == 1.0) {
                    btnCalculate.clearProgress();
                    btnCalculate.setDisable(false);
                    stage.getIcons().remove(logo1);
                    stage.getIcons().add(logo0);
                    try {
                      Map<String, Double> map = work.get();
                      setResult(nmc, map);
                    } 
                    catch (Exception ex) {
                      ex.printStackTrace(System.out);
                    }
                  }
                });
        new Thread(work).start();
      } 
      catch (Exception e) {
        e.printStackTrace(System.out);
      }
      
    }
    
    private void setResult(Besmax nmc, Map<String, Double> map) {
      nomaco = nmc;
      if (map == null) {
        if (results == null)
          return;
        for (TextField tf: results)
          tf.clear();
        chcSituations.getSelectionModel().clearSelection();
        ctfRa.setText("");
        ctfTc.setText("");
        ctfCr.setText("");
        btnSave.setDisable(true);
        return;
      }
      results[0].setText(String.format(LOC, "%9.3e", map.get("con")));
      results[1].setText(String.format(LOC, "%9.1f", 100*map.get("dev")));
      results[2].setText(String.format(LOC, "%9.1f", map.get("xp")));
      results[3].setText(String.format(LOC, "%9.1f", map.get("yp")));
      results[4].setText(String.format(LOC, "%9.1f", map.get("kl")));
      results[5].setText(String.format(LOC, "%9.1f", map.get("ua")));
      results[6].setText(String.format(LOC, "%9.1f", map.get("ra")));
      int is = map.get("is").intValue();
      chcSituations.getSelectionModel().select(is);
      ctfRa.setText("" + map.get("ra").intValue());
      ctfTc.setText(String.format(LOC, "%1.3f", 10*map.get("dd")));
      ctfCr.setText(String.format(LOC, "%9.3e", map.get("con")));
      btnSave.setDisable(false);
    }
    
    private int saveResult() {
      List<String> list = new ArrayList<>();
      list.add(String.format("%s %s Version %s", 
              sdf.format(new Date()), my_name, my_version));
      if (PLURIS_VERSION != null)
        list.add(PLURIS_VERSION);
      list.add(title.getText()); 
      list.add("");
      list.add(heading1.getText());
      int nc = 0;
      for (Label lbl: headers) {
        int n = lbl.getText().length();
        if (n > nc)
          nc = n;
      }
      String frm = String.format("%%-%ds", nc);
      StringBuilder sb = new StringBuilder();
      for (int il=0; il<headers.length; il++) {
        sb.setLength(0);
        sb.append(String.format(frm, headers[il].getText()));
        sb.append(" ").append(names[il].getText());
        for (Node[] nn: fields) {
          sb.append(String.format(" %9s", ((TextField)nn[il]).getText()));
        }
        if (il > 0)
          sb.append(" ").append(((Label)units[il]).getText());
        list.add(sb.toString());
      }
      list.add("");
      list.add(heading2.getText());
      nc = 0;
      for (Label lbl: result_labels) {
        int n = lbl.getText().length();
        if (n > nc)
          nc = n;
      }
      frm = String.format("%%-%ds", nc);
      for (int il=0; il<result_labels.length; il++) {
        sb.setLength(0);
        sb.append(String.format(frm, result_labels[il].getText()));
        sb.append(" ").append(result_names[il].getText());
        sb.append(" ").append(results[il].getText());
        sb.append(" ").append(result_units[il].getText());
        list.add(sb.toString());
      }
      File flog = new File(base);
      flog = new File(flog.getParentFile(), "log");
      if (!flog.exists())
        flog.mkdir();
      String[] files = flog.list();
      int nf = 0;
      for (String file: files) {
        String[] ss = file.split("[()]");
        if (ss.length==3 && ss[0].equals("besmax") && ss[2].equals(".log")) {
          try {
            int n = Integer.parseInt(ss[1]);
            if (n > nf)
              nf = n;
          } 
          catch (Exception e) {}
        }
      }
      nf++;
      File f = new File(flog, String.format("besmax(%d).log", nf));
      try {
        Files.write(f.toPath(), list);
        btnSave.setDisable(true);
      } 
      catch (IOException ex) {
        ex.printStackTrace(System.out);
        nf = -1;
      }
      return nf;
    }
    
  } // View

  //==========================================================================
  
  public static class Work extends Task<Map<String, Double>> {
    private final Besmax nmc;
    private final int nthreads;
     
    public Work(Besmax nmc, int nthreads) {
      this.nmc = nmc;
      this.nthreads = nthreads;
    }

    @Override
    protected Map<String, Double> call() throws Exception {
      Map<String, Double> map = null;
      try {
        nmc.progress.addListener((ObservableValue<? extends Number> obs,
                Number old, Number val) -> {
          double p = val.doubleValue();
          updateProgress(p, 1.0);
        });
        log("init ...");
        String msg = nmc.init();
        if (msg != null) {
          System.out.printf("ERROR %s%n", msg);
          updateProgress(1.0, 1.0);
          return null;
        }
        log("mapping ...");
        log("using "+nthreads+" threads ...");
        map = (nthreads > 0) ? nmc.getMaxCon(nthreads) : nmc.getMaxCon();
        log("returning ...");
      } 
      catch (Exception e) {
        e.printStackTrace(System.out);
        updateProgress(1.0, 1.0);
      }
      return map;
    }
    
  } //-------------------------- Work
  
  //==========================================================================
  
  private static String my_home() {                               //-2019-02-28
    Class<Besmax> cls = Besmax.class;
    String nm = cls.getName();
    URL u2 = cls.getResource(cls.getSimpleName() + ".class"); 
    String s2;
    File root = null;
    try {
      URI uri = u2.toURI();
      String sScheme = uri.getScheme();
      s2 = uri.getSchemeSpecificPart();
      if (sScheme.equals("jar")) { 
        if (s2.startsWith("file:")) s2 = s2.substring(5); 
      } 
      else if (!sScheme.equals("file")) { 
        return null;
      }
      int k = s2.indexOf(nm.replace('.', '/'));
      if (k > 0) s2 = s2.substring(0, k-1);
      if (s2.endsWith(".jar!")) { 
        s2 = s2.substring(0, s2.length() - 1);
        root = (new File(s2)).getParentFile();
      }
      else {
        root = new File(s2);
      }
      if (root.getName().equals("classes") 
          && root.getParentFile().getName().equals("build"))
        root = root.getParentFile().getParentFile();
    } 
    catch (Exception e) {
      e.printStackTrace(System.out);
    }
    return (root != null) ? root.getAbsolutePath() : null;
  }
  
  private static void checkPlumeData(String dir) throws Exception {
    FileInputStream fis;
    CRC32 crc32 = new CRC32();
    byte[] arr;
    String sr, sc;
    boolean found;
    String[] ext = { ".dmna", ".dmnb.gz" };
    int icount = 0;
    for (int ih=0; ih<NH; ih++) {
      double hq = vhq[ih];
      String dn = String.format("mv%02.0frl%03.0fhq%03.0f", 10*metvers, 100*vz0[0], hq);    
      for (int in=0; in<NN; in++) {
        for (String sx : ext) { 
          String fn = String.format("%s/c%04da00%s", dn, in+1, sx);           
          File f = new File(dir+"/plumes", fn);
          fis = new FileInputStream(f);
          arr = new byte[(int)f.length()];
          fis.read(arr, 0, arr.length);
          fis.close();
          crc32.reset();
          crc32.update(arr);
          sc = String.format("%x", crc32.getValue());
          found = false;
          for (String s : PLUME_LIST) {
            if (s.contains(fn)) {
              found = true;
              sr = s.substring(s.lastIndexOf("|")+1);
              if (!sc.equalsIgnoreCase(sr))
                throw new Exception("unregistered content in file "+fn);
            }
          }
          if (!found)
            throw new Exception("file "+fn+" not found in internal file list");
          icount++;
        }
      }
    }
  }
  
  private static final String[] PLUME_LIST = {
    "mv53rl050hq006/c0001a00.dmna|79c19492", "mv53rl050hq006/c0001a00.dmnb.gz|2d8fc4de", 
    "mv53rl050hq006/c0002a00.dmna|a078c989", "mv53rl050hq006/c0002a00.dmnb.gz|22b3b4e0", 
    "mv53rl050hq006/c0003a00.dmna|1c5257b8", "mv53rl050hq006/c0003a00.dmnb.gz|91c8e438", 
    "mv53rl050hq006/c0004a00.dmna|cd1be24f", "mv53rl050hq006/c0004a00.dmnb.gz|20809208", 
    "mv53rl050hq006/c0005a00.dmna|588e4d8f", "mv53rl050hq006/c0005a00.dmnb.gz|57fde7c7", 
    "mv53rl050hq006/c0006a00.dmna|6e3fd86c", "mv53rl050hq006/c0006a00.dmnb.gz|557303da", 
    "mv53rl050hq006/c0007a00.dmna|d215465d", "mv53rl050hq006/c0007a00.dmnb.gz|e7830459", 
    "mv53rl050hq006/c0008a00.dmna|17ddb5c3", "mv53rl050hq006/c0008a00.dmnb.gz|229707df", 
    "mv53rl050hq006/c0009a00.dmna|d13679e1", "mv53rl050hq006/c0009a00.dmnb.gz|57939f95", 
    "mv53rl050hq006/c0010a00.dmna|61608a3d", "mv53rl050hq006/c0010a00.dmnb.gz|262f4b9f", 
    "mv53rl050hq006/c0011a00.dmna|57ab4447", "mv53rl050hq006/c0011a00.dmnb.gz|f8277a3", 
    "mv53rl050hq006/c0012a00.dmna|bc8ad7bd", "mv53rl050hq006/c0012a00.dmnb.gz|2d29a2da", 
    "mv53rl050hq006/c0013a00.dmna|d1bd0e3c", "mv53rl050hq006/c0013a00.dmnb.gz|ccbb2953", 
    "mv53rl050hq006/c0014a00.dmna|b1b8f608", "mv53rl050hq006/c0014a00.dmnb.gz|c962f090", 
    "mv53rl050hq006/c0015a00.dmna|80f6d6f0", "mv53rl050hq006/c0015a00.dmnb.gz|d955ad72", 
    "mv53rl050hq006/c0016a00.dmna|6bd7450a", "mv53rl050hq006/c0016a00.dmnb.gz|56552fac", 
    "mv53rl050hq006/c0017a00.dmna|6e09c8b", "mv53rl050hq006/c0017a00.dmnb.gz|3fad54b4", 
    "mv53rl050hq006/c0018a00.dmna|abdcb562", "mv53rl050hq006/c0018a00.dmnb.gz|9126c818", 
    "mv53rl050hq006/c0019a00.dmna|22616768", "mv53rl050hq006/c0019a00.dmnb.gz|4e6ce2c7", 
    "mv53rl050hq006/c0020a00.dmna|60d507b1", "mv53rl050hq006/c0020a00.dmnb.gz|5469895c", 
    "mv53rl050hq006/c0021a00.dmna|f35225c6", "mv53rl050hq006/c0021a00.dmnb.gz|f91b6568", 
    "mv53rl050hq006/c0022a00.dmna|1873b63c", "mv53rl050hq006/c0022a00.dmnb.gz|f42fe480", 
    "mv53rl050hq006/c0023a00.dmna|75446fbd", "mv53rl050hq006/c0023a00.dmnb.gz|1f57916", 
    "mv53rl050hq006/c0024a00.dmna|15419789", "mv53rl050hq006/c0024a00.dmnb.gz|e6bdf9f", 
    "mv53rl050hq006/c0025a00.dmna|240fb771", "mv53rl050hq006/c0025a00.dmnb.gz|4644c824", 
    "mv53rl050hq006/lasat.log|fb8cfd84", "mv53rl050hq006/param.def|5f4562b3", 
    "mv53rl050hq008/c0001a00.dmna|2e7f9b4e", "mv53rl050hq008/c0001a00.dmnb.gz|f54fb467", 
    "mv53rl050hq008/c0002a00.dmna|10dfc56b", "mv53rl050hq008/c0002a00.dmnb.gz|ecf659b", 
    "mv53rl050hq008/c0003a00.dmna|acf55b5a", "mv53rl050hq008/c0003a00.dmnb.gz|5c8fddec", 
    "mv53rl050hq008/c0004a00.dmna|7dbceead", "mv53rl050hq008/c0004a00.dmnb.gz|3a0231bf", 
    "mv53rl050hq008/c0005a00.dmna|e829416d", "mv53rl050hq008/c0005a00.dmnb.gz|d25f0edc", 
    "mv53rl050hq008/c0006a00.dmna|de98d48e", "mv53rl050hq008/c0006a00.dmnb.gz|1e340b78", 
    "mv53rl050hq008/c0007a00.dmna|62b24abf", "mv53rl050hq008/c0007a00.dmnb.gz|82784f5b", 
    "mv53rl050hq008/c0008a00.dmna|a77ab921", "mv53rl050hq008/c0008a00.dmnb.gz|430d923b", 
    "mv53rl050hq008/c0009a00.dmna|61917503", "mv53rl050hq008/c0009a00.dmnb.gz|9acf9359", 
    "mv53rl050hq008/c0010a00.dmna|90b2b01", "mv53rl050hq008/c0010a00.dmnb.gz|81e919f0", 
    "mv53rl050hq008/c0011a00.dmna|507e9fd3", "mv53rl050hq008/c0011a00.dmnb.gz|2a24d1f9", 
    "mv53rl050hq008/c0012a00.dmna|bb5f0c29", "mv53rl050hq008/c0012a00.dmnb.gz|725357b1", 
    "mv53rl050hq008/c0013a00.dmna|d668d5a8", "mv53rl050hq008/c0013a00.dmnb.gz|3abd685f", 
    "mv53rl050hq008/c0014a00.dmna|b66d2d9c", "mv53rl050hq008/c0014a00.dmnb.gz|79a84011", 
    "mv53rl050hq008/c0015a00.dmna|87230d64", "mv53rl050hq008/c0015a00.dmnb.gz|309d1694", 
    "mv53rl050hq008/c0016a00.dmna|6c029e9e", "mv53rl050hq008/c0016a00.dmnb.gz|53700850", 
    "mv53rl050hq008/c0017a00.dmna|135471f", "mv53rl050hq008/c0017a00.dmnb.gz|9d1a1dc0", 
    "mv53rl050hq008/c0018a00.dmna|ac096ef6", "mv53rl050hq008/c0018a00.dmnb.gz|6456300c", 
    "mv53rl050hq008/c0019a00.dmna|25b4bcfc", "mv53rl050hq008/c0019a00.dmnb.gz|42a39477", 
    "mv53rl050hq008/c0020a00.dmna|6700dc25", "mv53rl050hq008/c0020a00.dmnb.gz|62ebeea9", 
    "mv53rl050hq008/c0021a00.dmna|f487fe52", "mv53rl050hq008/c0021a00.dmnb.gz|a7af83cd", 
    "mv53rl050hq008/c0022a00.dmna|1fa66da8", "mv53rl050hq008/c0022a00.dmnb.gz|23992d6d", 
    "mv53rl050hq008/c0023a00.dmna|7291b429", "mv53rl050hq008/c0023a00.dmnb.gz|f0c2619e", 
    "mv53rl050hq008/c0024a00.dmna|12944c1d", "mv53rl050hq008/c0024a00.dmnb.gz|dae4da8e", 
    "mv53rl050hq008/c0025a00.dmna|23da6ce5", "mv53rl050hq008/c0025a00.dmnb.gz|b878cd40", 
    "mv53rl050hq008/lasat.log|a7f46b2", "mv53rl050hq008/param.def|aceac3", 
    "mv53rl050hq010/c0001a00.dmna|7f0e90b9", "mv53rl050hq010/c0001a00.dmnb.gz|ae35d136", 
    "mv53rl050hq010/c0002a00.dmna|1db0fc8f", "mv53rl050hq010/c0002a00.dmnb.gz|a0b06179", 
    "mv53rl050hq010/c0003a00.dmna|4cd2d62b", "mv53rl050hq010/c0003a00.dmnb.gz|7e3aa5e5", 
    "mv53rl050hq010/c0004a00.dmna|6fdaf821", "mv53rl050hq010/c0004a00.dmnb.gz|e78a7c89", 
    "mv53rl050hq010/c0005a00.dmna|f42baf3e", "mv53rl050hq010/c0005a00.dmnb.gz|db85c185", 
    "mv53rl050hq010/c0006a00.dmna|cd1ead69", "mv53rl050hq010/c0006a00.dmnb.gz|b8fef953", 
    "mv53rl050hq010/c0007a00.dmna|9c7c87cd", "mv53rl050hq010/c0007a00.dmnb.gz|6aa2019a", 
    "mv53rl050hq010/c0008a00.dmna|8b0ef17d", "mv53rl050hq010/c0008a00.dmnb.gz|dd512ed9", 
    "mv53rl050hq010/c0009a00.dmna|5ea85b55", "mv53rl050hq010/c0009a00.dmnb.gz|1703be2c", 
    "mv53rl050hq010/c0010a00.dmna|b1be2998", "mv53rl050hq010/c0010a00.dmnb.gz|76d830dc", 
    "mv53rl050hq010/c0011a00.dmna|c4cbd150", "mv53rl050hq010/c0011a00.dmnb.gz|ca654746", 
    "mv53rl050hq010/c0012a00.dmna|9948ebc1", "mv53rl050hq010/c0012a00.dmnb.gz|4c1047dd", 
    "mv53rl050hq010/c0013a00.dmna|39a6fae", "mv53rl050hq010/c0013a00.dmnb.gz|7b23a379", 
    "mv53rl050hq010/c0014a00.dmna|224e9ee3", "mv53rl050hq010/c0014a00.dmnb.gz|b01e9cbe", 
    "mv53rl050hq010/c0015a00.dmna|9119aaed", "mv53rl050hq010/c0015a00.dmnb.gz|91ab7ff1", 
    "mv53rl050hq010/c0016a00.dmna|cc9a907c", "mv53rl050hq010/c0016a00.dmnb.gz|bf012ee9", 
    "mv53rl050hq010/c0017a00.dmna|56481413", "mv53rl050hq010/c0017a00.dmnb.gz|fc3d8b58", 
    "mv53rl050hq010/c0018a00.dmna|8f3372e6", "mv53rl050hq010/c0018a00.dmnb.gz|bd844d4a", 
    "mv53rl050hq010/c0019a00.dmna|6f6f262a", "mv53rl050hq010/c0019a00.dmnb.gz|e5ea1d91", 
    "mv53rl050hq010/c0020a00.dmna|9022f83a", "mv53rl050hq010/c0020a00.dmnb.gz|d605ea66", 
    "mv53rl050hq010/c0021a00.dmna|5ed09b87", "mv53rl050hq010/c0021a00.dmnb.gz|d1b5502c", 
    "mv53rl050hq010/c0022a00.dmna|353a116", "mv53rl050hq010/c0022a00.dmnb.gz|4402e3d1", 
    "mv53rl050hq010/c0023a00.dmna|99812579", "mv53rl050hq010/c0023a00.dmnb.gz|28c2cff2", 
    "mv53rl050hq010/c0024a00.dmna|b855d434", "mv53rl050hq010/c0024a00.dmnb.gz|2a7c8ed2", 
    "mv53rl050hq010/c0025a00.dmna|b02e03a", "mv53rl050hq010/c0025a00.dmnb.gz|928a4444", 
    "mv53rl050hq010/lasat.log|d96b7189", "mv53rl050hq010/param.def|cc95ae5b", 
    "mv53rl050hq012/c0001a00.dmna|30d5c6ae", "mv53rl050hq012/c0001a00.dmnb.gz|e5d13ae3", 
    "mv53rl050hq012/c0002a00.dmna|899c7160", "mv53rl050hq012/c0002a00.dmnb.gz|992c724", 
    "mv53rl050hq012/c0003a00.dmna|dc8707ac", "mv53rl050hq012/c0003a00.dmnb.gz|23d34f7c", 
    "mv53rl050hq012/c0004a00.dmna|77bf454c", "mv53rl050hq012/c0004a00.dmnb.gz|133049be", 
    "mv53rl050hq012/c0005a00.dmna|a29832c2", "mv53rl050hq012/c0005a00.dmnb.gz|63f5d50", 
    "mv53rl050hq012/c0006a00.dmna|dd89a8d4", "mv53rl050hq012/c0006a00.dmnb.gz|a13215cf", 
    "mv53rl050hq012/c0007a00.dmna|8892de18", "mv53rl050hq012/c0007a00.dmnb.gz|b2c63358", 
    "mv53rl050hq012/c0008a00.dmna|50882b55", "mv53rl050hq012/c0008a00.dmnb.gz|6272af9a", 
    "mv53rl050hq012/c0009a00.dmna|5ea6581e", "mv53rl050hq012/c0009a00.dmnb.gz|68c127ef", 
    "mv53rl050hq012/c0010a00.dmna|b11ea0c7", "mv53rl050hq012/c0010a00.dmnb.gz|20a55083", 
    "mv53rl050hq012/c0011a00.dmna|563edc34", "mv53rl050hq012/c0011a00.dmnb.gz|9c184f12", 
    "mv53rl050hq012/c0012a00.dmna|fa722661", "mv53rl050hq012/c0012a00.dmnb.gz|791d5170", 
    "mv53rl050hq012/c0013a00.dmna|445ef976", "mv53rl050hq012/c0013a00.dmnb.gz|9c05682a", 
    "mv53rl050hq012/c0014a00.dmna|799ad48a", "mv53rl050hq012/c0014a00.dmnb.gz|ce5c1adb", 
    "mv53rl050hq012/c0015a00.dmna|72fe96b0", "mv53rl050hq012/c0015a00.dmnb.gz|b04d0088", 
    "mv53rl050hq012/c0016a00.dmna|deb26ce5", "mv53rl050hq012/c0016a00.dmnb.gz|9bb0c942", 
    "mv53rl050hq012/c0017a00.dmna|609eb3f2", "mv53rl050hq012/c0017a00.dmnb.gz|35246300", 
    "mv53rl050hq012/c0018a00.dmna|a53a371d", "mv53rl050hq012/c0018a00.dmnb.gz|ea3342b0", 
    "mv53rl050hq012/c0019a00.dmna|1fbe493c", "mv53rl050hq012/c0019a00.dmnb.gz|7b43f051", 
    "mv53rl050hq012/c0020a00.dmna|e4199e35", "mv53rl050hq012/c0020a00.dmnb.gz|c67b088a", 
    "mv53rl050hq012/c0021a00.dmna|6f263fc1", "mv53rl050hq012/c0021a00.dmnb.gz|beb99cff", 
    "mv53rl050hq012/c0022a00.dmna|c36ac594", "mv53rl050hq012/c0022a00.dmnb.gz|d625dbad", 
    "mv53rl050hq012/c0023a00.dmna|7d461a83", "mv53rl050hq012/c0023a00.dmnb.gz|5188760", 
    "mv53rl050hq012/c0024a00.dmna|4082377f", "mv53rl050hq012/c0024a00.dmnb.gz|acd3bd86", 
    "mv53rl050hq012/c0025a00.dmna|4be67545", "mv53rl050hq012/c0025a00.dmnb.gz|c7e2c548", 
    "mv53rl050hq012/lasat.log|75a627df", "mv53rl050hq012/param.def|254865e1", 
    "mv53rl050hq014/c0001a00.dmna|251b17d5", "mv53rl050hq014/c0001a00.dmnb.gz|7724d47d", 
    "mv53rl050hq014/c0002a00.dmna|c234c7b7", "mv53rl050hq014/c0002a00.dmnb.gz|ac37a6e5", 
    "mv53rl050hq014/c0003a00.dmna|1c1ddee7", "mv53rl050hq014/c0003a00.dmnb.gz|1bef1283", 
    "mv53rl050hq014/c0004a00.dmna|dab1a68f", "mv53rl050hq014/c0004a00.dmnb.gz|67b290d6", 
    "mv53rl050hq014/c0005a00.dmna|b1d022f2", "mv53rl050hq014/c0005a00.dmnb.gz|733c4de1", 
    "mv53rl050hq014/c0006a00.dmna|bd92926e", "mv53rl050hq014/c0006a00.dmnb.gz|75e399a8", 
    "mv53rl050hq014/c0007a00.dmna|63bb8b3e", "mv53rl050hq014/c0007a00.dmnb.gz|8e5b60b0", 
    "mv53rl050hq014/c0008a00.dmna|ebbb64ff", "mv53rl050hq014/c0008a00.dmnb.gz|a115840c", 
    "mv53rl050hq014/c0009a00.dmna|313adc99", "mv53rl050hq014/c0009a00.dmnb.gz|2deaa32c", 
    "mv53rl050hq014/c0010a00.dmna|c63b8543", "mv53rl050hq014/c0010a00.dmnb.gz|83fd3f58", 
    "mv53rl050hq014/c0011a00.dmna|3fc4a284", "mv53rl050hq014/c0011a00.dmnb.gz|16c412ec", 
    "mv53rl050hq014/c0012a00.dmna|a560bcdb", "mv53rl050hq014/c0012a00.dmnb.gz|8b43336b", 
    "mv53rl050hq014/c0013a00.dmna|b68f4924", "mv53rl050hq014/c0013a00.dmnb.gz|9b03ac7e", 
    "mv53rl050hq014/c0014a00.dmna|4b598624", "mv53rl050hq014/c0014a00.dmnb.gz|51b92e1e", 
    "mv53rl050hq014/c0015a00.dmna|f6227385", "mv53rl050hq014/c0015a00.dmnb.gz|b3bf80aa", 
    "mv53rl050hq014/c0016a00.dmna|6c866dda", "mv53rl050hq014/c0016a00.dmnb.gz|f25992db", 
    "mv53rl050hq014/c0017a00.dmna|7f699825", "mv53rl050hq014/c0017a00.dmnb.gz|b3179da1", 
    "mv53rl050hq014/c0018a00.dmna|4c5af59b", "mv53rl050hq014/c0018a00.dmnb.gz|6ac5b02", 
    "mv53rl050hq014/c0019a00.dmna|777806c7", "mv53rl050hq014/c0019a00.dmnb.gz|4cc83798", 
    "mv53rl050hq014/c0020a00.dmna|be0d015e", "mv53rl050hq014/c0020a00.dmnb.gz|e9f93ef8", 
    "mv53rl050hq014/c0021a00.dmna|6f0c3da6", "mv53rl050hq014/c0021a00.dmnb.gz|4c79fa1e", 
    "mv53rl050hq014/c0022a00.dmna|f5a823f9", "mv53rl050hq014/c0022a00.dmnb.gz|441d6eca", 
    "mv53rl050hq014/c0023a00.dmna|e647d606", "mv53rl050hq014/c0023a00.dmnb.gz|147690f1", 
    "mv53rl050hq014/c0024a00.dmna|1b911906", "mv53rl050hq014/c0024a00.dmnb.gz|e6d0ef2e", 
    "mv53rl050hq014/c0025a00.dmna|a6eaeca7", "mv53rl050hq014/c0025a00.dmnb.gz|4be11ea1", 
    "mv53rl050hq014/lasat.log|d3ab96f3", "mv53rl050hq014/param.def|9a0aef0d", 
    "mv53rl050hq017/c0001a00.dmna|66b7574b", "mv53rl050hq017/c0001a00.dmnb.gz|459b9cbc", 
    "mv53rl050hq017/c0002a00.dmna|f2315dcb", "mv53rl050hq017/c0002a00.dmnb.gz|11e82404", 
    "mv53rl050hq017/c0003a00.dmna|2c18449b", "mv53rl050hq017/c0003a00.dmnb.gz|e40a5a24", 
    "mv53rl050hq017/c0004a00.dmna|eab43cf3", "mv53rl050hq017/c0004a00.dmnb.gz|d44931c1", 
    "mv53rl050hq017/c0005a00.dmna|81d5b88e", "mv53rl050hq017/c0005a00.dmnb.gz|a1b62c3a", 
    "mv53rl050hq017/c0006a00.dmna|8d970812", "mv53rl050hq017/c0006a00.dmnb.gz|212dce95", 
    "mv53rl050hq017/c0007a00.dmna|53be1142", "mv53rl050hq017/c0007a00.dmnb.gz|93d3f5bd", 
    "mv53rl050hq017/c0008a00.dmna|dbbefe83", "mv53rl050hq017/c0008a00.dmnb.gz|699972e0", 
    "mv53rl050hq017/c0009a00.dmna|13f46e5", "mv53rl050hq017/c0009a00.dmnb.gz|9b68582c", 
    "mv53rl050hq017/c0010a00.dmna|fc70c632", "mv53rl050hq017/c0010a00.dmnb.gz|b82c1537", 
    "mv53rl050hq017/c0011a00.dmna|4a614632", "mv53rl050hq017/c0011a00.dmnb.gz|aa24021b", 
    "mv53rl050hq017/c0012a00.dmna|d0c5586d", "mv53rl050hq017/c0012a00.dmnb.gz|a08705fa", 
    "mv53rl050hq017/c0013a00.dmna|c32aad92", "mv53rl050hq017/c0013a00.dmnb.gz|d591208d", 
    "mv53rl050hq017/c0014a00.dmna|3efc6292", "mv53rl050hq017/c0014a00.dmnb.gz|3d052bb1", 
    "mv53rl050hq017/c0015a00.dmna|83879733", "mv53rl050hq017/c0015a00.dmnb.gz|257ba163", 
    "mv53rl050hq017/c0016a00.dmna|1923896c", "mv53rl050hq017/c0016a00.dmnb.gz|2c78aadb", 
    "mv53rl050hq017/c0017a00.dmna|acc7c93", "mv53rl050hq017/c0017a00.dmnb.gz|73687937", 
    "mv53rl050hq017/c0018a00.dmna|39ff112d", "mv53rl050hq017/c0018a00.dmnb.gz|cb09b4c4", 
    "mv53rl050hq017/c0019a00.dmna|2dde271", "mv53rl050hq017/c0019a00.dmnb.gz|6bfee80c", 
    "mv53rl050hq017/c0020a00.dmna|cba8e5e8", "mv53rl050hq017/c0020a00.dmnb.gz|800e743", 
    "mv53rl050hq017/c0021a00.dmna|1aa9d910", "mv53rl050hq017/c0021a00.dmnb.gz|410c7aa5", 
    "mv53rl050hq017/c0022a00.dmna|800dc74f", "mv53rl050hq017/c0022a00.dmnb.gz|3ca0ab9c", 
    "mv53rl050hq017/c0023a00.dmna|93e232b0", "mv53rl050hq017/c0023a00.dmnb.gz|fbf0288b", 
    "mv53rl050hq017/c0024a00.dmna|6e34fdb0", "mv53rl050hq017/c0024a00.dmnb.gz|ce9d66e1", 
    "mv53rl050hq017/c0025a00.dmna|d34f0811", "mv53rl050hq017/c0025a00.dmnb.gz|cbb2aa40", 
    "mv53rl050hq017/lasat.log|983aa2e3", "mv53rl050hq017/param.def|408e6b67", 
    "mv53rl050hq020/c0001a00.dmna|7ff2c66a", "mv53rl050hq020/c0001a00.dmnb.gz|35019559", 
    "mv53rl050hq020/c0002a00.dmna|903af0e6", "mv53rl050hq020/c0002a00.dmnb.gz|da204c5c", 
    "mv53rl050hq020/c0003a00.dmna|81ea11bf", "mv53rl050hq020/c0003a00.dmnb.gz|8a1dceca", 
    "mv53rl050hq020/c0004a00.dmna|a7d3d7f2", "mv53rl050hq020/c0004a00.dmnb.gz|20f0c6f6", 
    "mv53rl050hq020/c0005a00.dmna|9f8686ca", "mv53rl050hq020/c0005a00.dmnb.gz|89e645d1", 
    "mv53rl050hq020/c0006a00.dmna|84721540", "mv53rl050hq020/c0006a00.dmnb.gz|a8bc8feb", 
    "mv53rl050hq020/c0007a00.dmna|95a2f419", "mv53rl050hq020/c0007a00.dmnb.gz|e168a21c", 
    "mv53rl050hq020/c0008a00.dmna|c80199da", "mv53rl050hq020/c0008a00.dmnb.gz|6cde780e", 
    "mv53rl050hq020/c0009a00.dmna|a35fa820", "mv53rl050hq020/c0009a00.dmnb.gz|53357369", 
    "mv53rl050hq020/c0010a00.dmna|76fb4548", "mv53rl050hq020/c0010a00.dmnb.gz|765f22ba", 
    "mv53rl050hq020/c0011a00.dmna|fdd8472b", "mv53rl050hq020/c0011a00.dmnb.gz|ef810eab", 
    "mv53rl050hq020/c0012a00.dmna|538e7841", "mv53rl050hq020/c0012a00.dmnb.gz|5d8a8c7f", 
    "mv53rl050hq020/c0013a00.dmna|5b3ed2c2", "mv53rl050hq020/c0013a00.dmnb.gz|db20385f", 
    "mv53rl050hq020/c0014a00.dmna|d45300d4", "mv53rl050hq020/c0014a00.dmnb.gz|ec03a19c", 
    "mv53rl050hq020/c0015a00.dmna|6b646ab8", "mv53rl050hq020/c0015a00.dmnb.gz|3a5c02df", 
    "mv53rl050hq020/c0016a00.dmna|c53255d2", "mv53rl050hq020/c0016a00.dmnb.gz|dd35649e", 
    "mv53rl050hq020/c0017a00.dmna|cd82ff51", "mv53rl050hq020/c0017a00.dmnb.gz|706a1d05", 
    "mv53rl050hq020/c0018a00.dmna|98f7bf", "mv53rl050hq020/c0018a00.dmnb.gz|ba3b8567", 
    "mv53rl050hq020/c0019a00.dmna|bd11a4c", "mv53rl050hq020/c0019a00.dmnb.gz|d341ee16", 
    "mv53rl050hq020/c0020a00.dmna|20780883", "mv53rl050hq020/c0020a00.dmnb.gz|dc8673", 
    "mv53rl050hq020/c0021a00.dmna|a5a82666", "mv53rl050hq020/c0021a00.dmnb.gz|61ac2e68", 
    "mv53rl050hq020/c0022a00.dmna|bfe190c", "mv53rl050hq020/c0022a00.dmnb.gz|d32319ad", 
    "mv53rl050hq020/c0023a00.dmna|34eb38f", "mv53rl050hq020/c0023a00.dmnb.gz|83938d8c", 
    "mv53rl050hq020/c0024a00.dmna|8c236199", "mv53rl050hq020/c0024a00.dmnb.gz|8f44b86", 
    "mv53rl050hq020/c0025a00.dmna|33140bf5", "mv53rl050hq020/c0025a00.dmnb.gz|9ab647cc", 
    "mv53rl050hq020/lasat.log|ee034b44", "mv53rl050hq020/param.def|29af1d3f", 
    "mv53rl050hq024/c0001a00.dmna|38e12791", "mv53rl050hq024/c0001a00.dmnb.gz|11cf3ba7", 
    "mv53rl050hq024/c0002a00.dmna|6ad54553", "mv53rl050hq024/c0002a00.dmnb.gz|79c83f85", 
    "mv53rl050hq024/c0003a00.dmna|3bc49ac1", "mv53rl050hq024/c0003a00.dmnb.gz|bf15bdeb", 
    "mv53rl050hq024/c0004a00.dmna|27d880c7", "mv53rl050hq024/c0004a00.dmnb.gz|41d9bca9", 
    "mv53rl050hq024/c0005a00.dmna|a37ddd29", "mv53rl050hq024/c0005a00.dmnb.gz|77fcbb0d", 
    "mv53rl050hq024/c0006a00.dmna|85fb3fe3", "mv53rl050hq024/c0006a00.dmnb.gz|5f14aabe", 
    "mv53rl050hq024/c0007a00.dmna|d4eae071", "mv53rl050hq024/c0007a00.dmnb.gz|31deb2cc", 
    "mv53rl050hq024/c0008a00.dmna|bdc30bef", "mv53rl050hq024/c0008a00.dmnb.gz|ecd82436", 
    "mv53rl050hq024/c0009a00.dmna|497e54b8", "mv53rl050hq024/c0009a00.dmnb.gz|1f7d84fb", 
    "mv53rl050hq024/c0010a00.dmna|9f50acef", "mv53rl050hq024/c0010a00.dmnb.gz|7a977d48", 
    "mv53rl050hq024/c0011a00.dmna|64630109", "mv53rl050hq024/c0011a00.dmnb.gz|75ab73ac", 
    "mv53rl050hq024/c0012a00.dmna|68e56e99", "mv53rl050hq024/c0012a00.dmnb.gz|3c092a1e", 
    "mv53rl050hq024/c0013a00.dmna|a5a3b6e8", "mv53rl050hq024/c0013a00.dmnb.gz|699166de", 
    "mv53rl050hq024/c0014a00.dmna|71e9b1b9", "mv53rl050hq024/c0014a00.dmnb.gz|1582d6e6", 
    "mv53rl050hq024/c0015a00.dmna|3c93688a", "mv53rl050hq024/c0015a00.dmnb.gz|10cd0567", 
    "mv53rl050hq024/c0016a00.dmna|3015071a", "mv53rl050hq024/c0016a00.dmnb.gz|7f12bfc2", 
    "mv53rl050hq024/c0017a00.dmna|fd53df6b", "mv53rl050hq024/c0017a00.dmnb.gz|598775f7", 
    "mv53rl050hq024/c0018a00.dmna|43f00ff9", "mv53rl050hq024/c0018a00.dmnb.gz|9bb34119", 
    "mv53rl050hq024/c0019a00.dmna|d583d20f", "mv53rl050hq024/c0019a00.dmnb.gz|86111265", 
    "mv53rl050hq024/c0020a00.dmna|92f90e88", "mv53rl050hq024/c0020a00.dmnb.gz|6230018d", 
    "mv53rl050hq024/c0021a00.dmna|35f1995c", "mv53rl050hq024/c0021a00.dmnb.gz|9842e553", 
    "mv53rl050hq024/c0022a00.dmna|3977f6cc", "mv53rl050hq024/c0022a00.dmnb.gz|1b25e230", 
    "mv53rl050hq024/c0023a00.dmna|f4312ebd", "mv53rl050hq024/c0023a00.dmnb.gz|2855efd0", 
    "mv53rl050hq024/c0024a00.dmna|207b29ec", "mv53rl050hq024/c0024a00.dmnb.gz|5e0129f7", 
    "mv53rl050hq024/c0025a00.dmna|6d01f0df", "mv53rl050hq024/c0025a00.dmnb.gz|d31edb1c", 
    "mv53rl050hq024/lasat.log|d6311fb7", "mv53rl050hq024/param.def|b504ccb3", 
    "mv53rl050hq028/c0001a00.dmna|cd6ac998", "mv53rl050hq028/c0001a00.dmnb.gz|c4827a2d", 
    "mv53rl050hq028/c0002a00.dmna|71e45ebd", "mv53rl050hq028/c0002a00.dmnb.gz|aa6b9378", 
    "mv53rl050hq028/c0003a00.dmna|afcd47ed", "mv53rl050hq028/c0003a00.dmnb.gz|e8851e21", 
    "mv53rl050hq028/c0004a00.dmna|69613f85", "mv53rl050hq028/c0004a00.dmnb.gz|8b61f853", 
    "mv53rl050hq028/c0005a00.dmna|200bbf8", "mv53rl050hq028/c0005a00.dmnb.gz|c985186d", 
    "mv53rl050hq028/c0006a00.dmna|e420b64", "mv53rl050hq028/c0006a00.dmnb.gz|30d17f28", 
    "mv53rl050hq028/c0007a00.dmna|d06b1234", "mv53rl050hq028/c0007a00.dmnb.gz|a2615a1e", 
    "mv53rl050hq028/c0008a00.dmna|586bfdf5", "mv53rl050hq028/c0008a00.dmnb.gz|a2fadd9f", 
    "mv53rl050hq028/c0009a00.dmna|82ea4593", "mv53rl050hq028/c0009a00.dmnb.gz|af55f8f6", 
    "mv53rl050hq028/c0010a00.dmna|226b4a19", "mv53rl050hq028/c0010a00.dmnb.gz|f62f006d", 
    "mv53rl050hq028/c0011a00.dmna|3a104d37", "mv53rl050hq028/c0011a00.dmnb.gz|72d7997b", 
    "mv53rl050hq028/c0012a00.dmna|a0b45368", "mv53rl050hq028/c0012a00.dmnb.gz|ae05f773", 
    "mv53rl050hq028/c0013a00.dmna|b35ba697", "mv53rl050hq028/c0013a00.dmnb.gz|385e18ec", 
    "mv53rl050hq028/c0014a00.dmna|4e8d6997", "mv53rl050hq028/c0014a00.dmnb.gz|65ff1ae8", 
    "mv53rl050hq028/c0015a00.dmna|f3f69c36", "mv53rl050hq028/c0015a00.dmnb.gz|9c9a8ea4", 
    "mv53rl050hq028/c0016a00.dmna|69528269", "mv53rl050hq028/c0016a00.dmnb.gz|b19960c3", 
    "mv53rl050hq028/c0017a00.dmna|7abd7796", "mv53rl050hq028/c0017a00.dmnb.gz|3c336d34", 
    "mv53rl050hq028/c0018a00.dmna|498e1a28", "mv53rl050hq028/c0018a00.dmnb.gz|cc661ff5", 
    "mv53rl050hq028/c0019a00.dmna|72ace974", "mv53rl050hq028/c0019a00.dmnb.gz|d4fc44a3", 
    "mv53rl050hq028/c0020a00.dmna|bbd9eeed", "mv53rl050hq028/c0020a00.dmnb.gz|375bd78b", 
    "mv53rl050hq028/c0021a00.dmna|6ad8d215", "mv53rl050hq028/c0021a00.dmnb.gz|5b08953f", 
    "mv53rl050hq028/c0022a00.dmna|f07ccc4a", "mv53rl050hq028/c0022a00.dmnb.gz|3c10c672", 
    "mv53rl050hq028/c0023a00.dmna|e39339b5", "mv53rl050hq028/c0023a00.dmnb.gz|9609bcb2", 
    "mv53rl050hq028/c0024a00.dmna|1e45f6b5", "mv53rl050hq028/c0024a00.dmnb.gz|59f046a9", 
    "mv53rl050hq028/c0025a00.dmna|a33e0314", "mv53rl050hq028/c0025a00.dmnb.gz|cd248c2f", 
    "mv53rl050hq028/lasat.log|f34e4aac", "mv53rl050hq028/param.def|76466ce9", 
    "mv53rl050hq034/c0001a00.dmna|f4914154", "mv53rl050hq034/c0001a00.dmnb.gz|6ab0da9d", 
    "mv53rl050hq034/c0002a00.dmna|d1345ffd", "mv53rl050hq034/c0002a00.dmnb.gz|dac76dcb", 
    "mv53rl050hq034/c0003a00.dmna|f1d46ad", "mv53rl050hq034/c0003a00.dmnb.gz|354cdd07", 
    "mv53rl050hq034/c0004a00.dmna|c9b13ec5", "mv53rl050hq034/c0004a00.dmnb.gz|e37b9bd2", 
    "mv53rl050hq034/c0005a00.dmna|a2d0bab8", "mv53rl050hq034/c0005a00.dmnb.gz|a9b5662b", 
    "mv53rl050hq034/c0006a00.dmna|ae920a24", "mv53rl050hq034/c0006a00.dmnb.gz|8746bfae", 
    "mv53rl050hq034/c0007a00.dmna|70bb1374", "mv53rl050hq034/c0007a00.dmnb.gz|3bfe6bb2", 
    "mv53rl050hq034/c0008a00.dmna|f8bbfcb5", "mv53rl050hq034/c0008a00.dmnb.gz|9d6c82af", 
    "mv53rl050hq034/c0009a00.dmna|223a44d3", "mv53rl050hq034/c0009a00.dmnb.gz|9556612b", 
    "mv53rl050hq034/c0010a00.dmna|81e93eb1", "mv53rl050hq034/c0010a00.dmnb.gz|f121ed90", 
    "mv53rl050hq034/c0011a00.dmna|7e6337ae", "mv53rl050hq034/c0011a00.dmnb.gz|603842a1", 
    "mv53rl050hq034/c0012a00.dmna|e4c729f1", "mv53rl050hq034/c0012a00.dmnb.gz|8f16e9ea", 
    "mv53rl050hq034/c0013a00.dmna|f728dc0e", "mv53rl050hq034/c0013a00.dmnb.gz|e66c1db7", 
    "mv53rl050hq034/c0014a00.dmna|afe130e", "mv53rl050hq034/c0014a00.dmnb.gz|384ffb73", 
    "mv53rl050hq034/c0015a00.dmna|b785e6af", "mv53rl050hq034/c0015a00.dmnb.gz|5bcd0dbd", 
    "mv53rl050hq034/c0016a00.dmna|2d21f8f0", "mv53rl050hq034/c0016a00.dmnb.gz|10a5f2f", 
    "mv53rl050hq034/c0017a00.dmna|3ece0d0f", "mv53rl050hq034/c0017a00.dmnb.gz|81266697", 
    "mv53rl050hq034/c0018a00.dmna|dfd60b1", "mv53rl050hq034/c0018a00.dmnb.gz|8941d76b", 
    "mv53rl050hq034/c0019a00.dmna|36df93ed", "mv53rl050hq034/c0019a00.dmnb.gz|c82c44a8", 
    "mv53rl050hq034/c0020a00.dmna|ffaa9474", "mv53rl050hq034/c0020a00.dmnb.gz|4c563b40", 
    "mv53rl050hq034/c0021a00.dmna|2eaba88c", "mv53rl050hq034/c0021a00.dmnb.gz|9def5cc7", 
    "mv53rl050hq034/c0022a00.dmna|b40fb6d3", "mv53rl050hq034/c0022a00.dmnb.gz|cd2abee8", 
    "mv53rl050hq034/c0023a00.dmna|a7e0432c", "mv53rl050hq034/c0023a00.dmnb.gz|aeda86f2", 
    "mv53rl050hq034/c0024a00.dmna|5a368c2c", "mv53rl050hq034/c0024a00.dmnb.gz|7b700a0c", 
    "mv53rl050hq034/c0025a00.dmna|e74d798d", "mv53rl050hq034/c0025a00.dmnb.gz|3e779ac2", 
    "mv53rl050hq034/lasat.log|c814e79d", "mv53rl050hq034/param.def|3e32d49f", 
    "mv53rl050hq040/c0001a00.dmna|33a2f52e", "mv53rl050hq040/c0001a00.dmnb.gz|4181b0c8", 
    "mv53rl050hq040/c0002a00.dmna|99cb13ca", "mv53rl050hq040/c0002a00.dmnb.gz|39fda759", 
    "mv53rl050hq040/c0003a00.dmna|881bf293", "mv53rl050hq040/c0003a00.dmnb.gz|6d479fa9", 
    "mv53rl050hq040/c0004a00.dmna|ae2234de", "mv53rl050hq040/c0004a00.dmnb.gz|1e6f0b8c", 
    "mv53rl050hq040/c0005a00.dmna|967765e6", "mv53rl050hq040/c0005a00.dmnb.gz|facaca70", 
    "mv53rl050hq040/c0006a00.dmna|8d83f66c", "mv53rl050hq040/c0006a00.dmnb.gz|7adb1305", 
    "mv53rl050hq040/c0007a00.dmna|9c531735", "mv53rl050hq040/c0007a00.dmnb.gz|348e6661", 
    "mv53rl050hq040/c0008a00.dmna|c1f07af6", "mv53rl050hq040/c0008a00.dmnb.gz|250d3230", 
    "mv53rl050hq040/c0009a00.dmna|aaae4b0c", "mv53rl050hq040/c0009a00.dmnb.gz|cc537323", 
    "mv53rl050hq040/c0010a00.dmna|54fdd0be", "mv53rl050hq040/c0010a00.dmnb.gz|72c3808", 
    "mv53rl050hq040/c0011a00.dmna|a9127221", "mv53rl050hq040/c0011a00.dmnb.gz|f9c0f47f", 
    "mv53rl050hq040/c0012a00.dmna|7444d4b", "mv53rl050hq040/c0012a00.dmnb.gz|a9ad2299", 
    "mv53rl050hq040/c0013a00.dmna|ff4e7c8", "mv53rl050hq040/c0013a00.dmnb.gz|18adfb24", 
    "mv53rl050hq040/c0014a00.dmna|809935de", "mv53rl050hq040/c0014a00.dmnb.gz|e96241bb", 
    "mv53rl050hq040/c0015a00.dmna|3fae5fb2", "mv53rl050hq040/c0015a00.dmnb.gz|42e6e22c", 
    "mv53rl050hq040/c0016a00.dmna|91f860d8", "mv53rl050hq040/c0016a00.dmnb.gz|649fc132", 
    "mv53rl050hq040/c0017a00.dmna|9948ca5b", "mv53rl050hq040/c0017a00.dmnb.gz|1dd3e6e2", 
    "mv53rl050hq040/c0018a00.dmna|5452c2b5", "mv53rl050hq040/c0018a00.dmnb.gz|fd6cb774", 
    "mv53rl050hq040/c0019a00.dmna|5f1b2f46", "mv53rl050hq040/c0019a00.dmnb.gz|35288b63", 
    "mv53rl050hq040/c0020a00.dmna|74b23d89", "mv53rl050hq040/c0020a00.dmnb.gz|afd0ffd0", 
    "mv53rl050hq040/c0021a00.dmna|f162136c", "mv53rl050hq040/c0021a00.dmnb.gz|a671f667", 
    "mv53rl050hq040/c0022a00.dmna|5f342c06", "mv53rl050hq040/c0022a00.dmnb.gz|14cd7bad", 
    "mv53rl050hq040/c0023a00.dmna|57848685", "mv53rl050hq040/c0023a00.dmnb.gz|38b87724", 
    "mv53rl050hq040/c0024a00.dmna|d8e95493", "mv53rl050hq040/c0024a00.dmnb.gz|a703415e", 
    "mv53rl050hq040/c0025a00.dmna|67de3eff", "mv53rl050hq040/c0025a00.dmnb.gz|30a7c7d4", 
    "mv53rl050hq040/lasat.log|18d275f1", "mv53rl050hq040/param.def|d7a12eee", 
    "mv53rl050hq048/c0001a00.dmna|e90b770", "mv53rl050hq048/c0001a00.dmnb.gz|286f70b9", 
    "mv53rl050hq048/c0002a00.dmna|2ef07549", "mv53rl050hq048/c0002a00.dmnb.gz|3b652b4c", 
    "mv53rl050hq048/c0003a00.dmna|f0d96c19", "mv53rl050hq048/c0003a00.dmnb.gz|68d893d5", 
    "mv53rl050hq048/c0004a00.dmna|36751471", "mv53rl050hq048/c0004a00.dmnb.gz|d8626be9", 
    "mv53rl050hq048/c0005a00.dmna|5d14900c", "mv53rl050hq048/c0005a00.dmnb.gz|b8f887c8", 
    "mv53rl050hq048/c0006a00.dmna|51562090", "mv53rl050hq048/c0006a00.dmnb.gz|cb538931", 
    "mv53rl050hq048/c0007a00.dmna|8f7f39c0", "mv53rl050hq048/c0007a00.dmnb.gz|7e2fa491", 
    "mv53rl050hq048/c0008a00.dmna|77fd601", "mv53rl050hq048/c0008a00.dmnb.gz|c9d07654", 
    "mv53rl050hq048/c0009a00.dmna|ddfe6e67", "mv53rl050hq048/c0009a00.dmnb.gz|51022d86", 
    "mv53rl050hq048/c0010a00.dmna|80090525", "mv53rl050hq048/c0010a00.dmnb.gz|7ea74221", 
    "mv53rl050hq048/c0011a00.dmna|92ed7163", "mv53rl050hq048/c0011a00.dmnb.gz|b0ce7dfd", 
    "mv53rl050hq048/c0012a00.dmna|8496f3c", "mv53rl050hq048/c0012a00.dmnb.gz|12bc2527", 
    "mv53rl050hq048/c0013a00.dmna|1ba69ac3", "mv53rl050hq048/c0013a00.dmnb.gz|9e4022ae", 
    "mv53rl050hq048/c0014a00.dmna|e67055c3", "mv53rl050hq048/c0014a00.dmnb.gz|81c29402", 
    "mv53rl050hq048/c0015a00.dmna|5b0ba062", "mv53rl050hq048/c0015a00.dmnb.gz|38a188a6", 
    "mv53rl050hq048/c0016a00.dmna|c1afbe3d", "mv53rl050hq048/c0016a00.dmnb.gz|aa3fc68f", 
    "mv53rl050hq048/c0017a00.dmna|d2404bc2", "mv53rl050hq048/c0017a00.dmnb.gz|486c5052", 
    "mv53rl050hq048/c0018a00.dmna|e173267c", "mv53rl050hq048/c0018a00.dmnb.gz|27a6ff0f", 
    "mv53rl050hq048/c0019a00.dmna|da51d520", "mv53rl050hq048/c0019a00.dmnb.gz|dd5b440", 
    "mv53rl050hq048/c0020a00.dmna|1324d2b9", "mv53rl050hq048/c0020a00.dmnb.gz|5e50ce30", 
    "mv53rl050hq048/c0021a00.dmna|c225ee41", "mv53rl050hq048/c0021a00.dmnb.gz|bfe968a1", 
    "mv53rl050hq048/c0022a00.dmna|5881f01e", "mv53rl050hq048/c0022a00.dmnb.gz|1efa07b8", 
    "mv53rl050hq048/c0023a00.dmna|4b6e05e1", "mv53rl050hq048/c0023a00.dmnb.gz|69a030f2", 
    "mv53rl050hq048/c0024a00.dmna|b6b8cae1", "mv53rl050hq048/c0024a00.dmnb.gz|30c89b28", 
    "mv53rl050hq048/c0025a00.dmna|bc33f40", "mv53rl050hq048/c0025a00.dmnb.gz|3958e90f", 
    "mv53rl050hq048/lasat.log|40fcbdef", "mv53rl050hq048/param.def|715116e4", 
    "mv53rl050hq057/c0001a00.dmna|644451be", "mv53rl050hq057/c0001a00.dmnb.gz|d45c468b", 
    "mv53rl050hq057/c0002a00.dmna|aea6c0e9", "mv53rl050hq057/c0002a00.dmnb.gz|91f3aaf8", 
    "mv53rl050hq057/c0003a00.dmna|708fd9b9", "mv53rl050hq057/c0003a00.dmnb.gz|c1d7d169", 
    "mv53rl050hq057/c0004a00.dmna|b623a1d1", "mv53rl050hq057/c0004a00.dmnb.gz|ac8960e8", 
    "mv53rl050hq057/c0005a00.dmna|dd4225ac", "mv53rl050hq057/c0005a00.dmnb.gz|4572cddf", 
    "mv53rl050hq057/c0006a00.dmna|d1009530", "mv53rl050hq057/c0006a00.dmnb.gz|c4b0f27e", 
    "mv53rl050hq057/c0007a00.dmna|f298c60", "mv53rl050hq057/c0007a00.dmnb.gz|12e058a4", 
    "mv53rl050hq057/c0008a00.dmna|872963a1", "mv53rl050hq057/c0008a00.dmnb.gz|ca8c326d", 
    "mv53rl050hq057/c0009a00.dmna|5da8dbc7", "mv53rl050hq057/c0009a00.dmnb.gz|b655116b", 
    "mv53rl050hq057/c0010a00.dmna|9431c60", "mv53rl050hq057/c0010a00.dmnb.gz|c9b85da5", 
    "mv53rl050hq057/c0011a00.dmna|b3b8c1d0", "mv53rl050hq057/c0011a00.dmnb.gz|56664d26", 
    "mv53rl050hq057/c0012a00.dmna|291cdf8f", "mv53rl050hq057/c0012a00.dmnb.gz|f60053b9", 
    "mv53rl050hq057/c0013a00.dmna|3af32a70", "mv53rl050hq057/c0013a00.dmnb.gz|147669be", 
    "mv53rl050hq057/c0014a00.dmna|c725e570", "mv53rl050hq057/c0014a00.dmnb.gz|1542b19d", 
    "mv53rl050hq057/c0015a00.dmna|7a5e10d1", "mv53rl050hq057/c0015a00.dmnb.gz|b970aedd", 
    "mv53rl050hq057/c0016a00.dmna|e0fa0e8e", "mv53rl050hq057/c0016a00.dmnb.gz|a345007c", 
    "mv53rl050hq057/c0017a00.dmna|f315fb71", "mv53rl050hq057/c0017a00.dmnb.gz|e97d610f", 
    "mv53rl050hq057/c0018a00.dmna|c02696cf", "mv53rl050hq057/c0018a00.dmnb.gz|d5e3d3b4", 
    "mv53rl050hq057/c0019a00.dmna|fb046593", "mv53rl050hq057/c0019a00.dmnb.gz|7cecbede", 
    "mv53rl050hq057/c0020a00.dmna|3271620a", "mv53rl050hq057/c0020a00.dmnb.gz|b67ce9f9", 
    "mv53rl050hq057/c0021a00.dmna|e3705ef2", "mv53rl050hq057/c0021a00.dmnb.gz|58f10af9", 
    "mv53rl050hq057/c0022a00.dmna|79d440ad", "mv53rl050hq057/c0022a00.dmnb.gz|312cb934", 
    "mv53rl050hq057/c0023a00.dmna|6a3bb552", "mv53rl050hq057/c0023a00.dmnb.gz|c4245da2", 
    "mv53rl050hq057/c0024a00.dmna|97ed7a52", "mv53rl050hq057/c0024a00.dmnb.gz|ef90cc3b", 
    "mv53rl050hq057/c0025a00.dmna|2a968ff3", "mv53rl050hq057/c0025a00.dmnb.gz|eed16872", 
    "mv53rl050hq057/lasat.log|87ae21e9", "mv53rl050hq057/param.def|e0c63274", 
    "mv53rl050hq067/c0001a00.dmna|1b8afd12", "mv53rl050hq067/c0001a00.dmnb.gz|43df8e02", 
    "mv53rl050hq067/c0002a00.dmna|9f1f46cb", "mv53rl050hq067/c0002a00.dmnb.gz|1a8bd35d", 
    "mv53rl050hq067/c0003a00.dmna|41365f9b", "mv53rl050hq067/c0003a00.dmnb.gz|6f1e987", 
    "mv53rl050hq067/c0004a00.dmna|879a27f3", "mv53rl050hq067/c0004a00.dmnb.gz|355868d9", 
    "mv53rl050hq067/c0005a00.dmna|ecfba38e", "mv53rl050hq067/c0005a00.dmnb.gz|7bfac958", 
    "mv53rl050hq067/c0006a00.dmna|e0b91312", "mv53rl050hq067/c0006a00.dmnb.gz|53c13d19", 
    "mv53rl050hq067/c0007a00.dmna|3e900a42", "mv53rl050hq067/c0007a00.dmnb.gz|c834c63d", 
    "mv53rl050hq067/c0008a00.dmna|b690e583", "mv53rl050hq067/c0008a00.dmnb.gz|d2e5928e", 
    "mv53rl050hq067/c0009a00.dmna|6c115de5", "mv53rl050hq067/c0009a00.dmnb.gz|cd3ab44b", 
    "mv53rl050hq067/c0010a00.dmna|4641a826", "mv53rl050hq067/c0010a00.dmnb.gz|31fa6536", 
    "mv53rl050hq067/c0011a00.dmna|f9f5cc22", "mv53rl050hq067/c0011a00.dmnb.gz|b11eff8", 
    "mv53rl050hq067/c0012a00.dmna|6351d27d", "mv53rl050hq067/c0012a00.dmnb.gz|a5729188", 
    "mv53rl050hq067/c0013a00.dmna|70be2782", "mv53rl050hq067/c0013a00.dmnb.gz|f9da2ef1", 
    "mv53rl050hq067/c0014a00.dmna|8d68e882", "mv53rl050hq067/c0014a00.dmnb.gz|a8264bde", 
    "mv53rl050hq067/c0015a00.dmna|30131d23", "mv53rl050hq067/c0015a00.dmnb.gz|5a531f13", 
    "mv53rl050hq067/c0016a00.dmna|aab7037c", "mv53rl050hq067/c0016a00.dmnb.gz|ad370a72", 
    "mv53rl050hq067/c0017a00.dmna|b958f683", "mv53rl050hq067/c0017a00.dmnb.gz|65d4cd39", 
    "mv53rl050hq067/c0018a00.dmna|8a6b9b3d", "mv53rl050hq067/c0018a00.dmnb.gz|9dc2aeb7", 
    "mv53rl050hq067/c0019a00.dmna|b1496861", "mv53rl050hq067/c0019a00.dmnb.gz|190ae0c7", 
    "mv53rl050hq067/c0020a00.dmna|783c6ff8", "mv53rl050hq067/c0020a00.dmnb.gz|b47857ea", 
    "mv53rl050hq067/c0021a00.dmna|a93d5300", "mv53rl050hq067/c0021a00.dmnb.gz|2b8f006f", 
    "mv53rl050hq067/c0022a00.dmna|33994d5f", "mv53rl050hq067/c0022a00.dmnb.gz|c20847a8", 
    "mv53rl050hq067/c0023a00.dmna|2076b8a0", "mv53rl050hq067/c0023a00.dmnb.gz|921076aa", 
    "mv53rl050hq067/c0024a00.dmna|dda077a0", "mv53rl050hq067/c0024a00.dmnb.gz|2026ca4d", 
    "mv53rl050hq067/c0025a00.dmna|60db8201", "mv53rl050hq067/c0025a00.dmnb.gz|b28fcd6", 
    "mv53rl050hq067/lasat.log|6b9e0792", "mv53rl050hq067/param.def|d9a4ce8d", 
    "mv53rl050hq080/c0001a00.dmna|ab0293a6", "mv53rl050hq080/c0001a00.dmnb.gz|3312ddbf", 
    "mv53rl050hq080/c0002a00.dmna|8a28d592", "mv53rl050hq080/c0002a00.dmnb.gz|8002e417", 
    "mv53rl050hq080/c0003a00.dmna|9bf834cb", "mv53rl050hq080/c0003a00.dmnb.gz|270011e3", 
    "mv53rl050hq080/c0004a00.dmna|bdc1f286", "mv53rl050hq080/c0004a00.dmnb.gz|71078fdd", 
    "mv53rl050hq080/c0005a00.dmna|8594a3be", "mv53rl050hq080/c0005a00.dmnb.gz|7fad64d7", 
    "mv53rl050hq080/c0006a00.dmna|9e603034", "mv53rl050hq080/c0006a00.dmnb.gz|b781ca23", 
    "mv53rl050hq080/c0007a00.dmna|8fb0d16d", "mv53rl050hq080/c0007a00.dmnb.gz|77550017", 
    "mv53rl050hq080/c0008a00.dmna|d213bcae", "mv53rl050hq080/c0008a00.dmnb.gz|7b247f5a", 
    "mv53rl050hq080/c0009a00.dmna|b94d8d54", "mv53rl050hq080/c0009a00.dmnb.gz|fe1ea6e1", 
    "mv53rl050hq080/c0010a00.dmna|10f0fb52", "mv53rl050hq080/c0010a00.dmnb.gz|728587", 
    "mv53rl050hq080/c0011a00.dmna|861835", "mv53rl050hq080/c0011a00.dmnb.gz|8e6d249b", 
    "mv53rl050hq080/c0012a00.dmna|aed0275f", "mv53rl050hq080/c0012a00.dmnb.gz|598f9fb3", 
    "mv53rl050hq080/c0013a00.dmna|a6608ddc", "mv53rl050hq080/c0013a00.dmnb.gz|2ac34cb8", 
    "mv53rl050hq080/c0014a00.dmna|290d5fca", "mv53rl050hq080/c0014a00.dmnb.gz|94617a29", 
    "mv53rl050hq080/c0015a00.dmna|963a35a6", "mv53rl050hq080/c0015a00.dmnb.gz|198eb4e3", 
    "mv53rl050hq080/c0016a00.dmna|386c0acc", "mv53rl050hq080/c0016a00.dmnb.gz|f814cce0", 
    "mv53rl050hq080/c0017a00.dmna|30dca04f", "mv53rl050hq080/c0017a00.dmnb.gz|b35c05a0", 
    "mv53rl050hq080/c0018a00.dmna|fdc6a8a1", "mv53rl050hq080/c0018a00.dmnb.gz|a2b9073c", 
    "mv53rl050hq080/c0019a00.dmna|f68f4552", "mv53rl050hq080/c0019a00.dmnb.gz|8f88bb85", 
    "mv53rl050hq080/c0020a00.dmna|dd26579d", "mv53rl050hq080/c0020a00.dmnb.gz|3f4193a7", 
    "mv53rl050hq080/c0021a00.dmna|58f67978", "mv53rl050hq080/c0021a00.dmnb.gz|afdd1cd5", 
    "mv53rl050hq080/c0022a00.dmna|f6a04612", "mv53rl050hq080/c0022a00.dmnb.gz|da5b0485", 
    "mv53rl050hq080/c0023a00.dmna|fe10ec91", "mv53rl050hq080/c0023a00.dmnb.gz|c29bbcb5", 
    "mv53rl050hq080/c0024a00.dmna|717d3e87", "mv53rl050hq080/c0024a00.dmnb.gz|fbe370dd", 
    "mv53rl050hq080/c0025a00.dmna|ce4a54eb", "mv53rl050hq080/c0025a00.dmnb.gz|3140a0c7", 
    "mv53rl050hq080/lasat.log|ca4ddd56", "mv53rl050hq080/param.def|f0cc4f0d", 
    "mv53rl050hq095/c0001a00.dmna|3288771a", "mv53rl050hq095/c0001a00.dmnb.gz|e699953a", 
    "mv53rl050hq095/c0002a00.dmna|47f62688", "mv53rl050hq095/c0002a00.dmnb.gz|9254fe38", 
    "mv53rl050hq095/c0003a00.dmna|99df3fd8", "mv53rl050hq095/c0003a00.dmnb.gz|edda3096", 
    "mv53rl050hq095/c0004a00.dmna|5f7347b0", "mv53rl050hq095/c0004a00.dmnb.gz|5da1b444", 
    "mv53rl050hq095/c0005a00.dmna|3412c3cd", "mv53rl050hq095/c0005a00.dmnb.gz|536da281", 
    "mv53rl050hq095/c0006a00.dmna|38507351", "mv53rl050hq095/c0006a00.dmnb.gz|e0d104de", 
    "mv53rl050hq095/c0007a00.dmna|e6796a01", "mv53rl050hq095/c0007a00.dmnb.gz|1a0b5ba0", 
    "mv53rl050hq095/c0008a00.dmna|6e7985c0", "mv53rl050hq095/c0008a00.dmnb.gz|8f5da10d", 
    "mv53rl050hq095/c0009a00.dmna|b4f83da6", "mv53rl050hq095/c0009a00.dmnb.gz|742fad6f", 
    "mv53rl050hq095/c0010a00.dmna|cdfaa426", "mv53rl050hq095/c0010a00.dmnb.gz|45d06e43", 
    "mv53rl050hq095/c0011a00.dmna|a1a45803", "mv53rl050hq095/c0011a00.dmnb.gz|be46ddff", 
    "mv53rl050hq095/c0012a00.dmna|3b00465c", "mv53rl050hq095/c0012a00.dmnb.gz|931078dc", 
    "mv53rl050hq095/c0013a00.dmna|28efb3a3", "mv53rl050hq095/c0013a00.dmnb.gz|ad8f267a", 
    "mv53rl050hq095/c0014a00.dmna|d5397ca3", "mv53rl050hq095/c0014a00.dmnb.gz|6c7a1fb0", 
    "mv53rl050hq095/c0015a00.dmna|68428902", "mv53rl050hq095/c0015a00.dmnb.gz|cab5e190", 
    "mv53rl050hq095/c0016a00.dmna|f2e6975d", "mv53rl050hq095/c0016a00.dmnb.gz|fdcfe6ec", 
    "mv53rl050hq095/c0017a00.dmna|e10962a2", "mv53rl050hq095/c0017a00.dmnb.gz|49fd1db9", 
    "mv53rl050hq095/c0018a00.dmna|d23a0f1c", "mv53rl050hq095/c0018a00.dmnb.gz|bce0f9a5", 
    "mv53rl050hq095/c0019a00.dmna|e918fc40", "mv53rl050hq095/c0019a00.dmnb.gz|e8774b15", 
    "mv53rl050hq095/c0020a00.dmna|206dfbd9", "mv53rl050hq095/c0020a00.dmnb.gz|fc308a26", 
    "mv53rl050hq095/c0021a00.dmna|f16cc721", "mv53rl050hq095/c0021a00.dmnb.gz|c6c7e368", 
    "mv53rl050hq095/c0022a00.dmna|6bc8d97e", "mv53rl050hq095/c0022a00.dmnb.gz|5db637ee", 
    "mv53rl050hq095/c0023a00.dmna|78272c81", "mv53rl050hq095/c0023a00.dmnb.gz|17e932b1", 
    "mv53rl050hq095/c0024a00.dmna|85f1e381", "mv53rl050hq095/c0024a00.dmnb.gz|f242f5e0", 
    "mv53rl050hq095/c0025a00.dmna|388a1620", "mv53rl050hq095/c0025a00.dmnb.gz|367319d", 
    "mv53rl050hq095/lasat.log|184a9ec", "mv53rl050hq095/param.def|fdc3c9fc", 
    "mv53rl050hq113/c0001a00.dmna|cd4a3183", "mv53rl050hq113/c0001a00.dmnb.gz|74324891", 
    "mv53rl050hq113/c0002a00.dmna|1169acbb", "mv53rl050hq113/c0002a00.dmnb.gz|8fa87294", 
    "mv53rl050hq113/c0003a00.dmna|83ee3956", "mv53rl050hq113/c0003a00.dmnb.gz|897fa50", 
    "mv53rl050hq113/c0004a00.dmna|234fe36c", "mv53rl050hq113/c0004a00.dmnb.gz|5ca8235c", 
    "mv53rl050hq113/c0005a00.dmna|299a6a3c", "mv53rl050hq113/c0005a00.dmnb.gz|948e0ada", 
    "mv53rl050hq113/c0006a00.dmna|dd31cef7", "mv53rl050hq113/c0006a00.dmnb.gz|a43e9835", 
    "mv53rl050hq113/c0007a00.dmna|4fb65b1a", "mv53rl050hq113/c0007a00.dmnb.gz|f17718bc", 
    "mv53rl050hq113/c0008a00.dmna|47037cc2", "mv53rl050hq113/c0008a00.dmnb.gz|2e1d0c8c", 
    "mv53rl050hq113/c0009a00.dmna|a603caa9", "mv53rl050hq113/c0009a00.dmnb.gz|bee6aea4", 
    "mv53rl050hq113/c0010a00.dmna|ff77edb8", "mv53rl050hq113/c0010a00.dmnb.gz|8404ebfe", 
    "mv53rl050hq113/c0011a00.dmna|7e028850", "mv53rl050hq113/c0011a00.dmnb.gz|9b22ad78", 
    "mv53rl050hq113/c0012a00.dmna|65af61d1", "mv53rl050hq113/c0012a00.dmnb.gz|f7dd4b17", 
    "mv53rl050hq113/c0013a00.dmna|e6c2c8c9", "mv53rl050hq113/c0013a00.dmnb.gz|3e948950", 
    "mv53rl050hq113/c0014a00.dmna|52f4b2d3", "mv53rl050hq113/c0014a00.dmnb.gz|8893d1fd", 
    "mv53rl050hq113/c0015a00.dmna|94f30f23", "mv53rl050hq113/c0015a00.dmnb.gz|6409b807", 
    "mv53rl050hq113/c0016a00.dmna|8f5ee6a2", "mv53rl050hq113/c0016a00.dmnb.gz|3675b810", 
    "mv53rl050hq113/c0017a00.dmna|c334fba", "mv53rl050hq113/c0017a00.dmnb.gz|cc2c94be", 
    "mv53rl050hq113/c0018a00.dmna|3c4314d7", "mv53rl050hq113/c0018a00.dmnb.gz|f8273167", 
    "mv53rl050hq113/c0019a00.dmna|709080f7", "mv53rl050hq113/c0019a00.dmnb.gz|22b0c90a", 
    "mv53rl050hq113/c0020a00.dmna|9b79f84", "mv53rl050hq113/c0020a00.dmnb.gz|ade57d96", 
    "mv53rl050hq113/c0021a00.dmna|b3ec9620", "mv53rl050hq113/c0021a00.dmnb.gz|57db48f2", 
    "mv53rl050hq113/c0022a00.dmna|a8417fa1", "mv53rl050hq113/c0022a00.dmnb.gz|10bb97ed", 
    "mv53rl050hq113/c0023a00.dmna|2b2cd6b9", "mv53rl050hq113/c0023a00.dmnb.gz|f348d7a0", 
    "mv53rl050hq113/c0024a00.dmna|9f1aaca3", "mv53rl050hq113/c0024a00.dmnb.gz|b5680f3c", 
    "mv53rl050hq113/c0025a00.dmna|591d1153", "mv53rl050hq113/c0025a00.dmnb.gz|561b8b66", 
    "mv53rl050hq113/lasat.log|6245285d", "mv53rl050hq113/param.def|6a39b98d", 
    "mv53rl050hq135/c0001a00.dmna|f42727e5", "mv53rl050hq135/c0001a00.dmnb.gz|8b3df3c5", 
    "mv53rl050hq135/c0002a00.dmna|f920c7e9", "mv53rl050hq135/c0002a00.dmnb.gz|11f7c69d", 
    "mv53rl050hq135/c0003a00.dmna|2709deb9", "mv53rl050hq135/c0003a00.dmnb.gz|d55ec7b8", 
    "mv53rl050hq135/c0004a00.dmna|e1a5a6d1", "mv53rl050hq135/c0004a00.dmnb.gz|38c96100", 
    "mv53rl050hq135/c0005a00.dmna|8ac422ac", "mv53rl050hq135/c0005a00.dmnb.gz|24067ac0", 
    "mv53rl050hq135/c0006a00.dmna|86869230", "mv53rl050hq135/c0006a00.dmnb.gz|bc541f44", 
    "mv53rl050hq135/c0007a00.dmna|58af8b60", "mv53rl050hq135/c0007a00.dmnb.gz|3ef33cd8", 
    "mv53rl050hq135/c0008a00.dmna|d0af64a1", "mv53rl050hq135/c0008a00.dmnb.gz|779a7802", 
    "mv53rl050hq135/c0009a00.dmna|a2edcc7", "mv53rl050hq135/c0009a00.dmnb.gz|249d4885", 
    "mv53rl050hq135/c0010a00.dmna|208819b0", "mv53rl050hq135/c0010a00.dmnb.gz|ac9edfb", 
    "mv53rl050hq135/c0011a00.dmna|177067c9", "mv53rl050hq135/c0011a00.dmnb.gz|238ba3a3", 
    "mv53rl050hq135/c0012a00.dmna|8dd47996", "mv53rl050hq135/c0012a00.dmnb.gz|1183fa59", 
    "mv53rl050hq135/c0013a00.dmna|9e3b8c69", "mv53rl050hq135/c0013a00.dmnb.gz|116edb68", 
    "mv53rl050hq135/c0014a00.dmna|63ed4369", "mv53rl050hq135/c0014a00.dmnb.gz|a3e0c871", 
    "mv53rl050hq135/c0015a00.dmna|de96b6c8", "mv53rl050hq135/c0015a00.dmnb.gz|b2e6cd8d", 
    "mv53rl050hq135/c0016a00.dmna|4432a897", "mv53rl050hq135/c0016a00.dmnb.gz|525cb78b", 
    "mv53rl050hq135/c0017a00.dmna|57dd5d68", "mv53rl050hq135/c0017a00.dmnb.gz|3afc7748", 
    "mv53rl050hq135/c0018a00.dmna|64ee30d6", "mv53rl050hq135/c0018a00.dmnb.gz|93db14e4", 
    "mv53rl050hq135/c0019a00.dmna|5fccc38a", "mv53rl050hq135/c0019a00.dmnb.gz|4bea81a9", 
    "mv53rl050hq135/c0020a00.dmna|96b9c413", "mv53rl050hq135/c0020a00.dmnb.gz|ac7d86bb", 
    "mv53rl050hq135/c0021a00.dmna|47b8f8eb", "mv53rl050hq135/c0021a00.dmnb.gz|49b919ae", 
    "mv53rl050hq135/c0022a00.dmna|dd1ce6b4", "mv53rl050hq135/c0022a00.dmnb.gz|de39ceae", 
    "mv53rl050hq135/c0023a00.dmna|cef3134b", "mv53rl050hq135/c0023a00.dmnb.gz|aa4771eb", 
    "mv53rl050hq135/c0024a00.dmna|3325dc4b", "mv53rl050hq135/c0024a00.dmnb.gz|4c108628", 
    "mv53rl050hq135/c0025a00.dmna|8e5e29ea", "mv53rl050hq135/c0025a00.dmnb.gz|c6ef3c2d", 
    "mv53rl050hq135/lasat.log|fd9de06f", "mv53rl050hq135/param.def|c25461e9", 
    "mv53rl050hq160/c0001a00.dmna|3fb3219b", "mv53rl050hq160/c0001a00.dmnb.gz|777e37ec", 
    "mv53rl050hq160/c0002a00.dmna|aa333f51", "mv53rl050hq160/c0002a00.dmnb.gz|f4f0b31d", 
    "mv53rl050hq160/c0003a00.dmna|b89506e0", "mv53rl050hq160/c0003a00.dmnb.gz|ff862f02", 
    "mv53rl050hq160/c0004a00.dmna|b0de020b", "mv53rl050hq160/c0004a00.dmnb.gz|f37c0c1c", 
    "mv53rl050hq160/c0005a00.dmna|98e4efc4", "mv53rl050hq160/c0005a00.dmnb.gz|a7a1898e", 
    "mv53rl050hq160/c0006a00.dmna|95927169", "mv53rl050hq160/c0006a00.dmnb.gz|54d606b8", 
    "mv53rl050hq160/c0007a00.dmna|873448d8", "mv53rl050hq160/c0007a00.dmnb.gz|61665646", 
    "mv53rl050hq160/c0008a00.dmna|850478bf", "mv53rl050hq160/c0008a00.dmnb.gz|810948e6", 
    "mv53rl050hq160/c0009a00.dmna|d8073d8c", "mv53rl050hq160/c0009a00.dmnb.gz|23873be", 
    "mv53rl050hq160/c0010a00.dmna|a42dded0", "mv53rl050hq160/c0010a00.dmnb.gz|5ed8c7ae", 
    "mv53rl050hq160/c0011a00.dmna|e1f97b8c", "mv53rl050hq160/c0011a00.dmnb.gz|f3403977", 
    "mv53rl050hq160/c0012a00.dmna|4c30a5f5", "mv53rl050hq160/c0012a00.dmnb.gz|9622e3bb", 
    "mv53rl050hq160/c0013a00.dmna|3889c7c5", "mv53rl050hq160/c0013a00.dmnb.gz|ad9c185f", 
    "mv53rl050hq160/c0014a00.dmna|ccd21f46", "mv53rl050hq160/c0014a00.dmnb.gz|959cab82", 
    "mv53rl050hq160/c0015a00.dmna|8869055f", "mv53rl050hq160/c0015a00.dmnb.gz|48d06b35", 
    "mv53rl050hq160/c0016a00.dmna|25a0db26", "mv53rl050hq160/c0016a00.dmnb.gz|2c0b1287", 
    "mv53rl050hq160/c0017a00.dmna|5119b916", "mv53rl050hq160/c0017a00.dmnb.gz|f9944988", 
    "mv53rl050hq160/c0018a00.dmna|16666c61", "mv53rl050hq160/c0018a00.dmnb.gz|67a1c32d", 
    "mv53rl050hq160/c0019a00.dmna|32d9862a", "mv53rl050hq160/c0019a00.dmnb.gz|35f4bb73", 
    "mv53rl050hq160/c0020a00.dmna|3929f019", "mv53rl050hq160/c0020a00.dmnb.gz|e6b9d888", 
    "mv53rl050hq160/c0021a00.dmna|e9cc36c0", "mv53rl050hq160/c0021a00.dmnb.gz|abeb2199", 
    "mv53rl050hq160/c0022a00.dmna|4405e8b9", "mv53rl050hq160/c0022a00.dmnb.gz|3257cc92", 
    "mv53rl050hq160/c0023a00.dmna|30bc8a89", "mv53rl050hq160/c0023a00.dmnb.gz|f558c904", 
    "mv53rl050hq160/c0024a00.dmna|c4e7520a", "mv53rl050hq160/c0024a00.dmnb.gz|c024f53c", 
    "mv53rl050hq160/c0025a00.dmna|805c4813", "mv53rl050hq160/c0025a00.dmnb.gz|492d7708", 
    "mv53rl050hq160/lasat.log|25d8f425", "mv53rl050hq160/param.def|42442fd3", 
    "mv53rl050hq190/c0001a00.dmna|f76babdf", "mv53rl050hq190/c0001a00.dmnb.gz|1468a651", 
    "mv53rl050hq190/c0002a00.dmna|1a96ba9a", "mv53rl050hq190/c0002a00.dmnb.gz|b9930b6f", 
    "mv53rl050hq190/c0003a00.dmna|4f8dcc56", "mv53rl050hq190/c0003a00.dmnb.gz|95f58ad1", 
    "mv53rl050hq190/c0004a00.dmna|e4b58eb6", "mv53rl050hq190/c0004a00.dmnb.gz|b0f8a799", 
    "mv53rl050hq190/c0005a00.dmna|3192f938", "mv53rl050hq190/c0005a00.dmnb.gz|f147acb7", 
    "mv53rl050hq190/c0006a00.dmna|4e83632e", "mv53rl050hq190/c0006a00.dmnb.gz|4f346b21", 
    "mv53rl050hq190/c0007a00.dmna|1b9815e2", "mv53rl050hq190/c0007a00.dmnb.gz|b9390131", 
    "mv53rl050hq190/c0008a00.dmna|c382e0af", "mv53rl050hq190/c0008a00.dmnb.gz|d5fed3af", 
    "mv53rl050hq190/c0009a00.dmna|cdac93e4", "mv53rl050hq190/c0009a00.dmnb.gz|f7d958ae", 
    "mv53rl050hq190/c0010a00.dmna|c5d678d4", "mv53rl050hq190/c0010a00.dmnb.gz|ca69c299", 
    "mv53rl050hq190/c0011a00.dmna|50a8899", "mv53rl050hq190/c0011a00.dmnb.gz|7cda594", 
    "mv53rl050hq190/c0012a00.dmna|a94672cc", "mv53rl050hq190/c0012a00.dmnb.gz|262eecbc", 
    "mv53rl050hq190/c0013a00.dmna|176aaddb", "mv53rl050hq190/c0013a00.dmnb.gz|d60c20a2", 
    "mv53rl050hq190/c0014a00.dmna|2aae8027", "mv53rl050hq190/c0014a00.dmnb.gz|eed54c81", 
    "mv53rl050hq190/c0015a00.dmna|21cac21d", "mv53rl050hq190/c0015a00.dmnb.gz|b7236fde", 
    "mv53rl050hq190/c0016a00.dmna|8d863848", "mv53rl050hq190/c0016a00.dmnb.gz|7d248755", 
    "mv53rl050hq190/c0017a00.dmna|33aae75f", "mv53rl050hq190/c0017a00.dmnb.gz|e576db35", 
    "mv53rl050hq190/c0018a00.dmna|f60e63b0", "mv53rl050hq190/c0018a00.dmnb.gz|97604ad8", 
    "mv53rl050hq190/c0019a00.dmna|4c8a1d91", "mv53rl050hq190/c0019a00.dmnb.gz|17cadb53", 
    "mv53rl050hq190/c0020a00.dmna|b72dca98", "mv53rl050hq190/c0020a00.dmnb.gz|8fced0a0", 
    "mv53rl050hq190/c0021a00.dmna|3c126b6c", "mv53rl050hq190/c0021a00.dmnb.gz|fb5ac72", 
    "mv53rl050hq190/c0022a00.dmna|905e9139", "mv53rl050hq190/c0022a00.dmnb.gz|4048a1a9", 
    "mv53rl050hq190/c0023a00.dmna|2e724e2e", "mv53rl050hq190/c0023a00.dmnb.gz|14abfdc5", 
    "mv53rl050hq190/c0024a00.dmna|13b663d2", "mv53rl050hq190/c0024a00.dmnb.gz|ca0108dd", 
    "mv53rl050hq190/c0025a00.dmna|18d221e8", "mv53rl050hq190/c0025a00.dmnb.gz|c0e46744", 
    "mv53rl050hq190/lasat.log|8fea4434", "mv53rl050hq190/param.def|ef5771d2", 
    "mv53rl050hq226/c0001a00.dmna|e1fdba14", "mv53rl050hq226/c0001a00.dmnb.gz|777ae1a8", 
    "mv53rl050hq226/c0002a00.dmna|2f1557e9", "mv53rl050hq226/c0002a00.dmnb.gz|608fb8ec", 
    "mv53rl050hq226/c0003a00.dmna|3145b45e", "mv53rl050hq226/c0003a00.dmnb.gz|7566fb71", 
    "mv53rl050hq226/c0004a00.dmna|d83a0d71", "mv53rl050hq226/c0004a00.dmnb.gz|e9af9756", 
    "mv53rl050hq226/c0005a00.dmna|9f0c6753", "mv53rl050hq226/c0005a00.dmnb.gz|fd0c27ed", 
    "mv53rl050hq226/c0006a00.dmna|e49bca1f", "mv53rl050hq226/c0006a00.dmnb.gz|f82ec03a", 
    "mv53rl050hq226/c0007a00.dmna|facb29a8", "mv53rl050hq226/c0007a00.dmnb.gz|9b87c2bf", 
    "mv53rl050hq226/c0008a00.dmna|ed15be00", "mv53rl050hq226/c0008a00.dmnb.gz|cbd95c97", 
    "mv53rl050hq226/c0009a00.dmna|18eec708", "mv53rl050hq226/c0009a00.dmnb.gz|d0f9781c", 
    "mv53rl050hq226/c0010a00.dmna|76ca5d96", "mv53rl050hq226/c0010a00.dmnb.gz|e51b851f", 
    "mv53rl050hq226/c0011a00.dmna|f1a6764b", "mv53rl050hq226/c0011a00.dmnb.gz|7ba705db", 
    "mv53rl050hq226/c0012a00.dmna|1a56360", "mv53rl050hq226/c0012a00.dmnb.gz|a4d96bd0", 
    "mv53rl050hq226/c0013a00.dmna|266a6412", "mv53rl050hq226/c0013a00.dmnb.gz|1d9abcf5", 
    "mv53rl050hq226/c0014a00.dmna|3ad24f77", "mv53rl050hq226/c0014a00.dmnb.gz|be852b5", 
    "mv53rl050hq226/c0015a00.dmna|854f54b8", "mv53rl050hq226/c0015a00.dmnb.gz|62575835", 
    "mv53rl050hq226/c0016a00.dmna|754c4193", "mv53rl050hq226/c0016a00.dmnb.gz|1d450db5", 
    "mv53rl050hq226/c0017a00.dmna|528346e1", "mv53rl050hq226/c0017a00.dmnb.gz|87473be0", 
    "mv53rl050hq226/c0018a00.dmna|4c3c1759", "mv53rl050hq226/c0018a00.dmnb.gz|e268145f", 
    "mv53rl050hq226/c0019a00.dmna|187433ad", "mv53rl050hq226/c0019a00.dmnb.gz|7fb00678", 
    "mv53rl050hq226/c0020a00.dmna|6bef5ff2", "mv53rl050hq226/c0020a00.dmnb.gz|69672807", 
    "mv53rl050hq226/c0021a00.dmna|eaf641a8", "mv53rl050hq226/c0021a00.dmnb.gz|c049823e", 
    "mv53rl050hq226/c0022a00.dmna|1af55483", "mv53rl050hq226/c0022a00.dmnb.gz|cc7272a7", 
    "mv53rl050hq226/c0023a00.dmna|3d3a53f1", "mv53rl050hq226/c0023a00.dmnb.gz|36c98cec", 
    "mv53rl050hq226/c0024a00.dmna|21827894", "mv53rl050hq226/c0024a00.dmnb.gz|2086c9f5", 
    "mv53rl050hq226/c0025a00.dmna|9e1f635b", "mv53rl050hq226/c0025a00.dmnb.gz|7664bf49", 
    "mv53rl050hq226/lasat.log|d56f82a0", "mv53rl050hq226/param.def|31c7f6a0", 
    "mv53rl050hq269/c0001a00.dmna|143c5a94", "mv53rl050hq269/c0001a00.dmnb.gz|120eb80c", 
    "mv53rl050hq269/c0002a00.dmna|d43a71ce", "mv53rl050hq269/c0002a00.dmnb.gz|16ad6106", 
    "mv53rl050hq269/c0003a00.dmna|a13689e", "mv53rl050hq269/c0003a00.dmnb.gz|38fbc5f5", 
    "mv53rl050hq269/c0004a00.dmna|ccbf10f6", "mv53rl050hq269/c0004a00.dmnb.gz|8118ed9", 
    "mv53rl050hq269/c0005a00.dmna|a7de948b", "mv53rl050hq269/c0005a00.dmnb.gz|dbbf4978", 
    "mv53rl050hq269/c0006a00.dmna|ab9c2417", "mv53rl050hq269/c0006a00.dmnb.gz|576f15cf", 
    "mv53rl050hq269/c0007a00.dmna|75b53d47", "mv53rl050hq269/c0007a00.dmnb.gz|84522139", 
    "mv53rl050hq269/c0008a00.dmna|fdb5d286", "mv53rl050hq269/c0008a00.dmnb.gz|972d6878", 
    "mv53rl050hq269/c0009a00.dmna|27346ae0", "mv53rl050hq269/c0009a00.dmnb.gz|fde6a991", 
    "mv53rl050hq269/c0010a00.dmna|edc5837c", "mv53rl050hq269/c0010a00.dmnb.gz|f44031d", 
    "mv53rl050hq269/c0011a00.dmna|fd2f96a8", "mv53rl050hq269/c0011a00.dmnb.gz|a889ee72", 
    "mv53rl050hq269/c0012a00.dmna|678b88f7", "mv53rl050hq269/c0012a00.dmnb.gz|20f5d492", 
    "mv53rl050hq269/c0013a00.dmna|74647d08", "mv53rl050hq269/c0013a00.dmnb.gz|3cec443", 
    "mv53rl050hq269/c0014a00.dmna|89b2b208", "mv53rl050hq269/c0014a00.dmnb.gz|914864fe", 
    "mv53rl050hq269/c0015a00.dmna|34c947a9", "mv53rl050hq269/c0015a00.dmnb.gz|2035038d", 
    "mv53rl050hq269/c0016a00.dmna|ae6d59f6", "mv53rl050hq269/c0016a00.dmnb.gz|e43a9001", 
    "mv53rl050hq269/c0017a00.dmna|bd82ac09", "mv53rl050hq269/c0017a00.dmnb.gz|f9581b62", 
    "mv53rl050hq269/c0018a00.dmna|8eb1c1b7", "mv53rl050hq269/c0018a00.dmnb.gz|e7238438", 
    "mv53rl050hq269/c0019a00.dmna|b59332eb", "mv53rl050hq269/c0019a00.dmnb.gz|ab3468ef", 
    "mv53rl050hq269/c0020a00.dmna|7ce63572", "mv53rl050hq269/c0020a00.dmnb.gz|b0fa99d5", 
    "mv53rl050hq269/c0021a00.dmna|ade7098a", "mv53rl050hq269/c0021a00.dmnb.gz|2adaa85f", 
    "mv53rl050hq269/c0022a00.dmna|374317d5", "mv53rl050hq269/c0022a00.dmnb.gz|ac1d4443", 
    "mv53rl050hq269/c0023a00.dmna|24ace22a", "mv53rl050hq269/c0023a00.dmnb.gz|b2fcd49d", 
    "mv53rl050hq269/c0024a00.dmna|d97a2d2a", "mv53rl050hq269/c0024a00.dmnb.gz|d0d50f90", 
    "mv53rl050hq269/c0025a00.dmna|6401d88b", "mv53rl050hq269/c0025a00.dmnb.gz|e24e227", 
    "mv53rl050hq269/lasat.log|38e7b876", "mv53rl050hq269/param.def|5a56cac", 
    "mv53rl050hq320/c0001a00.dmna|6f3511f8", "mv53rl050hq320/c0001a00.dmnb.gz|def643f8", 
    "mv53rl050hq320/c0002a00.dmna|17594211", "mv53rl050hq320/c0002a00.dmnb.gz|c5f7749a", 
    "mv53rl050hq320/c0003a00.dmna|b5824746", "mv53rl050hq320/c0003a00.dmnb.gz|68df517b", 
    "mv53rl050hq320/c0004a00.dmna|f65a133", "mv53rl050hq320/c0004a00.dmnb.gz|ee4786e3", 
    "mv53rl050hq320/c0005a00.dmna|9dbcdc4d", "mv53rl050hq320/c0005a00.dmnb.gz|18fc9663", 
    "mv53rl050hq320/c0006a00.dmna|91a2addc", "mv53rl050hq320/c0006a00.dmnb.gz|da1afdff", 
    "mv53rl050hq320/c0007a00.dmna|3379a88b", "mv53rl050hq320/c0007a00.dmnb.gz|12e6443e", 
    "mv53rl050hq320/c0008a00.dmna|3f1c6777", "mv53rl050hq320/c0008a00.dmnb.gz|6207a6e1", 
    "mv53rl050hq320/c0009a00.dmna|cdc1ea5b", "mv53rl050hq320/c0009a00.dmnb.gz|e9d24428", 
    "mv53rl050hq320/c0010a00.dmna|2bbc78f4", "mv53rl050hq320/c0010a00.dmnb.gz|2f8e9a0b", 
    "mv53rl050hq320/c0011a00.dmna|948cc2ec", "mv53rl050hq320/c0011a00.dmnb.gz|5d408806", 
    "mv53rl050hq320/c0012a00.dmna|b1a5249", "mv53rl050hq320/c0012a00.dmnb.gz|2bd37993", 
    "mv53rl050hq320/c0013a00.dmna|77e01077", "mv53rl050hq320/c0013a00.dmnb.gz|f50e1b1", 
    "mv53rl050hq320/c0014a00.dmna|ef467542", "mv53rl050hq320/c0014a00.dmnb.gz|a319caaa", 
    "mv53rl050hq320/c0015a00.dmna|8924619b", "mv53rl050hq320/c0015a00.dmnb.gz|3f145f15", 
    "mv53rl050hq320/c0016a00.dmna|16b2f13e", "mv53rl050hq320/c0016a00.dmnb.gz|1de0843a", 
    "mv53rl050hq320/c0017a00.dmna|6a48b300", "mv53rl050hq320/c0017a00.dmnb.gz|7d09914c", 
    "mv53rl050hq320/c0018a00.dmna|fc8f3d15", "mv53rl050hq320/c0018a00.dmnb.gz|3ab378a5", 
    "mv53rl050hq320/c0019a00.dmna|afdd8402", "mv53rl050hq320/c0019a00.dmnb.gz|b4d6d9be", 
    "mv53rl050hq320/c0020a00.dmna|e2d03ecb", "mv53rl050hq320/c0020a00.dmnb.gz|1af97441", 
    "mv53rl050hq320/c0021a00.dmna|c02df7fd", "mv53rl050hq320/c0021a00.dmnb.gz|b148f4bf", 
    "mv53rl050hq320/c0022a00.dmna|5fbb6758", "mv53rl050hq320/c0022a00.dmnb.gz|54d6ce24", 
    "mv53rl050hq320/c0023a00.dmna|23412566", "mv53rl050hq320/c0023a00.dmnb.gz|c9ead7b6", 
    "mv53rl050hq320/c0024a00.dmna|bbe74053", "mv53rl050hq320/c0024a00.dmnb.gz|28a9d311", 
    "mv53rl050hq320/c0025a00.dmna|dd85548a", "mv53rl050hq320/c0025a00.dmnb.gz|f6983591", 
    "mv53rl050hq320/lasat.log|7b7ff625", "mv53rl050hq320/param.def|eaa86dc7", 
    "mv53rl050hq381/c0001a00.dmna|61a802fa", "mv53rl050hq381/c0001a00.dmnb.gz|e57177d0", 
    "mv53rl050hq381/c0002a00.dmna|969e8e33", "mv53rl050hq381/c0002a00.dmnb.gz|fc0d934f", 
    "mv53rl050hq381/c0003a00.dmna|c78f51a1", "mv53rl050hq381/c0003a00.dmnb.gz|c1fd8553", 
    "mv53rl050hq381/c0004a00.dmna|db934ba7", "mv53rl050hq381/c0004a00.dmnb.gz|21cbdeb3", 
    "mv53rl050hq381/c0005a00.dmna|5f361649", "mv53rl050hq381/c0005a00.dmnb.gz|6e812bdc", 
    "mv53rl050hq381/c0006a00.dmna|79b0f483", "mv53rl050hq381/c0006a00.dmnb.gz|eb68d0b9", 
    "mv53rl050hq381/c0007a00.dmna|28a12b11", "mv53rl050hq381/c0007a00.dmnb.gz|47bec50c", 
    "mv53rl050hq381/c0008a00.dmna|4188c08f", "mv53rl050hq381/c0008a00.dmnb.gz|61d370", 
    "mv53rl050hq381/c0009a00.dmna|b5359fd8", "mv53rl050hq381/c0009a00.dmnb.gz|b92f8fe3", 
    "mv53rl050hq381/c0010a00.dmna|c75dd439", "mv53rl050hq381/c0010a00.dmnb.gz|e7acfc74", 
    "mv53rl050hq381/c0011a00.dmna|e823a2b6", "mv53rl050hq381/c0011a00.dmnb.gz|bdb523b7", 
    "mv53rl050hq381/c0012a00.dmna|e4a5cd26", "mv53rl050hq381/c0012a00.dmnb.gz|64ffe819", 
    "mv53rl050hq381/c0013a00.dmna|29e31557", "mv53rl050hq381/c0013a00.dmnb.gz|18352746", 
    "mv53rl050hq381/c0014a00.dmna|fda91206", "mv53rl050hq381/c0014a00.dmnb.gz|b033c9c0", 
    "mv53rl050hq381/c0015a00.dmna|b0d3cb35", "mv53rl050hq381/c0015a00.dmnb.gz|aa4284ff", 
    "mv53rl050hq381/c0016a00.dmna|bc55a4a5", "mv53rl050hq381/c0016a00.dmnb.gz|73cbfce0", 
    "mv53rl050hq381/c0017a00.dmna|71137cd4", "mv53rl050hq381/c0017a00.dmnb.gz|2a0831c1", 
    "mv53rl050hq381/c0018a00.dmna|cfb0ac46", "mv53rl050hq381/c0018a00.dmnb.gz|cf2ade9b", 
    "mv53rl050hq381/c0019a00.dmna|59c371b0", "mv53rl050hq381/c0019a00.dmnb.gz|cd7af961", 
    "mv53rl050hq381/c0020a00.dmna|1eb9ad37", "mv53rl050hq381/c0020a00.dmnb.gz|43a8ad40", 
    "mv53rl050hq381/c0021a00.dmna|b9b13ae3", "mv53rl050hq381/c0021a00.dmnb.gz|9046500c", 
    "mv53rl050hq381/c0022a00.dmna|b5375573", "mv53rl050hq381/c0022a00.dmnb.gz|a7c7cfde", 
    "mv53rl050hq381/c0023a00.dmna|78718d02", "mv53rl050hq381/c0023a00.dmnb.gz|18e07969", 
    "mv53rl050hq381/c0024a00.dmna|ac3b8a53", "mv53rl050hq381/c0024a00.dmnb.gz|72eef5c5", 
    "mv53rl050hq381/c0025a00.dmna|e1415360", "mv53rl050hq381/c0025a00.dmnb.gz|ad0b7cb4", 
    "mv53rl050hq381/lasat.log|9988968f", "mv53rl050hq381/param.def|4d525fd0", 
    "mv53rl050hq453/c0001a00.dmna|53094103", "mv53rl050hq453/c0001a00.dmnb.gz|706ee8e", 
    "mv53rl050hq453/c0002a00.dmna|808ca4ff", "mv53rl050hq453/c0002a00.dmnb.gz|8ef0a375", 
    "mv53rl050hq453/c0003a00.dmna|d597d233", "mv53rl050hq453/c0003a00.dmnb.gz|dd5af9e4", 
    "mv53rl050hq453/c0004a00.dmna|7eaf90d3", "mv53rl050hq453/c0004a00.dmnb.gz|abee53c9", 
    "mv53rl050hq453/c0005a00.dmna|ab88e75d", "mv53rl050hq453/c0005a00.dmnb.gz|b1ca8723", 
    "mv53rl050hq453/c0006a00.dmna|d4997d4b", "mv53rl050hq453/c0006a00.dmnb.gz|aee76d6e", 
    "mv53rl050hq453/c0007a00.dmna|81820b87", "mv53rl050hq453/c0007a00.dmnb.gz|7aae0894", 
    "mv53rl050hq453/c0008a00.dmna|5998feca", "mv53rl050hq453/c0008a00.dmnb.gz|95e7b85f", 
    "mv53rl050hq453/c0009a00.dmna|57b68d81", "mv53rl050hq453/c0009a00.dmnb.gz|4c422699", 
    "mv53rl050hq453/c0010a00.dmna|d01b7b3d", "mv53rl050hq453/c0010a00.dmnb.gz|a3d83d48", 
    "mv53rl050hq453/c0011a00.dmna|3eb9791b", "mv53rl050hq453/c0011a00.dmnb.gz|51f7e6ad", 
    "mv53rl050hq453/c0012a00.dmna|92f5834e", "mv53rl050hq453/c0012a00.dmnb.gz|cf167f94", 
    "mv53rl050hq453/c0013a00.dmna|2cd95c59", "mv53rl050hq453/c0013a00.dmnb.gz|6fabf7be", 
    "mv53rl050hq453/c0014a00.dmna|111d71a5", "mv53rl050hq453/c0014a00.dmnb.gz|6cd8617f", 
    "mv53rl050hq453/c0015a00.dmna|1a79339f", "mv53rl050hq453/c0015a00.dmnb.gz|9ef8a623", 
    "mv53rl050hq453/c0016a00.dmna|b635c9ca", "mv53rl050hq453/c0016a00.dmnb.gz|94225e2e", 
    "mv53rl050hq453/c0017a00.dmna|81916dd", "mv53rl050hq453/c0017a00.dmnb.gz|1f898e9a", 
    "mv53rl050hq453/c0018a00.dmna|cdbd9232", "mv53rl050hq453/c0018a00.dmnb.gz|9f4cb6ec", 
    "mv53rl050hq453/c0019a00.dmna|7739ec13", "mv53rl050hq453/c0019a00.dmnb.gz|dc7b3afd", 
    "mv53rl050hq453/c0020a00.dmna|8c9e3b1a", "mv53rl050hq453/c0020a00.dmnb.gz|1ecf672", 
    "mv53rl050hq453/c0021a00.dmna|7a19aee", "mv53rl050hq453/c0021a00.dmnb.gz|ca46af75", 
    "mv53rl050hq453/c0022a00.dmna|abed60bb", "mv53rl050hq453/c0022a00.dmnb.gz|edc9ceb2", 
    "mv53rl050hq453/c0023a00.dmna|15c1bfac", "mv53rl050hq453/c0023a00.dmnb.gz|1e8ca13d", 
    "mv53rl050hq453/c0024a00.dmna|28059250", "mv53rl050hq453/c0024a00.dmnb.gz|c9c8a575", 
    "mv53rl050hq453/c0025a00.dmna|2361d06a", "mv53rl050hq453/c0025a00.dmnb.gz|637743f7", 
    "mv53rl050hq453/lasat.log|435b6181", "mv53rl050hq453/param.def|e782e7a2", 
    "mv53rl050hq538/c0001a00.dmna|ab56446", "mv53rl050hq538/c0001a00.dmnb.gz|e620f9a1", 
    "mv53rl050hq538/c0002a00.dmna|24157c31", "mv53rl050hq538/c0002a00.dmnb.gz|fade3ff4", 
    "mv53rl050hq538/c0003a00.dmna|fa3c6561", "mv53rl050hq538/c0003a00.dmnb.gz|e4128945", 
    "mv53rl050hq538/c0004a00.dmna|3c901d09", "mv53rl050hq538/c0004a00.dmnb.gz|783df407", 
    "mv53rl050hq538/c0005a00.dmna|57f19974", "mv53rl050hq538/c0005a00.dmnb.gz|58a6de4e", 
    "mv53rl050hq538/c0006a00.dmna|5bb329e8", "mv53rl050hq538/c0006a00.dmnb.gz|cd17fe7a", 
    "mv53rl050hq538/c0007a00.dmna|859a30b8", "mv53rl050hq538/c0007a00.dmnb.gz|2e92cc27", 
    "mv53rl050hq538/c0008a00.dmna|d9adf79", "mv53rl050hq538/c0008a00.dmnb.gz|9f81a62f", 
    "mv53rl050hq538/c0009a00.dmna|d71b671f", "mv53rl050hq538/c0009a00.dmnb.gz|5bac101a", 
    "mv53rl050hq538/c0010a00.dmna|82105bdc", "mv53rl050hq538/c0010a00.dmnb.gz|e8928b8e", 
    "mv53rl050hq538/c0011a00.dmna|163b85ab", "mv53rl050hq538/c0011a00.dmnb.gz|b2ae66f1", 
    "mv53rl050hq538/c0012a00.dmna|8c9f9bf4", "mv53rl050hq538/c0012a00.dmnb.gz|3058e896", 
    "mv53rl050hq538/c0013a00.dmna|9f706e0b", "mv53rl050hq538/c0013a00.dmnb.gz|62b14378", 
    "mv53rl050hq538/c0014a00.dmna|62a6a10b", "mv53rl050hq538/c0014a00.dmnb.gz|87ac1f97", 
    "mv53rl050hq538/c0015a00.dmna|dfdd54aa", "mv53rl050hq538/c0015a00.dmnb.gz|118469d0", 
    "mv53rl050hq538/c0016a00.dmna|45794af5", "mv53rl050hq538/c0016a00.dmnb.gz|cad01a34", 
    "mv53rl050hq538/c0017a00.dmna|5696bf0a", "mv53rl050hq538/c0017a00.dmnb.gz|1479f3bf", 
    "mv53rl050hq538/c0018a00.dmna|65a5d2b4", "mv53rl050hq538/c0018a00.dmnb.gz|cd2b4807", 
    "mv53rl050hq538/c0019a00.dmna|5e8721e8", "mv53rl050hq538/c0019a00.dmnb.gz|6c1bfc5a", 
    "mv53rl050hq538/c0020a00.dmna|97f22671", "mv53rl050hq538/c0020a00.dmnb.gz|2470a2b5", 
    "mv53rl050hq538/c0021a00.dmna|46f31a89", "mv53rl050hq538/c0021a00.dmnb.gz|ffebfdc7", 
    "mv53rl050hq538/c0022a00.dmna|dc5704d6", "mv53rl050hq538/c0022a00.dmnb.gz|8b3b5cba", 
    "mv53rl050hq538/c0023a00.dmna|cfb8f129", "mv53rl050hq538/c0023a00.dmnb.gz|588d1455", 
    "mv53rl050hq538/c0024a00.dmna|326e3e29", "mv53rl050hq538/c0024a00.dmnb.gz|abebe19d", 
    "mv53rl050hq538/c0025a00.dmna|8f15cb88", "mv53rl050hq538/c0025a00.dmnb.gz|1e201023", 
    "mv53rl050hq538/lasat.log|a7cc1d7b", "mv53rl050hq538/param.def|dc3b5832", 
    "mv53rl050hq640/c0001a00.dmna|bdb7bb15", "mv53rl050hq640/c0001a00.dmnb.gz|d2fa828b", 
    "mv53rl050hq640/c0002a00.dmna|eda79c3e", "mv53rl050hq640/c0002a00.dmnb.gz|d3b2680a", 
    "mv53rl050hq640/c0003a00.dmna|4f7c9969", "mv53rl050hq640/c0003a00.dmnb.gz|3511a9d3", 
    "mv53rl050hq640/c0004a00.dmna|f59b7f1c", "mv53rl050hq640/c0004a00.dmnb.gz|987b57d3", 
    "mv53rl050hq640/c0005a00.dmna|67420262", "mv53rl050hq640/c0005a00.dmnb.gz|2f2d4570", 
    "mv53rl050hq640/c0006a00.dmna|6b5c73f3", "mv53rl050hq640/c0006a00.dmnb.gz|9602d744", 
    "mv53rl050hq640/c0007a00.dmna|c98776a4", "mv53rl050hq640/c0007a00.dmnb.gz|b577ed15", 
    "mv53rl050hq640/c0008a00.dmna|c5e2b958", "mv53rl050hq640/c0008a00.dmnb.gz|80909a13", 
    "mv53rl050hq640/c0009a00.dmna|373f3474", "mv53rl050hq640/c0009a00.dmnb.gz|2b9bd5ee", 
    "mv53rl050hq640/c0010a00.dmna|7fe68e4e", "mv53rl050hq640/c0010a00.dmnb.gz|658b5f0", 
    "mv53rl050hq640/c0011a00.dmna|eecce8c8", "mv53rl050hq640/c0011a00.dmnb.gz|6007f804", 
    "mv53rl050hq640/c0012a00.dmna|715a786d", "mv53rl050hq640/c0012a00.dmnb.gz|d63744a", 
    "mv53rl050hq640/c0013a00.dmna|da03a53", "mv53rl050hq640/c0013a00.dmnb.gz|8c9bc4b7", 
    "mv53rl050hq640/c0014a00.dmna|95065f66", "mv53rl050hq640/c0014a00.dmnb.gz|c7361267", 
    "mv53rl050hq640/c0015a00.dmna|f3644bbf", "mv53rl050hq640/c0015a00.dmnb.gz|50cdba3a", 
    "mv53rl050hq640/c0016a00.dmna|6cf2db1a", "mv53rl050hq640/c0016a00.dmnb.gz|93dbffef", 
    "mv53rl050hq640/c0017a00.dmna|10089924", "mv53rl050hq640/c0017a00.dmnb.gz|a8487b85", 
    "mv53rl050hq640/c0018a00.dmna|86cf1731", "mv53rl050hq640/c0018a00.dmnb.gz|d78ad6aa", 
    "mv53rl050hq640/c0019a00.dmna|d59dae26", "mv53rl050hq640/c0019a00.dmnb.gz|7d18b442", 
    "mv53rl050hq640/c0020a00.dmna|989014ef", "mv53rl050hq640/c0020a00.dmnb.gz|dfb5e7ce", 
    "mv53rl050hq640/c0021a00.dmna|ba6dddd9", "mv53rl050hq640/c0021a00.dmnb.gz|b0f4400f", 
    "mv53rl050hq640/c0022a00.dmna|25fb4d7c", "mv53rl050hq640/c0022a00.dmnb.gz|76ca2205", 
    "mv53rl050hq640/c0023a00.dmna|59010f42", "mv53rl050hq640/c0023a00.dmnb.gz|865d2f2", 
    "mv53rl050hq640/c0024a00.dmna|c1a76a77", "mv53rl050hq640/c0024a00.dmnb.gz|4599d6c7", 
    "mv53rl050hq640/c0025a00.dmna|a7c57eae", "mv53rl050hq640/c0025a00.dmnb.gz|bc5ca42c", 
    "mv53rl050hq640/lasat.log|2aaa9074", "mv53rl050hq640/param.def|a7301a2c", 
    "mv53rl050hq761/c0001a00.dmna|a76fd89d", "mv53rl050hq761/c0001a00.dmnb.gz|5eba23e1", 
    "mv53rl050hq761/c0002a00.dmna|d60e2c61", "mv53rl050hq761/c0002a00.dmnb.gz|ccc7cc28", 
    "mv53rl050hq761/c0003a00.dmna|8273531", "mv53rl050hq761/c0003a00.dmnb.gz|33e9fee7", 
    "mv53rl050hq761/c0004a00.dmna|ce8b4d59", "mv53rl050hq761/c0004a00.dmnb.gz|f74b03be", 
    "mv53rl050hq761/c0005a00.dmna|a5eac924", "mv53rl050hq761/c0005a00.dmnb.gz|5143d566", 
    "mv53rl050hq761/c0006a00.dmna|a9a879b8", "mv53rl050hq761/c0006a00.dmnb.gz|3fe0587", 
    "mv53rl050hq761/c0007a00.dmna|778160e8", "mv53rl050hq761/c0007a00.dmnb.gz|7d04391a", 
    "mv53rl050hq761/c0008a00.dmna|ff818f29", "mv53rl050hq761/c0008a00.dmnb.gz|39d21ee9", 
    "mv53rl050hq761/c0009a00.dmna|2500374f", "mv53rl050hq761/c0009a00.dmnb.gz|4396c9e6", 
    "mv53rl050hq761/c0010a00.dmna|1c0bb9e", "mv53rl050hq761/c0010a00.dmnb.gz|4518e101", 
    "mv53rl050hq761/c0011a00.dmna|26cdf1e9", "mv53rl050hq761/c0011a00.dmnb.gz|5bfe51ff", 
    "mv53rl050hq761/c0012a00.dmna|bc69efb6", "mv53rl050hq761/c0012a00.dmnb.gz|84802b0a", 
    "mv53rl050hq761/c0013a00.dmna|af861a49", "mv53rl050hq761/c0013a00.dmnb.gz|569cedbf", 
    "mv53rl050hq761/c0014a00.dmna|5250d549", "mv53rl050hq761/c0014a00.dmnb.gz|d3e161c7", 
    "mv53rl050hq761/c0015a00.dmna|ef2b20e8", "mv53rl050hq761/c0015a00.dmnb.gz|fc18ff32", 
    "mv53rl050hq761/c0016a00.dmna|758f3eb7", "mv53rl050hq761/c0016a00.dmnb.gz|86965b5d", 
    "mv53rl050hq761/c0017a00.dmna|6660cb48", "mv53rl050hq761/c0017a00.dmnb.gz|d01cf2db", 
    "mv53rl050hq761/c0018a00.dmna|5553a6f6", "mv53rl050hq761/c0018a00.dmnb.gz|a47a20", 
    "mv53rl050hq761/c0019a00.dmna|6e7155aa", "mv53rl050hq761/c0019a00.dmnb.gz|f3007640", 
    "mv53rl050hq761/c0020a00.dmna|a7045233", "mv53rl050hq761/c0020a00.dmnb.gz|4e4ac80", 
    "mv53rl050hq761/c0021a00.dmna|76056ecb", "mv53rl050hq761/c0021a00.dmnb.gz|e2a2d346", 
    "mv53rl050hq761/c0022a00.dmna|eca17094", "mv53rl050hq761/c0022a00.dmnb.gz|d96dbf5d", 
    "mv53rl050hq761/c0023a00.dmna|ff4e856b", "mv53rl050hq761/c0023a00.dmnb.gz|2a3193eb", 
    "mv53rl050hq761/c0024a00.dmna|2984a6b", "mv53rl050hq761/c0024a00.dmnb.gz|b6ffa9da", 
    "mv53rl050hq761/c0025a00.dmna|bfe3bfca", "mv53rl050hq761/c0025a00.dmnb.gz|142d5e24", 
    "mv53rl050hq761/lasat.log|82ebd0fe", "mv53rl050hq761/param.def|73ed045e", 
    "mv53rl050hq905/c0001a00.dmna|bf27a02c", "mv53rl050hq905/c0001a00.dmnb.gz|dd01c71d", 
    "mv53rl050hq905/c0002a00.dmna|cd852d76", "mv53rl050hq905/c0002a00.dmnb.gz|e36c12f7", 
    "mv53rl050hq905/c0003a00.dmna|13ac3426", "mv53rl050hq905/c0003a00.dmnb.gz|3f4ca378", 
    "mv53rl050hq905/c0004a00.dmna|d5004c4e", "mv53rl050hq905/c0004a00.dmnb.gz|3249eae2", 
    "mv53rl050hq905/c0005a00.dmna|be61c833", "mv53rl050hq905/c0005a00.dmnb.gz|3da70de6", 
    "mv53rl050hq905/c0006a00.dmna|b22378af", "mv53rl050hq905/c0006a00.dmnb.gz|d9ddf6a1", 
    "mv53rl050hq905/c0007a00.dmna|6c0a61ff", "mv53rl050hq905/c0007a00.dmnb.gz|3489c834", 
    "mv53rl050hq905/c0008a00.dmna|e40a8e3e", "mv53rl050hq905/c0008a00.dmnb.gz|2f7f325e", 
    "mv53rl050hq905/c0009a00.dmna|3e8b3658", "mv53rl050hq905/c0009a00.dmnb.gz|15881ca", 
    "mv53rl050hq905/c0010a00.dmna|cbcf6bbd", "mv53rl050hq905/c0010a00.dmnb.gz|b35f64a9", 
    "mv53rl050hq905/c0011a00.dmna|b4d31993", "mv53rl050hq905/c0011a00.dmnb.gz|3878fcc6", 
    "mv53rl050hq905/c0012a00.dmna|2e7707cc", "mv53rl050hq905/c0012a00.dmnb.gz|f1cf2ce1", 
    "mv53rl050hq905/c0013a00.dmna|3d98f233", "mv53rl050hq905/c0013a00.dmnb.gz|e6bfb77", 
    "mv53rl050hq905/c0014a00.dmna|c04e3d33", "mv53rl050hq905/c0014a00.dmnb.gz|23b66a71", 
    "mv53rl050hq905/c0015a00.dmna|7d35c892", "mv53rl050hq905/c0015a00.dmnb.gz|a7e4f0cd", 
    "mv53rl050hq905/c0016a00.dmna|e791d6cd", "mv53rl050hq905/c0016a00.dmnb.gz|e6b02afe", 
    "mv53rl050hq905/c0017a00.dmna|f47e2332", "mv53rl050hq905/c0017a00.dmnb.gz|58dd91ea", 
    "mv53rl050hq905/c0018a00.dmna|c74d4e8c", "mv53rl050hq905/c0018a00.dmnb.gz|297c4244", 
    "mv53rl050hq905/c0019a00.dmna|fc6fbdd0", "mv53rl050hq905/c0019a00.dmnb.gz|cdd18e05", 
    "mv53rl050hq905/c0020a00.dmna|351aba49", "mv53rl050hq905/c0020a00.dmnb.gz|14aa2cf5", 
    "mv53rl050hq905/c0021a00.dmna|e41b86b1", "mv53rl050hq905/c0021a00.dmnb.gz|7a13fc68", 
    "mv53rl050hq905/c0022a00.dmna|7ebf98ee", "mv53rl050hq905/c0022a00.dmnb.gz|bd94f781", 
    "mv53rl050hq905/c0023a00.dmna|6d506d11", "mv53rl050hq905/c0023a00.dmnb.gz|6bc5c873", 
    "mv53rl050hq905/c0024a00.dmna|9086a211", "mv53rl050hq905/c0024a00.dmnb.gz|53db5e77", 
    "mv53rl050hq905/c0025a00.dmna|2dfd57b0", "mv53rl050hq905/c0025a00.dmnb.gz|6e18e226", 
    "mv53rl050hq905/lasat.log|1ace2b4", "mv53rl050hq905/param.def|55159907"  
    };

  
  public static void main(String[] args) {    
    int ir = 0;
    boolean non_standard = false;
    boolean interactive = false;
    boolean skipstdw = false;
    double bfac = Double.NaN;
    final ArrayList<Source> list = new ArrayList<>();
    final int nthreads;
    int nt = Runtime.getRuntime().availableProcessors();
    if (nt > 1) nt -= 1;
    String base_dir = my_home();                                  //-2019-02-28
    log(String.format("%s Version %s", my_name, my_version));
    for (String arg: args) {
      if (arg.startsWith("--source=")) {
        Source src = new Source(arg);
        if (src.name.equals(Source.INVALID)) {
          System.out.printf("*** %s:%n%s%n", Source.INVALID, arg);
          return;
        }
        list.add(src);
      }
      else if (arg.startsWith("-v")) {
        chk = true;
      }
      else if (arg.startsWith("--interactive")) {
        interactive = true;
      }
      else if (arg.startsWith("--num-threads=")) {
        nt = Integer.parseInt(arg.substring(14).trim());
        if (nt < 0)
          nt = 0;
        if (nt > 64)
          nt = 64;
      }
      else if (arg.equals("--skip-stacktip-downwash")) {
        skipstdw = true;
      }
      else if (arg.startsWith("--break-factor=")) {
        bfac = Double.parseDouble(arg.substring(15).trim().replace(',', '.'));
      }
    }
    nthreads = nt;
    //
    // interactive with GUI
    if (args.length == 0 || interactive) {
      new JFXPanel();
      Platform.runLater(() -> {
        View view = new View();
        view.create(list, nthreads);
        view.init(base_dir);
      });
      return;
    }
    //
    // console mode
    if (skipstdw) {
      skip_stacktip_downwash = skipstdw;
      non_standard = true;
    }
    if (!Double.isNaN(bfac) && bfac> 0) { 
      break_factor = bfac;
      non_standard = true;
    }
    log("creating...");
    Besmax nmc = new Besmax(base_dir, list, ir);
    log("init...");
    String msg = nmc.init();
    if (msg != null) {
      System.out.printf("ERROR %s%n", msg);
      return;
    }
    log("mapping...");
    //
    System.out.printf(LOC, "center = [%1.1f, %1.1f]%n", 
            nmc.center.getX(), nmc.center.getY());
    for (Source src: nmc.sources) {
      System.out.println(src);
    }
    log("using "+nthreads+" threads ...");
    Map<String, Double> map = (nthreads > 0) ? nmc.getMaxCon(nthreads) : nmc.getMaxCon();
    if (non_standard)
      log("WARNING: non-standard settings specified on program call!\n");
    System.out.printf(LOC, "cm = %9.3e g/m³%n", map.get("con"));
    System.out.printf(LOC, "dm = %9.1f %%%n", 100*map.get("dev"));
    System.out.printf(LOC, "xp = %9.1f m%n", map.get("xp"));
    System.out.printf(LOC, "yp = %9.1f m%n", map.get("yp"));
    System.out.printf(LOC, "kl = %9.1f KM%n", map.get("kl"));
    System.out.printf(LOC, "ua = %9.1f m/s%n", map.get("ua"));
    System.out.printf(LOC, "ra = %9.1f grd%n", map.get("ra"));
    log("done...");
    System.out.println("program finished");
  }
  
  private static void log(String msg) {
    System.out.printf("%s %s%n", sdf.format(new Date()), msg);
  }

}
