////////////////////////////////////////////////////////////////////////////////
//
// history
//
// 2021-10-07 uj distribution
// 
////////////////////////////////////////////////////////////////////////////////

/**
 * SVG
 * 
 * Utility class used by Besmax
 *
 * Copyright (C) Umweltbundesamt Dessau-Roßlau, Germany, 2021
 * Copyright (C) Janicke Consulting, Überlingen, Germany, 2021
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 * 
 */

package de.janicke.SVG;

import java.io.File;
import java.io.FileOutputStream;
import java.security.InvalidParameterException;
import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.function.Function;
import javafx.application.Platform;
import javafx.embed.swing.JFXPanel;
import javafx.geometry.Point2D;
import javafx.geometry.Rectangle2D;
import javafx.scene.Scene;
import javafx.scene.input.MouseEvent;
import javafx.scene.layout.StackPane;
import javafx.scene.paint.Color;
import javafx.scene.text.Font;
import javafx.scene.text.FontPosture;
import javafx.scene.text.FontWeight;
import javafx.scene.web.WebEngine;
import javafx.scene.web.WebView;
import javafx.stage.FileChooser;
import javafx.stage.Stage;

public class SVG {
  
  public static final String my_name = "SVG";
  private static int count = 0;
  private static final FileChooser chooser = new FileChooser();  
  
  private final boolean isRoot;
  private final Map<String,String> parms;
  private final List<String> children;
  
  public SVG(boolean root, Map<String,String> parms) {
    this.isRoot = root;
    this.parms = new LinkedHashMap<>(parms);
    children = new ArrayList<>();
  }
  
  @Override
  public String toString() {
    String title = parms.get("title");
    StringBuilder sb = new StringBuilder();
    sb.append("<svg");
    if (isRoot) {
      sb.append(" xmlns=\"http://www.w3.org/2000/svg\"");
      sb.append(" version=\"1.1\"");
    }
    for (String key: parms.keySet()) {
      if (key.equals("title"))
        continue;
      String value = parms.get(key);
      if (key.equals("font-family")) {
        if (value.toLowerCase().equals("monospaced"))
          value = "monospace";
        else if (value.toLowerCase().equals("sansserif"))
          value = "sans-serif";
      }
      sb.append(" ").append(key).append("=\"").append(value).append("\"");
    }
    sb.append(">\n");
    if (title != null)
      sb.append("<title>").append(title).append("</title>\n");
    for (String child: children) {
      sb.append(child).append("\n");
    }
    sb.append("</svg>\n");
    return sb.toString();
  }
  
  public void addChild(String child) {
    children.add(child);
  }
  
  public String display() {
    return display(null);
  }
    
  public String display(File file) {
    Stage stage = new Stage();
    WebView browser = new WebView();
    browser.setOnMouseClicked((MouseEvent e) -> {
      if (e.isControlDown()) {
        chooser.setTitle("Als SVG-Bild speichern");
        if (file != null) {
          File dir = file.getParentFile();
          chooser.setInitialDirectory(dir);
          chooser.setInitialFileName(file.getName());
        }
        File f = chooser.showSaveDialog(stage);
        if (f != null) {
          try {
            String fn = f.getPath().toLowerCase();
            if (!fn.endsWith(".svg"))
              f = new File(f.getPath() + ".svg");
            try (FileOutputStream fos = new FileOutputStream(f)) {
              fos.write(toString().getBytes("utf-8"));
            }
            System.out.printf("file '%s' written%n", f);
          } 
          catch (Exception x) {
            x.printStackTrace(System.out);
          }
        }
      }
    });
    WebEngine webEngine = browser.getEngine();
    String content = toString();
    webEngine.loadContent(content);
    //
    StackPane root = new StackPane();
    root.getChildren().add(browser);
    Scene scene = new Scene(root, -1, -1);
    String title = parms.getOrDefault("title", "SVG");
    stage.setTitle(title);
    stage.setScene(scene);
    stage.show();
    return content;
  }
  
  public static int getCount() {
    count++;
    return count;
  }
  
  public static String font(Font f) {
    double size = f.getSize();
    String name = f.getFamily();
    String family = name;
    switch (name.toLowerCase()) {
      case "monospaced":
        family = "monospace";
        break;
      case "sansserif":
        family = "sans-serif";
        break;
      default:;
    }
    String s = " font-family='" + family + "' font-size='" + size + "' ";
    return s;
  }
  
  //=========================================================================
  
  public static class ViewPort {
    
    public Rectangle2D view;
    public final Rectangle2D scale;
    public final int logx;
    public final int logy;
    private String border;
    private String style;
    private List<String> children;
    private double umin, umax, ulen;
    private double vmin, vmax, vlen;
    private double xmin, xmax, xlen;
    private double ymin, ymax, ylen;
    private double font_size;
    
    private Function<Point2D, Point2D> u2x;
    
    public ViewPort(Rectangle2D view, Rectangle2D scale,
            int logx, int logy, String border) {
      this.view = view;
      this.scale = (scale == null) ? view : scale;
      this.logx = logx;
      if (logx > 0 && (this.scale.getMinX() <= 0 || this.scale.getWidth() <= 0))
        throw new InvalidParameterException("invalid x-scaling");
      if (logy > 0 && (this.scale.getMinY() <= 0 || this.scale.getHeight() <= 0))
        throw new InvalidParameterException("invalid y-scaling");
      this.logy = logy;
      this.border = border;
      children = new ArrayList<>();
      //
      setView(view);
      //
      umin = this.scale.getMinX();
      umax = this.scale.getMaxX();
      vmin = this.scale.getMinY();
      vmax = this.scale.getMaxY();
      if (logx > 0) {
        umin = Math.log10(umin);
        umax = Math.log10(umax);
      }
      if (logy > 0) {
        vmin = Math.log10(vmin);
        vmax = Math.log10(vmax);
      }
      ulen = umax - umin;
      vlen = vmax - vmin;
    }
    
    public final void setView(Rectangle2D view) {
      this.view = view;
      xmin = view.getMinX();
      xmax = view.getMaxX();
      xlen = xmax - xmin;
      ymin = view.getMinY();
      ymax = view.getMaxY();
      ylen = ymax - ymin;
      u2x = (pu) -> {
        double u = pu.getX();
        if (logx > 0 && u <= 0)
          return null;
        double v = pu.getY();
        if (logy > 0 && v <= 0)
          return null;
        double x = getX(u);
        double y = getY(v);
        Point2D pp = new Point2D(x, y);
        return pp;
      };
    }
    
    public void setStyle(String style) {
      this.style = style;
    }
    
    public void setFontSize(double size) {
      font_size = size;
    }
    
    public double getX(double u) {
      double x;
      if (logx > 0)
        u = Math.log10(u);
      x = xmin + xlen*(u - umin)/ulen;
      return x;
    }
    
    public double getY(double v) {
      double y;
      if (logy > 0)
        v = Math.log10(v);
      y = ymax - ylen*(v - vmin)/vlen;
      return y;
    }
    
    public String addCurve(double u0, double du, double[] vv, 
            Map<String,String> opt) {
      int n = vv.length;
      double[] uu = new double[n];
      for (int i=0; i<n; i++)
        uu[i] = u0 + i*du;
      return addCurve(uu, vv, opt);
    }
    
    public String addCurve(double[] uu, double[] vv, Map<String,String> opt) {
      boolean drawing = false;
      StringBuilder sb = new StringBuilder();
      sb.append("<path");
      for (String key: opt.keySet()) {
        String val = opt.get(key);
        sb.append(" ").append(key).append("='").append(val).append("'\n");
      }
      int n = uu.length;
      if (vv.length < n)
        n = vv.length;
      sb.append(" d='");
      for (int i=0; i<n; i++) {
        double u = uu[i];
        double v = vv[i];
        Point2D pp = u2x.apply(new Point2D(u, v));
        if (pp == null) {
          drawing = false;
          continue;
        }
        double x = pp.getX();
        double y = pp.getY();
        if (drawing) {
          sb.append(" L ");
        }
        else {
          sb.append(" M ");
          drawing = true;
        }
        sb.append(x).append(" ").append(y).append("\n");
      }
      sb.append("'/>");
      String s = sb.toString();
      children.add(s);
      return s;
    }
    
    public String addRectangle(Rectangle2D r, Color clr) {
      return addRectangle(r, clr, null);
    }
    
    public String addRectangle(Rectangle2D r, Color clr, String t) {
      double xa = getX(r.getMinX());
      double xe = getX(r.getMaxX());
      double ya = getY(r.getMaxY());
      double ye = getY(r.getMinY());
      Locale l = Locale.ENGLISH;
      int red = (int)Math.round(255*clr.getRed());
      int grn = (int)Math.round(255*clr.getGreen());
      int blu = (int)Math.round(255*clr.getBlue());
      StringBuilder sb = new StringBuilder();
      sb.append("<rect")
              .append(" x=").append(String.format(l, "'%1.2f'", xa))
              .append(" y=").append(String.format(l, "'%1.2f'", ya))
              .append(" width=").append(String.format(l, "'%1.2f'", xe-xa))
              .append(" height=").append(String.format(l, "'%1.2f'", ye-ya))
              .append(" fill=")
              .append(String.format("'rgb(%d,%d,%d)'", red, grn, blu))
              .append(" />");
      if (t != null) {
        sb.append("\n<text")
                .append(" x=").append(String.format(l, "'%1.2f'", (xa+xe)/2))
                .append(" y=").append(String.format(l, 
                        "'%1.2f'", (ya+ye)/2 + 0.25*font_size))
                .append(" text-anchor='middle'>").append(t).append("</text>");
      }
      String s = sb.toString();
      children.add(s);
      return s;
    }
    
    public String addPaintable(Paintable o) {
      String s = o.paint(u2x);
      children.add(s);
      return s;
    }
    
    @Override
    public String toString() {
      return toString(null);
    }
    
    public String toString(Map<String, Object> opt) {
      StringBuilder sb = new StringBuilder();
      //
      // a rectangle for clipping
      //
      sb.append("<rect fill='none'");
      sb.append(" x='").append(view.getMinX()).append("'");
      sb.append(" y='").append(view.getMinY()).append("'");
      sb.append(" width='").append(view.getWidth()).append("'");
      sb.append(" height='").append(view.getHeight()).append("'");
      sb.append("/>");
      String rect = sb.toString();
      sb.setLength(0);
      //
      // define a styled border if requested
      //
      if (border != null) {
        sb.append("<rect");
        sb.append(" x='").append(view.getMinX()).append("'");
        sb.append(" y='").append(view.getMinY()).append("'");
        sb.append(" width='").append(view.getWidth()).append("'");
        sb.append(" height='").append(view.getHeight()).append("'");
        sb.append(" style='").append(border).append("'");
        sb.append("/>\n");
      }
      //
      sb.append("<svg");
      if (style != null)
        sb.append(" style='").append(style).append("'");
      sb.append(">\n");
      //
      // define the clipping area
      //
      int id = SVG.getCount();
      sb.append("<defs>\n");
      sb.append("<clipPath id='view-port-").append(id).append("'>\n");
      sb.append(rect).append("\n");
      sb.append("</clipPath>\n");
      sb.append("</defs>\n");
      sb.append("<g clip-path='url(#view-port-").append(id).append(")'>\n");
      //
      sb.append("<g");
      if (opt != null) {
        for (String key: opt.keySet()) {
          sb.append(" ").append(key).append("='");
          sb.append(opt.get(key)).append("'");
        }
      }
      sb.append(">\n");
      
//      sb.append("<g>\n");
      for (String child: children) {
        sb.append(child).append("\n");
      }
      sb.append("</g>\n");  // children
      sb.append("</g>\n");  // clip path
      sb.append("</svg>");
      return sb.toString();
    }            
  
    public void addChild(String child) {
      children.add(child);
    }
    
  }
  
  //=======================================================================
  
  public static class Axis {
    
    private final ViewPort port;
    private final String position;
    private final Map<String,Object> parms;
    private final int tlen;
    private boolean tick_long = false;
    private final double[][] tlog = 
    { {},
      { 1.0 },
      { Math.log10(3), 1.0 },
      { Math.log10(2), Math.log10(5), 1.0 },
      { Math.log10(2), Math.log10(4), Math.log10(7), 1.0 }
    };
    private List<Double> ticks;
    private List<String> labels;
    private double font_size;
    private double tick_step;
    
    public Axis(ViewPort port, String position, Map<String,Object> parms) {
      this.port = port;
      this.position = position;
      this.parms = new LinkedHashMap<>(parms);
      font_size = 0;
      if (parms.containsKey("font-size")) {
        Object o = parms.get("font-size");
        if (o instanceof Number)
          font_size = ((Number)o).doubleValue();
        else if (o instanceof String)
          font_size = Double.parseDouble((String)o);
      }
      //
      if (parms.containsKey("tick-step")) {
        Object o = parms.get("tick-step");
        if (o instanceof Number)
          tick_step = ((Number)o).doubleValue();
        else if (o instanceof String)
          tick_step = Double.parseDouble((String)o);
      }
      tlen = setTicks(port.scale);
    }
    
    public int getTlen() {
      return tlen;
    }
    
    private int setTicks(Rectangle2D scale) {
      int sulen = 0;
      try {
        if (position == null)
          return 0;
        ticks = new ArrayList<>();
        labels = new ArrayList<>();
        //
        String format = (String)parms.get("axis-format");
        double umin = scale.getMinX();
        double umax = scale.getMaxX();
        double vmin = scale.getMinY();
        double vmax = scale.getMaxY();
        //
        if (position.equals("bottom")) {    //----------- xaxis bottom
          int log = port.logx;
          if (log < 0)
            log = 0;
          else if (log > 4)
            log = 4;
          if (tick_step <= 0 && log <= 0) 
            return 0;
          double utck = umin;
          double udec = 0;
          int ilog = 0;
          if (log == 0) {   // linear
            utck = tick_step*Math.floor(umin/tick_step + 0.00001);
          }
          else {    // logarithmic
            umin = Math.log10(umin);
            umax = Math.log10(umax);
            udec = Math.floor(umin);
            utck = udec;
          }
          //
          while (utck<=umax) {
            if (utck >= umin) {
              double u = (log == 0) ? utck : Math.pow(10, utck);
              ticks.add(utck);
              String su = null;
              if (format != null) {
                su = String.format(Locale.ENGLISH, format, u);
                if (su.length() > sulen)
                  sulen = su.length();
              }
              labels.add(su);
            }
            if (log == 0) {
              utck += tick_step;
            }
            else {
              utck = udec + tlog[log][ilog];
              ilog++;
              if (ilog >= log) {
                ilog = 0;
                udec += 1.0;
              }
            }
          } // while
        }
        else if (position.equals("left")) {   //-------------- yaxis left
          int log = port.logy;
          if (log < 0)
            log = 0;
          else if (log > 4)
            log = 4;
          if (tick_step <= 0 && log <= 0) 
            return 0;
          double vtck = vmin;
          double vdec = 0;
          int ilog = 0;
          if (log == 0) {   // linear
            vtck = tick_step*Math.floor(vmin/tick_step + 0.00001);
          }
          else {    // logarithmic
            vmin = Math.log10(vmin);
            vmax = Math.log10(vmax);
            vdec = Math.floor(vmin);
            vtck = vdec;
          }
          //
          while (vtck<=vmax) {
            if (vtck >= vmin) {
              double v = (log == 0) ? vtck : Math.pow(10, vtck);
              ticks.add(vtck);
              String sv = null;
              if (format != null) {
                sv = String.format(Locale.ENGLISH, format, v);
                if (sv.length() > sulen)
                  sulen = sv.length();
              }
              labels.add(sv);
            }
            if (log == 0) {
              vtck += tick_step;
            }
            else {
              vtck = vdec + tlog[log][ilog];
              ilog++;
              if (ilog >= log) {
                ilog = 0;
                vdec += 1.0;
              }
            }
          } // while
        } // left
      } 
      catch (Exception e) {
        e.printStackTrace(System.out);
        sulen = 0;
      }
      return sulen;
    }
    
    @Override
    public String toString() {
      StringBuilder sb = new StringBuilder();
      try {
        if (position == null)
          return "";
        //
        if (parms.containsKey("tick-long")) {
          Object o = parms.get("tick-long");
          if (o instanceof Boolean)
            tick_long = (Boolean)o;
          else if (o instanceof String)
            tick_long = ((String)o).equals("true");
          else 
            tick_long = false;
        }
        sb.append("<g");
        if (parms.containsKey("stroke"))
          sb.append(" stroke='").append(parms.get("stroke")).append("'");
        if (parms.containsKey("stroke-width"))
          sb.append(" stroke-width='").append(parms.get("stroke-width")).append("'");
        if (parms.containsKey("fill"))
          sb.append(" fill='").append(parms.get("fill")).append("'");
        if (parms.containsKey("transform"))
          sb.append(" transform='").append(parms.get("transform")).append("'");
        //
        if (font_size > 0) {
          sb.append(" font-size='").append(font_size).append("'");
        }
        sb.append(">\n");
        //
        String caption = (String)parms.get("axis-caption");
        double xmin = port.view.getMinX();
        double ymin = port.view.getMinY();
        double xmax = port.view.getMaxX();
        double ymax = port.view.getMaxY();
        double umin = port.scale.getMinX();
        double umax = port.scale.getMaxX();
        double ulen = port.scale.getWidth();
        double vmin = port.scale.getMinY();
        double vmax = port.scale.getMaxY();
        double vlen = port.scale.getHeight();
        double dt = font_size/2;
        //
        if (position.equals("bottom")) {    //----------- xaxis bottom
          sb.append("<line x1='").append(xmin).append("' y1='")
                  .append(ymax).append("' x2='").append(xmax).append("' y2='")
                  .append(ymax).append("'/>\n");
          //
          int log = port.logx;
          if (log < 0)
            log = 0;
          else if (log > 4)
            log = 4;
          if (log > 0) {   // logarithmic
            umin = Math.log10(umin);
            umax = Math.log10(umax);
            ulen = umax - umin;
          }
          //
          double ystart = (tick_long) ? ymin : ymax;
          for (int it=0; it<ticks.size(); it++) {
            double utck = ticks.get(it);
            float x = (float)(xmin + (xmax-xmin)*(utck-umin)/ulen);
            sb.append("<line x1='").append(x).append("' y1='")
                    .append(ystart).append("' x2='").append(x)
                    .append("' y2='").append(ymax+dt).append("'/>\n");
            String su = labels.get(it);
            if (su != null) {
              sb.append("<text text-anchor='middle' stroke='none")
                      .append("' x='").append(x)
                      .append("' y='").append(ymax+1.5*font_size)
                      .append("'>").append(su).append("</text>\n");
            }
          }
          //
          if (caption != null) {
            sb.append("<text text-anchor='middle' stroke='none")
                    .append("' x='").append((xmin+xmax)/2)
                    .append("' y='").append(ymax+3*font_size).append("'>")
                    .append(caption).append("</text>\n");
          }
        }
        else if (position.equals("left")) {   //-------------- yaxis left
          sb.append("<line x1='").append(xmin).append("' y1='")
                  .append(ymin).append("' x2='").append(xmin).append("' y2='")
                  .append(ymax).append("'/>\n");
          //
          int log = port.logy;
          if (log < 0)
            log = 0;
          else if (log > 4)
            log = 4;
          if (log > 0) {   // logarithmic
            vmin = Math.log10(vmin);
            vmax = Math.log10(vmax);
            vlen = vmax - vmin;
          }
          //
          double xstart = (tick_long) ? xmax : xmin;
          for (int it=0; it<ticks.size(); it++) {
            double vtck = ticks.get(it);
            if (vtck >= vmin) {
              double v = (log == 0) ? vtck : Math.pow(10, vtck);
              float y = (float)(ymax - (ymax-ymin)*(vtck-vmin)/vlen);
              sb.append("<line x1='").append(xstart).append("' y1='")
                      .append(y).append("' x2='").append(xmin-dt)
                      .append("' y2='").append(y).append("'/>\n");
              String sv = labels.get(it);
              if (sv != null) {
                sb.append("<text text-anchor='end' stroke='none")
                        .append("' x='").append(xmin - 1.5*dt)
                        .append("' y='").append(y + 0.25*font_size)
                        .append("'>").append(sv).append("</text>\n");
              }
            }
          } // while
          //
          if (caption != null) {
            double xc = xmin - 3*dt - tlen*0.6*font_size;
            double yc = (ymin+ymax)/2;
            sb.append("<text text-anchor='middle' stroke='none")
                   .append("' transform='translate(").append(xc)
                   .append(",").append(yc).append(")").append(" rotate(-90)")
                   .append("' x='0' y='0'>")
                   .append(caption).append("</text>\n");
         }            
        } // left
        //
        sb.append("</g>");
      }
      catch (Exception e) {
        e.printStackTrace(System.out);
        return "";
      }
      return sb.toString();
    }
    
  }
  
  //=========================================================================
  
  public interface Paintable {
    String paint(Function<Point2D, Point2D> f);
  }
  
  //-----------------------------------------------------------------------
  
  private static void test_1() {
    new JFXPanel();
    Platform.runLater(() -> {
      String msg = null;
      try {
        //
        // define the root canvas, the view box and general properties
        //
        Map<String,String> parms = new LinkedHashMap<String,String>();
        parms.put("width", "150mm");
        parms.put("height", "150mm");
        parms.put("viewBox", "0 0 150 150");
        parms.put("stroke-width", "0.1");
        parms.put("font-family", "monospace");
        SVG svg = new SVG(true, parms);
        //
        // set the background color of the canvas
        //
        svg.addChild("<rect stroke='none' fill='beige'  width='100%'  height='100%'/>");
        //
        // define a viewport with scaling (opt.logarithmic) and style
        //
        ViewPort vp = new ViewPort(
                new Rectangle2D(60, 10, 80, 80),
                new Rectangle2D(0.1, 1, 1, 10), 
                4, 4,
                "fill:white; stroke:blue;");
        //
        // add a curve with painting options to the viewport
        //
        Map<String,String> opt = new LinkedHashMap<>();
        opt.put("stroke", "blue");
        opt.put("stroke-width", "0.5");
        opt.put("fill", "none");
        double[] yy = { 2, 7, Double.NaN, 4, 7, 4, 2};
        vp.addCurve(0.2, 0.2, yy, opt);
        //
        // add the view port to the canvas
        //
        svg.addChild(vp.toString());
        //
        // define the x-axis and add it to the canvas
        //
        Map<String,Object> x_map = new LinkedHashMap<>();
        x_map.put("fill", "red");
        x_map.put("stroke", "red");
        x_map.put("stroke-width", 0.3);
        x_map.put("font-size", 4);
        x_map.put("tick-step", 0.2);
        x_map.put("axis-format", "%1.1f");
        x_map.put("axis-caption", "x-Werte (m)");
        x_map.put("tick-long", true);
        Axis x_axis = new Axis(vp, "bottom", x_map);
        svg.addChild(x_axis.toString());
        //
        // define the y-axis and add it to the canvas
        //
        Map<String,Object> y_map = new LinkedHashMap<>();
        y_map.put("fill", "green");
        y_map.put("stroke", "green");
        y_map.put("stroke-width", 0.3);
        y_map.put("font-size", 4);
        y_map.put("tick-step", 1);
        y_map.put("axis-format", "%1.1f");
        y_map.put("axis-caption", "Eine ganz lange y-Beschriftung zum Testen");
        y_map.put("tick-long", true);
//        y_map.put("transform", "translate(-10,0)");   // shift the axis if required
        Axis y_axis = new Axis(vp, "left", y_map);
        svg.addChild(y_axis.toString());
        //
        // display the canvas
        //
        System.out.printf("%s%n", svg);
        svg.display();
      }
      catch (Exception e) {
        e.printStackTrace(System.out);
        msg = "*** " + e.toString();
      }
    });
  }
  
  private static void test_2() {
    Font f = Font.font("monospaced", FontWeight.BOLD, FontPosture.ITALIC, 4.0);
    System.out.printf("font=%s, stype=%s%n", f, f.getStyle());
  }
  
  private static void show(String fn) {
    Stage stage = new Stage();
    WebView browser = new WebView();
    
  }
  
  
  public static void main(String[] args) {
    System.out.printf("%s started%n", my_name);
    test_1();
  }
  
  
}
