/*====================================================================== IBJtm.c
 *
 * Time and date routines for IBJ programs
 * =======================================
 *
 * Copyright (C) Janicke Consulting, 88662 berlingen, Germany, 2011-2019
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * last change 2013-10-14 uj
 *
 *============================================================================*/


char *IBJtmVersion = "3.3.1";
static char *eMODn = "IBJtm";

#include <stdio.h>
#include <stdarg.h>
#include <string.h>
#include <stdlib.h>
#include <limits.h>
#include <ctype.h>
#include <math.h>
#include <time.h>
#include <locale.h>
#include <signal.h>

#include "IBJtm.h"
#include "IBJtm.nls"

char TmDateSeparator = 'T';
static int dayspassed[13] = {
        0, 31, 59, 90, 120, 151, 181, 212, 243, 273, 304, 334, 365 };

int TmParseDate(      // Parse a date of form yyyy-MM-dd[Thh:mm:ss[.SSS][ZZZZZ]]
    TMBROKEN *pbd,    // the parsed date as broken date
    char *s)          // the source string to be parsed
{                     // RETURNS 0 or <0 on error
  char *p, *q;
  TMBROKEN bd;
  double ds;
  int n, tz_min, tz_hour;
  if (s == NULL || pbd == NULL)
    return -1;
  memset(&bd, 0, sizeof(TMBROKEN));
  bd.tzmin = INT_MIN;                                             //-2011-11-10
  if (!strcmp(s, "+inf")) {
    pbd->year = INT_MAX;
    return 0;
  }
  if (!strcmp(s, "-inf")) {
    pbd->year = INT_MIN;
    return 0;
  }
  //
  while ((*s) && isspace(*s))
    s++;
  if (!*s)
    return -2;
  n = sscanf(s, "%d-%d-%d", &bd.year, &bd.month, &bd.day);
  if (n != 3)
    return -3;
  if (bd.year < 100)
    return -4;
  if (bd.month < 1 || bd.month > 12)
    return -5;
  bd.leap = TmIsLeapYear(bd.year);
  //
  p = strchr(s, 'T');
  if (p == NULL)
    p = strchr(s, '\'');
  if (p == NULL)
    p = strchr(s, ' ');
  if (p == NULL)
    p = strchr(s, '.');
  if (p == NULL) {
    *pbd = bd;
    return 0;
  }
  p++;
  n = sscanf(p, "%d:%d:%lf", &bd.hour, &bd.minute, &ds);
  if (n != 3)
    return -6;
  bd.second = ds;
  if (bd.hour < 0 || bd.minute < 0 || bd.second < 0)
    return -7;
  bd.millisec = (ds - bd.second)*1000 + 0.5;
  if (bd.millisec > 999) {
    bd.second += 1;
    bd.millisec -= 1000;
  }
  //
  q = strchr(p, '+');
  if (q == NULL)
    q = strchr(p, '-');
  if (q != NULL) {
    tz_min = 0;
    n = sscanf(q+1, "%2d%2d", &tz_hour, &tz_min);
    if (n < 1)
      return -9;
    tz_min += 60*tz_hour;
    if (*q == '-')
      tz_min = -tz_min;
    if (tz_min < -720 || tz_min > 840)
      tz_min = INT_MIN;
    bd.tzmin = tz_min;
  }
  *pbd = bd;
  return 1;
}

int TmIsLeapYear(     // Checks for leap year
int year)             // the year to be checked
{                     // RETURNS 1 if <year> is a leap year
  return (year % 4 == 0 && (year % 100 != 0 || year % 400 == 0));
}

int TmZone(void)      // Get the local time zone
{                     // RETURNS difference to GMT in minutes
  int tzsec, tzmin;
#ifdef __linux__
  tzsec = timezone;
#else
  #ifdef __GNUC__
    tzsec = timezone;
  #else  
    _get_timezone(&tzsec);
  #endif
#endif
  tzmin = -tzsec/60;
  return tzmin;  
}

TMBROKEN TmDate(void)   // Get the current date
{                       // RETURNS the broken date with local time zone
  TMBROKEN bd;
  time_t now;
  struct tm tb;
  time(&now);
  tb = *localtime(&now);
  bd.year = 1900 + tb.tm_year;
  bd.month = 1 + tb.tm_mon;
  bd.day = tb.tm_mday;
  bd.hour = tb.tm_hour;
  bd.minute = tb.tm_min;
  bd.second = tb.tm_sec;
  bd.millisec = 0;
  bd.tzmin = TmZone();
  bd.leap = TmIsLeapYear(bd.year);
  return bd;
}

TMDATE TmGetDate(     // Convert a broken date into date
    TMBROKEN bd)      // the broken date
{                     // RETURNS the date using GMT if time zone is unknown
  TMDATE date=0, days;
  int y;
  if (bd.year > 9999)
    return TM_MAX_DATE;
  if (bd.year < -999)
    return TM_MIN_DATE;
  if (bd.month > 12)
    return TM_MAX_DATE;
  if (bd.month < 1)
    return TM_MIN_DATE;
  days = (bd.year - 1970)*365;
  if (days < 0) {
    for (y = bd.year; y<1970; y++) {
      if (TmIsLeapYear(y))
        days--;
    }
  }
  else {
    for (y=1970; y<bd.year; y++) {
      if (TmIsLeapYear(y))
        days++;
    }
  }
  days += dayspassed[bd.month-1] + bd.day - 1;
  if (bd.month > 2 && TmIsLeapYear(y))
    days++;
  date = (((days*24.0 + bd.hour)*60.0 + bd.minute)*60.0 + bd.second)*1000.0 + bd.millisec; //-2013-10-14
  if (bd.tzmin >= -720 && bd.tzmin <= 840)
    date -= 60000*bd.tzmin;
  return date;
}

TMBROKEN TmBreakDate(   // Break a date into components
    TMDATE date,        // the date
    int tzmin)          // the time zone to be used
{                       // RETURNS the broken date
  TMBROKEN bd;
  TMDATE days;
  int ms, nd=0, m;
  memset(&bd, 0, sizeof(TMBROKEN));
  if (date == TM_MAX_DATE) {
    bd.year = INT_MAX;
    return bd;
  }
  if (date == TM_MIN_DATE) {
    bd.year = INT_MIN;
    return bd;
  }
  if (tzmin < -720 || tzmin > 840)
    tzmin = INT_MIN;
  bd.tzmin = tzmin;
  if (tzmin != INT_MIN) {
    date += tzmin*60000;
  }
  days = date/86400000;
  ms = date - 86400000*days;
  if (ms < 0)
    ms += 86400000;
  if (date >= 0) {
    bd.year = 1970;
    while (1) {
      nd = 365 + TmIsLeapYear(bd.year);
      if (days < nd)
        break;
      days -= nd;
      bd.year++;
    }
  }
  else {
    bd.year = 1969;
    while (days < 0) {
      nd = 365 + TmIsLeapYear(bd.year);
      days += nd;
      bd.year--;
    }
  }
  bd.leap = TmIsLeapYear(bd.year);
  for (m=1; m<=12; m++) {
    nd = dayspassed[m];
    if (m > 1 && bd.leap)
      nd++;
    if (nd > days)
      break;
  }
  bd.month = m;
  bd.day = 1 + days - dayspassed[m-1];
  if (m > 2 && bd.leap)
    bd.day--;
  //
  bd.second = ms/1000;
  bd.millisec = ms - 1000*bd.second;
  bd.minute = bd.second/60;
  bd.second -= 60*bd.minute;
  bd.hour = bd.minute/60;
  bd.minute -= 60*bd.hour;
  return bd;
}

int TmFormatDate(     // Format a date
    char *buf,        // result (buffer size >= 29)
    char *format,     // format as "%<l>[.<p>]D" with 10..l..28, 0..p..3
                      // or NULL for default "%19D"
    TMBROKEN bd)      // the broken date value
{                     // RETURNS <0 on error
  char *pb;
  int ld, lm, lf, l;
  if (buf == NULL)
    return -21;
  if (format == NULL)
    format = "%19D";
  else if (*format != '%' || format[strlen(format)-1] != 'D')
    return -1;
  lf = 19;
  lm = 0;
  sscanf(format+1, "%d.%d", &lf, &lm);
  if (lm < 0)
    lm = 0;
  else if (lm > 0)
    lm = 3;
  l = lm + (lm>0);
  if (lf <= 0)
    lf = 19 + l;
  else if (lf > 28)
    lf = 28;
  ld = lf - l;
  if (bd.year > 9999) {
    strcpy(buf, "+inf");
    return 0;
  }
  if (bd.year < -999) {
    strcpy(buf, "-inf");
    return 0;
  }
  pb = buf;
  *pb = 0;
  sprintf(pb, "%4d-%02d-%02d", bd.year, bd.month, bd.day);
  pb += 10;
  if (ld >= 19) {
    sprintf(pb, "%c%02d:%02d:%02d", TmDateSeparator,
        bd.hour, bd.minute, bd.second);
    pb += 9;
  }
  if (lm > 0) {
    sprintf(pb, ".%03d", bd.millisec);
    pb += 4;
  }
  if (ld >= 24) {
    if (bd.tzmin >= -720 && bd.tzmin <= 840) {
      int sgn=1, h=0, m=0;
      if (bd.tzmin < 0) {
        sgn = -1;
        bd.tzmin = -bd.tzmin;
      }
      h = bd.tzmin/60;
      m = bd.tzmin - 60*h;
      sprintf(pb, "%+03d%02d", sgn*h, m);
      pb += 5;
    }
  }
  *pb = 0;
  return 0;
}

int TmHasZone(      // Check the time zone of a broken date
    TMBROKEN *pbd)  // pointer to broken date
{                   // RETURNS != 0 if time zone is valid
  return pbd->tzmin>=-720 && pbd->tzmin<=840;
}

int TmSetZone(      // Set (if unknown) or change the time zone of a broken date
    TMBROKEN *pbd,  // pointer to broken date
    int tzmin)      // the new time zone (-720 <= tzmin <= 840)
{                   // RETURNS <0 on error
  TMBROKEN bd;
  TMDATE date;
  if (pbd == NULL)
    return - 1;
  if (tzmin < -720 || tzmin > 840)
    return -2;
  bd = *pbd;
  if (bd.tzmin < -720 || bd.tzmin > 840) {
    bd.tzmin = tzmin;
    *pbd = bd;
    return 0;
  }
  date = TmGetDate(bd);
  bd = TmBreakDate(date, tzmin);
  *pbd = bd;
  return 1;
}

int TmGetTzShift(   // Get the time shift of a time zone
    char *tmzn)     // time zone as [GMT][+|-]hh[:]mm
{                   // RETURNS the time shift in minutes
  int l, n, hours=0, minutes=0;
  if (tmzn == NULL)
    return 0;
  l = strlen(tmzn);
  if (l < 5)
    return 0;
  if (strncmp(tmzn, "GMT", 3))
    return 0;
  if (!strchr("+-", tmzn[3]))
    return 0;
  if (strchr(tmzn+4, ':'))
    n = sscanf(tmzn+4, "%2d:%2d", &hours, &minutes);
  else
    n = sscanf(tmzn+4, "%2d%2d", &hours, &minutes);
  if (n < 1)
    return 0;
  if (n < 2)  minutes = 0;
  minutes += 60*hours;
  if (tmzn[3] == '-') minutes = -minutes;
  return minutes;
}

int TmParseTime(    // Parse a time string of form [ddd.]hh:mm:ss[.SSS]
    TMTIME *ptm,    // pointer to time value
    char *s)        // the string to be parsed
{                   // RETURNS <0 on error
  char *p, *q;
  int day, hour, minute;
  double second;
  double sgn = 1.;                                                //-2013-10-14
  if (s == NULL || ptm == NULL)
    return -1;
  if (!strcmp(s, "+inf")) {
    *ptm = TM_MAX_TIME;
    return 0;
  }
  if (!strcmp(s, "-inf")) {
    *ptm = TM_MIN_TIME;
    return 0;
  }
  //
  while ((*s) && isspace(*s))
    s++;
  if (!*s)
    return -2;
  if (*s == '-') {
    sgn = -1.;
    s++;
  }
  p = strchr(s, '.');
  q = strchr(s, ':');
  if (q == NULL) {  // only seconds
    if (1 != sscanf(s, "%lf", &second))
      return -2;
    *ptm = second;
    return 0;
  }
  day = 0;
  if (p != NULL && p < q) {  // read day
    if (1 != sscanf(s, "%d", &day))
      return -3;
    if (day < 0) 
      return -6;
    s = p + 1;
  }
  //
  if (s == q)
    hour = 0;
  else if (1 != sscanf(s, "%d", &hour))
    return -10;
  s = q + 1;
  q = strchr(s, ':');
  if (q == NULL) {
    minute = hour;
    hour = 0;
  }
  else {
    if (s == q)
      minute = 0;
    else if (1 != sscanf(s, "%d", &minute))
      return -11;
    s = q + 1;
  }
  if (*s == 0)
    second = 0;
  else if (1 != sscanf(s, "%lf", &second))
    return -12;
  //

//  n = sscanf(s, "%d:%d:%lf", &hour, &minute, &second);
//  if (n == 2) {
//    hour = 0;
//    n = sscanf(s, "%d:%lf", &minute, &second);
//    if (n != 2)
//      return -6;
//  }
//  else if (n != 3)
//    return -4;
  if (hour < 0 || minute < 0 || second < 0)
    return -5;
  second = sgn*(second + 60.0*(minute + 60.0*(hour + 24.0*day))); //-2013-10-14
  *ptm = second;
  return 0;
}

int TmFormatTime(   // Format a time value as [ddd.]mm:hh:ss[.SSS]
    char *buffer,   // result buffer (size >=24)
    char *format,   // format as "%<l>[.<p>]T" with 0..p..9
    TMTIME tm)      // the time value to be formatted
{                   // RETURNS <0 on error
  int day, hour, minute, sgn=1;
  int ld=0, lm=0, lf=0, l;
  char frm_sec[32], *buf;
  if (buffer == NULL)
    return -1;
  if (tm == TM_MIN_TIME || tm <= -10000.0*86400) {                //-2013-10-14
    strcpy(buffer, "-inf");
    return 0;
  }
  if (tm == TM_MAX_TIME || tm >= 100000.0*86400) {                //-2013-10-14
    strcpy(buffer, "+inf");
    return 0;
  }
  buf = buffer;
  if (format == NULL)
    format = "%8.0T";
  else if (*format != '%' || format[strlen(format)-1] != 'T')
    return -2;
  lf = 8;
  lm = 0;
  sscanf(format+1, "%d.%d", &lf, &lm);
  if (lm < 0)
    lm = 0;
  else if (lm > 9)
    lm = 9;
  l = lm + (lm > 0);
  if (lf < l+2)
    lf = l + 2;
  else if (lf > l+14)                                             //-2013-10-14 13->14
    lf = l + 14;
  ld = lf - 8 - l - 1;
  if (ld < 0)
    ld = 0;
  sprintf(frm_sec, "%%0%d.%dlf", l+2, lm);
  //
  if (tm < 0) {
    sgn = -1;
    tm = -tm;
  }
  day = tm/86400;
  tm -= day*86400.0;                                              //-2013-10-14
  hour = tm/3600;
  tm -= hour*3600;
  minute = tm/60;
  tm -= 60*minute;
  //
  if (sgn < 0) {
    *buf = '-';
    buf++;
    if (day > 9999)                                               //-2013-10-14 999->9999
      day = 9999;                                                
  }
  else if (day > 99999)                                           //-2013-10-14 9999->99999
    day = 99999;
  if (ld > 0 || day > 0) {
    sprintf(buf, "%d.", day);
    buf += strlen(buf);
  }
  sprintf(buf, "%02d:%02d:", hour, minute);
  buf += 6;
  sprintf(buf, frm_sec, tm);
  return 0;
}

int TmFormatZone(     // Format the time zone value as "GMT+xx:yy"
    char *buffer,     // the result buffer (size >= 10)
    int tzmin)        // the time zone value (minutes)
{                     // RETURNS 0
  int hour, min, sign;
  if (tzmin < -720 || tzmin > 840)
    tzmin = 0;
  sign = (tzmin < 0) ? -1 : 1;
  tzmin *= sign;
  hour = tzmin/60;
  min = tzmin - 60*hour;
  hour *= sign;
  sprintf(buffer, "GMT%+03d:%02d", hour, min);
  return 0;
}

TMTIME TmDiffTime(    // Get a time difference
    TMDATE newdate,   // the new date
    TMDATE olddate)   // the old date
{                     // RETURNS the difference new-old
  if (newdate == TM_MAX_DATE)
    return (olddate == TM_MAX_DATE) ? 0 : TM_MAX_TIME;
  if (newdate == TM_MIN_DATE)
    return (olddate == TM_MIN_DATE) ? 0 : TM_MIN_TIME;
  if (olddate == TM_MAX_DATE)
    return TM_MIN_TIME;
  if (olddate == TM_MIN_DATE)
    return TM_MAX_TIME;
  return (newdate - olddate)/((double)1000);
}

TMDATE TmShiftDate(   // Shift a date by a time span
    TMDATE date,      // the unshifted date
    TMTIME tm)        // the time span
{                     // RETURNS the shifted date
  TMDATE d;
  if (date == TM_MAX_DATE || date == TM_MIN_DATE)
    return date;
  if (tm == TM_MAX_TIME)
    return TM_MAX_DATE;
  if (tm == TM_MIN_TIME)
    return TM_MIN_DATE;
  d = date + (TMDATE)(1000*tm + ((tm < 0) ? -0.5 : 0.5));
  return d;
}

TMTIME TmClock(void)    // Get the current time
{                       // RETURNS the time span from the start of the program
  TMTIME c = clock();
  return c/CLOCKS_PER_SEC;
}

TMDATE TmConvertDate(   // Convert date
    double d)           // date in days as defined in MSG
{                       // RETURNS date in milli-seconds as defined in TM
  TMDATE date;
  date = (d - 1025569)*1440*60000 + 0.5;
  return date;
}

int TmRelation(               // 1, if data are valid within [*pt1,...]
TMTIME vt1, TMTIME vt2,       // start and end of validity time interval
TMTIME rt1, TMTIME rt2 )      // start and end of requested time interval
{
  if (vt2 < vt1)  vt2 = vt1;
  if (rt1 < vt1)
    return -1;
  if (rt1 > vt2)
    return 1;
  if (rt1 == vt2) {
    if (vt1 < vt2)
      return 1;
    if (rt2 > rt1)
      return 1;
  }
  return 0;
}

/*
 * history:
 * 
 * 2013-10-14  uj 3.3.1  day index and second parsing adjusted
 * 2018-10-04  uj        imported from LASAT
 *
 */
