/*==================================================================== TalBlm.h
 *
 * Meteorological boundary layer mode according to VDI 3783 Part 8 (2017)
 * ======================================================================
 *
 * Copyright (C) Umweltbundesamt, Dessau-Rolau, Germany, 2002-2021
 * Copyright (C) Janicke Consulting, 88662 berlingen, Germany, 2002-2021
 * Email: info@austal.de
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * last change: 2021-09-21 uj
 *
 *============================================================================*/

#include <math.h>

#define  STDMYMAIN  BlmMain
static char *eMODn = "TalBlm";
#include "IBJmsg.h"
#include "IBJary.h"
#include "IBJtxt.h"
#include "IBJstd.h"    

//============================================================================

STDPGM(tstblm, BlmServer, 3, 0, 1);

//============================================================================

#include "genutl.h"
#include "genio.h"

#include "TalTmn.h"
#include "TalNms.h"
#include "TalGrd.h"
#include "TalIo1.h"
#include "TalBlm.h"
#include "TalBlm.nls"

#ifndef PI
  #define PI       3.1415926
#endif

#define  TP_VAL  0x0001
#define  TS_VAL  0x0002
#define  SP_VAL  0x0004
#define  ST_VAL  0x0008
#define  SW_VAL  0x0010

#define ALPHA         0.3
#define U2LFA         0.2
#define U2LITERMAX    30
#define U2LITERDEV    0.005
#define U2LITERSTEP   0.1

BLMPARM *BlmPprm;
ARYDSC *BlmParr;
VARTAB *BlmPvpp;

static int   nz0TAL = 9;
static float z0TAL[9] = { 0.01, 0.02, 0.05, 0.10, 0.20, 0.50, 1.0, 1.5, 2.0 };

static float lmTAL[6][9] =                                        //-2018-10-04
                  { {       5,     7,    9,   13,   17,   28,   44,   60,   77 }, 
                    {      25,    31,   44,   59,   81,  133,  207,  280,  358 },
                    {     354,   448,   631, 842, 1160, 1893, 2951, 4000, 5107 },
                    {     -37,   -47,  -66,  -88, -122, -199, -310, -420, -536 },
                    {     -15,   -19,  -27,  -36,  -49,  -80, -125, -170, -217 },
                    {      -6,    -8,  -11,  -15,  -20,  -33,  -52,  -70,  -89 } };

static float TLhMax = 1200,
             TLvMax = 1200,
             TlMin  = 0;

static float Hm10 =  -1,
             Hm20 =  -1,
             Hm31 =  -1,
             Hm32 =  800, /*  konsistent mit VDI 3782/3  */
             Hm40 = 1100, /*  konsistent mit VDI 3782/3  */
             Hm50 = 1100; /*  konsistent mit VDI 3782/3  */

static char DefName[256] = "param.def";
static char InpName[256] = "meteo.def";                           //-2011-11-23
static long MetT1, MetT2, ZtrT1, ZtrT2, PosZtr;
static int PrfMode;
static GRDPARM *Pgp;
static ARYDSC  *Pga;
static int Gl, Gi;
static float vKk=0.4, rvKk=2.5, TuMax, TvMax, TwMax;
static float Suf=-999, Svf=-999, Swf=-999, Tuf=-999, Tvf=-999, Twf=-999;

//-2018-10-04
static float HmMin = 0.0;
static float HmNeutral = BLMDEFHM;
static float Fcor = BLMFCOR;
static int   HmMeanAdjust = 0;
static float Avf;
static int u2lInit;
static float u2lA, u2lH1, u2lK2, u2lFa, u2lAa, u2lAlpha, u2lHa;
static float u2lUp, u2lUpd, u2lRadd, u2lUg;
static int SkipHm = 0;

/*==========================================================================*/


static int TalUu(float h, float z0, float d0, float ust, float lmo, float hm, float *puu);

static int TalRa(float h, float z0, float d0, float usg, float lmo, float hm, float *prr);

static int TalSw(float h, float z0, float d0, float ust, float lmo, float hm, float *psw);

static int TalKw(float h, float z0, float d0, float ust, float lmo, float hm, float *pkw);

static int TalTw(float h, float z0, float d0, float ust, float lmo, float hm, float *ptw);

static int TalSv(float h, float z0, float d0, float ust, float lmo, float hm, float *psv);

static int TalKv(float h, float z0, float d0, float ust, float lmo, float hm, float *pkv);

static int TalTv(float h, float z0, float d0, float ust, float lmo, float hm, float *ptv);

static int TalSu(float h, float z0, float d0, float ust, float lmo, float hm, float *psu);

static int TalKu(float h, float z0, float d0, float ust, float lmo, float hm, float *pku);

static int TalTu(float h, float z0, float d0, float ust, float lmo, float hm, float *ptu);

static int StdUuPrandtl(float h, float z0, float d0, float ust, float lmo, float *puu, int simple);

static int StdKwm(float h, float z0, float d0, float ust, float lmo, float hm, float *pkw, int simple);

static int u2lInitialize(float ha, float ua, float ra, float z0, float d0, float usg, float lmo, float hm, float *pusg);

static int u2lSetModelParameters(float ha, float ra, float z0, float d0, float usg, float lmo, float hm, int adjust);

static int u2lCalculate(float h, float z0, float d0, float usg, float lmo, float hm, float *puu, float *prr);

static float getTheta(float z, float lm);

static float getHm(float ust, float lmo);

static int is_equal(float x, float y) {
  float xa, ya, q, eps;
  if (x == 0 && y == 0)
    return 1;
  eps = 1.e-5;
  xa = (x < 0) ? -x : x;
  ya = (y < 0) ? -y : y;
  q = (x-y)/(xa+ya);
  if (q < 0) q = -q;
  if (q < eps)
    return 1;
  return 0;
}

/*======================================================================= getTmp
*/
float BlmTemp( // temperature (C or K)
  float h,     // height above ground (m)
  float z0,    // roughness length (m)
  float d0,    // displacement height (m) 
  float ust,   // friction velocity (m/s)
  float lm,    // Obukhov length (m)
  float ta,    // temperature at height hta (C or K)
  int vers)    // boundary layer version
{
  float z, za, za2, tst, t, g1, g2, tbase;
  z = (h < d0 + 6*z0) ? 6*z0 : h - d0;
  // uj 2022-04-02
  // this settings yields temperature tb at height hb+d0 or,
  // if hb+d0 < d0+6*z0, temperature at d0+6*z0 
  za = BLMHT;
  //
  // this setting would yield temperature tb at height hb
  //// za = (BLMHT < D0 + 6*Z0) ? 6*Z0 : BLMHT - D0;
  //
  za2 = BLMHT2;
  tst = ((BLMTZERO+10.)/(BLMVKK*BLMGRAV))*ust*ust/lm;
  t = ta + (tst/BLMVKK)*(getTheta(z,lm) - getTheta(za,lm));  
  if (BLMHT2 < 0) {
    t += BLMDRYAD*(z - za);
  }
  else {
    g1 = BLMDRYAD;
    g2 = BLMGRAD2;
    if (za < za2) {
      if (z < za2)
        t += g1*(z - za);
      else
        t += g2*(z - za) + (g1 - g2)*(za2 - za);
    }
    else {
      if (z < za2)
        t += g1*(z - za) + (g2 - g1)*(za2 - za);
      else
        t += g2*(z - za);
    }
  }
  tbase = (ta > BLMTZERO-BLMTMAX) ? BLMTZERO : 0.;
  if (t > tbase+BLMTMAX)
    t = tbase+BLMTMAX;
  else if (t < tbase+BLMTMIN)
    t = tbase+BLMTMIN;
  return (float)t;
}
  
/*===================================================================== getTheta
*/
static float getTheta(float z, float lm) {
  double theta;
  double zeta = z/lm; 
  if (lm > 0.) {
    if (zeta < 0.5)
      theta = log(zeta) + 5*zeta -7*log(2.) + 4;
    else if (zeta < 10.)
      theta = 8.*log(zeta) + 4.25/zeta - 0.5/(zeta*zeta);
    else
      theta = 0.7585*zeta + 8.*log(10.) - 7.165;   
  }
  else {
    double y = sqrt(1 - 15*zeta);
    theta = log(fabs(zeta)) - 2.*log(1+y);  
  }
  return theta;
}

/*================================================================ u2lInitialize
*/
static int u2lInitialize(float ha, float ua, float ra, float z0, float d0, 
  float usg, float lmo, float hm, float *pusg) {
  dP(u2lInitialize);
  float hminit, us, u, diff, diffold, step;
  int n;
  if (u2lInit >= 0)                                                    eX(1); 
  u2lInit = 0;
  u2lH1 = NOTV;
  u2lHa = NOTV;
  hminit = hm;
  if (ua < 0. && usg > 0.) {
    u2lSetModelParameters(ha, ra, z0, d0, usg, lmo, hm, 0);            eG(12);
    TalUu(ha, z0, d0, usg, lmo, hm, &ua);
  }
  else if (ua > 0 && usg < 0) {
    StdUuPrandtl(ha, z0, d0, 1, lmo, &u, 1);
    usg = ua/u;
    u2lSetModelParameters(ha, ra, z0, d0, usg, lmo, hm, 0);            eG(13);
  }
  //
  // if resulting h1 < ha, try to derive the friction velocity iteratively
  if (u2lH1 < ha) {
    diff = 1.e20;
    diffold = diff;
    step = U2LITERSTEP;
    n = 0;
    us = 0.;
    while (fabs(diff) > U2LITERDEV) {          
      if ((diff*diffold < 0) || (fabs(diff) > fabs(diffold)))
        step *= -0.5;                      
      us += step;
      if (us <= 0.) {
        us -= step;
        step *= 0.5;
      }
      hm = (hminit > 0) ? hminit : getHm(us, lmo);
      u2lSetModelParameters(ha, ra, z0, d0, us, lmo, hm, 0);           eG(14);
      TalUu(ha, z0, d0, us, lmo, hm, &u);
      diffold = diff;
      diff = ua - u;             
      if (++n > U2LITERMAX)
        break; 
    }
    //
    // if iteration did not succeed, adjust H1 to ha
    if (n > U2LITERMAX) {
      vLOG(4)("BLM: u2lInitialize, iterative setting of u* did not succeed, adjusting H1");
        StdUuPrandtl(ha, z0, d0, 1, lmo, &u, 1);
      usg = ua/u;
      u2lSetModelParameters(ha, ra, z0, d0, usg, lmo, hm, 1);         eG(15);
    }
    else {
      usg = us;
      vLOG(4)("BLM: u2lInitialize, h1<ha: %d steps for interpolating ua (diff=%1.1e)", n, diff);
    }
  }   
  if (u2lH1 < ha) {   
    u2lRadd = 0.; //getR(ha) - ra;
  }
  u2lUg = sqrt(u2lUp*u2lUp + 
    (u2lUpd*u2lUpd + u2lAa*u2lAa*u2lUp*u2lUp)/(2*u2lA*u2lA) + 
    u2lUp*(u2lUpd - fabs(u2lAa)*u2lUp)/u2lA);
  //
  BlmPprm->u2lH1 = u2lH1;
  BlmPprm->u2lK2 = u2lK2;
  BlmPprm->u2lFa = u2lFa;
  BlmPprm->u2lUg = u2lUg;
  *pusg = usg;
  u2lInit = 1;
  return 0;
eX_1:
  eMSG("unexpected call!");
eX_12: eX_13: eX_14: eX_15:
  eMSG("can't set model parameters!");
}

/*======================================================== u2lSetModelParameters
*/
static int u2lSetModelParameters(float ha, float ra, float z0, float d0, 
float usg, float lmo, float hm, int adjust) {
  dP(u2lSetModelParameters);
  float kwm;
  if (u2lInit != 0)                                                      eX(1);
  if (usg <= 0. || lmo == 0. || ha <= 0.)                                eX(2);
  if (hm <= 0.)
    hm = getHm(usg, lmo);
  //
  // height of the first layer h1: half the height where Kwm has its maximum,
  // use neutral case for unstable stratification to provide an analytical 
  // expression
  if (lmo < 0.)    
    u2lH1 = hm/(6*ALPHA);
  else
    u2lH1 = 0.1*lmo*(-1. + sqrt(1. + 10.*hm/(3.*ALPHA*lmo)));
  u2lH1 /= 2.0;
  //
  // factor fa determining the wind rotation in the first layer
  u2lFa = U2LFA;
  if (adjust) {
    u2lFa *= u2lH1/ha;
    u2lH1 = ha;
  }
  //
  // constant diffusion coefficient K2 in the second layer
  StdKwm(u2lH1, z0, d0, usg, lmo, hm, &u2lK2, 0);
  //
  // A
  u2lA = sqrt(0.5*fabs(Fcor)/u2lK2);
  //
  // a
  u2lAa = u2lA * u2lFa;
  if (Fcor > 0)
    u2lAa *= -1;
  //
  // rotation 
  u2lAlpha = (270. - ra)*(PI/180.);
  u2lHa = ha;
  //
  // wind speed at h1 (use simple profile in the first layer)
  StdUuPrandtl(u2lH1, z0, d0, usg, lmo, &u2lUp, 1);
  //
  // wind speed derivation at h1 (analytical for the case of simple profiles)
  StdKwm(u2lH1, z0, d0, usg, lmo, hm, &kwm, 1);
  u2lUpd = usg*usg/kwm;
  return 0;
eX_1:
  eMSG("unexpected call!");
eX_2:
  eMSG("invalid usg/lmo/ha!");
}

/*================================================================= StdUuPrandtl
*/
static int StdUuPrandtl(float h, float z0, float d0, float ust, float lmo, 
float *puu, int simple) {
  float u=0, z, zm, zm0, psi, psi0, u1, hh;
  hh = d0 + 6*z0;
  z = (h < hh) ? hh-d0 : h-d0;
  if (lmo >= 0) {
    if (lmo > 0) {
      zm = z/lmo;
      zm0 = z0/lmo;
    }
    else {
      zm = 0.;
      zm0 = 0.;
    }
    if (simple)
      psi = (zm > zm0) ? log(zm/zm0) + 5*(zm - zm0) : 0.;
    else {              
      if (zm < 0.5)
        psi = log(zm/zm0) + 5*(zm - zm0);
      else if (zm < 10)
        psi = 8*log(2*zm) + 4.25/zm - 1./(2*zm*zm) - log(2*zm0) - 5*zm0 - 4;
      else
        psi = 0.7585*zm + 8*log(20.) - 11.165 - log(2*zm0) - 5*zm0;
    }
    u = rvKk * ust * psi;
  }
  else {
    zm = z/lmo;
    zm0 = z0/lmo;
    psi = pow(1. - 15.*zm, 0.25);
    psi0 = pow(1. - 15.*zm0, 0.25);
    u1 = (psi0+1)*(psi0+1)*(psi0*psi0+1)*zm/zm0;
    u1 = log(u1/( (psi+1)*(psi+1)*(psi*psi+1) ));
    u = rvKk * ust * (2*(atan(psi) - atan(psi0)) + u1);
  }    
  if (h < hh)
    u *= h / hh;
  *puu = u;
  return 0;
}

/*======================================================================= StdKwm
*/
static int StdKwm( float h, float z0, float d0, float ust, float lmo, float hm,
float *pkw, int simple )
  {
  float z, s, a, b, dc, dm, kw;
  z = (h < d0+6*z0) ? 6*z0 : h-d0;
  a = (hm > 0) ? z/hm : 0;
  s = (lmo != 0) ? z/lmo : 0;
  b = 1. - 0.8*a;
  if (simple) {
    dc = 1.;
    dm = 1.;
  }
  else {
    dc = (b > 0) ? b*b : 0;
    dm = (a > 0) ? exp(-6*ALPHA*a) : 1.0;
  }
  if (lmo >= 0)
    kw = dm/(1 + 5*s);
  else    
    kw = pow(dm*dm*dm*dm - 15*s*dc*dc*dc*dc, 1./4.);  
  kw *= Twf*vKk*ust*z;       
  *pkw = kw;
  return 0;
}

/*======================================================================== getHm
*/
static float getHm(float ust, float lmo) {
  float a, hm = HmNeutral;
  if (lmo > 0.) {
    float fc = fmax(BLMFCORMIN, fabs(Fcor));
    hm = 0.3*ust/fc;
    a = sqrt(fc*lmo/ust);
    if (a < 1)  
      hm *= a;
    if (hm > HmNeutral) 
      hm = HmNeutral; 
  }
  if (hm < HmMin) 
    hm = HmMin;
  return hm;
}

/*================================================================= u2lCalculate
*/
static int u2lCalculate(float h, float z0, float d0, float usg, float lmo, 
float hm, float *puu, float *prr) {
  dP(u2lCalculate);
  float hh, z, u, v, za, z1, uu, rr, vv;
  float cz, sz, cp, sp, az, wp, wm, p, q;
  if (u2lInit < 0)                                                       eX(1); 
  if (z0 <= 0)                                                           eX(2);
  if (usg <= 0)                                                          eX(3);
  //
  // displacement zone treatment
  hh = d0+6*z0;
  z = (h < hh) ? hh - d0 : h - d0;
  za = (u2lHa < hh) ? hh - d0 : u2lHa - d0;
  z1 = (u2lH1 < hh) ? hh - d0 : u2lH1 - d0;
  if (z < 0)                                                             eX(4);  
  //
  //  profile for the first layer
  if (h < u2lH1) {
    StdUuPrandtl(h, z0, d0, usg, lmo, &uu, 1);
    if (uu == 0)
      uu = 1.e-8;
    cz = cos(u2lAa*(z-za) + u2lAlpha);
    sz = sin(u2lAa*(z-za) + u2lAlpha);
    u = uu * cz;
    v = uu * sz;
  }
  //
  // profile for the second layer
  else {      
    cp = cos(u2lAa*(z1-za) + u2lAlpha);
    sp = sin(u2lAa*(z1-za) + u2lAlpha);      
    az = u2lA * (z-z1);
    wp = cp + sp;
    wm = cp - sp;     
    cz = exp(-az) * cos(az);
    sz = exp(-az) * sin(az);
    p = u2lUpd*wp + u2lAa*u2lUp*wm;
    q = u2lUpd*wm - u2lAa*u2lUp*wp;
    if (Fcor > 0) {
      u = u2lUp*cp + (0.5/u2lA)*((1-cz)*p + sz*q);
      v = u2lUp*sp + (0.5/u2lA)*((cz-1)*q + sz*p);
    }
    else {
      u = u2lUp*cp + (0.5/u2lA)*((1-cz)*q + sz*p);
      v = u2lUp*sp - (0.5/u2lA)*((cz-1)*p + sz*q);
    }
    //
    // linear extension in displacement zone
    if (h < hh) {
      if (h == 0)
        h = 1.e-8;
      u *= h/hh;
      v *= h/hh;
    }
  }
  //
  // store results
  if (u2lRadd != 0.0) {
    rr = u2lRadd*(PI/180.);
    uu = u*cos(rr) - v*sin(rr);
    vv = u*sin(rr) + v*cos(rr);
    u = uu;
    v = vv;
  }
  uu = sqrt(u*u + v*v);
  rr = atan2(-u, -v)*(180./PI);
  if (rr < 0)
    rr += 360.;
  *puu = uu;
  *prr = rr;
  return 0;
eX_1:
  eMSG("unexpected call!");
eX_2:
  eMSG("z0 <= 0!");
eX_3:
  eMSG("usg <= 0!");
eX_4:
  eMSG("z < 0!");
}

/*======================================================================== TalUu
*/
static int TalUu(float h, float z0, float d0, float ust, float lmo, float hm, 
float *puu)
{
  float rr;
  u2lCalculate(h, z0, d0, ust, lmo, hm, puu, &rr);
  return 0;
}

/*======================================================================== TalRa
*/
static int TalRa(float h, float z0, float d0, float usg, float lmo, float hm, 
float *prr)
{
  float uu;
  u2lCalculate(h, z0, d0, usg, lmo, hm, &uu, prr);
  return 0;
}

/*======================================================================== TalSw
*/
static int TalSw(float h, float z0, float d0, float ust, float lmo, float hm,
float *psw)
{
  float z, s, a, sw, e, f;
  z = (h < d0+6*z0) ? 6*z0 : h-d0;  
  a = (hm > 0) ? z/hm : 0;
  s = (lmo != 0) ? z/lmo : 0;
  e = (a > 0) ? exp(-0.9*a) : 1;
  f = ((1 - 0.8*a) > 0) ? (1 - 0.8*a) : 0;
  if (lmo >= 0) 
    sw = e;
  else
    sw = pow(e*e*e - 2.5*s*f*f*f, 1./3.); 
  *psw = Swf*ust*sw;
  return 0;
}

/*======================================================================== TalKw
*/
static int TalKw(float h, float z0, float d0, float ust, float lmo, float hm,
float *pkw)
  {
  float z, s, a, kw, e, f;
  z = (h < d0+6*z0) ? 6*z0 : h-d0;
  a = (hm > 0) ? z/hm : 0;
  s = (lmo != 0) ? z/lmo : 0;
  e = exp(-1.8*a); 
  f = ((1 - 0.8*a) > 0) ? (1 - 0.8*a)*(1 - 0.8*a) : 0;
  if (lmo >= 0)
    kw = e/(1+5*s);
  else 
    kw = pow(e*e - 9*s*f*f, 1./2.);
  kw *= Twf*vKk*ust*z;    
  *pkw = kw;
  return 0;
}

/*======================================================================== TalTw
*/
static int TalTw(float h, float z0, float d0, float ust, float lmo, float hm,
float *ptw)
  {
  float kw, tw, sw; 
  TalKw(h, z0, d0, ust, lmo, hm, &kw);
  TalSw(h, z0, d0, ust, lmo, hm, &sw);
  tw = (sw > 0.) ? kw/(sw*sw) : TwMax;
  *ptw = tw;
  return 0;
}

/*======================================================================== TalSv
*/
static int TalSv(float h, float z0, float d0, float ust, float lmo, float hm,
float *psv)
{
  float z, a, sv;
  z = (h < d0+6*z0) ? 6*z0 : h-d0;
  a = (hm > 0) ? z/hm : 0;
  if (lmo >= 0)
    sv = Avf;
  else
    sv = Avf*pow(1 - 0.0642*(hm/lmo)*exp(-0.9*a), 1./3.);
  *psv = Svf*ust*sv;
  return 0;
}

/*======================================================================== TalKv
*/
static int TalKv(float h, float z0, float d0, float ust, float lmo, float hm,
float *pkv)
{
  float kv, sv, uu, z, hh;
  z = (h < d0+6*z0) ? d0+6*z0 : h; 
  TalSv(z, z0, d0, ust, lmo, hm, &sv);
  TalUu(z, z0, d0, ust, lmo, hm, &uu);
  hh = hm;
  if (hh > BLMDEFHM) hh = BLMDEFHM;
  kv = Tvf*Avf*(uu/ust)*(hh/100)*sv;   
  *pkv = kv;
  return 0;
}

/*======================================================================== TalTv
*/
static int TalTv(float h, float z0, float d0, float ust, float lmo, float hm,
float *ptv)
{
  float kv, tv, sv;
  TalSv(h, z0, d0, ust, lmo, hm, &sv);
  TalKv(h, z0, d0, ust, lmo, hm, &kv);
  tv = (sv > 0) ? kv/(sv*sv) : TvMax;
  *ptv = tv;
  return 0;
}

/*======================================================================== TalSu
*/
static int TalSu(float h, float z0, float d0, float ust, float lmo, float hm,
float *psu)
{
  float z, a, su;
  z = (h < d0+6*z0) ? 6*z0 : h-d0;
  a = (hm > 0) ? z/hm : 0;
  if (lmo >= 0) 
    su = Avf;
  else
    su = Avf*pow(1 - 0.0371*(hm/lmo)*exp(-0.9*a), 1./3.);
  *psu = Suf*ust*su;
  return 0;
}

/*======================================================================== TalKu
*/
static int TalKu(float h, float z0, float d0, float ust, float lmo, float hm,
float *pku)
{
  float ku, su, uu, hh, z;
  z = (h < d0+6*z0) ? d0+6*z0 : h;
  TalUu(z, z0, d0, ust, lmo, hm, &uu);
  TalSu(z, z0, d0, ust, lmo, hm, &su);
  hh = hm;
  if (hh > BLMDEFHM) hh = BLMDEFHM;
  ku = Tuf*Avf*(uu/ust)*(hh/100)*su;
  *pku = ku;
  return 0;
}

/*======================================================================== TalTu
*/
static int TalTu(float h, float z0, float d0, float ust, float lmo, float hm,
float *ptu)
{
  float ku, tu, su;
  TalSu(h, z0, d0, ust, lmo, hm, &su);
  TalKu(h, z0, d0, ust, lmo, hm, &ku);
  tu = (su > 0) ? ku/(su*su) : TuMax;   
  *ptu = tu;
  return 0;
}


//================================================================= BlmProfile
//
long BlmProfile(      /* Berechnung des Grenzschicht-Profils.               */
BLMPARM *p,           /* Eingabe-Daten (Grenzschicht-Parameter).            */
BLMREC *v )           /* Ausgabe-Daten (Wind-Varianzen und Korr.-Zeiten).   */
  {
  dP(BlmProfile);
  int vers1, vers2;
  float z, z0, d0, lmo, hm, ha, ua, da, sua, sva, swa;
  float ust, u, a, d, uha, dha;
  float xu, zs, zu, zg;
  float usg, su, sv, sw;
  float hp, hma, hmr;
  float tu, tv, tw;
  int no_shear=0;
  vLOG(5)("BLM:BlmProfile(...)");
  if ((!p) || (!v))                                             eX(1);
  z    = v->z;          /* absolute z */
  zg   = p->AnmZpos;
  hp   = z - zg;
  if (hp < 0)                                                   eX(2);
  z0   = p->RghLen;
  d0   = p->ZplDsp;
  vers1= p->MetVers;
  if (vers1!=53 && vers1!=52 && vers1!=5 && vers1!=7 && vers1!=1) eX(6);
  if (vers1 == 52) {
    no_shear = 1;
    vers1 = 53;
  }
  vers2 = vers1 % 10;  vers1 = vers1 / 10;
  lmo  = p->lmo;  if (lmo == 0)                                 eX(4);  //-2002-04-19
  hma  = p->hm;    // absolute: above z=0
  hmr  = hma - zg; // relative: above ground 
  hm   = (lmo < 0) ? hmr : hma;
  ust  = p->Ustar;
  usg  = p->Ustar;
  if (ust < 0) { ust = p->UstCalc;  usg = p->UsgCalc; }
  ha  = p->AnmHeight;
  ua  = p->WndSpeed;
  da  = p->WndDir;
  sua = p->SigmaU;
  sva = p->SigmaV;
  swa = p->SigmaW;
  Suf = 2.4;
  Svf = 2.0;
  Swf = 1.3;
  Tuf = 0.9;
  Tvf = 0.9;
  Twf = 1.0;
  a = BlmPprm->AvrTime;
  if (a > BLMINTMAX)  a = BLMINTMAX;
  if (a < BLMINTMIN)  a = BLMINTMIN;
  Avf = pow(a/3600, 0.18);
  TLhMax = a * 0.5;
  TLvMax = a * 0.5;
  TuMax = TLhMax;
  TvMax = TLhMax;
  TwMax = TLvMax;
  HmMin = 20.;
  if (vers1 < 1)  
   goto test_model;
  //----------------------------------------------------------------------------
  // default model
  //
  // initialize wind profile
  if (u2lInit < 0) {
    u2lInitialize(ha, ua, da, z0, d0, ust, lmo, hm, &usg);            eG(9);   
  }
  else if (usg <= 0.)                                                 eX(10);
  //
  // friction velocity for wind profile
  if (usg < 0.0) { 
    TalUu(ha, z0, d0, 1, lmo, hm, &u);
    usg =  ua/u;
    if (usg <= 0)                                                     eX(5);
  }
  //
  // friction velocity for turbulence profiles
  if (ust < 0.0) {
    if (swa > 0.0)  ust = swa/1.3;
    if (ust < 0.0)  ust = usg;
  }
  //
  // mixing layer height
  if (hma <= 0) {
    hm = getHm(ust, lmo);
    if (lmo > 0) { 
      hma = hm;
      hmr = hm - zg;
    }
    else  {
      hma = hm + zg;
      hmr = HmNeutral;
    }
    p->hm = hma;
  }
  if (lmo<0 && 2*ha>hmr)                                                 eX(3);
  //
  // save values
  p->UsgCalc = usg;
  p->UstCalc = ust;
  p->us = ust;
  //
  // wind speed
  TalUu(hp, z0, d0, usg, lmo, hm, &u);
  TalUu(ha, z0, d0, usg, lmo, hm, &uha);
  v->u = u * (ua/uha);
  //
  // wind direction
  if (no_shear)  
    d = da;
  else {
    TalRa(hp, z0, d0, usg, lmo, hm, &d);
    TalRa(ha, z0, d0, usg, lmo, hm, &dha);
    d += (da - dha);
    if (d >= 360.)                                               //-2021-09-21
      d -= 360.;
  }
  v->d = d;
  //
  // keep turbulence profiles constant above 3*hm
  // (for savety to avoid extremely small values)
  if (hp > d0+3*hm)  hp = d0+3*hm;
  //
  // velocity fluctuations
  TalSw(hp, z0, d0, ust, lmo, hm, &sw);
  v->sw = sw;
  TalSv(hp, z0, d0, ust, lmo, hm, &sv);
  v->sv = sv;
  TalSu(hp, z0, d0, ust, lmo, hm, &su);
  v->su = su;
  v->suw = 0;
  //
  // Lagrange correlation times
  TlMin = z0/ust;
  TalTu(hp, z0, d0, ust, lmo, hm, &tu);
  TalTv(hp, z0, d0, ust, lmo, hm, &tv);
  TalTw(hp, z0, d0, ust, lmo, hm, &tw);
  if (tu < TlMin)  tu = TlMin;
  if (tv < TlMin)  tv = TlMin;
  if (tw < TlMin)  tw = TlMin;
  if (tu > TuMax)  tu = TuMax;
  if (tv > TvMax)  tv = TvMax;
  if (tw > TwMax)  tw = TwMax;
  v->tu = tu;
  v->tv = tv;
  v->tw = tw;
  v->ths = p->ThetaGrad;
  return 0;
  //----------------------------------------------------------------------------
  // test versions
test_model:
  //
  if (ha <= 0)  ha = 10;
  if (da < 0)   da = 270;
  if (ust < 0) {
    ust = z0*ua/ha;
    p->UstCalc = ust;
    p->UsgCalc = ust;
  }
  p->us = ust;
  TlMin = z0/ust;
  v->u = ua;
  v->d = da;
  //
  if (vers2 == 5) {
    xu = 0.3;
    if (hp == 0.0) {
      zu = 0.0;
      zs = 1.e-12;
    }
    else {
      zu = pow(hp/ha, xu);
      zs = hp/ha;
    }
    v->su = 1.e-6;
    v->sv = 1.e-6;
    v->sw = swa*sqrt(zs);
    v->u  = ua*zu;
    v->tu = TlMin;
    v->tv = TlMin;
    v->tw = TlMin;
    return 0;
  }
  if (vers2 == 7) {
    if (hm <= 0)  hm = 200;
    v->su = sua;
    v->sv = sva;
    v->sw = swa*(1 - (z0/ha)*sin(0.5*PI*hp/hm));
    v->tu = 20*TlMin;
    v->tv = 20*TlMin;
    v->tw = TlMin*(1 + 20*sin(0.5*PI*hp/hm));
    return 0;
    }
  if (vers2 == 1) {
    v->su = sua;
    v->sv = sva;
    v->sw = swa;
    v->tu = 100*TlMin;
    v->tv = 100*TlMin;
    if (lmo > 9000)  v->tw = 10*TlMin;
    else             v->tw = TlMin*(1.0 + hp/ABS(lmo));
    v->suw = 0;
    return 0;
    }
eX_1:
  eMSG(_undefined_arguments_);
eX_2:
  eMSG(_negative_height_);
eX_3:
  eMSG(_anemometer_too_high_);
eX_4:
  eMSG(_undefined_stability_);
eX_5:
  eMSG(_no_ua_);
eX_6:
  eMSG(_unknown_model_$_, vers1/10.);
eX_9:
  eMSG("can't initialize 2-layer profile!");
eX_10:
  eMSG("unexpected error!");
}


/*================================================================ KTA1508T7i2
*/
static float KTA1508T7i2(   /* Tmp.-Gradient nach KTA1508 9/88 Tab. 7-2  */
float u10,                  /* Windgeschwindigkeit [m/s] in 10 m Hoehe    */
int kl )                    /* Stabilitaetsklasse                         */
  {
  int i, i10, k1, k2;
  float dt100;
  static float tsg[9][5] = {
    { -1.13, -1.03, -0.91, -0.37, 0.78 },
    { -1.18, -1.05, -0.91, -0.22, 1.12 },
    { -1.39, -1.18, -0.97, -0.16, 1.25 },
    { -1.61, -1.33, -1.00, -0.10, 1.32 },
    { -1.82, -1.48, -1.04, -0.04, 1.39 },
    {  NOTV, -1.62, -1.08,  0.02, 1.46 },
    {  NOTV, -1.77, -1.16,  0.08, NOTV },
    {  NOTV,  NOTV, -1.25,  NOTV, NOTV },
    {  NOTV,  NOTV, -1.40,  NOTV, NOTV } };
  if (u10 <0.0 || kl<=0)  return -1.0;
  i10 = ((int)(10*u10+0.5))/10;
  if (i10 >= 9)  i10 = 8;
  k2 = (kl < 6) ? kl-1 : 4;
  k1 = (kl > 1) ? kl-2 : 0;
  for (i=i10; i>0; i--)
    if (tsg[i][k1]>NOTV || tsg[i][k2]>NOTV)  break;
  if (tsg[i][k1] <= NOTV)  dt100 =  tsg[i][k2] - 0.05;
  else  if (tsg[i][k2] <= NOTV)  dt100 = tsg[i][k1] + 0.05;
        else {
          dt100 = 0.5*(tsg[i][k1] + tsg[i][k2]);
          if (k1 == 4)  dt100 += 0.05;
          if (k2 == 0)  dt100 -= 0.05;
          }
  return dt100; 
}

//=============================================================== BlmStability
//
long BlmStability(    // Berechnung von Mischungsschichthoehe und Stab.-Klasse
BLMPARM *p )          // Parameter der Grenzschicht.
  {
  dP(BlmStability);
  float lmo, hm, f1, f2, z0, d0, u10, h10=10, h, ha, ua, z1, z2, hr;
  int cl, kl=0, k, kk;
  vLOG(4)("BLM:BlmStability(...)");
  if (!p)                                         eX(1);
  lmo = p->MonObLen;
  hm  = p->MixDpt;
  z0  = p->RghLen;
  d0  = p->ZplDsp;
  ha  = p->AnmHeight;
  ua  = p->WndSpeed;
  cl  = 0.001+10*p->Class;
  if (lmo==0 && cl<=0) {     // missing data    -2002-04-19
    p->cl  = 3.1;
    p->lmo = 99999;
    p->hm  = 800;
    p->kta = 4;
    p->ths = 0;
    return 0;
  }
  if (lmo == 0.0) {          // unknown Obukhov length
    if (cl!=50 && cl!=40 && cl!=32 && cl!=31 && cl!=20 && cl!=10)
      cl = 31;
    kk = (cl + 10*(cl>31))/10;
    if (z0 <= z0TAL[0]) lmo = lmTAL[kk-1][0];
    else {
      for (k=1; k<nz0TAL; k++)
        if (z0>z0TAL[k-1] && z0<=z0TAL[k]) break;
      if (k > nz0TAL-1) lmo = lmTAL[kk-1][nz0TAL-1];
      if (lmo == 0.0) {
        z1 = log(z0TAL[k-1]);
        z2 = log(z0TAL[k]);
        f1 = 1./lmTAL[kk-1][k-1];
        f2 = 1./lmTAL[kk-1][k];
        lmo = 1./(f1 + (log(z0)-z1)/(z2-z1)*(f2-f1));
      }
    }
  }
  if (cl <= 0) {            // unknown Klug/Manier stability class
    if (z0 <= z0TAL[0]) k = 0;
    else if (z0 > z0TAL[nz0TAL-1]) k = nz0TAL-1;
    else
      for (k=1; k<nz0TAL; k++)
        if (z0>z0TAL[k-1] && z0<=z0TAL[k]) break;
    if (lmo == 0) cl = 31;
    else {
      f1 = 1./lmo;
      for (kk=5; kk>0; kk--) {
        f2 = (lmTAL[kk][k]+lmTAL[kk-1][k])/(2*lmTAL[kk][k]*lmTAL[kk-1][k]);
        if (f1 < f2) break;
      }
      switch (kk) {
        case 0:  cl = 10;  break;
        case 1:  cl = 20;  break;
        case 2:  cl = 31;  break;
        case 3:  cl = 32;  break;
        case 4:  cl = 40;  break;
        case 5:  cl = 50;  break;
        default: cl = 31;
      }
    }
  }
  if (kl <= 0)  kl = 7 - (cl + 10*(cl>31))/10;
  //
  // set Coriolis parameter and mixing layer
  Fcor = BLMFCOR;
  if (p->Fcor != NOTV)
    Fcor = p->Fcor;
  else
    p->Fcor = Fcor;
  HmNeutral = BLMDEFHM*BLMFCOR/fmax(BLMFCORMIN, fabs(Fcor));
  if (HmMeanAdjust) { // HmMean not user provided
    float diff40 = Hm40 - Hm32;
    float diff50 = Hm50 - Hm32;
    Hm32 = HmNeutral;
    Hm40 = Hm32 + diff40;                                         //-2018-12-31
    Hm50 = Hm32 + diff50;                                         //-2018-12-31
    HmMeanAdjust = 0;
    if (p->ZgMean > 0) {                                          //-2018-03-21
      Hm32 += p->ZgMean;
      Hm40 += p->ZgMean;
      Hm50 += p->ZgMean;
    }
    vLOG(2)(_hm_array_absolute_$$$$$$_, Hm10, Hm20, Hm31, Hm32, Hm40, Hm50);
    vLOG(2)(_hm_array_relative_$$$$$$_, 
            Hm10, Hm20, Hm31, Hm32-p->AnmZpos, Hm40-p->AnmZpos, Hm50-p->AnmZpos);
  }
  //
  // unknown mixing layer height
  if (hm<=0.0 && lmo<0) {
    TMBROKEN bdat, bact;
    TMDATE rdat, ract;
    int month;
    switch ( cl ) {
      case 10:  hm = Hm10;  break;
      case 20:  hm = Hm20;  break;
      case 31:  hm = Hm31;  break;
      case 32:  hm = Hm32;  break;
      case 40:  hm = Hm40;  break;
      case 50:  hm = Hm50;  break;
      default:  hm = Hm31;
    }   
    //                                                             //-2018-10-04
    bdat = DefGetRefDate();
    rdat = TmGetDate(bdat);
    ract = TmShiftDate(rdat, MetT2);
    bact = TmBreakDate(ract, bdat.tzmin);
    month = bact.month;
    if (is_equal(Fcor/BLMFCOR, 1.) &&   // default Coriolis parameter
        !SkipHm &&                      // no pseudo time series
        (month > 0 && month < 13) &&    // ref date set
        (cl == 40 || cl == 50)) {       // class IV/V according to VDI 3783/8 (2017)
      float old = hm;
      hm += 300.*sin((PI/6.)*month - 0.5*PI);
      vLOG(3)(_hm_seasonal_$$_, month, old, hm);
    }
    //
  }
  if (lmo < 0 && hm > 0 && (hm-p->AnmZpos) < 400.0) {              //-2019-01-14
    vLOG(1)(_hm_unstable_small_, lmo, hm, p->AnmZpos);
  }
  u10 = 3.0;
  hr = d0 + 2*z0;
  h = (ha < hr) ? (hr-d0)*(ha/hr) + (hr-ha)*(z0/hr) : ha-d0;     //-2015-01-15
  if (h10 > z0 && h > z0 && z0 > 0.)
    u10 = ua*log(h10/z0)/log(h/z0);
  p->lmo = lmo;
  p->cl  = 0.1*cl;
  p->kta = kl;
  p->hm  = hm;
  if (p->PtmpGrad > NOTV) {
    p->ThetaGrad = p->PtmpGrad;
  }
  else {
    p->ThetaGrad = 0.01*(1.0 + KTA1508T7i2(u10, kl));
  }
  p->U10 = u10;
  return 0;
eX_1:
  eMSG(_undefined_parameters_);
  }

//==================================================================== BlmRead
//
#define GETDATA(n,f,a)  {rc=GetData(#n,buf,#f,a); if(rc<0){strcpy(name,#n); eX(99);}}

static long BlmRead(   /* File WETTER.DEF einlesen und Daten uebernehmen. */
char *altname )
  {
  dP(BlmRead);
  long rc, id, pos;
  float v, za, vv[6];
  char name[40], *buf;
  int sgn;
  ARYDSC *pa;
  vLOG(4)("BLM:BlmRead(%s)", altname);
  if ((!Pgp) || (!Pga))                                                 eX(1);
  buf = ALLOC(GENHDRLEN);  if (!buf)                                    eX(21);
  id = IDENT(BLMpar, 0, 0, 0);
  pa = TmnCreate(id, sizeof(BLMPARM), 1, 0, 0);                         eG(2);
  BlmPprm = (BLMPARM*) pa->start;
  TmnDetach(id, NULL, NULL, TMN_MODIFY|TMN_FIXED, NULL);                eG(13);
  PrfMode = Pgp->prfmode;
  vLOG(3)(_reading_$_, InpName);
  OpenInput(InpName , altname );                                        eG(3);
  rc = GetLine('.', buf, GENHDRLEN);  if (rc <= 0)                      eX(4);
  pos = CloseInput();
  vLOG(3)(_$_closed_, InpName);
  BlmPprm->MetVers = BLMDEFVERS;
  GETDATA(Version, %f, &v);
  if (rc > 0) BlmPprm->MetVers = 0.001 + 10.0*v;
  BlmPprm->AvrTime = BLMINTDEF;
  GETDATA(INTERVAL, %f, &BlmPprm->AvrTime);
  BlmPprm->ZplDsp = 0.0;
  GETDATA(D0, %f, &BlmPprm->ZplDsp);
  BlmPprm->AnmXpos = 0.0;
  GETDATA(XA, %f, &BlmPprm->AnmXpos);
  BlmPprm->AnmYpos = 0.0;
  GETDATA(YA, %f, &BlmPprm->AnmYpos);
  rc = GrdBottom(BlmPprm->AnmXpos, BlmPprm->AnmYpos, &za);
  if (rc < 0)                                                           eX(9);
  BlmPprm->AnmZpos = za;
  BlmPprm->AnmGridNumber = rc;
  //
  // Coriolis parameter and mean mixing height                    //-2018-10-04
  BlmPprm->Fcor = NOTV;
  GETDATA(FCOR, %f, &BlmPprm->Fcor); 
  v = NOTV;
  GETDATA(GLAT, %f, &v);
  if (v != NOTV && BlmPprm->Fcor == NOTV)
    BlmPprm->Fcor = 2*7.29e-5*sin(v*PI/180.);
  if (BlmPprm->Fcor != NOTV) {
    sgn = (BlmPprm->Fcor<0) ? -1 : 1;
    if (fabs(BlmPprm->Fcor) > BLMFCORMAX)
      BlmPprm->Fcor = sgn*BLMFCORMAX;
    if (fabs(BlmPprm->Fcor) < 1.e-12)
      BlmPprm->Fcor = sgn*1.e-12; 
  }
  BlmPprm->ZgMean = 0;                                            
  GETDATA(ZGMEAN, %f, &BlmPprm->ZgMean);
  HmMeanAdjust = 1;
  //
  strcpy(name, "HmMean");
  rc = GetData("HmMean", buf, "%[6]f", vv);  if (rc < 0)                eX(99);
  if (rc == 6) {
    Hm10 = vv[0];
    Hm20 = vv[1];
    Hm31 = vv[2];
    Hm32 = vv[3];
    Hm40 = vv[4];
    Hm50 = vv[5];
    HmMeanAdjust = 0;
  }
  else if (rc > 0)                                                      eX(99); 
  //                                       
  GETDATA(HA, %f, &BlmPprm->AnmHeight);
  DefParm( "HW",  buf, "%f", &BlmPprm->AnmHeightW, "-999", &BlmPvpp ); //-2012-11-05
  DefParm( "UA",  buf, "%f", &BlmPprm->WndSpeed, "-999", &BlmPvpp );
  DefParm( "RA",  buf, "%f", &BlmPprm->WndDir,   "-999", &BlmPvpp );
  DefParm( "Us",  buf, "%f", &BlmPprm->Ustar,    "-999", &BlmPvpp );
  DefParm( "LM",  buf, "%f", &BlmPprm->MonObLen,  "0.0", &BlmPvpp );
  DefParm( "Z0",  buf, "%f", &BlmPprm->RghLen,   BLMSZ0, &BlmPvpp );
  DefParm( "D0",  buf, "%f", &BlmPprm->ZplDsp,    "0.0", &BlmPvpp );
  DefParm( "MS",  buf, "%f", &BlmPprm->RezMol,   "-999", &BlmPvpp );
  DefParm( "HM",  buf, "%f", &BlmPprm->MixDpt,    "0.0", &BlmPvpp );
  DefParm( "KL",  buf, "%f", &BlmPprm->Class,     "0.0", &BlmPvpp );
  DefParm("PREC", buf, "%f", &BlmPprm->Precep,   "-1.0", &BlmPvpp ); //-2011-11-21
  DefParm( "SG",  buf, "%f", &BlmPprm->StatWeight,"1.0", &BlmPvpp );
  DefParm( "SU",  buf, "%f", &BlmPprm->SigmaU,   "-999", &BlmPvpp );
  DefParm( "SV",  buf, "%f", &BlmPprm->SigmaV,   "-999", &BlmPvpp );
  DefParm( "SW",  buf, "%f", &BlmPprm->SigmaW,   "-999", &BlmPvpp );
  DefParm( "TS",  buf, "%f", &BlmPprm->PtmpGrad, "-999", &BlmPvpp ); 
  DefParm( "TA",  buf, "%f", &BlmPprm->Ta,     "  10.0", &BlmPvpp ); //-2018-10-04
  DefParm( "RH",  buf, "%f", &BlmPprm->Rh,     "  70.0", &BlmPvpp ); //-2018-10-04
  GETDATA(WindLib,  %s, BlmPprm->WindLib);
  MsgCheckPath(BlmPprm->WindLib);
  DefParm("WIND", buf, "%d", &BlmPprm->Wind, "-999", &BlmPvpp);
  DefParm("INIT", buf, "%d", &BlmPprm->Wini, "-999", &BlmPvpp);
  DefParm("DIFF", buf, "%d", &BlmPprm->Diff, "-999", &BlmPvpp);
  DefParm("TURB", buf, "%d", &BlmPprm->Turb, "-999", &BlmPvpp);         eG(11);
  strcpy(buf, "@");
  if (BlmPprm->WindLib[0] == '~') {
    strcat(buf, StdPath);
    strcat(buf, "/");                                 //-2002-04-28
    strcat(buf, BlmPprm->WindLib+1);
  }
  else strcat(buf, BlmPprm->WindLib);
  MsgCheckPath(buf);                                  //-2002-04-28    
  strcat(buf, "/");                                   //-2002-04-28
  vLOG(5)("wind library in %s", buf);
  NmsSeq(buf, 0);
  FREE(buf);
  buf = NULL;
  /*
  //                                                              //-2018-10-04
  if (TmGetDate(DefGetRefDate()) >= TM_MAX_DATE || 
      (BlmPprm->MetVers != 52 && BlmPprm->MetVers != 53) ||
      BlmPprm->StatWeight != 1)  
    SkipHm = 1;
  if ((BlmPprm->MetVers == 52 || BlmPprm->MetVers == 53) && SkipHm && )
  	vLOG(1)(_seasonal_hm_skipped_);
  //
  */
  return pos;
eX_99:
  eMSG(_error_definition_$_, name);
eX_1:
  eMSG(_no_grid_);
eX_2:  eX_13:
  eMSG(_cant_create_blmpar_);
eX_3:
  eMSG(_cant_open_$_, InpName);
eX_4:
  eMSG(_invalid_structure_$_, InpName);
eX_9:
  eMSG(_anemometer_$$_outside_, BlmPprm->AnmXpos, BlmPprm->AnmYpos);
eX_11:
  eMSG(_syntax_error_);
eX_21:
  eMSG(_cant_allocate_);
  }

#undef  GETDATA

//================================================================= BlmSrfFac
//
float BlmSrfFac(        /* factor to impose surface layer */
float z0,               /* roughness length               */
float d0,               /* zero plane displacement        */
float h )               /* height above ground            */
  {
  float uh, z, u, hh;
  if (z0<=0 || d0<0)    return 1;
  if (h<0 || h>200)    return 1;
  hh = d0 + 6*z0;
  z = 200-d0;                                                 //-2002-04-02
  uh = log(z/z0);
  z = (h < hh) ? hh-d0 : h-d0;
  u = log(z/z0)/uh;
  if (h < hh)  u *= h/hh;
  return u;
  }

//=================================================================== Clc1dPrf
//
static long Clc1dPrf(   // Grenzschicht-Profil berechnen.
long t1, long t2 )
  {
  dP(Clc1dPrf);
  int k, nk, old_gl, old_gi, anm_gn;
  BLMREC *pm;
  float s, z0, d0, d, h, g, xa, ya, za;
  TXTSTR hdr = { NULL, 0 };                                       //-2011-06-29
  TXTSTR t = { NULL, 0 };
  long id;
  vLOG(4)("BLM:Clc1dPrf(...)");
  if ((!BlmPprm) || (!Pgp) || (!Pga))                                 eX(1);
  nk = Pga->bound[0].hgh;
  id = IDENT(BLMarr, 0, 0, 0);
  BlmParr = TmnCreate(id, sizeof(BLMREC), 1, 0, nk);                  eG(6);
  TxtCpy(&hdr, "\n");
  TxtPrintf(&t, "prgm  \"TALBLM_%d.%d.%s\"\n", StdVersion, StdRelease, StdPatch);
  TxtCat(&hdr, t.s);
  TxtCat(&hdr, "form  \"Z%6.1fU%6.2fG%6.2fD%5.0fSu%[3]5.2fTu%[3]5.0fSuw%5.2fThs%7.3f\"\n");
  old_gl = Pgp->level;
  old_gi = Pgp->index;
  anm_gn = BlmPprm->AnmGridNumber;
  xa = BlmPprm->AnmXpos;
  ya = BlmPprm->AnmYpos;
  za = BlmPprm->AnmZpos;
  if (BlmPprm->MonObLen==0 && BlmPprm->RezMol!=-999 && BlmPprm->RezMol!=0)
    BlmPprm->MonObLen = 1/BlmPprm->RezMol;
  if (BlmPprm->Class<=0 && BlmPprm->MonObLen==0) {                //-2001-06-29
    TmnDetach(id, &t1, &t2, TMN_MODIFY|TMN_INVALID, &hdr);          eG(4);
    TxtClr(&hdr);
    TxtClr(&t);
    return 0;
  }
  BlmStability(BlmPprm);
  BlmPprm->UsgCalc = -999;
  BlmPprm->UstCalc = -999;
  z0 = BlmPprm->RghLen;
  d0 = BlmPprm->ZplDsp;
  u2lInit = -1;                                                   //-2018-10-04
  BlmPprm->u2lH1 = NOTV;
  BlmPprm->u2lK2 = NOTV;
  BlmPprm->u2lFa = NOTV;
  BlmPprm->u2lUg = NOTV;
  GrdSetNet(anm_gn);                                                eG(7);
  for (k=0; k<=nk; k++) {
    s = *(float*) AryPtrX(Pga, k);
    h = GrdHh(xa, ya, s);
    pm = AryPtr(BlmParr, k);  if (!pm)                              eX(3);
    pm->z = h + za;
    }
  GrdSet(old_gl, old_gi);                                           eG(8);
  for (k=0; k<=nk; k++) {
    pm = AryPtr(BlmParr, k);  if (!pm)                              eX(3);
    h = pm->z - za;
    BlmProfile(BlmPprm, pm);                                        eG(5);
    if (k > 0)  pm->g = pm->u/BlmSrfFac(z0, d0, h);
    if (k == 1) {
      g = pm->g;
      d = pm->d;
      pm = AryPtrX(BlmParr, 0);
      pm->g = g;
      pm->d = d;
      }
    }
  TmnDetach(id, &t1, &t2, TMN_MODIFY|TMN_SETVALID, &hdr);           eG(4);
  TxtClr(&hdr);
  TxtClr(&t);
  return 0;
eX_1:
  eMSG(_undefined_grid_);
eX_3:  eX_4:  eX_6:
  eMSG(_cant_create_profile_);
eX_5:
  eMSG(_cant_calculate_profile_);
eX_7:  eX_8:
  eMSG(_cant_switch_grids_);
  }

//==================================================================== BlmInit
//
long BlmInit(           /* initialize server    */
long flags,             /* action flags         */
char *istr )            /* server options       */
  {
  dP(BlmInit);
  long id, mask;
  char *jstr, *ps;
  if (StdStatus & STD_INIT)  return 0;
  if (istr) {
    jstr = istr;
    ps = strstr(istr, " -v");
    if (ps) sscanf(ps+3, "%d", &StdLogLevel);
    ps = strstr(istr, " -y");
    if (ps) sscanf(ps+3, "%d", &StdDspLevel);
    ps = strstr(istr, " -d");
    if (ps)  strcpy(DefName, ps+3);
    ps = strstr(istr, " -i");
    if (ps)  strcpy(InpName, ps+3);                                                                  
    if (strstr(istr, " -u"))                                      //-2018-10-04
      SkipHm = 1;
  }
  else  jstr = "";
  vLOG(3)("BLM_%d.%d.%s (%08lx,%s)", StdVersion, StdRelease, StdPatch, flags, jstr);
  StdStatus |= flags;
  mask = ~(NMS_LEVEL | NMS_GRIDN);
  id = IDENT(BLMarr, 0, 0, 0);
  TmnCreator(id, mask, 0        , istr, BlmServer, NULL);       eG(1);
  id = IDENT(BLMpar, 0, 0, 0);
  TmnCreator(id, mask, TMN_FIXED, istr, BlmServer, NULL);       eG(2);
  StdStatus |= STD_INIT;
  return 0;
eX_1:  eX_2:
  eMSG(_not_initialized_);
  }

/*=================================================================== BlmServer
*/
long BlmServer(
char *ss )
  {
  dP(BlmServer);
  long mask, rc, idpar, idarr, igrda, igrdp;
  int read_ztr, read_parm;
  ARYDSC *pa;
  enum DATA_TYPE dt;
  if (StdArg(ss))  return 0;
  if (*ss) {
    switch (ss[1]) {
      case 'd':  strcpy(DefName, ss+2);
                 break;
      default:   ;
      }
    return 0;
    }
  dt = XTR_DTYPE(StdIdent);
  Gl = XTR_LEVEL(StdIdent);
  Gi = XTR_GRIDN(StdIdent);
  igrdp = IDENT(GRDpar, 0, Gl, Gi);
  pa = TmnAttach(igrdp, NULL, NULL, 0, NULL);  if (!pa)         eX(10); //-2014-06-26
  Pgp = (GRDPARM*) pa->start;
  igrda = IDENT(GRDarr, 0, 0, 0);
  Pga = TmnAttach(igrda, NULL, NULL, 0, NULL);  if (!Pga)       eX(11); //-2014-06-26
  idarr = IDENT(BLMarr, 0, 0, 0);

  if (dt == BLMarr) {
    TmnDelete(TmMax(), idarr, 0);                               eG(24);
    }
  if (!BlmPprm) {
    MetT1 = TmMin();
    MetT2 = MetT1;
    ZtrT1 = MetT1;
    ZtrT2 = MetT1;
    }
  if (StdStatus & STD_TIME)  MetT1 = StdTime;
  read_parm = (MetT1 >= MetT2);
  rc = 1;
  if (read_parm) {
    MetT2 = TmMax();
    mask = -1;
    idpar = IDENT( BLMpar, 0, 0, 0 );
    TmnCreator(idpar, mask, TMN_FIXED, NULL, NULL, NULL);       eG(1);
    if (!BlmPprm) {
      PosZtr = BlmRead(DefName);                                eG(3);
      if (!BlmPprm)                                             eX(4);
      if (!BlmPvpp)  ZtrT2 = TmMax();
      }
    read_ztr = (BlmPvpp) && (ZtrT2 <= MetT1);
    ZtrT2 = MetT1;
    if (read_ztr) {
      rc = ReadZtr(InpName, DefName, &MetT1, &ZtrT2,
                  StdDspLevel, &PosZtr, BlmPvpp);
      if (rc < 0)                                                       eX(6);
      if (MetT2 > ZtrT2)  MetT2 = ZtrT2;
    }
    if (!rc) {
      MetT2 = MetT1;
      vLOG(3)(_no_data_after_$_, TmString(&MetT1));
    }
    TmnAttach(idpar,   NULL,   NULL, TMN_MODIFY, NULL);                 eG(20);
    TmnDetach(idpar, &MetT1, &MetT2, TMN_MODIFY, NULL);                 eG(21);
    TmnCreator(idpar, mask, TMN_FIXED, NULL, BlmServer, NULL);          eG(9);
    }

  dt = XTR_DTYPE(StdIdent);
  if (dt == BLMarr) {
    mask = ~(NMS_LEVEL | NMS_GRIDN);
    if (rc) {
      Clc1dPrf(MetT1, MetT2);                                           eG(8);
      }
    else {
      TmnCreate(idarr, 0, 0);                                           eG(7);
      TmnDetach(idarr, &MetT1, &MetT2, TMN_MODIFY, NULL);               eG(7);
      }
    }
  TmnDetach(igrdp, NULL, NULL, 0, NULL);                                eG(10);
  TmnDetach(igrda, NULL, NULL, 0, NULL);                                eG(11);
  return 0;
eX_1:  eX_9:
  eMSG(_cant_redefine_);
eX_3:  eX_4:
  eMSG(_cant_get_parameters_);
eX_6:
  eMSG(_error_reading_$_, InpName);
eX_7:
  eMSG(_cant_create_null_);
eX_8:
  eMSG(_error_model_);
eX_10: eX_11:
  eMSG(_no_grid_);
eX_20: eX_21:
  eMSG(_cant_set_times_);
eX_24:
  eMSG(_cant_delete_$_, NmsName(idarr));
  }

#ifdef MAIN  /*############################################################*/

#include <signal.h>

static void MyExit( void ) {
  if (MsgCode < 0) {
    vMsg("@%s error exit", StdMyProg);
    TmnList(MsgFile);
  }
  else vMsg("@%s finished", StdMyProg);
  TmnFinish();
  if (MsgFile) {
    fprintf(MsgFile, "\n\n\n");
    fclose(MsgFile);
  }
  MsgFile = NULL;
  MsgSetQuiet(1);
}

static void MyAbort( int sig_number )
  {
  FILE *prn;
  MsgSetQuiet(0);
  vMsg("@%s aborted", StdMyProg);
  if (sig_number != 4) {
    prn = (MsgFile) ? MsgFile : stderr;
    fflush(prn);
    fprintf(prn, "\nABORT, SIGNAL %x\n", sig_number);
  }
  if (MsgFile) {
    fprintf(MsgFile, "\n\n");
    fclose(MsgFile);
  }
  MsgFile = NULL;
  MsgSetQuiet(1);
  exit(0);
  }

static long BlmMain(void)
  {
  dP(BlmMain);
  ARYDSC *pa;
  GRDPARM *pp;
  long id, m, t1, t2;
  char istr[80];
  TXTSTR hdr = { NULL, 0 };
  MsgSetVerbose(1);
  MsgSetQuiet(0);
  MsgBreak = '\'';
  if ((StdStatus&STD_ANYARG) == 0) {
    printf("usage: TSTblm <path>\n");
    exit(0);
  }
  atexit(MyExit);
  signal(SIGSEGV, MyAbort);
  signal(SIGINT,  MyAbort);
  signal(SIGTERM, MyAbort);
  signal(SIGABRT, MyAbort);
  if (StdStatus & STD_TIME)  t1 = StdTime;
  else  t1 = TmMin();
  t2 = t1;
  m = TMN_DISPLAY + TMN_LOGALLOC + TMN_LOGACTION + TMN_DONTZERO;
  TmnInit( StdPath, NmsName, NULL, m, MsgFile );                        eG(1);
  sprintf(istr, " lstgrd -v%d -y%d -d%s", StdLogLevel, StdDspLevel, DefName);
  GrdInit(0, istr);                                                     eG(2);
  sprintf(istr, " lstblm -v%d -y%d -d%s", StdLogLevel, StdDspLevel, DefName);
  BlmInit(0, istr);                                                     eG(3);
  id = IDENT(GRDpar, 0, 0, 0);
  pa = TmnAttach(id, NULL, NULL, 0, &hdr );                             eG(4);
  if (!pa)                                              eX(11);
  pp = (GRDPARM*) pa->start;
  TmnDetach(id, NULL, NULL, 0, NULL);
  id = IDENT(BLMarr, 0, 0, 0);
  TmnAttach(id, &t1, &t2, 0, &hdr );                    eG(5);
  TmnDetach(id, NULL, NULL, 0, NULL);                   eG(6);
  TmnArchive(id, "blmtest.dmna", 0);                    eG(20);
  TmnList(MsgFile);                                     eG(9);
  t1 = t2;
  TmnAttach(id, &t1, &t2, 0, &hdr );                    eG(7);
  TmnDetach(id, NULL, NULL, 0, NULL);                   eG(8);
  TmnArchive(id, "blmtest.dmna", 1);                    eG(21);
  TmnList(MsgFile);                                     eG(9);
  TxtClr(&hdr);
  return 0;
eX_1:  eX_2:  eX_3:  eX_4:  eX_5:  eX_6:  eX_7:  eX_8:  eX_9:
eX_11:
eX_20: eX_21:
  TmnList(stdout);
  eMSG("error!");
  }
#endif  /*##################################################################*/

/*==============================================================================
 * history:
 * 
 * 2002-09-24 lj 1.0.0  final release candidate
 * 2004-10-25 lj 2.0.4  optional factors Ftv  
 * 2005-03-17 uj 2.2.0  version number upgrade
 * 2006-02-07 uj 2.2.7  error if ua less equal 0 for determination of u*
 * 2006-02-13 uj 2.2.8  new svmin, tvmin (blm versions 3.6, 3.8)
 *                      error if unknown blm version encountered
 * 2006-10-26 lj 2.3.0  external strings
 * 2007-01-30 uj 2.3.4  Svf
 * 2011-06-29 uj 2.5.0  DMN header
 * 2011-07-07 uj        profile versions 4.6 and 4.8 (A2K option PRFMOD)
 * 2011-11-21 lj 2.6.0  precipitation
 * 2012-11-05 uj 2.6.5  anemometer height hw for base field superposition
 * 2014-06-26 uj 2.6.11 eG/eX adjusted
 * 2018-10-04 uj 3.0.0  BlmTemp()
 * 2021-09-21 uj 3.0.1  adjust ra>=360 to ra<360
 *
 *============================================================================*/

