/**
 * Plume Rise Model PLURIS
 *
 * For theory see: 
 * Janicke, U., Janicke, L.: A three-dimensional plume rise model for
 * dry and wet plumes, Atmospheric Environment 35 (2000), 877-890.
 *
 * Copyright (C) Janicke Consulting, Überlingen, Germany, 2015-2024
 * email: info@janicke.de
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 * 
 */


////////////////////////////////////////////////////////////////////////////////
//
// history
//
// 2015-06-01 uj  3.0.1   imported from C version 2.7, several adjustments
// 2015-08-10 uj  3.0.2   adjusted output 
// 2015-08-16 uj  3.0.3   work without input file, break-on-rise adjusted
// 2015-09-04 uj  3.0.4   break-on-rise adjusted
// 2015-09-20 uj  3.0.5   adjustments, breaks facs = 0.01, 0.7, 0.5, 1; -0.008
// 2015-12-14 uj  3.0.6   option stacktip downwash
// 2015-12-30 uj  3.0.7   final rise criterium w < <s>, setting of Ts
// 2016-01-06 uj  3.0.8   adjusted stacktip downwash 
// 2016-01-25 uj  3.0.9   adjusted temperature profile from BLProfile2Layer (1.4)
// 2016-01-28 uj  3.0.10  final rise criterium w < f*ust
// 2016-06-27 uj  3.0.11  optionally user-defined Gamma (K/m)
// 2016-06-28 uj  3.0.12  optionally constant ra (ambient relative humidity)
// 2016-07-12 uj  3.0.13  by default constant ra and gamma = -0.008, break-factor=2
// 2016-07-19 uj  3.0.14  by default constant ra and gamma = -0.0085, break-factor=1.7
// 2016-07-20 uj  3.0.15  by default A2K vertical grid
// 2016-08-02 uj  3.0.16  optional new temp gradient gu above height hu
// 2017-01-08 uj  3.0.17  by default break-factor=1.5
// 2017-02-07 uj  3.0.18  by default break-factor=1.3
// 2017-10-13 uj  3.0.19  by default use max(0.05m/s, ust) for break condition
// 2018-02-07 uj  3.0.20  save final rise point as well without stop at final rise
// 2018-03-24 uj  3.0.21  additional blank in DMNA output
// 2018-09-08 uj  3.0.22  fcor provided as input parameter
// 2018-09-10 uj  3.1.01  parameter zq instead of sq, handle wet plumes for t > 100 C
// 2019-01-15 uj  3.1.02  ust constant, ra constant options (default true)
// 2019-02-12 uj  3.1.03  ust constant, ra constant options (default false)
// 2019-03-02 uj  3.1.04  security checks added
// 2019-10-15 uj          (no version change) dump A2 with correct value
// 2020-09-11 uj  3.1.05  explicit reduction factor rf (source parameter)
// 2021-07-05 uj  3.1.06  VDI 3782/3-old output in DMNA header corrected
// 2023-03-15 uj  3.1.07  range check for FC, avoid FC=0
// 2024-01-17 uj  3.1.08  small adjustments, no change in calculation
// 2024-01-17 uj  3.2.00  revised reduction factor rf according to VDI 3782/3 (E5)
//                        static public functions for use with BESTAL
//                        JAVA 21
//                        option --no-shear
//
////////////////////////////////////////////////////////////////////////////////


package de.janicke.pluris;

import de.janicke.graph.Graph;
import de.janicke.graph.Graph.Diagram;
import de.janicke.graph.Graph.Element;
import de.janicke.ibjutil.IBJarr;
import de.janicke.ibjutil.IBJarr.AbstractArray;
import de.janicke.ibjutil.IBJdmn;
import de.janicke.meteo.BLProfile2Layer;
import de.janicke.pluris.IBJparamSet.Param;
import de.janicke.pluris.IBJparamSet.PTYPE;
import java.util.Locale;
import static de.janicke.pluris.IBJplurisHeader.*;
import java.awt.BasicStroke;
import java.awt.Color;
import java.awt.Font;
import java.awt.geom.Rectangle2D;
import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.PrintWriter;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.logging.Level;

/**
 *
 * @author Janicke Consulting
 */
public class IBJpluris {
  
  private static final String PRGNAME = "IBJpluris";
  public static final String VERSION = "3.2.0";
  private static final String PARAMSETFILE = "IBJpluris-3.2-paramset.txt";
  private static final String LOGFILE = "IBJpluris";
  private static final String PRMFILE = "IBJpluris";
  private static final String OUTFILE = "IBJpluris";
  private static final String GRPFILE = "IBJpluris";
  private static final String PRFFILE = "profiles";
  private String LogFile, PrmFile, OutFile, GrpFile, PrfFile;
  public static Level LogLevel = Level.FINEST;
  public static int Verbose = 2;
  public static boolean DEBUG = true;
  public static PrintWriter Printer = new PrintWriter(System.out);  //-2015-11-23
  
  private static double[] vzz = new double[] { 
    0, 3, 6, 10, 16, 25, 40, 65, 100, 150, 200, 300, 400, 500, 
    600, 700, 800, 1000, 1200, 1500 };
  
  private SimpleDateFormat Sdf = new SimpleDateFormat("yyyy-MM-dd.HH:mm:ss ");
  private BufferedWriter bwLog = null;
  private File fLog = null;
  private boolean tmp_initialized;
  private static boolean redirected = false;                      //-2015-11-23
  
  private static final int NSTORE = 500;
  private static final double DSTORE = 1.;
  
  private static final double USTTIME = 120.;                     //-2019-01-15
  
  private static final double ZGAMU = 200;
  private static final double GAMU = -0.0085;
  private static final double BREAKFACTOR = 1.3;                  //-2017-02-07
  private static final double BREAKZMAX = 800.;
  
  public static final double TZERO = 273.15;          // 0 deg Celsius
  private static final double TZEROT = 273.16;        // 0.01 deg Celsius
  private static final double GRAV = 9.8066;          // gravitational acceleration (m/s2)
  private static final double RGAS = 8.314472;        // gas constant (J/(mol K))
  private static final double MDRY = 28.96546e-3;     // molar mass of dry air (kg/mol)
  private static final double MVAPOUR = 18.01528e-3;  // molar mass of vapour (kg/mol)
  private static final double RDRY = RGAS/MDRY;       // gas constant of dry air
  private static final double RVAPOUR = RGAS/MVAPOUR; // gas constant of water vapour
  private static final double LHEAT = 2454300.0;      // latent heat (J/kg) 
  private static final double CPRESSD = 1004.1;       // specific heat capacity at constant pressure dry air (J/(kgK))
  private static final double CPRESSV = 1863.0;       // specific heat capacity at constant pressure water vapour (J/(kgK))
  private static final double CPRESSL = 4191.0;       // specific heat capacity at constant pressure liquid water (J/(kgK))
  private static final double CFAC = 3.;              // factor to convert top-heat to Gaussian profile
  private static final double ETR_ALPHA = 0.15;       // entrainment constant alpha
  private static final double ETR_BETA = 0.6;         // entrainment constant beta
  private static final double ETR_GAMMA = 0.38;       // entrainment constant gamma
  private static final double MIN_UST = 0.05;         // minimum ust for break condition
  public static final double RVORD = RVAPOUR/RDRY;
  private static final double LN10 = Math.log(10.);
  
  private double ZGamU = ZGAMU;
  private double GamU = GAMU;
  private double ZmaxFinal = BREAKZMAX;
  private double fFinal = BREAKFACTOR;
  private double hFinal;
  private double xFinal; 
  private double[] tstore, hstore;                                 //-2015-12-30
   
  private double ds;   // calculation step
  private double dp;   // output step
  private double[] zv; // ambient profiles, mesh (m)
  private double[] uv; // ambient profiles, wind speed (m/s)
  private double[] dv; // ambient profiles, direction (against north, rad)
  private double[] xv; // ambient profiles, vx (m/s)
  private double[] yv; // ambient profiles, vx (m/s)
  private double[] wv; // ambient profiles, vz (m/s)
  private double[] iv; // ambient profiles, turbulence intensity (1)
  private double[] jv; // ambient profiles, velocity fluctuations (m/s)
  private double[] tv; // ambient profiles, temperature (K)
  private double[] pv; // ambient profiles, pressure (Pa)
  private double[] rv; // ambient profiles, relative humidity (1)
  private double[] lv; // ambient profiles, specific liquid water content (kg/kg)
  private double[] qv; // ambient profiles, specific humidity (kg/kg)
  private double[] cv; // ambient profiles, concentration (MU/m3)                          
  private double[] av; // ambient profiles, density (kg/m3)
  private double xx;   // plume, x-coordinate (m)
  private double yy;   // plume, y-coordinate (m)
  private double zz;   // plume, z-coordinate (m)
  private double ss;   // plume, s-coordinate (path along the plume axis) (m)
  private double ll;   // plume, l-coordinate (path along the horizontal projection) (m)
  private double zt;   // plume, transport time (s)
  private double[] kk; // plume, cartesian velocity components (m/s)
  private double uu;   // plume, average velocity (m/s)
  private double jj;   // plume, turbulent velocity fluctuations (m/s)
  private double tt;   // plume, temperature (K)
  private double dd;   // plume, density (kg/m3)
  private double cc;   // plume, concentration (MU/m3)
  private double qi;   // plume, specific humidity (kg/kg)
  private double ei;   // plume, specific liquid water content (kg/kg)
  private double zi;   // plume, specific total water content (kg/kg)
  private double rr;   // plume, radius (m)
  private double aa;   // plume, visible radius (m)
  private double w1;   // plume, angle to the horizontal (rad)
  private double w2;   // plume, angle against north (rad)
  private double[] kl; // ambient at plume height, cartesian velocity components (m/s)
  private double ul;   // ambient at plume height, wind speed (m/s)
  private double jl;   // ambient at plume height, turbulent velocity fluctuations (m/s)
  private double tl;   // ambient at plume height, temperature (K)
  private double dl;   // ambient at plume height, density (kg/m3)
  private double cl;   // ambient at plume height, concentration (MU/m3)
  private double ql;   // ambient at plume height, specific humidity (kg/kg)
  private double el;   // ambient at plume height, specific liquid water content (kg/kg)
  private double zl;   // ambient at plume height, specific total water content (kg/kg)
  private double pl;   // ambient at plume height, pressure (Pa)
  
  private double km;    // Klug/Manier stability class (derived)
  private int    ki;    // Klug/Manier stability class index (derived from lm)
  private double lm;    // Monin-Obukhov length (m) (derived from ki)
  private double hm;    // hm (m) internally set
  private double lenm;  // Briggs length lm (m) (internal)
  private double lenb;  // Briggs length lb (m) (internal)
  private double uH;    // wind speed at source height (m/s) (internal)
  private double ust;   // friction velocity (m/s) (either from UST or derived)
  
  private String Dir = null;
  private boolean ignoreProfiles = false;
  private boolean stopAtFinalRise = false;
  private boolean stopAtGround = false;
  private boolean useSimProfiles = false;
  private boolean paintExtended = false;
  private boolean skipGraph = false;
  private boolean skipDMN = false;
  private boolean skipLog = false;
  private boolean skipStacktipDownwash = false;
  private boolean useConstantRa = false;
  private boolean useConstantUst = false;
  private boolean useNoShear = false;                             //-2024-03-19
  
  private ArrayList<String> vparamdef;
  private ArrayList<PlrOutput> vout;
  private ArrayList<String> vcmd;
  
  //
  // result for a particle model
  private double ptlVq, ptlTs, ptlRed;
  
  public static void setPrinter(PrintWriter pw) {                 //-2015-11-23
    Printer = pw;
    redirected = true;
  }
  
  private class PlrOutput {
    double xx, yy, zz, ll, ss, ww, zt, uu, rr, aa, jj, tt, dd, cc;
    double qi, ei, zi, ri, ee, ek;
    double[] kk;
    double hvd, xvd, hbr, hbm, hbb;
  }
  
  public IBJpluris(String dir, String ext) {    
    Dir = dir;
    LogFile = LOGFILE+ext+".log";
    PrmFile = PRMFILE+ext+".txt";
    OutFile = OUTFILE+ext+".dmna";
    PrfFile = PRFFILE+ext+".dmna";
    GrpFile = GRPFILE+ext+".pdf";
    bwLog = null; 
  }
  
  public void reset() {
    if (vparamdef != null)
      vparamdef.clear();
    if (vout != null)
      vout.clear();
    if (vcmd != null)
      vcmd.clear();
    bwLog = null;
    fLog = null;
    zv = null;
    uv = null;
    dv = null;
    xv = null;
    yv = null;
    wv = null;
    iv = null;
    jv = null;
    tv = null;
    pv = null;
    rv = null;
    qv = null;
    cv = null;
    av = null;
    kk = null;
    kl = null;
    hFinal = -1;
    xFinal = -1;
    tstore = null;
    hstore = null;
  }
  
  private void addcmd(String s) {
    if (vcmd == null)
      vcmd = new ArrayList<String>();
    vcmd.add(s);
  }
  
  /**
   * Adds a parameter setting in the form parameter=value
   * @param s
   * @return 
   */
  public boolean addParameterDefinition(String s) {
    if (s == null)
      return false;
    addcmd("--"+s);
    boolean inq = false;
    StringBuilder sb = new StringBuilder();
    for (int i=0; i<s.length(); i++) {
      if (s.charAt(i) == '"')
        inq = !inq;
      if (inq && s.charAt(i) == ' ')
        sb.append('\t');
      else
        sb.append(s.charAt(i));
    }  
    String[] sa = sb.toString().trim().split("[ ]+");
    for (String ss1 : sa) { 
      int ig = ss1.indexOf("=");
      if (ig <= 0)
        return false;    
      if (vparamdef == null)
        vparamdef = new ArrayList<String>();
      vparamdef.add(ss1.substring(0,ig) + ";" + unquote(ss1.substring(ig+1).replace('\t', ' ')));
    }
    return true;
  }

  public void setGamU(double z, double g) {
    ZGamU = z;
    GamU = g;
  }
  
  public double getBreakFactor() {
    return fFinal;
  }
  
  public void setBreakFactor(double f) {
    if (f >= 0) fFinal = f;
  }
  
  public void setSkipGraph(boolean b) {
    skipGraph = b;
  }
  
  public void setSkipDMN(boolean b) {
    skipDMN = b;
  }
  
   public void setSkipLog(boolean b) {
    skipLog = b;
  }
   
  public double getUh() {
    return uH;
  }
  
   public double getUst() {
    return ust;
  }
  
  public void setPaintExtended(boolean b) {
    paintExtended = b;
  }
  
  public void setUseSimProfiles(boolean b) {
    useSimProfiles = b;
  }
  
  public void setUseConstantUst(boolean b) {
    useConstantUst = b;
  }
  
  public void setUseConstantRa(boolean b) {
    useConstantRa = b;
  }
  
  public void setUseNoShear(boolean b) {
    useNoShear = b;
  }

  public void setSkipStacktipDownwash(boolean b) {
    skipStacktipDownwash = b;
  }
  
  public void setStopAtFinalRise(boolean b) {
    stopAtFinalRise = b;
  }
  
  public void setStopAtGround(boolean b) {
    stopAtGround = b;
  }
  
  public void setIgnoreProfiles(boolean b) {
    ignoreProfiles = b;
  }
  
  public void setVerbose(int i) {
    Verbose = i;
  }
  
  private static String unquote(String s) {
    if (s == null)
      return null;
    int l = s.length();
    if (l < 1 || s.charAt(0) != '\"')
      return s;
    if (s.charAt(l - 1) == '\"')
      l--;
    String t = (l > 1) ? s.substring(1, l) : "";
    return t;
  }
  
  private static String quote(String s) {
    if (s == null)
      return null;
    else
      return "\"" + unquote(s) + "\"";
  }
  
  public void log(String s) {
    log(Level.SEVERE, s);
  }
  
  private void log(Level l, String s) {
    if (s == null || l.intValue() < LogLevel.intValue())
      return;
    String date = "";
    if (s.startsWith("@")) {
      date = Sdf.format(new Date());
      s = s.substring(1);
    }
    StringBuilder sb = new StringBuilder();
    if (l == Level.WARNING)
      sb.append("%n*** ");
    sb.append(date).append(s);
    if (l == Level.WARNING)
      sb.append("%n");
    if (Verbose > 0)
      Printer.printf(sb.toString()+"%n");
    if (!skipLog && bwLog == null && fLog == null) {
      File f = new File(Dir, LogFile);
      try {
        bwLog = new BufferedWriter(new FileWriter(f));
      }
      catch (Exception e) {
        Printer.printf("*** can't create log file!");
      } 
    }
    if (bwLog != null) {
      try {
        bwLog.write(sb.toString());
        bwLog.newLine();
        bwLog.flush();
      }
      catch (Exception e) {
      }     
    }
  }
  
  /** 
   * returns the pressure for the specified height index
   * @param ih  the heigth index
   * @return    the pressure (Pa)
   */ 
  private double getPressure(int ih) {
    if (!tmp_initialized || ih < 0 || ih >= NZ)
      return -1;
    else if (ih < 1)
      return PA;
    double tint = 0.;
    for (int i=1; i<=ih; i++)
      tint += 0.5*(zv[i]-zv[i-1])*(1./tv[i] + 1./tv[i-1]);
    double p = PA * Math.exp(-(GRAV/RDRY)*tint);
    return p;
  }
  
  /**
   * calculates zq and zq from nf and nt
   * @param dq  source diameter (m)
   * @param tt  source temperature (K)
   * @param nf  norm volume flow wet (m3/s)
   * @param nt  norm volume flow dry (m3/s)
   * @return { vq (m/s), zq (kg/kg dry), lq (kg/kg) }
   */
  public static double[] GetVqZq(double dq, double tt, double nf, double nt) {
    if (tt-TZERO < 0.) return null;
    double vq;
    if (dq <= 0.)
      vq = 0.;
    else 
      vq = tt*4.*nf/(TZERO*Math.PI*dq*dq);
    double zq;
    if (vq == 0.)
      zq = 0.;
    else
      zq = (nt == 0.) ? 99999 : (nf-nt)/(nt*RVORD);
    double lq = GetSqLq(zq, tt, 101300.)[1];
    return new double[] { vq, zq, lq };
  }
  
  /**
   * calculates nf and nt from vq and zq
   * @param dq  source diameter (m)
   * @param tt  source temperature (K)
   * @param vq  exit velocity (m/s)
   * @param zq  water load (kg/kg dry)
   * @return { nf (m3/s), nt (m3/s), lq (kg/kg) }
   */
  public static double[] GetNfNt(double dq, double tt, double vq, double zq) {
    if (tt-TZERO < 0.) return null;
    double nf = (Math.PI/4.)*dq*dq*vq*TZERO/tt;
    double nt = nf/(1. + zq*RVORD);
    double lq = GetSqLq(zq, tt, 101300.)[1];
    return new double[] { nf, nt, lq };
  }
  
  
  /**
   * access sq and lq also for external use
   * @param zq   water load (kg/kg dry)
   * @param t    temperature (K)
   * @param p    pressure (Pa)
   * @return { sq (kg/kg), lq (kg/kg), qs0 (kg/kg) }
   */
  public static double[] GetSqLq(double zq, double t, double p) { //-2024-01-17
    double zeta = zq/(1+zq);
    double qs0 = getQs(t, p, 0);
    double lq = (zeta > qs0) ? (zeta - qs0)/(1-qs0) : 0.;
    double sq = (zeta > qs0) ? qs0*(1-lq) : zeta;
    return new double[] { sq, lq, qs0 };
  }

  /**
   * Saturation humidity for the specified temperature and pressure
   * using a more accurate relation.
   * 
   * @param t  the temperature (K)
   * @param p  the pressure (Pa)
   * @param eta the specific liquid water content
   * @return   the specific saturation humidity (kg per kg wet air)
   */
  public static double getQs(double t, double p, double eta) {
    double psp = getPs(t)/p;
    double k = 1/(1. - psp*(1. - 1./RVORD));
    double qs = psp*k/RVORD;
    return qs*(1. - eta);
    
  }
  
  /**
   * Saturation pressure for the specified temperature and pressure
   * (implementation of the Goff-Gratch equations)
   * 
   * @param t  the temperature (K)
   * @return   the saturation pressure (Pa)
   */
  public static double getPs(double t) {
    if (t > 100+TZERO)
      t = 100+TZERO;
    double tot0  = t/TZEROT;
    double t0ot  = TZEROT/t;    
    double ps;
    if (t > TZEROT)
      ps = LN10*(10.79574*(1-t0ot) + 0.000150474*(1-Math.pow(10,-8.2969*(tot0-1)))
                  + 0.00042873*(Math.pow(10,4.76955*(1-t0ot))-1))
           - 5.02800*Math.log(tot0)+ LN10*0.78614;
    else
      ps = LN10*(-9.09718*(t0ot-1) +  0.876793*(1-tot0)) - 3.56654*Math.log(t0ot)
              + Math.log(6.1071);  
    ps = Math.exp(ps);   
    return ps*100;
  }
  
  /**
   * Returns the density of wet air
   * @param t   the temperature (K)
   * @param q   the specific humidity (kg per kg wet air)
   * @param eta the liquid water content (kg per kg wet air)
   * @param p   the pressure (Pa)
   * @return    the density (kg/m3)
   */
  private double getDensity(double t, double q, double eta, double p) {
    if (q > 1.) q = 1.;
    if (q < 0.) q = 0.;
    double zeta = eta + q;
    if (zeta > 1.) zeta = 1.;
    if (zeta < 0.) zeta = 0.;
    return (p/(RDRY*t*(1 - zeta + RVORD*q)));
  }
  
  private String getS(double e, String frm) {
    if (Double.isNaN(e))
      return "NaN";
    else if (e == Double.MIN_VALUE)
      return "-inf";
    else if (e == Double.MAX_VALUE)
      return "+inf";
    else if (frm != null)
      return String.format(frm, e);
    else 
      return String.format("%e", e);
  }
   
  private void readParams() throws Exception {
    //
    // read parameter set
    IBJparamSet ps = new IBJparamSet();
    ps.readParamSet(this, PARAMSETFILE);
    //
    // set default values of all possible input parameters
    PrmSetDefaults();
    //
    // read input file
    int nr = 0;
    File fin = new File(Dir, PrmFile);
    if (!fin.exists())
      log(Level.FINEST, String.format("input file %s not found, skipped.", PrmFile));
    else {
      log(Level.FINEST, String.format("reading input file %s.", PrmFile));
      BufferedReader br;
      br = new BufferedReader(new FileReader(fin));
      String line;
      String sec = null;   
      StringBuilder sbread = new StringBuilder();
      while ((line = br.readLine()) != null) {
        if (line.startsWith("-") || line.startsWith("#") || line.trim().length() == 0)
          continue;
        if (line.startsWith("*")) {
          sec = line.substring(1,2);
          if (ps.getSection(sec) == null && !sec.equalsIgnoreCase("e"))
            throw new Exception("unknown section in input file: *"+sec);
          continue;
        }
        int ii = line.indexOf("'");
        if (ii >= 0) 
          line = line.substring(0, ii);
        String[] sa;
        if (line.indexOf("\"") > 0) {
          sa = new String[2];
          sa[0] = line.substring(0, line.indexOf(" ")).trim();
          sa[1] = line.substring(line.indexOf("\""));
        }
        else {
          sa = line.split("[ \t]+");
        }
        Param p = ps.getParam(sa[0]);
        if (p == null)
          throw new Exception("unknown parameter: "+sa[0]);
        if (!p.Section.equalsIgnoreCase(sec))
          throw new Exception("parameter "+p.Name+" must be defined in section *"+p.Section);
        if (sa.length != p.Size+1)
          throw new Exception("inconsistent number of elements for parameter: "+p.Name);
        if (p.Size != 1)
          throw new Exception("assignment of more than one value not implemented yet!");
        if (sbread.indexOf(p.Name.toLowerCase()) >= 0)
          throw new Exception("duplicate parameter setting: "+p.Name);
        //
        Object ov = PrmSetValue(p.Name, sa[1]);
        if (ov == null)
          throw new Exception("can't set value for parameter: "+p.Name);
        nr++;
        sbread.append(" ").append(p.Name.toLowerCase()).append(" ");
        //   
        if (p.Type == PTYPE.INT || p.Type == PTYPE.FLOAT)  {
          double d = (p.Type == PTYPE.INT) ? (Integer)ov : (Double)ov;
          if (d < p.Gmin || d > p.Gmax)
            throw new Exception(String.format("parameter %s: value %f out of global range [%s,%s]", 
                    p.Name, d, getS(p.Gmin, "%e"), getS(p.Gmax, "%e")));
          else if (d < p.Pmin || d > p.Pmax)
            Printer.printf(String.format("parameter %s: value %f out of plausible range [%s,%s]\n", 
                    p.Name, d, getS(p.Pmin, "%e"), getS(p.Pmax, "%e"))); 
        }
      }
      br.close();
    }
    //
    // read command-line definitions
    if (vparamdef != null) {
      for (String s : vparamdef) {
        String[] sa = s.split("[;]");
        Param p = ps.getParam(sa[0]);
        if (p == null)
          throw new Exception("unknown command-line parameter: "+sa[0]);
        if (sa.length != p.Size+1)
          throw new Exception("inconsistent number of elements for command-line parameter: "+p.Name);
        if (p.Size != 1)
          throw new Exception("assignment of more than one value not implemented yet!");       
        Object ov = PrmSetValue(p.Name, sa[1]);
        if (ov == null)
          throw new Exception("can't set value for command-line parameter: "+p.Name);
        nr++; 
        if (p.Type == PTYPE.INT || p.Type == PTYPE.FLOAT)  {
          double d = (p.Type == PTYPE.INT) ? (Integer)ov : (Double)ov;
          if (d < p.Gmin || d > p.Gmax)
            throw new Exception(String.format("command-line parameter %s: value %f out of global range [%s,%s]", 
                  p.Name, d, getS(p.Gmin, "%e"), getS(p.Gmax, "%e")));
          else if (d < p.Pmin || d > p.Pmax)
            Printer.printf(String.format("command-line parameter %s: value %f out of plausible range [%s,%s]\n", 
                  p.Name, d, getS(p.Pmin, "%e"), getS(p.Pmax, "%e"))); 
        }
      }
    }
    //
    // convert to internal units
    TQ += TZERO;
    TA += TZERO;
    A1 = Math.toRadians(A1);
    A2 = Math.toRadians(A2);
    DA = Math.toRadians(DA);
    //
    log(Level.FINEST, String.format("%d parameters read.", nr));  
  }
  
  private void addOutput() {
    if (skipDMN || vout == null)
      return;  
    PlrOutput po = new PlrOutput();
    /*
    double sw = kk[0]*kl[0] + kk[1]*kl[1] + kk[2]*kl[2];
    double w = Math.sqrt(uu*uu + ul*ul - 2*(kk[0]*kl[0] + kk[1]*kl[1] + kk[2]*kl[2]));
    System.out.printf("### w=%f, comp=%f, zz=%f, uu=%f, ul=%f, sw=%f\n", 
            w, fFinal*jl/Math.sqrt(3.), zz, uu, ul, sw);
    */  
    po.xx = xx;
    po.yy = yy;
    po.zz = zz;
    po.ll = ll;
    po.ss = ss;
    po.rr = rr;
    po.aa = aa;
    po.uu = uu;
    po.dd = dd;
    po.tt = tt;
    po.qi = qi;
    po.ei = ei;
    po.zi = zi;
    po.cc = cc;
    po.ri = qi/getQs(tt, pl, ei);
    po.ee = getEntrainment();
    po.kk = new double[3];
    po.kk[0] = kk[0];
    po.kk[1] = kk[1];
    po.kk[2] = kk[2];
    po.zt = zt;
    po.jj = jj;
    //    
    if (lenb < 0 || lenm < 0) {
      double fq = (FQ < 0) ? -FQ : FQ;
      if (fq < 1.e-20)
        fq = 1.e-20;
      double k = UQ/uH;
      lenm = DQ*k;
      lenb = DQ*k*k*k/fq;
    }
    double bfac = Math.pow(3./(4.*ETR_BETA*ETR_BETA), 0.33333333); 
    po.hbr = bfac * Math.pow(lenb*ll*ll + lenm*lenm*ll, 0.33333333); 
    po.hbm = bfac * Math.pow(lenm*lenm*ll, 0.33333333); 
    po.hbb = bfac * Math.pow(lenb*ll*ll, 0.33333333);
    double[] da = getVDI37823D(ll, ki, HQ, DQ, UQ, uH, QQ);
    po.hvd = da[0];
    po.xvd = da[1]; 
    po.ww = Math.sqrt(uu*uu + ul*ul - 2*(kk[0]*kl[0] + kk[1]*kl[1] + kk[2]*kl[2]));
    //
    vout.add(po);
  }
  
  private void initialize() throws Exception {
    if (vout != null)
      vout.clear();
    else if (!skipDMN)
      vout = new ArrayList<PlrOutput>();
    //
    // read input parameters
    readParams();
    //
    StringBuilder sb = new StringBuilder();
    sb.append("default options changed to true:");
    if (useConstantUst) sb.append(" const-ust");
    if (useConstantRa) sb.append(" const-ra");
    if (useSimProfiles) sb.append(" sim-profiles");
    if (skipStacktipDownwash) sb.append(" skip-stacktip-downwash");
    if (stopAtGround) sb.append(" stop-at-ground");
    if (stopAtFinalRise) sb.append(" stop-at-final-rise");
    if (!sb.toString().endsWith(":")) 
      log(Level.FINEST, sb.toString());
    //
    // temperature from heat current if provided
    if (QQ > 0) {
      double f = 1 - QQ/(Math.PI*0.25*DQ*DQ*UQ*1.36e-3*TZERO);
      TQ = (f != 0) ? (TZERO+10.)/f : -999;
      if (TQ < TZERO+10.)
        TQ = TZERO+10.;
    } 
    if (TQ < TZERO) 
      throw new Exception("exit temperature must be larger 0 deg Celsius!");
    QQ = Math.PI*0.25*DQ*DQ*UQ*(TZERO/TQ)*1.36e-3*(TQ-10.-TZERO);  
    //
    // set ambient profiles and friction velocity
    ust = (US > 0) ? US : 0;
    File fprf = new File(Dir, PrfFile);
    if (!fprf.exists() || ignoreProfiles) {
     // Printer.println("&&& creating ambient profiles");
      log("creating ambient profiles.");
      setAmbPrf();
    }
    else {
      readAmbPrf(fprf);
      log(String.format("profile file %s read (%d heights).", PrfFile, NZ));
    }
    if (ust <= 0) {
      int iust = 0;
      while (zv[iust]<=(6*Z0 + D0) && iust<NZ-1)
        iust++;
      ust = uv[iust]*0.4/Math.log((zv[iust]-D0)/Z0);
    }
    //
    // set parameters   
    ds = SD*SC;
    dp = SE*SC/SN;
    xx = XQ;
    yy = YQ;
    zz = HQ;
    ss = 0;
    ll = 0;
    zt = 0;
    kl = new double[3];
    setAmb(xx, yy, zz);
    hFinal = -1;
    xFinal = -1;
    ptlTs = -1;
    ptlVq = -1;
    ptlRed = -1;
    lenm = -1;
    lenb = -1;
    uH = ul;
    rr = DQ * 0.5;
    aa = 0.;
    qi = 0;
    ei = 0;
    zi = 0;
    w1 = A1;
    w2 = A2;
    cc = CQ;
    if (FQ > 0.) {
      TQ = tl/(1. - UQ*UQ/(FQ*GRAV*DQ*0.5));
      if ((TQ < TZERO-20) || (TQ > TZERO+600))
        throw new Exception(String.format("tq derived from fq outside range [-20,600] (%1.1f)", TQ-TZERO));
      QQ = (TQ < 10+TZERO) ? 0 : Math.PI*0.25*DQ*DQ*UQ*(TZERO/TQ)*1.36e-3*(TQ-10.-TZERO); //-2016-01-11
    }
    tt = TQ;
    //
    // set sq and lq
    if (LQ < 0) LQ = 0;
    if (LQ > 1) LQ = 1;
    if (SQ < 0) SQ = 0;
    if (SQ > 1) SQ = 1;
    if (TQ >= 100+TZERO) {
      LQ = 0;
      RQ = 0;
      if (ZQ > 0) {
        SQ = ZQ/(1+ZQ);
      }
    }
    else {
      if (ZQ > 0) {
        double[] da = GetSqLq(ZQ, tt, pl);                        //-2024-01-17
        SQ = da[0];
        LQ = da[1];
        /*
        double zeta = ZQ/(1+ZQ);
        double qs0 = getQs(tt, pl, 0);
        LQ = (zeta > qs0) ? (zeta - qs0)/(1-qs0) : 0.;
        SQ = (zeta > qs0) ? qs0*(1-LQ) : zeta;
        */
      }
      else {
        if (SQ == 0 && RQ > 0)
          SQ = RQ * getQs(tt, pl, LQ);
      }
    }
    if (SQ+LQ > 1) {
      log(String.format("adjusting water content (lq=%1.6e->%1.6e, sq=%1.6e)\n", LQ, 1-SQ, SQ));
      LQ = 1 - SQ;
    } 
    qi = SQ;
    ei = LQ;
    zi = qi + ei;
    aa = (ei > 0.) ? rr : 0.;
    dd = getDensity(tt, qi, ei, pl);
    FQ = (dl != dd && DQ != 0) ? UQ*UQ*dl/(Math.abs(dl-dd)*DQ*0.5*GRAV) : 1.e20;
    uu = UQ;
    jj = IQ * UQ; 
    kk = new double[3];
    kk[0] = Math.abs(Math.cos(w1)*uu) * Math.cos(w2);
    kk[1] = Math.abs(Math.cos(w1)*uu) * Math.sin(w2);
    kk[2] = Math.sin(w1)*uu;
    for (int i=0; i<3; i++) 
      if (Math.abs(kk[i]) < 1.e-10)
        kk[i] = 0.; 
    tstore = new double[NSTORE];
    hstore = new double[NSTORE];
  }
   
  private void setAmb(double x, double y, double z) {
    int iz;
    boolean outside = false;
    if (z < zv[0]) {
      iz = 0;
      outside = true;
    }
    else if (z >= zv[NZ-1]) {
      iz = NZ-1;
      outside = true;
    }
    else { 
      for (iz=0; iz<NZ-1; iz++)
        if ((z >= zv[iz]) && (z < zv[iz+1]))
          break;
    }
    tl    = tv[iz];
    //ul    = uv[iz];
    kl[0] = xv[iz];
    kl[1] = yv[iz];
    kl[2] = wv[iz];
    jl    = jv[iz];
    pl    = pv[iz];
    cl    = cv[iz];
    ql    = qv[iz];
    el    = lv[iz];
    dl    = av[iz]; 
    if (!outside) {
      double d = (z - zv[iz])/(zv[iz+1] - zv[iz]);     
      tl += d*(tv[iz+1]-tv[iz]);
      //ul += d*(uv[iz+1]-uv[iz]);
      kl[0] += d*(xv[iz+1]-xv[iz]);
      kl[1] += d*(yv[iz+1]-yv[iz]);
      kl[2] += d*(wv[iz+1]-wv[iz]);  
      ql += d*(qv[iz+1]-qv[iz]);
      el += d*(lv[iz+1]-lv[iz]);
      jl += d*(jv[iz+1]-jv[iz]);
      pl += d*(pv[iz+1]-pv[iz]);
      cl += d*(cv[iz+1]-cv[iz]);
      dl += d*(av[iz+1]-av[iz]);
    }
    ul = Math.sqrt(kl[0]*kl[0] + kl[1]*kl[1] + kl[2]*kl[2]);
    zl = ql + el;
    dl = getDensity(tl, ql, el, pl); // not using av, because in subtile runs linear interpolation is too rough
  }
  
  private void setAmbPrf() throws Exception {
    if (NZ <= 1) {
      NZ = vzz.length;
      zv = new double[NZ];
      for (int i=0; i<NZ; i++)
        zv[i] = vzz[i];
    }
    else {
      zv = new double[NZ];
      for (int i=0; i<NZ; i++)
      zv[i] = i*ZA;
    }
    uv = new double[NZ];
    dv = new double[NZ];
    xv = new double[NZ];
    yv = new double[NZ];
    wv = new double[NZ];
    iv = new double[NZ];
    jv = new double[NZ];
    tv = new double[NZ];
    pv = new double[NZ];
    rv = new double[NZ];
    qv = new double[NZ];
    lv = new double[NZ];
    cv = new double[NZ];
    av = new double[NZ];
    tmp_initialized = false;    
    //
    // ambient humidity
     if (SA > 0.) {
      RA = SA/getQs(TA, PA, 0);
      if (RA > 1.) {   
        RA = 1.;       
      }
    }
    SA = RA * getQs(TA, PA, 0); 
    //
    // stability measure
    lm = 0.;
    ki = 0;
    km = 0;
    if (LM != 0. || KI != 0) {
      double[] da = null;
      try {
        da = BLProfile2Layer.GetStability(Z0, LM, KI);
      } 
      catch (Exception e) {
        e.printStackTrace(System.out);
        throw new Exception("*** setAmbPrf()");
      }
      ki = (int)da[1];
      lm = da[0];
      double[] ka = { 0, 1, 2, 3.1, 3.2, 4, 5 };
      km = ka[ki]; 
    }
    hm = 800;
    //
    if (lm != 0) {
      //                                                            -2023-03-15
      int sgn = (FC < 0) ? -1 : 1;
      if (Math.abs(FC) > 1.4580e-4)
        FC = sgn*1.4580e-4;
      if (Math.abs(FC) < 1.e-12)
        FC = sgn*1.e-12; 
      //
      BLProfile2Layer prf = new BLProfile2Layer(FC, Z0, D0, HA, 
        (US > 0) ? US : UA, Math.toDegrees(DA), lm, HM, (US > 0), true);
      prf.setInterval(600);
      prf.init();
      log(String.format("BLProfile2Layer %s (fc=%1.3e,z0=%1.3f,ha=%1.1f,ua=%1.3f,lm=%1.1f,km=%1.1f,hm=%1.2f,us=%1.5f)", 
        BLProfile2Layer.VERSION, FC, Z0, HA, UA, lm, km, prf.getHm(), prf.getUst()));
      if (ust <= 0)
        ust = prf.getUst();
      hm = prf.getHm();
      for (int i=0; i<NZ; i++) {
        uv[i] = prf.getU(zv[i]);
        dv[i] = Math.toRadians(prf.getR(zv[i]));
        if (useNoShear)
          dv[i] = DA;
        tv[i] = prf.getTmp(zv[i], TH, TA, ZGamU, GamU); 
        jv[i] = Math.sqrt(prf.getSw(zv[i])*prf.getSw(zv[i]) + 
                          prf.getSv(zv[i])*prf.getSv(zv[i]) +
                          prf.getSu(zv[i])*prf.getSu(zv[i]) );
        iv[i] = (uv[i] > 1.e-4) ? jv[i]/(Math.sqrt(3.)*uv[i]) : 1.0;
      }
    }
    else {
      for (int i=0; i<NZ; i++) {
        double uh = (zv[i] > 200.) ? 200. : zv[i];
        uv[i] = (UX != 0.) ? UA * Math.pow(uh/UH, UX) : UA;
        dv[i] = DA;
        tv[i] = TA + (zv[i]-TH)*TX;
        iv[i] = IA;
        jv[i] = Math.sqrt(3.) * iv[i] * uv[i];                    //-2015-12-27
      }
    }
    tmp_initialized = true;
    for (int i=0; i<NZ; i++) {
      wv[i] = 0.;
      cv[i] = CA;
      xv[i] = -uv[i]*Math.sin(dv[i]);
      yv[i] = -uv[i]*Math.cos(dv[i]);
      pv[i] = getPressure(i); if (pv[i] < 0) throw new Exception("unexpected error!");
      lv[i] = 0;
      double qs = getQs(tv[i], pv[i], 0);     
      if (useConstantRa){
        qv[i] = qs * RA;
      }
      else {  // use constant specific water content
        qv[i] = SA; 
        if (qv[i] > qs) {
          // lv[i] = qv[i] - qs;  unsure about effects
          qv[i] = qs;
        }   
      }
      rv[i] = qv[i]/qs;
      av[i] = getDensity(tv[i], qv[i], lv[i], pv[i]);  
    }
  }
  
  private void readAmbPrf(File f) throws Exception {
    IBJarr arr = IBJdmn.readDmn(f.getPath());
    if (arr.getStructure().getDims() != 1)
      throw new Exception(f.getName()+": data part must be one-dimensional!");
    String[] cols = { "ha", "ua", "da", "ta", "ia", "ra", "ca" };
    for (String s : cols) {
      AbstractArray a = arr.getArray(s);
      if (a == null)
        throw new Exception(f.getName()+": data part must contain column "+s+"!");
      if (a.getElementLength() != 4)
        throw new Exception(f.getName()+": column "+s+" has invalid data size!");
    }
    int i0 = arr.getStructure().getFirstIndex()[0];
    int i1 = arr.getStructure().getLastIndex()[0];
    NZ = i1 - i0 + 1;
    zv = new double[NZ];
    uv = new double[NZ];
    dv = new double[NZ];
    xv = new double[NZ];
    yv = new double[NZ];
    wv = new double[NZ];
    iv = new double[NZ];
    jv = new double[NZ];
    tv = new double[NZ];
    pv = new double[NZ];
    rv = new double[NZ];
    qv = new double[NZ];
    lv = new double[NZ];
    cv = new double[NZ];
    av = new double[NZ];
    tmp_initialized = false;
    IBJarr.FloatArray faa = null;
    faa = (IBJarr.FloatArray)arr.getArray("ha");
    for (int i=0; i<NZ; i++) {
      zv[i] = faa.get(i0+i);
      if (i > 0 && zv[i] <= zv[i-1])
        throw new Exception(f.getName()+": invalid height order!");
    }
    faa = (IBJarr.FloatArray)arr.getArray("ua");
    for (int i=0; i<NZ; i++) {
      uv[i] = faa.get(i0+i);
      if (uv[i] < 0)
        throw new Exception(f.getName()+": ua < 0!");
    }
    faa = (IBJarr.FloatArray)arr.getArray("da");
    for (int i=0; i<NZ; i++) {
      dv[i] = faa.get(i0+i);
      if (dv[i] < 0 || dv[i] > 360)
        throw new Exception(f.getName()+": da not in range [0,360]!");
      dv[i] = Math.toRadians(dv[i]);
      xv[i] = -uv[i]*Math.sin(dv[i]);
      yv[i] = -uv[i]*Math.cos(dv[i]);
      wv[i] = 0.;
    }
    faa = (IBJarr.FloatArray)arr.getArray("ia");
    for (int i=0; i<NZ; i++) {
      iv[i] = faa.get(i0+i);
      if (iv[i] < 0. || iv[i] > 1.)
        throw new Exception(f.getName()+": ia not in range [0,1]!");
      jv[i] = Math.sqrt(3) * iv[i] * uv[i];                       //-2015-12-27
    }
    faa = (IBJarr.FloatArray)arr.getArray("ta");
    for (int i=0; i<NZ; i++) {
      tv[i] = faa.get(i0+i);
      if (tv[i] < -40 || tv[i] > +50)
        throw new Exception(f.getName()+": ta not in range [-40,50]!");
      tv[i] += TZERO; 
    }
    tmp_initialized = true;
    faa = (IBJarr.FloatArray)arr.getArray("pa");
    if (faa != null) {
      for (int i=0; i<NZ; i++) {
        pv[i] = faa.get(i0+i);
        if (pv[i] < 70000 || pv[i] > 120000)
          throw new Exception(f.getName()+": ta not in range [70000,120000]!");
      }
    }
    else {
      for (int i=0; i<NZ; i++) {  
        pv[i] = getPressure(i); if (pv[i] < 0) throw new Exception("unexpected error!");
      }
    }
    faa = (IBJarr.FloatArray)arr.getArray("ra");
    for (int i=0; i<NZ; i++) {
      rv[i] = faa.get(i0+i);
      if (rv[i] < 0. || rv[i] > 1.)
        throw new Exception(f.getName()+": ra not in range [0,1]!");
      qv[i] = rv[i] * getQs(tv[i], pv[i], 0); 
      lv[i] = 0;
      
    }
    faa = (IBJarr.FloatArray)arr.getArray("ca");
    for (int i=0; i<NZ; i++) {
      cv[i] = faa.get(i0+i);
      if (cv[i] < 0)
        throw new Exception(f.getName()+": ca < 0!");
    }
    for (int i=0; i<NZ; i++) {
      av[i] = getDensity(tv[i], qv[i], lv[i], pv[i]);
    }
  }
  
   /**
   * 
   * @param hh  enthalpy
   * @param zz  specific total water content
   * @param q0  specific humidity centre line
   * @param t0  temperature centre line
   * @return temperature
   */
  private double getTmpImplicitly2(double hh, double zz) { 
    double a = hh + LHEAT*zz;
    double diff = -Double.MAX_VALUE;
    double diffold = diff;
    double step = 0.5;     
    double iterdev = 1;
    int itermax = 200;
    int n = 0;
    double t = tt - 2;
    double cp = CP(qi, ei);
    while (Math.abs(diff) > iterdev) {          
      if (diff*diffold < 0 || Math.abs(diff) > Math.abs(diffold))
        step *= -0.5;                      
      t += step;
      if (t <= 0.) {
        t -= step;
        step *= 0.5;
      } 
      diffold = diff;        
      diff = t*cp + LHEAT*getQs(t, pl, 0) - a;
      //System.out.printf("### n=%2d, tt=%8.3f, t=%8.3f, diff=%10.2e\n", n, tt, t, diff);
      if (++n > itermax)
        break; 
    }
    if (n > itermax) {
      Printer.printf("*** iterative setting of T did not succeed!");
      System.exit(0);
    }   
    return t;   
  }
 
  /**
   * 
   * @param hh  enthalpy
   * @param zz  specific total water content
   * @param q0  specific saturation humidity
   * @param t0  temperature
   * @return temperature
   */
  private double getTmpImplicitly(double hh, double zz, double cp, double q0, double t0) { 
    // note: hh = enthalphy;
    //       zz = specific total water content;
    // what happens:
    //    Equation h = cp*T - hv*(zeta - qs(T));
    //    Clausius-C. qs = q0 exp(hv*(T-T0)/(R*T0*T0));
    //    Expansion of qs to second order -> quadratic equation for T;
    double A = LHEAT/(RVAPOUR*t0*t0);
    double A2 = A*A; 
    double a2 = 0.5*LHEAT*q0*A2;
    double a1 = cp + A*LHEAT*q0*(1 - A*t0);
    double a0 = LHEAT*q0*(1 - A*t0 + 0.5*A2*t0*t0) - hh - LHEAT*zz;
    double arg = a1*a1 - 4*a2*a0;
    if (arg < 0)
      return -999;
    double ti = (-a1 + Math.sqrt(arg))/(2.*a2);
    return ti;
  }
  

  /** calculates the plume temperature, liquid water content, and visible radius
   * 
   * @return { eta, a, tmp )
   */
  private double[] getEtaATmp(double t0, double cp, double hs, double zeta) {
    double[] res = new double[3];
    //
    // if the plume is dry, piece of cake
    if (zeta <= 0.) {
      res[0] = 0.;                 // eta
      res[1]  = 0.;                // a
      res[2] = hs/cp;              // tmp
      return res;
    }
    //
    // top hat profiles 
    else if (!useSimProfiles) {
      double q0 = getQs(t0, pl, 0);
      double ti = getTmpImplicitly(hs, zeta, cp, q0, t0);  
      if (ti >= 100+TZERO) {                                       //-2018-09-10
        res[0] = 0.;
        res[1] = 0.;
        res[2] = hs/cp; 
      }
      else {
        res[0] = (ti*cp > hs) ? (ti*cp - hs)/LHEAT : 0;
        res[1] = (ti*cp > hs) ? Double.NEGATIVE_INFINITY : 0;
        res[2] = (ti*cp > hs) ? ti : hs/cp; 
      }
      return res;
    }
    //
    // similarity profiles
    double cpl = CP(ql,el);
    double cfac = CFAC + Math.exp(-ss/(0.5*DQ))*(1 - CFAC);
    double hst = cfac*(hs - tl*cpl); if (hst < 0) hst = 0;
    double zst = cfac*(zeta - zl); if (zst < 0) zst = 0;
    double tst = cfac*(tt-tl); if (tst < 0) tst = 0;
    double q0 = getQs(t0, pl, 0.);
    double b = rr/Math.sqrt(cfac);
    //
    // start from outside the plume, decrease the distance to the plume axis,
    // find the visible plume radius, and integrate the liquid water content
    double eint = 0.;
    double a = 0.;
    double dr = b/20.;
    double t = t0;
    double ti = t0;
    double hr = tl*cpl + hst;
    double zr;
    for (double r=2*b; r>-dr; r-=dr) {
      double f = (b > 0.) ? Math.exp(-r*r/(b*b)) : 1.;
      hr = tl*cpl + hst*f;
      zr = zl + zst*f;
      ti = getTmpImplicitly(hr, zr, cp, q0, t0);
      double hi = ti*cp;
      //ti = getTmpImplicitly2(hs, zr);
      if ((hi-hr) > 0. && a == 0.) 
        a = r;
      if (a > 0.) {
        if (hi > hr)
          t = ti;
        if (t*cp > hr)
          eint += (t*cp-hr)*r;
      }
    }
    res[0] = eint * 2*dr/(LHEAT*rr*rr);
    res[1] = a;
    if (res[0] < 1.e-10) {
      res[0] = 0.;
      res[1] = 0.;
    }
    res[2] = (hs + LHEAT*res[0])/cp;
    return res;
  }
  
  private double getEntrainment() {
    double sw = kk[0]*kl[0] + kk[1]*kl[1] + kk[2]*kl[2];
    double wp = sw/uu - uu;
    double wp2 = wp*wp;
    double ws2 = Math.abs(ul*ul - sw*sw/(uu*uu));
    
   double if2 = (GRAV*Math.abs(dl-dd)*rr)/(uu*dl);
   double etr = ETR_GAMMA*if2;
 
 // double if2 = (GRAV*Math.abs(dl-dd)*rr)/(uu*uu*dl);         
//  double etr = 0.5*if2 * (0.5*wp2 + ws2)/Math.sqrt(wp2+ws2); 
    
    if ((wp2 + ws2) > 0) {
      etr += (0.5*ETR_ALPHA*wp2 + ETR_BETA*ws2)/Math.sqrt(wp2+ws2);
    }
    etr *= (EF * rr);
    return etr;   
  }
  
  private void derivs(double snew, double[] v, double[] dv) {
    //
    // calculate current parameters
    double v0i = 1./v[0];
    kk[0] = v[1]*v0i;
    kk[1] = v[2]*v0i;
    kk[2] = v[3]*v0i;
    uu = Math.sqrt(kk[0]*kk[0] + kk[1]*kk[1] + kk[2]*kk[2]);
    double ui = 1./uu;
    
    double dt = (snew - ss)/uu;
    xx = v[8] + kk[0]*dt;
    yy = v[9] + kk[1]*dt;
    zz = v[10] + kk[2]*dt;
    setAmb(xx, yy, zz);
    
    ll = v[11];
    zi = v[5]*v0i; 
    jj = Math.sqrt(Math.abs(v[7]));
    zt = v[12];
    double[] res = getEtaATmp(tt, CP(qi,ei), v[4]*v0i, zi);
    ei = res[0];
    aa = res[1];
    tt = res[2]; 
    qi = zi - ei;
    if (qi < 0) qi = 0;
    dd = getDensity(tt, qi, ei, pl);
    rr = Math.sqrt(Math.abs(v[0]*ui/dd));
    if (aa == Double.NEGATIVE_INFINITY)
      aa = rr;
    double r2 = rr*rr;
    cc = v[6]*ui/(r2);
    //
    // calculate current derivatives  
    if (dv != null) {
      double sw = (kk[0]*kl[0] + kk[1]*kl[1] + kk[2]*kl[2]);
      double eps = 2. * dl * getEntrainment();
      dv[0]  = eps;                                     // AE 2001, Eq. 10
      dv[1]  = eps*kl[0];                               // AE 2001, Eq. 11 
      dv[2]  = eps*kl[1];                               // AE 2001, Eq. 11 
      dv[3]  = eps*kl[2] - r2*(dd-dl)*GRAV;             // AE 2001, Eq. 11 
      dv[4]  = -(r2*dd*uu)*GRAV*(kk[2]*ui)*(dl/dd)      // AE 2001, Eq. 35 
               + eps*(CP(ql,el)*tl - LHEAT*el);
      dv[5]  = eps*zl;                                  // AE 2001, Eq. 34
      dv[6]  = eps*(cl/dl);                             // AE 2001, Eq. 19
      dv[7]  = (eps*v0i)*(uu*uu + ul*ul - 2*sw + jl*jl - jj*jj);         // AE 2001, Eq. 18
      dv[8]  = kk[0]*ui;
      dv[9]  = kk[1]*ui;
      dv[10] = (kk[2] - VS)*ui;
      dv[11] = Math.sqrt(kk[0]*kk[0] + kk[1]*kk[1])*ui;
      dv[12] = ui;
    }
  }
  
  private void rk(double[] y, double[] dydx, int n, double x, double h, double[] yout) {  
    int i;
    double xh, h1, h2;
    double[] dym = new double[n];
    double[] dyt = new double[n];
    double[] yt = new double[n];
    h1  = h*0.5;
    h2  = h/6.;
    xh  = x+h1;
    for (i=0; i<n; i++) 
      yt[i] = y[i] + h1*dydx[i];
    derivs(xh, yt, dyt);
    for (i=0; i<n; i++) 
      yt[i] = y[i] + h1*dyt[i];
    derivs(xh, yt, dym);
    for (i=0; i<n; i++) {
      yt[i] = y[i] + h*dym[i];
      dym[i] += dyt[i];
    }
    derivs(x+h, yt, dyt);
    for (i=0; i<n; i++)
      yout[i] = y[i] + h2*(dydx[i] + dyt[i] + 2.*dym[i]);
  }
  
  private double CP(double q, double eta) {
//"#########################################################
if (true) return CPRESSD;
//"#########################################################
    double qq = q;
    double ee = eta;
    double ww = q + eta;
    if (qq < 0) qq = 0;
    if (qq > 1) qq = 1;
    if (ee < 0) ee = 0;
    if (ee > 1) ee = 1;
    if (ww < 0) ww = 0;
    if (ww > 1) ww = 1;
    return CPRESSD*(1-ww) + CPRESSV*qq + CPRESSL*ee;
  }
  
  /**
   * 
   * @return result: Vq, Ts, DH(VDI), XH(VDI), FRindex
   * @throws Exception 
   */
  public double[] run() throws Exception {   
    double[] res = new double[4];
    for (int i=0; i<res.length; i++) res[i] = -1;
    //
    // initialize parameters
    log("@");
    log(String.format("%s Version %s", PRGNAME, VERSION));
    log(String.format("(c) Janicke Consulting 2016-2024"));
    log(String.format("%s comes with ABSOLUTELY NO WARRANTY", PRGNAME));
    log(String.format("%s is free software under the GNU PUBLIC LICENCE", PRGNAME));
    if (vcmd != null) 
      for (String s : vcmd) log(s);
    log(String.format("%nProject directory: %s%n", Dir));
    initialize();
    //
    // set integration parameters to their initial values
    double[] var = new double[13];
    double[] dvds = new double[var.length];
    var[0] = rr*rr*dd*uu;                   // M, mass flow density / PI
    var[1] = var[0] * kk[0];                // M*ux, momentum flow density / PI
    var[2] = var[0] * kk[1];                // M*uy, momentum flow density / PI
    var[3] = var[0] * kk[2];                // M*uz, momentum flow density / PI
    var[4] = var[0] * (CP(qi,ei)*tt-LHEAT*ei); // M*(cp*T-hv*eta), enthalpy flow density / PI
    var[5] = var[0] * zi;                   // M*zeta, water flow density / PI
    var[6] = var[0] * cc/dd;                // M*c/rho, scalar quantity flow density / PI
    var[7] = jj*jj;                         // sigma*sigma, turbulence
    var[8] = xx;                            // x-coordinate of the plume
    var[9] = yy;                            // y-coordinate of the plume
    var[10] = zz;                           // z-coordinate of the plume
    var[11] = 0.;                           // path along the horizontal projection 
    var[12] = 0.;                           // travel time
    //
    // do the calculation 
    int nstore = 0;
    tstore[nstore] = 0;
    hstore[nstore] = 0;
    int np = 0;
    double ps = 0;
    addOutput(); 
    while (np < SN) {
      ds = rr/500.;
      //
      // advance one step ds
      setAmb(xx, yy, zz);
      derivs(ss, var, dvds);
      rk(var, dvds, var.length, ss, ds, var);
      //
      // transfer calculated values at ss+ds from var[] to individual parameters
      derivs(ss, var, null);
      ss += ds;
      ps += ds;
      //    
      if (checkFinalRise(false)) {
        if (stopAtFinalRise) {
          log("");
          log("all final rise criteria reached, break.");
          break;
        }
        else if ((tstore[nstore] >= 0) && (nstore < tstore.length-4) ) {     //-2018-02-07
          // store final rise position to allow in all cases hFinal/2 evaluation
          nstore++;
          tstore[nstore] = zt;
          hstore[nstore] = zz - HQ;
          nstore++;
          tstore[nstore] = -1;
          hstore[nstore] = -1;
        }
      }
      if (zz <= 0. || zz > zv[NZ-1]) {
        log("");
        log("plume axis at ground or above maximum height, break.");
        break;
      }
      if ((tstore[nstore] >= 0) && (nstore < tstore.length-3) && (Math.abs(hstore[nstore] - (zz-HQ)) > DSTORE*DQ)) {
        nstore++;
        tstore[nstore] = zt;
        hstore[nstore] = zz-HQ;
      }
      if ((int)(ps/ds + 0.5)*ds >= dp) {
        np++; 
        ps = 0;
        addOutput();
        if (Verbose > 1) {
          Printer.printf("\r                      ");
          Printer.printf("\rstep %d", np);
          Printer.flush();
        }
      }
    }
    if (tstore[nstore] >= 0) {
      nstore++;
      tstore[nstore] = zt;
      hstore[nstore] = zz-HQ;
      nstore++;
      tstore[nstore] = -1;
      hstore[nstore] = -1;
    }
    if (np < SN) 
      addOutput();
    Printer.printf("\r"); 
    log(String.format("final rise        : dh=%5.1f m at x=%4.0f m", hFinal, xFinal));
    double[] da = getVDI37823D(1.e10, ki, HQ, DQ, UQ, uH, QQ);
    log(String.format("final rise 3782/3 : dh=%5.1f m at x=%4.0f m", da[0], da[1]));
    res[2] = da[0];
    res[3] = da[1]; 
    checkFinalRise(true);
    res[0] = ptlVq;
    res[1] = ptlTs;

    if (!skipDMN)
      writeOutput();
    if (!skipGraph)
      writeGraph();
    return res;
  }
  
  private boolean checkFinalRise(boolean evaluate) {
    if (evaluate) {
      if (xFinal < 0) {  
        log("can't determine final rise!");
        return false;
      }   
      if (hFinal <= 0.) {
        ptlTs = 0.;
        ptlVq = 0.;
      }
      else if (xFinal > 0) {
        //
        // find half time                                          //-2015-12-30
        double h = hFinal/2.;
        int i0 = 0;
        for (int i=0; i<hstore.length-1; i++) {
          if (hstore[i] <= h && h < hstore[i+1]) {
            i0 = i;
            break;
          }  
        }
        double t = 0;
        if (i0 == 0) {                                            //-2019-03-02
          log("can't determine half rise, using t=10*ln(2)");
          t = 10*Math.log(2.);
        }
        else {     
          t = tstore[i0];
          double d = (h - hstore[i0])/(hstore[i0+1] - hstore[i0]);
          t += d*(tstore[i0+1] - tstore[i0]);
        }
        ptlTs = t/Math.log(2.); 
        ptlVq = hFinal/ptlTs;
      }
      else {
        ptlTs = (UQ > 0) ? hFinal/UQ : hFinal/1.;
        ptlVq = (UQ > 0) ? UQ : 1.;
      }
      //
      // explicit reduction / stacktip downwash
      ptlRed = 1.;
      if (!skipStacktipDownwash) {
        double k = UQ/uH;
        double kkrit = 1.5/(1. + 2.*Math.pow(FQ, -0.3333));       //-2016-01-06
        ptlRed = Math.min(1.0, k/kkrit);                         
        if (ptlRed < 1. && RF >= 0. && RF < 1.)                   //-2024-01-17
          ptlRed *= RF;
        ptlVq *= ptlRed;
      }
      log(String.format("Vq=%1.2f; Ts=%1.2f; Dh=%1.2f; Red=%1.2f", 
        ptlVq, ptlTs, ptlVq*ptlTs, ptlRed));
      return true;
    }
    //
    // done already?
    if (xFinal >= 0)
      return true;
    //
    // set final rise conditions
    double w = Math.sqrt(uu*uu + ul*ul - 2*(kk[0]*kl[0] + kk[1]*kl[1] + kk[2]*kl[2]));
    double uss = ust;
    if (uss < MIN_UST)
      uss = MIN_UST;
    if (!useConstantUst) {
      uss *= Math.pow(Math.max(zt,USTTIME)/USTTIME, 0.18);
    }
    //
    // difference velocity
    if (xFinal < 0. && w < fFinal*uss) {
      hFinal = zz - HQ;
      xFinal = ll; 
    }
    if (xFinal < 0. && stopAtGround && zz-rr <= 0) {
      log("set final rise because plume lower edge at ground");
      hFinal = zz - HQ;
      xFinal = ll; 
    }
    //
    // ultimate zmax criterion
    if (xFinal < 0. && (zz <= 0. || zz > ZmaxFinal)) {
      hFinal = (zz <= 0) ? 0. : zz - HQ;
      xFinal = ll;
    }
    return (xFinal >= 0);
  }
  
  private String getOutputString(int oo, PlrOutput p) {
    StringBuilder sb = new StringBuilder();
    // 0: rise, 1: std, 2: std/scal, 3: outline, 4: outline/scal
    if (oo < 0 || oo > 4)
      oo = 1;
    double sc = (oo == 0 || oo == 2 || oo == 4) ? SC : 1;
    if (oo == 0) {
      if (p == null) sb.append(" \"ll%10.2f\""); else sb.append(String.format("%10.2f", p.ll/SC));
      if (p == null) sb.append(" \"hh%10.2f\""); else sb.append(String.format("%10.2f", (p.zz - HQ)/SC));
      if (p == null) sb.append(" \"brg%10.2f\""); else sb.append(String.format("%10.2f", p.hbr/SC));
      if (p == null) sb.append(" \"brgm%10.2f\""); else sb.append(String.format("%10.2f", p.hbm/SC));
      if (p == null) sb.append(" \"brgb%10.2f\""); else sb.append(String.format("%10.2f", p.hbb/SC));
      if (p == null) sb.append(" \"vdid%10.2f\""); else sb.append(String.format("%10.2f", p.hvd/SC));
    }
    else {
      double ww1 = -999;
      if (p != null) {
        /*
        if (p.uu != 0.) {
          ww1 = Math.toDegrees(Math.asin(p.kk[2]/p.uu));
          if (ww1 < 0.) ww1 += 360.;
        } 
        */
        ww1 = p.ww;
      }
      if (p == null) sb.append(" \"xx%10.2f\""); else sb.append(String.format(" %10.2f", p.xx/sc));
      if (p == null) sb.append(" \"yy%10.2f\""); else sb.append(String.format(" %10.2f", p.yy/sc));
      if (p == null) sb.append(" \"zz%10.2f\""); else sb.append(String.format(" %10.2f", p.zz/sc));
      if (p == null) sb.append(" \"w1%10.4f\""); else sb.append(String.format(" %10.4f", ww1));
      if (p == null) sb.append(" \"hh%10.2f\""); else sb.append(String.format(" %10.2f", (p.zz - HQ)/sc));
      if (p == null) sb.append(" \"ss%10.2f\""); else sb.append(String.format(" %10.2f", p.ss/sc));
      if (p == null) sb.append(" \"ll%10.2f\""); else sb.append(String.format(" %10.2f", p.ll/sc));
      if (p == null) sb.append(" \"zt%10.2f\""); else sb.append(String.format(" %10.2f", p.zt));
      if (p == null) sb.append(" \"rr%10.4f\""); else sb.append(String.format(" %10.4f", p.rr/sc));
      if (p == null) sb.append(" \"aa%10.4f\""); else sb.append(String.format(" %10.4f", p.aa/sc));
      if (oo < 3) {
        if (p == null) sb.append(" \"uu%10.2f\""); else sb.append(String.format(" %10.2f", p.uu));
        if (p == null) sb.append(" \"tt%12.4e\""); else sb.append(String.format(" %12.4e", p.tt -TZERO));    
        if (p == null) sb.append(" \"cc%10.2e\""); else sb.append(String.format(" %10.2e", p.cc));
        if (p == null) sb.append(" \"jj%10.2f\""); else sb.append(String.format(" %10.2f", p.jj));      
        if (p == null) sb.append(" \"qi%12.4e\""); else sb.append(String.format(" %12.4e", p.qi));
        if (p == null) sb.append(" \"ri%10.2f\""); else sb.append(String.format(" %10.2f", p.ri));
        if (p == null) sb.append(" \"li%10.2e\""); else sb.append(String.format(" %10.2e", p.zi - p.qi));
        if (p == null) sb.append(" \"dd%12.4e\""); else sb.append(String.format(" %12.4e", p.dd));
        if (p == null) sb.append(" \"ee%10.2e\""); else sb.append(String.format(" %10.2e", p.ee));    
      }
      else {
        double uh=0, co=0, si=0, r=0, l1r=0, z1r=0, l2r=0, z2r=0, l1a=0, z1a=0, l2a=0, z2a=0;
        double ww=0, w1a=0, w2a=0;
        if (p != null) {
          uh = Math.sqrt(p.kk[0]*p.kk[0] + p.kk[1]*p.kk[1]);
          co = (p.uu != 0) ? uh/p.uu : 1;
          si = (p.uu != 0) ? p.kk[2]/p.uu : 0;   
          r = p.rr/Math.sqrt(CFAC);
          l1r = p.ll + r*si;
          z1r = p.zz - r*co;
          l2r = p.ll - r*si;
          z2r = p.zz + r*co;
          l1a = p.ll + p.aa*si;
          z1a = p.zz - p.aa*co;
          l2a = p.ll - p.aa*si;
          z2a = p.zz + p.aa*co;
          ww = Math.sqrt((p.xx-XQ)*(p.xx-XQ) + (p.yy-YQ)*(p.yy-YQ));
          w1a = ww + p.aa*si;
          w2a = ww - p.aa*si;
        }
        if (p == null) sb.append(" \"ww%10.2f\"");  else sb.append(String.format(" %10.2f", ww/sc));  //-2018-03-24
        if (p == null) sb.append(" \"l1r%10.2f\""); else sb.append(String.format(" %10.2f", l1r/sc));
        if (p == null) sb.append(" \"z1r%10.2f\""); else sb.append(String.format(" %10.2f", z1r/sc));
        if (p == null) sb.append(" \"l2r%10.2f\""); else sb.append(String.format(" %10.2f", l2r/sc));
        if (p == null) sb.append(" \"z2r%10.2f\""); else sb.append(String.format(" %10.2f", z2r/sc));
        if (p == null) sb.append(" \"l1a%10.2f\""); else sb.append(String.format(" %10.2f", l1a/sc));
        if (p == null) sb.append(" \"z1a%10.2f\""); else sb.append(String.format(" %10.2f", z1a/sc));
        if (p == null) sb.append(" \"l2a%10.2f\""); else sb.append(String.format(" %10.2f", l2a/sc));
        if (p == null) sb.append(" \"z2a%10.2f\""); else sb.append(String.format(" %10.2f", z2a/sc));
        if (p == null) sb.append(" \"w1a%10.2f\""); else sb.append(String.format(" %10.2f", w1a/sc));
        if (p == null) sb.append(" \"w2a%10.2f\""); else sb.append(String.format(" %10.2f", w2a/sc));
      }
    }
    return sb.toString();
  }
  
  private void writeGraph() throws Exception {
    if (vout == null || vout.isEmpty())
      return;
    File fout = new File(Dir, GrpFile);
    Diagram dgm = new Diagram(0, 0);
    dgm.setView(0, 0, 100, 100);
    String title = ID;
    if (title != null && title.startsWith("\""))
      title = title.substring(1);
    if (title != null && title.endsWith("\""))
      title = title.substring(0,title.length()-1);    
    dgm.setTitle(title, false);
    BasicStroke[] bsa = new BasicStroke[4];
    bsa[0] = new BasicStroke(1f, 0, 0, 10, new float[] { 2, 2 }, 0);
    bsa[1] = new BasicStroke(1f, 0, 0, 10, new float[] { 10, 10 }, 0);
    bsa[2] = new BasicStroke(1f, 0, 0, 10, new float[] { 10, 2, 2, 2 }, 0);
    bsa[3] = new BasicStroke(1f, 0, 0, 10, new float[] { 10, 2, 2, 2, 2, 2 }, 0);
    //
    double ssc = (paintExtended) ? DQ : 1.;
    PlrOutput p = vout.get(vout.size()-1);
    double xmax = p.ll/ssc; 
    if (xmax > 100)
      xmax = (int)(xmax/100. + 0.5)*100;
    else
      xmax = (int)(xmax/10. + 0.5)*10;
    double ymax = (paintExtended) ? p.hvd : 0;
    for (PlrOutput vout1 : vout) {
      if (vout1.zz+vout1.rr-HQ > ymax)
        ymax = vout1.zz+vout1.rr-HQ;  
    }
    ymax = (ymax + HQ)/ssc;
    if (ymax > 500)
      ymax = (int)(ymax/500. + 0.5)*500;
    else if (ymax > 150)
      ymax = (int)(ymax/100. + 0.5)*100;
    else
      ymax = (int)(ymax/10. + 0.5)*10;
    // 
    dgm.setScale(-1.5, 0, 1.5+xmax, ymax);
    double dx = 5, dy = 5.;
    if (xmax > 50)   dx = 10;
    if (xmax > 100)  dx = 20;
    if (xmax > 200)  dx = 50;
    if (xmax > 500)  dx = 200;
    if (xmax > 2000) dx = 500;
    if (xmax > 5000) dx = 1000;
    if (xmax > 10000) dx = 5000;
    if (ymax > 50)   dy = 10;
    if (ymax > 100)  dy = 20;
    if (ymax > 200)  dy = 50;
    if (ymax > 500)  dy = 200;
    if (ymax > 2000) dy = 500;
    if (ymax > 5000) dy = 1000;
    if (ymax > 10000)dy = 5000;
    dgm.addAxis(Graph.AXIS_X+Graph.AXIS_GRID, dx, "%3.0f", (paintExtended) ? "Entfernung/Durchmesser" : "distance (m)");
    dgm.addAxis(Graph.AXIS_Y+Graph.AXIS_GRID, dy, "%3.0f", (paintExtended) ? "Höhe/Durchmesser" : "height (m)");
    // 
    // lines
    double[] x = new double[vout.size()];
    double[] y = new double[vout.size()];
    for (int i=0; i<x.length; i++) {
      x[i] = vout.get(i).ll/ssc;
      y[i] = vout.get(i).zz/ssc;  
    } 
    dgm.addPath(Graph.makePath(x, y), Color.black, new BasicStroke(3f));
    //
    // VDI 3782/3D
    if (paintExtended) {
      for (int i=0; i<x.length; i++) {
        y[i] = (vout.get(i).hvd+HQ)/ssc;       
     }
      dgm.addPath(Graph.makePath(x, y), Color.cyan, new BasicStroke(3f));
    }
    //
    // LASAT/VDI
    if (paintExtended) {
      double lst_ts = 0.4*vout.get(0).xvd/uH;  
      double lst_dh = vout.get(vout.size()-1).hvd;
      if (lst_ts == 0) 
        lst_ts = lst_dh/uH;
      for (int i=0; i<x.length; i++) {
        double f = (lst_ts > 0) ? 1 - Math.exp(-vout.get(i).zt/lst_ts) : 1; 
        y[i] = (HQ + lst_dh*f)/ssc;
      }
      dgm.addPath(Graph.makePath(x, y), Color.green, new BasicStroke(1f));
    }
    //
    // LASAT/PLURIS
    if (paintExtended) { 
      if (ptlVq >= 0) {
        double lst_ts = ptlTs;   
        double lst_dh = ptlTs * ptlVq;    
        for (int i=0; i<x.length; i++) {
          double f = (lst_ts > 0) ? 1 - Math.exp(-vout.get(i).zt/lst_ts) : 1;
          y[i] = (HQ + lst_dh*f)/ssc;
        }
        dgm.addPath(Graph.makePath(x, y), Color.green, new BasicStroke(2f));
      }
    }
    //
    // final rise
    if (paintExtended) {     
      if (hFinal >= 0) {
        for (int j=0; j<x.length; j++)
          y[j] = (vout.get(j).ll < xFinal) ? vout.get(j).zz/ssc : (hFinal+HQ)/ssc;
        dgm.addPath(Graph.makePath(x, y), Color.black, bsa[0]);
      }
    }    
    //
    // outline
    for (int i=0; i<2; i++) {
      x = new double[vout.size()*2];
      y = new double[vout.size()*2];
      for (int j=0; j<vout.size(); j++) {
        p = vout.get(j);
        double uh = Math.sqrt(p.kk[0]*p.kk[0] + p.kk[1]*p.kk[1]);
        double co = (p.uu != 0) ? uh/p.uu : 1;
        double si = (p.uu != 0) ? p.kk[2]/p.uu : 0;   
        double r = (i == 0) ? p.rr/Math.sqrt(CFAC) : p.aa;
        double l1r = p.ll + r*si;
        double z1r = p.zz - r*co;
        double l2r = p.ll - r*si;
        double z2r = p.zz + r*co;          
        x[j] = l1r/ssc;
        y[j] = z1r/ssc;
        x[2*vout.size()-1-j] = l2r/ssc;
        y[2*vout.size()-1-j] = z2r/ssc;
      }
      Element ge = new Graph.Element();
      if (i == 0) {
        ge.action = "fill";
        ge.color = new Color(230,230,230);
      }
      else {
        ge.color = Color.gray;
        ge.stroke = bsa[0];
      }
      ge.add(Graph.makePath(x, y));
      dgm.addGround(ge);
    }
    //
    // stack
    Element ge = new Graph.Element();
    ge.color = Color.magenta;
    ge.action = "fill";
    ge.add(new Rectangle2D.Double(-0.5*DQ/ssc, 0, DQ/ssc, HQ/ssc));
    dgm.addOverlay(ge);
    //
    // params
    dgm.sizes[0] = 10;
    dgm.sizes[7] = 0.5;
    Graph g = new Graph();
    dgm.createGraph(g);   
    Font f = g.getFont().deriveFont(2f);
    String s = String.format("hq=%1.1f, dq=%1.1f, uq=%1.1f, tq=%1.1f, qq=%1.3f, lm=%1.1f, us=%1.3f, ki=%1d, z0=%1.3f, ua=%1.1f, ha=%1.1f, Lm=%1.3f, Lb=%1.3f",
            HQ, DQ, UQ, TQ-TZERO, QQ, lm, ust, ki, Z0, UA, HA, lenm, lenb);
    Graph.Label lbl = new Graph.Label(dgm.getView().getX(), dgm.getView().getY()+dgm.getView().getHeight()+0.5, s, f);
    lbl.alignment = Graph.ALIGN_BOTTOM + Graph.ALIGN_LEFT;
    g.getRoot().add(lbl);
    g.writePDF(fout);
    log(fout.getName()+" written.");
  }
  
  private void writeOutput() throws Exception {
    File fout = new File(Dir, OutFile);
    BufferedWriter bw = null;
    StringBuilder sb = new StringBuilder();
    bw = new BufferedWriter(new FileWriter(fout));
    bw.write(String.format("- %s Version %s (c) Janicke Consulting 2016-2024", PRGNAME, VERSION));
    bw.newLine();
    bw.write("- "+Sdf.format(new Date()));
    bw.newLine();
    bw.write(String.format("--- final rise for VDI 3945/3"));  
    bw.newLine(); 
    bw.write(String.format("%-3s  %1.1f", "xf", xFinal));
    bw.newLine();
    bw.write(String.format("%-3s  %1.1f", "hf", hFinal));
    bw.newLine();  
    bw.write(String.format("%-3s  %1.3f", "ff", fFinal));
    bw.newLine();
    bw.write(String.format("%-3s  %1.0f", "ft", (useConstantUst) ? -1 : USTTIME));
    bw.newLine();
    bw.write(String.format("%-3s  %1.5f", "us", ust));
    bw.newLine();
    bw.write(String.format("%-3s  %1.1f", "red", ptlRed));
    bw.newLine();
    bw.write(String.format("%-3s  %1.3f", "ve", ptlVq));
    bw.newLine();
    bw.write(String.format("%-3s  %1.3f", "ts", ptlTs));
    bw.newLine();
    bw.write(String.format("%-3s  %1.3f", "hh", (ptlTs >= 0) ? ptlVq*ptlTs : -1));
    bw.newLine();
    bw.write(String.format("%-3s  %1.0f", "zgamu", ZGamU));
    bw.newLine();
    bw.write(String.format("%-3s  %1.5f", "gamu", GamU));
    bw.newLine();
    bw.write("--- final rise VDI 3782/3 (old)");
    bw.newLine();
    bw.write(String.format("%-5s  %1.1f", "xfvdi", vout.get(vout.size()-1).xvd)); //-2021-07-05
    bw.newLine();
    bw.write(String.format("%-5s  %1.1f", "hfvdi", vout.get(vout.size()-1).hvd)); //-2021-07-05
    bw.newLine();
    bw.write("--- Briggs lengths");
    bw.newLine();  
    bw.write(String.format("%-5s  %1.2f", "brglm", lenm));
    bw.newLine();
    bw.write(String.format("%-5s  %1.2f", "brglb", lenb));
    bw.newLine();
    //
    // input parameters
    for (String s : PrmSections) {
      bw.write("--- "+s);
      bw.newLine();
      if (s.equalsIgnoreCase("a")) {
        bw.write(String.format("%-2s  %1.2f", "uH", uH));
        bw.newLine();
      }
      for (String[] PrmParam : PrmParams) {
        if (PrmParam[1].equalsIgnoreCase(s)) {
          if (PrmParam[0].equalsIgnoreCase("tq"))
            bw.write(String.format("%2s  %s", PrmParam[0], String.format("%1.2f", TQ-TZERO)));
          else if (PrmParam[0].equalsIgnoreCase("ta"))
            bw.write(String.format("%2s  %s", PrmParam[0], String.format("%1.2f", TA-TZERO)));
          else if (PrmParam[0].equalsIgnoreCase("a1"))
            bw.write(String.format("%2s  %s", PrmParam[0], String.format("%1.2f", Math.toDegrees(A1))));
          else if (PrmParam[0].equalsIgnoreCase("a2"))
            bw.write(String.format("%2s  %s", PrmParam[0], String.format("%1.2f", Math.toDegrees(A2))));  //-2019-10-15
          else if (PrmParam[0].equalsIgnoreCase("da"))
            bw.write(String.format("%2s  %s", PrmParam[0], String.format("%1.2f", Math.toDegrees(DA))));
          else
            bw.write(String.format("%2s  %s", PrmParam[0], PrmGetValue(PrmParam[0]).trim()));
          bw.newLine();
          if (PrmParam[0].equalsIgnoreCase("ki")) {
            bw.write(String.format("%2s  %3.1f", "km", km));
            bw.newLine();
          }
        }
      }
    }  
    //
    // profiles
    bw.write("--- profiles");
    bw.newLine();
    Object[] vvv = {   zv,   uv,   dv,   tv,   rv,   pv,   qv, lv, cv,   iv,   jv, av };
    String[] svv = { "zv", "uv", "dv", "tv", "rv", "pv", "qv", "lv", "cv", "iv", "jv", "av" };
    int[]    evv = {    0,    0,    0,    0,    0,    1,    1,   1,   1,   0,  0, 0 };
    for (int i=0; i<vvv.length; i++) {
      sb.setLength(0);
      sb.append(String.format("%2s  ", svv[i]));
      for (int j=0; j<NZ; j++) {
        double d = ((double[])vvv[i])[j];
        if (svv[i].equalsIgnoreCase("tv")) {
          d -= TZERO; 
        }
        else if (svv[i].equalsIgnoreCase("dv"))
          d = Math.toDegrees(d);   
        sb.append(String.format((evv[i] == 1) ? "%12.4e" : "%12.6f", d));
      }
      bw.write(sb.toString());
      bw.newLine();
    }
    bw.write("-"); bw.newLine(); 
    bw.write("mode  text"); bw.newLine();
    bw.write("artp  P"); bw.newLine();
    bw.write("locl  \"C\""); bw.newLine();
    bw.write("sequ  i"); bw.newLine();
    bw.write("dims  1"); bw.newLine();
    bw.write("lowb  0"); bw.newLine();
    bw.write("hghb  "+ ((vout == null) ? -1 : vout.size()-1)); bw.newLine();
    sb.setLength(0);
    sb.append("form  ");
    sb.append(getOutputString(OO, null));    
    bw.write(sb.toString()); bw.newLine();
    bw.write("*"); bw.newLine();
    if (vout != null) {
      for (PlrOutput p : vout) {  
        bw.write(getOutputString(OO, p)); 
        bw.newLine();  
      }
    }
    bw.write("***"); 
    bw.newLine();
    bw.close();
    log(String.format("%s written.", fout.getName())); 
  }
  
  double[] getVDI37823D(double x, int km, double hq, double dq, double vq, 
    double uq, double qq) {
    double h = -999;
    double xmax = -999;
    double hmax = -999;
    double _x = x;
    if (qq > 1.4) {
      if (km == 5 || km == 6) {
        xmax = (qq > 6) ? 288 * Math.pow(qq, 0.4) : 194 * Math.pow(qq, 0.625);
        if (x > xmax) 
          x = xmax;
        h = 3.34 * Math.pow(qq, 0.333) * Math.pow(x, 0.667) / uq;
        if (h > 1100-hq) 
          h = 1100-hq;
      }
      else {
        //
        // neutral
        xmax = (qq > 6) ? 215 * Math.pow(qq, 0.4) : 145 * Math.pow(qq, 0.625);
        if (x > xmax) 
          x = xmax;
        hmax = 2.84 * Math.pow(qq, 0.333) * Math.pow(xmax, 0.667) / uq;
        h    = 2.84 * Math.pow(qq, 0.333) * Math.pow(x, 0.667) / uq;       
        if (h > 800-hq) 
          h = 800-hq;
        if (hmax > 800-hq)
          hmax = 800-hq;
        //
        // stable
        if (km == 1 || km == 2) {   
          double _xmax = (km == 1) ? 105 * uq : 129 * uq;
          double _xx = (_x > _xmax) ? _xmax : _x;
          double _hmax = 3.34 * Math.pow(qq, 0.333) * Math.pow(_xmax, 0.667) / uq;
          double _h   = 3.34 * Math.pow(qq, 0.333) * Math.pow(_xx, 0.667) / uq;
          if (_hmax < hmax) {
            xmax = _xmax;
            h = _h;
          }
        }
      }     
    }
    else {
      double h1 = (0.35*vq*dq + 84*Math.sqrt(qq)) / uq;
      double h2 = 3.0*vq*dq / uq;
      xmax = (qq > 0) ? 209.8 * Math.pow(qq, 0.522) : 0.0;  
      h = (h2 > h1) ? h2 : h1;
      if (xmax  > 0) {
        if (x > xmax) 
          x = xmax;
        h = h * Math.pow(x/xmax, 0.667);
      }
    }
    return new double[] { h, xmax };
  }

  
  private void exit(int n) {
    //Printer.println("&&& exit " + n);
    log(String.format("\r%s finished. %s", "IBJpluris",
      (n != 0) ? "exit code" + " " + n : ""));
    log("@");
    try { bwLog.close(); } catch (Exception ex) {}
    Printer.flush();                                              //-2015-11-23
    Printer.close();
    if (!redirected)                                              //-2015-11-23
    System.exit(n);
  }

  /**
   * @param args the command line arguments
   */
  public static void main(String[] args) {
    Locale.setDefault(Locale.ENGLISH);    
    String cmd;
    IBJpluris plr = null;
    try {
      if (args.length == 0 || args[0].startsWith("-h") || args[0].startsWith("-?")) {
        Printer.printf("IBJpluris %s\n", VERSION);
        Printer.printf("usage: IBJpluris <path> [options]\n");
        Printer.printf("options:\n");
        Printer.printf("  -e<extension>\n");
        Printer.printf("  --ignore-profiles\n");
        Printer.printf("  --stop-at-finalrise (false)\n");
        Printer.printf("  --paint-extended (false)\n");
        Printer.printf("  --sim-profiles (false)\n");
        Printer.printf("  --const-ra (false)\n");
        Printer.printf("  --const-ust (false)\n");
        Printer.printf("  --no-shear (false)\n");
        Printer.printf("  --skip-graph (false)\n");
        Printer.printf("  --skip-dmn (false)\n");
        Printer.printf("  --skip-log (false)\n");
        Printer.printf("  --skip-stacktip-downwash (false)\n");  
        Printer.printf("  --break-factor=<f>\n");
        Printer.printf("  --gamu=<zu (m)>;<gamu (K/m)>\n");
        Printer.printf("  --verbose=<i>\n");
        Printer.printf("  --<param>=<value>\n");   
        Printer.printf("\n");
        Printer.printf("tests: IBJpluris --test-density=<p>;<eta>;<t>;<phi>\n");
      }
      else if (args[0].startsWith("--")) {
        plr = new IBJpluris("", "");  
      }
      else {
        String sext = "";
        for (String s : args)
          if (s.startsWith("-e"))
            sext = s.substring(2);   
        plr = new IBJpluris(args[0], sext);
        plr.setStopAtFinalRise(false);
        plr.setStopAtGround(false);
        plr.setUseSimProfiles(false);
        plr.setPaintExtended(false);
        plr.setUseConstantRa(false);
        plr.setUseConstantUst(false);
        for (int ia=1; ia<args.length; ia++) {
          if (args[ia].equals("--ignore-profiles"))
            plr.setIgnoreProfiles(true);
          else if (args[ia].equals("--stop-at-finalrise"))
            plr.setStopAtFinalRise(true);
          else if (args[ia].equals("--stop-at-ground"))
            plr.setStopAtGround(true);
          else if (args[ia].equals("--sim-profiles"))
            plr.setUseSimProfiles(true);
          else if (args[ia].equals("--const-ust"))
            plr.setUseConstantUst(true);
          else if (args[ia].equals("--const-ra"))
            plr.setUseConstantRa(true);
          else if (args[ia].equals("--no-shear"))                 //-2024-03-19
            plr.setUseNoShear(true);
          else if (args[ia].equals("--skip-graph"))
            plr.setSkipGraph(true);
          else if (args[ia].equals("--skip-stacktip-downwash"))
            plr.setSkipStacktipDownwash(true);
          else if (args[ia].equals("--skip-dmn"))
            plr.setSkipDMN(true);
          else if (args[ia].equals("--skip-log"))
            plr.setSkipLog(true);
          else if (args[ia].equals("--paint-extended"))
            plr.setPaintExtended(true);
          else if (args[ia].startsWith("--verbose=")) {
            plr.setVerbose(Integer.parseInt(args[ia].substring(10)));
          }
          else if (args[ia].startsWith("--break-factor=")) {
            try {
              double f = Double.parseDouble(args[ia].substring(15).trim().replace(',', '.'));     
              plr.setBreakFactor(f);
            }
            catch (Exception e) {
              e.printStackTrace(Printer);
              plr.log("*** invalid value in option string --break-factor");
              plr.exit(1);
            }
          } 
          else if (args[ia].startsWith("--gamu=")) {
            try {
              String[] sa = args[ia].substring(7).split("[;]");
              double z = Double.parseDouble(sa[0].trim().replace(',', '.'));
              double g = Double.parseDouble(sa[1].trim().replace(',', '.'));  
              plr.setGamU(z,g);
            }
            catch (Exception e) {
              e.printStackTrace(Printer);
              plr.log("*** invalid value in option string --gamu");
              plr.exit(1);
            }
          }
          else if (args[ia].startsWith("-e"))
            ;
          else if (args[ia].startsWith("--") && args[ia].indexOf("=") > 0) {
            int ii =  args[ia].indexOf("=");
            String s = args[ia].substring(2,ii) + "=" + quote(args[ia].substring(ii+1));
            if (!plr.addParameterDefinition(s)) {
              plr.log("*** invalid value in option string "+args[ia]);
              plr.exit(1);
            }  
          }
          else
            throw new Exception("unknown option "+args[ia]);
          if (!args[ia].startsWith("--") || (args[ia].startsWith("--") && args[ia].indexOf("=") != 4)) 
            plr.addcmd(args[ia]);
        }        
        try {
          plr.run();
          plr.exit(0);
        }
        catch (Exception e) {      
          e.printStackTrace(System.out);
          Printer.println("*** " + e.toString());
          plr.log(Level.WARNING, e.getMessage());
          if (DEBUG) e.printStackTrace(Printer);
          plr.exit(1);
        }
      }   
    }
    catch (Exception e) {
      e.printStackTrace(Printer);     
    }
    finally {
      Printer.flush();
    }
    
  }
  
}
