////////////////////////////////////////////////////////////////////////////////
//
// history
//
// 2021-10-07 uj distribution
// 
////////////////////////////////////////////////////////////////////////////////

/**
 * Draw
 * 
 * Utility class used by Besmax
 *
 * Copyright (C) Umweltbundesamt Dessau-Roßlau, Germany, 2021
 * Copyright (C) Janicke Consulting, Überlingen, Germany, 2021
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 * 
 */

package de.janicke.SVG;

import de.janicke.SVG.SVG.Paintable;
import java.io.File;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.OpenOption;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.function.Function;
import javafx.geometry.HPos;
import javafx.geometry.Point2D;
import javafx.geometry.Point3D;
import javafx.geometry.Pos;
import javafx.geometry.Rectangle2D;
import javafx.geometry.VPos;
import javafx.scene.paint.Color;
import javafx.scene.text.Font;
import javafx.scene.text.FontPosture;
import javafx.scene.text.FontWeight;

public class Draw {
  
  private static Font current_font;
  
  private String title;
  private List<Object> objects;
  private View current_view;
  private Scale current_scale;
  private Paper paper;
  private Trans3D trans3d;
  //
  public static Symbol BOX = 
          (Stroke stroke, Fill fill, double u, double v, double size, 
          double... parms) -> {
    return (Function<Point2D, Point2D> g) -> {
      Point2D px = g.apply(new Point2D(u, v));
      return String.format(Locale.ENGLISH,
              "<rect %s %s x='%s' y='%s' width='%s' height='%s' />",
              stroke, fill, px.getX()-size/2, px.getY()-size/2, size, size);
    };
  };
  //
  public static Symbol CIRCLE = 
          (Stroke stroke, Fill fill, double u, double v, double size, 
          double... parms) -> {
    return (Function<Point2D, Point2D> g) -> {
      Point2D px = g.apply(new Point2D(u, v));
      return String.format(Locale.ENGLISH,
              "<circle %s %s cx='%s' cy='%s' r='%s' />",
              stroke, fill, px.getX(), px.getY(), size);
    };
  };
  //
  public static Symbol DIAMOND = 
          (Stroke stroke, Fill fill, double u, double v, double size, 
          double... parms) -> {
    return (Function<Point2D, Point2D> g) -> {
      Point2D px = g.apply(new Point2D(u, v));
      double x = px.getX();
      double y = px.getY();
      double sh = size/2;
      return String.format(Locale.ENGLISH,
              "<path %s %s d='M %s %s l %s %s l %s %s l %s %s z' />",
              stroke, fill, x, y-sh, -sh, sh, sh, sh, sh, -sh);
    };
  };
  //
  public static Symbol STAR = 
          (Stroke stroke, Fill fill, double u, double v, double size, 
          double... parms) -> {
    return (Function<Point2D, Point2D> g) -> {
      Point2D px = g.apply(new Point2D(u, v));
      double x = px.getX();
      double y = px.getY();
      double sh = size/2;
      double sf = size/8;
      return String.format(Locale.ENGLISH,
              "<path %s %s d='M %s %s l %s %s l %s %s l %s %s l %s %s l %s %s "
              + "l %s %s l %s %s z' />",
              stroke, fill, x, y-sh, -sf, sh-sf, -sh+sf, sf, sh-sf, sf,
              sf, sh-sf, sf, -sh+sf, sh-sf, -sf, -sh+sf, -sf);
    };
  };
  //
  public static Symbol PYRAMID = 
          (Stroke stroke, Fill fill, double u, double v, double size, 
          double... parms) -> {
    return (Function<Point2D, Point2D> g) -> {
      Point2D px = g.apply(new Point2D(u, v));
      double x = px.getX();
      double y = px.getY();
      double sh = size/2;
      double st = size/3;
      return String.format(Locale.ENGLISH,
              "<path %s %s d='M %s %s l %s %s l %s %s z' />",
              stroke, fill, x, y-2*st, -sh, size, size, 0);
    };
  };
  //
  public static Symbol TRIANGLE = 
          (Stroke stroke, Fill fill, double u, double v, double size, 
          double... parms) -> {
    return (Function<Point2D, Point2D> g) -> {
      Point2D px = g.apply(new Point2D(u, v));
      double x = px.getX();
      double y = px.getY();
      double sh = size/2;
      double st = size/3;
      return String.format(Locale.ENGLISH,
              "<path %s %s d='M %s %s l %s %s l %s %s z' />",
              stroke, fill, x, y+2*st, -sh, -size, size, 0);
    };
  };

  
  
  public Draw() {
    objects = new ArrayList<>();
    paper = new Paper(-1, -1, null, null);
    init_paper();
  }
  
  private void init_paper() {
    current_view = new View(0, 0, paper.width, paper.height, 
            paper.stroke, paper.fill);
    current_scale = new Scale(0, 0, paper.width, paper.height, 0, 0);
    Stroke.setDefault(paper.stroke);
    Fill.setDefault(paper.fill);
    if (current_font == null)
      current_font = Font.font("monospace", FontWeight.NORMAL, 
              FontPosture.REGULAR, 2.5);
    
    objects.add(paper);
    objects.add(current_view);
    objects.add(current_scale);
    objects.add(current_font);
  }
    
  public Draw paper(double width, double height, Stroke stroke, Fill fill) {
    objects.clear();
    paper = new Paper(width, height, stroke, fill);
    init_paper();
    return this;
  }
  
  public Draw trans3d(Trans3D t3d) {
    trans3d = t3d;
    return this;
  }
  
  public Draw view(double left, double bottom, double width, double height,
          Stroke stroke, Fill fill) {
    if (stroke == null)
      stroke = paper.stroke;
    if (fill == null)
      fill = paper.fill;
    current_view = new View(left, bottom, width, height, stroke, fill);
    Stroke.setDefault(stroke);
    Fill.setDefault(fill);
    objects.add(current_view);
    return this;
  }
  
  public Draw scale(double xmin, double ymin, double xmax, double ymax, 
          int logx, int logy) {
    current_scale = new Scale(xmin, ymin, xmax, ymax, logx, logy);
    objects.add(current_scale);
    return this;
  }
  
  public Draw scale(double xmin, double ymin, double xmax, double ymax) {
    return scale(xmin, ymin, xmax, ymax, 0, 0);
  }
  
  public Draw title(String title) {
    this.title = title;
    return this;
  }

  public Draw label(String label) {
    current_view.label = label;
    return this;
  }
  
  public static void setStroke(double width) {
    Stroke.setDefault(new Stroke(width));
  }
  
  public static Stroke stroke(double width) {
    return new Stroke(width);
  }
  
  public static void setStroke(String color, double width) {
    Stroke.setDefault(new Stroke(color, width));
  }
  
  public static Stroke stroke(String color, double width) {
    return new Stroke(color, width);
  }
  
  public static void setStroke(String color, double opacity, double width) {
    Stroke.setDefault(new Stroke(color, opacity, width, null));
  }
  
  public static Stroke stroke(String color, double opacity, double width) {
    return new Stroke(color, opacity, width, null);
  }
  
  public static void setStroke(String color) {
    Stroke.setDefault(new Stroke(color));
  }
  
  public static Stroke stroke(String color) {
    return new Stroke(color);
  }
  
  public static void setFill(String color) {
    Fill.setDefault(new Fill(color));
  }
  
  public static Fill fill(String color) {
    return new Fill(color);
  }
  
  public static void setFill(String color, double opacity) {
    Fill.setDefault(new Fill(color, opacity));
  }
  
  public static Fill fill(String color, double opacity) {
    return new Fill(color, opacity);
  }
  
  public static Stroke default_stroke(Stroke stroke) {
    Stroke old = Stroke.getDefault();
    Stroke.setDefault(stroke);
    return old;
  }
  
  public static Fill default_fill(Fill fill) {
    Fill old = Fill.getDefault();
    Fill.setDefault(fill);
    return old;
  }
  
  public static Font font(double size) {
    return font(null, size, null, null);
  }
  
  public static Font font(String family, double size) {
    return font(family, size, null, null);
  }
  
  public static Font font(String family, double size, String posture, 
          String weight) {
    if (size <= 0)
      size = 3.0;         // in mm
    if (family == null)
      family = "monospace";
    if (posture == null)
      posture = "REGULAR";
    if (weight == null)
      weight = "NORMAL";
    return Font.font(family, FontWeight.findByName(weight), 
            FontPosture.findByName(posture), size);
  }
  
  public static void setFont(String family, double size, String posture, 
          String weight) {
    current_font = font(family, size, posture, weight);
  }
  
  //--------------------------------------------------------------------------
  
  public Draw add(Paintable p) {
    objects.add(p);
    return this;
  }
  
  public Draw axis(String position, double step, int log, String format,
          String caption, boolean long_tick) {
    objects.add(new Axis(position, step, log, format, caption, long_tick, 
            Stroke.getDefault()));
    return this;
  }
  
  public Draw line(double[] xx, double[] yy, Stroke stroke, Fill fill) {
    objects.add(new Line(xx, yy, stroke, fill));
    return this;
  }

  public Draw line(double[] xx, double[] yy, double[] zz, Stroke stroke, Fill fill) {
    objects.add(new Line3D(xx, yy, zz, stroke, fill));
    return this;
  }

  public Draw line(double x0, double dx, double[] yy, Stroke stroke, Fill fill) {
    int n = yy.length;
    double[] xx = new double[n];
    for (int i=0; i<n; i++)
      xx[i] = x0 + i*dx;
    objects.add(new Line(xx, yy, stroke, fill));
    return this;
  }

  public Draw line(double[] xx, double y0, double dy, Stroke stroke, Fill fill) {
    int n = xx.length;
    double[] yy = new double[n];
    for (int i=0; i<n; i++)
      yy[i] = y0 + i*dy;
    objects.add(new Line(xx, yy, stroke, fill));
    return this;
  }
  
  public Draw rectangle(Rectangle2D rec, Stroke stroke, Fill fill) {
    objects.add(new Rectangle(rec, stroke, fill));
    return this;
  }
  
  public Draw text(Point2D pnt, String txt, Pos position, Fill fill, 
          Font font) {
    objects.add(new Text(pnt, txt, position, fill, font));
    return this;
  }
  
  public Draw mosaic(Mosaic m) {
    objects.add(m);
    return this;
  }
  
  public Draw symbol(Symbol smb, Stroke s, Fill f, double x, double y, 
          double sz, double... parms) {
    if (s == null)
      s = Stroke.getDefault();
    if (f == null)
      f = Fill.getDefault();
    Paintable p = smb.getPaintable(s, f, x, y, sz, parms);
    objects.add(p);
    return this;
  }
  
  public Draw symbol(Symbol smb, Stroke s, Fill f, double[] xx, double yy[], 
          double sz, double... parms) {
    if (s == null)
      s = Stroke.getDefault();
    if (f == null)
      f = Fill.getDefault();
    int n = (xx.length > yy.length) ? xx.length : yy.length;
    for (int i=0; i<n; i++) {
      double x = xx[i];
      double y = yy[i];
      Paintable p = smb.getPaintable(s, f, x, y, sz, parms);
      objects.add(p);
    }
    return this;
  }
  
  public String show() {
    return show(true, null);
  }
  
  public String show(boolean show, File file) {
    Paper ppr = null;
    View view = null;
    Scale s = null;
    Font f = null;
    boolean paper_set = false;
    List<View> views = new ArrayList<>();
    //
    double fsz = 2.5;
    double paper_width = 0;
    double paper_height = 0;
    //
    for (Object o: objects) {
      if (o instanceof Paper) {
        ppr = (Paper)o;
        paper_width = ppr.width;
        paper_height = ppr.height;
        paper_set = (ppr.width>0 && ppr.height>0);
      }
      else if (o instanceof View) {
        view = (View)o;
        views.add(view);
        if (s!=null && ppr!=null) {
          double ymin = ppr.height - view.bottom - view.height;
          view.port = new SVG.ViewPort(
              new Rectangle2D(view.left, ymin, view.width, view.height),
              new Rectangle2D(s.umin, s.vmin, s.umax-s.umin, s.vmax-s.vmin), 
              s.logu, s.logv,
              String.format("fill:%s; stroke:%s;", 
                      view.fill.color, view.stroke.color));
        }
      }
      else if (o instanceof Scale) {
        s = (Scale)o;
        if (view!=null && ppr!=null) {
          double ymin = ppr.height - view.bottom - view.height;
          view.port = new SVG.ViewPort(
              new Rectangle2D(view.left, ymin, view.width, view.height),
              new Rectangle2D(s.umin, s.vmin, s.umax-s.umin, s.vmax-s.vmin), 
              s.logu, s.logv,
              String.format("fill:%s; stroke:%s;", 
                      view.fill.color, view.stroke.color));
        }
      }
      else if (o instanceof Font) {
        f = (Font)o;
        fsz = f.getSize();
      }
      else if (o instanceof Axis) {
        if (view == null)
          continue;
        Axis a = (Axis)o;
        Map<String, Object> axis_opt = new LinkedHashMap<>();
        axis_opt.put("font-size", fsz);
        axis_opt.put("stroke", a.stroke.color);
        axis_opt.put("fill", a.stroke.color);
        axis_opt.put("stroke-width", a.stroke.width);
        axis_opt.put("position", a.position);
        axis_opt.put("axis-caption", a.caption);
        axis_opt.put("axis-format", a.format);
        axis_opt.put("tick-step", a.step);
        axis_opt.put("tick-long", a.tick_long);
        switch (a.position) {
          case "bottom":
            view.xa = a;
            view.xaxis = new SVG.Axis(view.port, a.position, axis_opt);
            break;
          case "left":
            view.ya = a;
            view.yaxis = new SVG.Axis(view.port, a.position, axis_opt);
            break;
        }
      }
      else if (o instanceof Paintable) {
        if (view == null)
          continue;
        view.content.add((Paintable)o);
      }
    }
    //
    if (!paper_set) {
      double y = 0;
      double max_indent = 0;
      double max_width = 0;
      double total_height = 0;
      if (title != null && title.length()>0)
        y += 1.5*fsz;
      for (View v: views) {
        v.btop = 0.5*fsz;
        v.bbtm = 0.5*fsz;
        v.brgt = fsz;
        v.blft = fsz;
        if (v.xaxis != null) {
          double xlen = v.xaxis.getTlen();
          v.brgt += 0.3*xlen*fsz;
          v.bbtm += 1.5*fsz;
          if (v.xa.caption != null)
            v.bbtm += 2*fsz;
        }
        if (v.yaxis != null) {
          v.btop += 0.5*fsz;
          double ylen = v.yaxis.getTlen();
          v.blft += 0.6*ylen*fsz;
          if (v.ya.caption != null)
            v.blft += 2*fsz;
          if (v.xaxis == null)
            v.bbtm += 0.5*fsz;
        }
        if (v.blft > max_indent)
          max_indent = v.blft;
        if (v.width + v.brgt > max_width)
          max_width = v.width + v.brgt;
        total_height += v.btop + v.height + v.bbtm;
      }
      paper_width = max_indent + max_width;
      paper_height = total_height + y;
      for (View v: views) {
        v.left = max_indent;
        y += v.btop + v.height;
        v.bottom = paper_height - y;
        v.port.setView(new Rectangle2D(v.left, y-v.height, v.width, v.height));
        y += v.bbtm;
      }
    }
    //
    Locale loc = Locale.ENGLISH;
    Map<String, String> parms = new LinkedHashMap<>();
    parms.put("width", String.format(loc, "%1.1fmm", paper_width));
    parms.put("height", String.format(loc, "%1.1fmm", paper_height));
    parms.put("viewBox", String.format(loc, "0 0 %1.1f %1.1f", 
            paper_width, paper_height));
    parms.put("fill", ppr.fill.color);
    parms.put("stroke", ppr.stroke.color);
    parms.put("stroke-width", String.format(loc, "%1.1f", ppr.stroke.width));
    parms.put("font-family", f.getFamily());
    if (title != null)
      parms.put("title", title);
    //
    SVG svg = new SVG(true, parms);
    svg.addChild(String.format(
            "<rect stroke='none' fill='%s'  width='%s'  height='%s'/>",
            ppr.fill.color, "100%", "100%"));
    if (title != null)
      svg.addChild(String.format(
              "<text x='%s' y='%s' text-anchor='%s' stroke='none' "
              + "font-family='%s' font-size='%s' fill='%s'>%s</text>\n",
              0.5*paper_width, 
              2*fsz, 
              "middle",
              "sans-serif", 
              1.2*fsz, 
              ppr.stroke.color,
              title));
    for (View v: views) {
      if (v.content.isEmpty())
        continue;
      if (v.label != null)
        svg.addChild(String.format(
                "<text x='%s' y='%s' text-anchor='%s' stroke='none' "
                + "font-family='%s' font-size='%s' fill='%s'>%s</text>\n",
                v.left, 
                paper_height - v.bottom - v.height - 0.2*fsz, 
                "start",
                f.getFamily(), 
                0.2*fsz, 
                v.stroke.color,
                v.label));
      for (Paintable p: v.content)
        v.port.addPaintable(p);
      svg.addChild(v.port.toString());
      if (v.xaxis != null)
        svg.addChild(v.xaxis.toString());
      if (v.yaxis != null)
        svg.addChild(v.yaxis.toString());
    }
    String content = null;
    if (show) {
      content = svg.display(file);
    }
    else {
      content = svg.toString();
      if (file != null) {
        try {
          Files.write(file.toPath(), content.getBytes(), (OpenOption)null);
        } 
        catch (IOException ex) {
          ex.printStackTrace(System.out);
        }
      }
    }
    return content;
  }
  
  //-----------------------------------------------------------------------
  
  public class Line3D implements SVG.Paintable {
    Line line;
    double[] uu, vv, ww;
    
    private Line3D(double[] xx, double[] yy, double[] zz, 
        Stroke stroke, Fill fill) {
      line = new Line(xx, yy, stroke, fill);
      uu = line.uu;
      vv = line.vv;
      ww = Arrays.copyOf(zz, zz.length);
    }


    @Override
    public String paint(Function<Point2D, Point2D> f) {
      if (trans3d != null) {
        double[][] uvw = new double[][] { uu, vv, ww };
        double[][] xyz = trans3d.transform(uvw);
        line.uu = xyz[0];
        line.vv = xyz[1];
      }
      return line.paint(f);
    }
    
  }
  
  //=======================================================================
  
  public static class Trans3D { // 3d-transform: rotation and translation
    
    double[][] mat = new double[][] { 
      {1, 0, 0, 0}, 
      {0, 1, 0, 0}, 
      {0, 0, 1, 0},
      {0, 0, 0, 1} };
    
    public Trans3D() {}
    
    public Trans3D(double[][] m) {
      mat = m;
    }
    
    public static Trans3D getRotX(double degree) {
      double rad = Math.toRadians(degree);
      double sin = Math.sin(rad);
      double cos = Math.cos(rad);
      double[][] m = new double[][] {
        { 1,    0,    0, 0 },
        { 0,  cos, -sin, 0 },
        { 0,  sin,  cos, 0 },
        { 0,    0,    0, 1 }
      };
      return new Trans3D(m);
    }
    
    public static Trans3D getRotY(double degree) {
      double rad = Math.toRadians(degree);
      double sin = Math.sin(rad);
      double cos = Math.cos(rad);
      double[][] m = new double[][] {
        {  cos,  0,  sin, 0 },
        {    0,  1,    0, 0 },
        { -sin,  0,  cos, 0 },
        {    0,  0,    0, 1 }
      };
      return new Trans3D(m);
    }
    
    public static Trans3D getRotZ(double degree) {
      double rad = Math.toRadians(degree);
      double sin = Math.sin(rad);
      double cos = Math.cos(rad);
      double[][] m = new double[][] {
        { cos, -sin, 0, 0 },
        { sin,  cos, 0, 0 },
        {   0,    0, 1, 0 },
        {   0,    0, 0, 1 }
      };
      return new Trans3D(m);
    }
    
    public static Trans3D getTranslation(Point3D p) {
      double[][] m = new double[][] {
        { 1, 0, 0, p.getX() },
        { 0, 1, 0, p.getY() },
        { 0, 0, 1, p.getZ() },
        { 0, 0, 0,        1 }
      };
      return new Trans3D(m);
    }
    
    public static Trans3D getScale(Point3D p) {
      double[][] m = new double[][] {
        { p.getX(), 0, 0, 0 },
        { 0, p.getY(), 0, 0 },
        { 0, 0, p.getZ(), 0 },
        { 0, 0, 0,        1 }
      };
      return new Trans3D(m);
    }
    
    public Trans3D concat(Trans3D t) {
      double[][] m = new double[4][4];
      for (int i=0; i<4; i++) {
        for (int j=0; j<4; j++) {
          double s = 0;
          for (int k=0; k<4; k++)
            s += t.mat[i][k]*mat[k][j];
          m[i][j] = s;
        }
      }
      return new Trans3D(m);
    }
    
    public Point3D transform(Point3D p) {
      double[] v = new double[] { p.getX(), p.getY(), p.getZ(), 1 };
      double x, y, z;
      int i;
      for (i=0, x=0; i<4; i++)
        x += mat[0][i]*v[i];
      for (i=0, y=0; i<4; i++)
        y += mat[1][i]*v[i];
      for (i=0, z=0; i<4; i++)
        z += mat[2][i]*v[i];
      return new Point3D(x, y, z);
    }
    
    public double[][] transform(double[][] xyz) {
      int n = xyz[0].length;
      double[][] r = new double[3][n];
      double x, y, z;
      int i;
      for (int k=0; k<n; k++) {
        for (i=0, x=0; i<3; i++)
          x += mat[0][i]*xyz[i][k];
        for (i=0, y=0; i<3; i++)
          y += mat[1][i]*xyz[i][k];
        for (i=0, z=0; i<3; i++)
          z += mat[2][i]*xyz[i][k];
        r[0][k] = x + mat[0][3];
        r[1][k] = y + mat[1][3];
        r[2][k] = z + mat[2][3];
      }
      return r;
    }
    
    
  }

  
  //========================================================================
  
  private static class Paper {
    final double width, height;
    final Stroke stroke;
    final Fill fill;
    
    private Paper(double width, double height, Stroke stroke, Fill fill) {
      if (width < 0)
        width = 150;
      if (height < 0)
        height = 150;
      if (stroke == null)
        stroke = new Stroke("black", 0.2, "");
      if (fill == null)
        fill = new Fill("white");
      this.width = width;
      this.height = height;
      this.stroke = stroke;
      this.fill = fill;
    }
    
  }
  
  //==========================================================================
  
  private static class View {
    double left, bottom, width, height;
    Stroke stroke;
    Fill fill;
    String label;
    //
    SVG.ViewPort port;
    List<Paintable> content;
    Axis xa;
    Axis ya;
    SVG.Axis xaxis;
    SVG.Axis yaxis;
    double blft, brgt, btop, bbtm;
  
    private View(double left, double bottom, double width, double height,
            Stroke stroke, Fill fill) {
      this.left = left;
      this.bottom = bottom;
      this.width = width;
      this.height = height;
      this.stroke = stroke;
      this.fill = fill;
      content = new ArrayList<>();
    }
    
  }
  
  //=========================================================================
  
  private static class Scale {
    int logu, logv;
    double umin, umax, vmin, vmax;
    
    private Scale(double umin, double vmin, double umax, double vmax, 
            int logu, int logv) {
      this.umin = umin;
      this.vmin = vmin;
      this.umax = umax;
      this.vmax = vmax;
      this.logu = logu;
      this.logv = logv;
    }
    
  }
  
  //==========================================================================
  
  private static class Axis {
    String position;
    double step;
    int log;
    String format;
    String caption;
    boolean tick_long;
    Stroke stroke;
  
    public Axis(String position, double step, int log, String format, 
            String caption, boolean tick_long, Stroke stroke) {
      this.position = position;
      this.step = step;
      this.log = log;
      this.format = format;
      this.caption = caption;
      this.tick_long = tick_long;
      this.stroke = stroke;
    }
  }
  
  //==========================================================================
  
  public static class Line implements Paintable {
    double[] uu;
    double[] vv;
    Stroke stroke;
    Fill fill;
    
    public Line(double[] xx, double[] yy, Stroke stroke, Fill fill) {
      uu = Arrays.copyOf(xx, xx.length);
      vv = Arrays.copyOf(yy, yy.length);
      if (stroke == null)
        stroke = Stroke.getDefault();
      this.stroke = stroke;
      if (fill == null)
        fill = Fill.getDefault();
      this.fill = fill;
    }   

    @Override
    public String paint(Function<Point2D, Point2D> f) {
      boolean drawing = false;
      StringBuilder sb = new StringBuilder();
      sb.append("<path");
      if (stroke != null)
        sb.append(stroke.toString());
      if (fill != null)
        sb.append(fill.toString());
      int n = uu.length;
      if (vv.length < n)
        n = vv.length;
      sb.append(" d='");
      for (int i=0; i<n; i++) {
        Point2D pu = new Point2D(uu[i], vv[i]);
        Point2D pp = f.apply(pu);
        if (pp == null) {
          drawing = false;
          continue;
        }
        double x = pp.getX();
        double y = pp.getY();
        if (drawing) {
          sb.append(" L ");
        }
        else {
          sb.append(" M ");
          drawing = true;
        }
        sb.append(x).append(" ").append(y).append("\n");
      }
      sb.append("'/>");
      String s = sb.toString();
      return s;
    }
  }
  
  //==========================================================================
  
  public static class Rectangle implements Paintable {
    Rectangle2D rec;
    Stroke stroke;
    Fill fill;
    
    public Rectangle(Rectangle2D rec, Stroke stroke, Fill fill) {
      this.rec = rec;
      this.stroke = stroke;
      this.fill = fill;
    }

    @Override
    public String paint(Function<Point2D, Point2D> f) {
      Point2D p00 = new Point2D(rec.getMinX(), rec.getMinY());
      Point2D p10 = new Point2D(rec.getMaxX(), rec.getMinY());
      Point2D p11 = new Point2D(rec.getMaxX(), rec.getMaxY());
      Point2D p01 = new Point2D(rec.getMinX(), rec.getMaxY());
      StringBuilder sb = new StringBuilder();
      sb.append("<path d='");
      Point2D q = f.apply(p00);
      sb.append(" M ").append(q.getX()).append(" ").append(q.getY());
      q = f.apply(p10);
      sb.append(" L ").append(q.getX()).append(" ").append(q.getY());
      q = f.apply(p11);
      sb.append(" L ").append(q.getX()).append(" ").append(q.getY());
      q = f.apply(p01);
      sb.append(" L ").append(q.getX()).append(" ").append(q.getY());
      sb.append(" z'\n");
      if (stroke != null)
        sb.append(stroke.toString());
      if (fill != null)
        sb.append(fill.toString());
      sb.append("/>\n");
      String s = sb.toString();
      return s;
    }
    
  }
  
  //==========================================================================
  
  public static class Text implements Paintable {
    
    private final Point2D point;
    private final String text;
    private final Pos position;
    private Fill fill;
    private Font font;
    
    public Text(Point2D point, String text, Pos position, Fill fill, Font font) {
      this.point = point;
      this.text = text;
      this.position = (position != null) ? position : Pos.BASELINE_LEFT;
      this.fill = fill;
      this.font = font;
    }

    @Override
    public String paint(Function<Point2D, Point2D> f) {
      if (font == null)
        font = Font.font("monospaced", 3.0);
      double fsz = font.getSize();
      StringBuilder sb = new StringBuilder();
      Point2D q = f.apply(point);
      double x = q.getX();
      double y = q.getY();
      VPos vp = position.getVpos();
      if (vp == VPos.BOTTOM)
        y -= 0.24*fsz;
      else if (vp == VPos.CENTER)
        y += 0.38*fsz;
      else if (vp == VPos.TOP)
        y += 0.76*fsz;
      HPos hp = position.getHpos();
      String anchor = "start";
      if (hp == HPos.CENTER)
        anchor = "middle";
      else if (hp == HPos.RIGHT)
        anchor = "end";
      //
      sb.append("<text stroke='none' ");      
      sb.append(" x='").append(x).append("' y='").append(y).append("' ");
      sb.append(" text-anchor='").append(anchor).append("' ");
      sb.append(SVG.font(font));
      if (fill != null)
        sb.append(fill.toString());
      sb.append(">").append(text).append("</text>\n");
      return sb.toString();
    }
    
  }
  
  //==========================================================================
  
  public static class Mosaic implements Paintable {
    double[] uu;
    double[] vv;
    double[][] cc;
    double cmin;
    double cmax;
    double fsz = 2.0;
    Function<Double, Color> g;
    Function<Double, String> t;
    
    private Mosaic() {}
    
    public static Mosaic getInstance(double[] uu, double[] vv, double[][] cc) {
      Mosaic m = null;
      try {
        m = new Mosaic();
        m.uu = Arrays.copyOf(uu, uu.length);
        m.vv = Arrays.copyOf(vv, vv.length);
        int nu = uu.length;
        int nv = vv.length;
        m.cc = new double[nu][nv];
        m.cmin = cc[0][0];
        m.cmax = m.cmin;
        for (int iu=0; iu<nu-1; iu++) {
          for (int iv=0; iv<nv-1; iv++) {
            double c = cc[iu][iv];
            if (c < m.cmin)
              m.cmin = c;
            else if (c > m.cmax)
              m.cmax = c;
            m.cc[iu][iv] = c;
          }
        }
      } 
      catch (Exception e) {
        e.printStackTrace(System.out);
        m = null;
      }
      return m;
    }
    
    public double getCmin() {
      return cmin;
    }
    
    public double getCmax() {
      return cmax;
    }
    
    public void setGauge(Function<Double, Color> g, Function<Double, String> t,
            double fsz) {
      this.g = g;
      this.t = t;
      this.fsz = fsz;
    }

    @Override
    public String paint(Function<Point2D, Point2D> f) {
      if (g == null)
        g = Gauge.getVioletToLightBlue(cmax);
      StringBuilder sb = new StringBuilder();
      sb.append("<g stroke='none' fill='black'");
      if (t != null)
        sb.append(String.format(
                " font-family='%s' font-size='%s' text-anchor='%s'", 
                "monospace", fsz, "middle"));
      sb.append(">\n");
      for (int iu=0; iu<uu.length-1; iu++) {
        for (int iv=0; iv<vv.length-1; iv++) {
          double c = cc[iu][iv];
          Color color = g.apply(c);
          String cs = null;
          if (color != null) {
            int red = (int)(255*color.getRed());
            int green = (int)(255*color.getGreen());
            int blue = (int)(255*color.getBlue());
            cs = String.format("#%02x%02x%02x", red, green, blue);
          }
          Point2D pu = new Point2D(uu[iu], vv[iv]);
          Point2D p0 = f.apply(pu);
          pu = new Point2D(uu[iu+1], vv[iv+1]);
          Point2D p1 = f.apply(pu);
          double x = Math.min(p0.getX(), p1.getX());
          double w = Math.abs(p0.getX() - p1.getX());
          double y = Math.min(p0.getY(), p1.getY());
          double h = Math.abs(p0.getY() - p1.getY());
          if (cs != null) {
            sb.append(String.format(Locale.ENGLISH, 
            "<rect x='%1.2f' y='%1.2f' width='%1.2f' height='%1.2f' fill='%s' />\n",
            x, y, w, h, cs));
          }
          if (t != null) {
            String s = t.apply(c);
            if (s != null) {
              x += w/2;
              y += h/2 + 0.35*fsz;
              sb.append(String.format(Locale.ENGLISH, 
                      "<text x='%1.2f' y='%1.2f'>%s</text>\n", x, y, s));
            }
          }
        }
      }
      sb.append("</g>\n");
      return sb.toString();
    }
    
    
  }
  
  //==========================================================================
  
  public static class Gauge {
    
    public static Function<Double, Color> getVioletToLightBlue(double ref) {
      double hue_from  = 243;
      double hue_to = -77;
      double sat_from = 0.0;
      double sat_to = 1.0;
      return (Double c) -> {
        double q = c/ref;
        if (q < 0)
          q = 0;
        else if (q > 1)
          q = 1;
        double hue =  hue_from + q*(hue_to - hue_from);
        double sat = sat_from + q*(sat_to - sat_from);
        return Color.hsb(hue, sat, 1.0); 
      };
    }
    
    public static Function<Double, Color> getPlusMinus(double cmin, double cmax,
            double opacity) {
      double hue_from  = -120;
      double hue_to = 240;
      double sat_from = 1.0;
      double sat_to = 1.0;
      return (Double c) -> {
        double q = (c-cmin)/(cmax-cmin);
        if (q < 0)
          q = 0;
        else if (q > 1)
          q = 1;
        double hue =  hue_from + q*(hue_to - hue_from);
        double sat = sat_from + q*(sat_to - sat_from);
        return Color.hsb(hue, sat, opacity); 
      };
    }
    
  }
  
  //==========================================================================
  
  public static class Stroke {
    
    public static final Stroke NONE = new Stroke("none", 0, 0, "");
    private static Stroke def = setDefault(null);
    
    public final String color;
    public final double opacity;
    public final double width;
    public final String dash;
    
    public static Stroke setDefault(Stroke s) {
      if (s == null)
        def = new Stroke("black", 1.0, 0.2, "");
      else {
        String clr = (s.color == null) ? "black" : s.color;
        double opc = (s.opacity<0 || s.opacity>1) ? 1.0 : s.opacity;
        double wdt = (s.width < 0) ? 0.2 : s.width;
        String dsh = (s.dash == null) ? "" : s.dash;
        def = new Stroke(clr, opc, wdt, dsh);
      }
      return def;
    }
    
    public static Stroke getDefault() {
      return def;
    }
    
    public Stroke(String color, double opacity, double width, String dash) {
      if (color == null)
        color = def.color;
      if (opacity<0 || opacity>1)
        opacity = def.opacity;
      if (width < 0)
        width = def.width;
      if (dash == null)
        dash = def.dash;
      this.color = color;
      this.opacity = opacity;
      this.width = width;
      this.dash = dash;
    }
    
    public Stroke(String color, double width, String dash) {
      this(color, 1.0, width, dash);
    }
    
    public Stroke(String color, double width) {
      this(color, -1, width, null);
    }
    
    public Stroke(String color) {
      this(color, -1, -1, null);
    }
    
    public Stroke(double width) {
      this(null, -1, width, null);
    }
        
    @Override
    public String toString() {
      StringBuilder sb = new StringBuilder();
      sb.append(" stroke='").append(color).append("' ")
              .append("stroke-opacity='").append(opacity).append("' ")
              .append("stroke-width='").append(width).append("' ")
              .append("stroke-dasharray='").append(dash).append("' ");
      return sb.toString();
    }
    
  }
  
  //==========================================================================
  
  public static class Fill {
    
    public static final Fill NONE = new Fill("none", 0);
    private static Fill def = setDefault(null);
    
    public static Fill setDefault(Fill f) {
      if (f == null)
        def = new Fill("gray", 1.0);
      else {
        String clr = (f.color == null) ? "gray" : f.color;
        double opc = (f.opacity<0 || f.opacity>1) ? 1.0 : f.opacity;
        def = new Fill(clr, opc);
      }
      return def;
    }
    
    public static Fill getDefault() {
      return def;
    }
    
    public final String color;
    public final double opacity;

    public Fill(String color, double opacity) {
      if (color == null) {
        color = def.color;
      }
      if (opacity<0 || opacity>1) 
        opacity = def.opacity;
      this.color = color;
      this.opacity = opacity;
    }
    
    public Fill(String color) {
      this(color, -1);
    }
        
    @Override
    public String toString() {
      StringBuilder sb = new StringBuilder();
      sb.append(" fill='").append(color).append("' ")
              .append("fill-opacity='").append(opacity).append("' ");
      return sb.toString();
    }
  }
  
  //==========================================================================
  
  public interface Symbol {
    Paintable getPaintable(Stroke stroke, Fill fill, double x, double y, 
            double size, double... parms);
  }


}
